/**********************************************************\
  HLP_CTRL.C    Source for processing multiple embedded
                windows each containing a WinHelp macro
                callback button.
                        
                Copyright (c) Andy Brundell 1994
\**********************************************************/
#define OEMRESOURCE    	_OEMRESOURCE

#include <windows.h>    /* Windows stuff    */
#include <stdlib.h>    	/* Standard library */
#include <string.h>    	/* String handling	*/
#include "hlp_ctrl.h"	/* Our stuff        */

char         szWork[300];
char         szAppName[] = "Help Controls";
HANDLE       hinst;
HBITMAP      hBM;

LPHELPTASK   TaskList   = NULL;
LPHELPBUTTON ButtonList = NULL;

/**********************************************************\
  LibMain() Standard DLL load function. This one differs
            from most in that it registers an AppGlobal
            class for the help buttons. It must be AppGlobal
            to allow WinHelp access to it.
\**********************************************************/
int CALLBACK LibMain(HANDLE hInst,WORD wDataSeg,
                        WORD cbHeapSize, LPSTR lpszCmdLine)
{
    WNDCLASS    wndclass;

    hinst = hInst;

    wndclass.style          = CS_GLOBALCLASS;
    wndclass.lpfnWndProc    = EWndProc;
    wndclass.cbClsExtra     = 0;
    wndclass.cbWndExtra     = 0;
    wndclass.hInstance      = hInst;
    wndclass.hIcon          = NULL;
    wndclass.hCursor        = LoadCursor(NULL, IDC_ARROW);
    wndclass.hbrBackground  = GetStockObject(WHITE_BRUSH);
    wndclass.lpszMenuName   = NULL;
    wndclass.lpszClassName  = (LPSTR) "EWnd";

    RegisterClass(&wndclass);

    return(1);
}

/**********************************************************\
  EwndProc()    Receives WinHelp callbacks to create and
                maintain the embedded windows.
\**********************************************************/
LONG CALLBACK __export EWndProc (HWND hwnd, WORD message,
                                    WPARAM wParam, LPARAM lParam)
{

    switch ( message ) {
        case WM_CREATE : {
            QCI qci = (QCI) ((CREATESTRUCT FAR *)lParam)
                                ->lpCreateParams;
            GetButtonInfo(hwnd, qci);
            return 0;
        }
    
        case EWM_QUERYSIZE: {
            LPHELPBUTTON Button = GetThisButton(hwnd);
            if (Button) {
                /* Set the lParam point structure right size */
                ((LPPOINT)lParam)->x = Button->ThisSize.x;
                ((LPPOINT)lParam)->y = Button->ThisSize.y;
                return TRUE;
            }
            else
                return FALSE;
        }
    
        case EWM_RENDER: {
            LPHELPBUTTON Button = GetThisButton(hwnd);
            switch( wParam ) {
                case CF_BITMAP: {
                   QRI qri = (QRI)lParam;
                   return RenderBitmapButton( Button, qri );
                }

                case CF_TEXT: {
                    HGLOBAL hGlobText = GlobalAlloc(
                            GMEM_MOVEABLE | GMEM_NOT_BANKED,
                            sizeof(Button->ThisText) + 2 );
                    if( hGlobText ) {
                        LPSTR sz = GlobalLock( hGlobText );
                        wsprintf( (LPSTR)sz, "(%s)",
                                 (LPSTR)Button->ThisText );
                        GlobalUnlock( hGlobText );
                    }
                    return (long)hGlobText;
                }
            }
            return (long) NULL;
        }
    
    /*	In this example, no painting is needed 
        case WM_PAINT : {
            LPHELPBUTTON Button = GetThisButton(hwnd);
            if (Button) {
                PAINTSTRUCT ps ;
                HDC hDC = BeginPaint(hwnd, &ps);
                EndPaint(hwnd, &ps);
            }
            return 0 ;
        } */
    
        case WM_COMMAND:
            if (wParam == IDB_BUTTON) {
                ExecuteButtonCommand(hwnd);
                return FALSE;
            }
            break;
    }
    
    return DefWindowProc(hwnd, message, wParam, lParam) ;
}
               
/**********************************************************\
  RenderBitmapButton()  Draws the button as a bitmap so
                        that WinHelp can print it nicely.
\**********************************************************/
LONG PASCAL RenderBitmapButton(LPHELPBUTTON Button, QRI qri)
{
    int     i;
    HDC     hDC;
    RECT    r;
    HPEN    hOldPen;
    HFONT   hOldFont;
    HBRUSH  hOldBrush;
    HBITMAP hOldBM;
    
    /* Setup a display context */
    hDC = CreateCompatibleDC( qri->hDC );
    if (!hDC)
        return NULL;

    r.top    = r.left = 0;
    r.right  = Button->ThisSize.x;
    r.bottom = Button->ThisSize.y;
    
    /* Setup a suitable bitmap */
    hBM = CreateCompatibleBitmap( qri->hDC,
                       r.right, r.bottom );
    if (!hBM) {
        DeleteDC( hDC );
        return (NULL);
    }

    hOldBM = SelectObject( hDC, hBM );
    PatBlt(hDC, r.left, r.top, r.right, r.bottom, WHITENESS); 
    
    /* Setup and draw the basic button shape */
    hOldPen   = SelectObject( hDC,
                    GetStockObject( BLACK_PEN ) );
    hOldBrush = SelectObject( hDC,
                    GetStockObject( LTGRAY_BRUSH ) );
    RoundRect( hDC, r.left, r.top, r.right, r.bottom, 2, 2 );

    /* Paint the text */
    hOldFont = SelectObject( hDC,
                    GetStockObject(SYSTEM_FONT) );
    SetBkMode( hDC,TRANSPARENT );
    TextOut( hDC, 5, 3, Button->ThisText,
        lstrlen(Button->ThisText) );
    
    /* Now draw the remaining lines */
    SelectObject( hDC, GetStockObject(WHITE_PEN) );
    for( i=1; i<3; i++ ) {
        MoveTo( hDC, r.left+i,   r.bottom-i-2);
        LineTo( hDC, r.left+i,   r.top+i);
        LineTo( hDC, r.right-i-1,r.top+i);
    }
                                                
    /* Now clean up the display context */
    SelectObject( hDC, hOldBrush );
    SelectObject( hDC, hOldFont );
    SelectObject( hDC, hOldPen );
    SelectObject( hDC, hOldBM );
    DeleteDC( hDC );
    
    return (long)hBM;
}

/**********************************************************\
  ExecuteButtonCommand() Locates the current task in the
                         TaskList and uses the internal
                         API entry point to send the
                         button command to the requesting
                         help file.
\**********************************************************/
BOOL PASCAL ExecuteButtonCommand(HWND hwnd)
{
    LPHELPTASK   Task;
    LPHELPBUTTON Button;
        
    /* Locate the current button in the button list */
    if ( Button = GetThisButton( hwnd ) ) {
        /* Locate the current task in the task list */
        if ( Task = GetThisTask() )
            Task->lpfnWinHelp( Button->ThisFile,
                        HELP_COMMAND,
                       (DWORD)(void FAR *)Button->ThisCmd );
        else
            MessageBox( hwnd, "Button Didn't Register",
                   szAppName, MB_OK );
    }
    else
        MessageBox( hwnd, "Button Setup Failed",
                szAppName, MB_OK );
    
    return TRUE;
}

/**********************************************************\
  GetButtonInfo()	Creates a new button info node in a SLL
                    so that we can provide services for
                    multiple buttons at the same time.
\**********************************************************/
BOOL PASCAL GetButtonInfo(HWND hwnd, QCI qci)
{
    char        *pStop;
    int        	 i;
    LPHELPBUTTON Button;
    DWORD        dwH;
    WORD         wH, wW;
    HDC        	 hDC;

    /* Allocate new BUTTONLIST node and load is */
    Button = (LPHELPBUTTON) malloc( sizeof(HELPBUTTON) );
    if ( Button ) {
        Button->ThisButton = hwnd;
        /* Copy the user control string */
        lstrcpy( szWork, qci->lpstrAuthorData );
        /* Replace all semi-colons with commas */
        for( i=0;szWork[i];i++ )
            if ( szWork[i] == ';' )
                szWork[i] = ',';
        /* Extract the button text */
        pStop = strtok( szWork,"!" );
        lstrcpy( Button->ThisText,pStop );

        /* Extract the macro command string */
        pStop = strtok( NULL,"!" );
        lstrcpy( Button->ThisCmd,pStop );

        /* Copy the help file name */
        lstrcpy( Button->ThisFile,qci->lpstrFileName );
        
        /* Copy the background color */
        Button->ThisColor = qci->coBackground;

        /* Determine and save the button size */
        hDC = CreateIC("DISPLAY",NULL,NULL,NULL);
        dwH = GetTextExtent( hDC, Button->ThisText,
                       lstrlen( Button->ThisText ) );
        wH  = HIWORD( dwH ); /* Get Text Height */
        wW  = LOWORD( dwH ); /* Get Text Width  */

        DeleteDC(hDC);
        Button->ThisSize.x = wW + 12;
        Button->ThisSize.y = wH + 6;
        
        /* If ButtonList started, we insert ourselves into
           position one, otherwise we start the button list */	
        Button->NextButton	= ButtonList;
        ButtonList        	= Button;

        /* Now we create the button */
        CreateWindow("Button", Button->ThisText,
                 WS_CHILD | WS_VISIBLE | BS_PUSHBUTTON,
                 0,0,
                 Button->ThisSize.x,
                 Button->ThisSize.y,
                 hwnd,
                 IDB_BUTTON,
                 hinst,
                 NULL);
    }
    return TRUE;
}

/**********************************************************\
  GetThisButton()	Gets pointer to current button based on
                    HWND to ensure that the message action
                    applies to the correct button.
\**********************************************************/
LPHELPBUTTON PASCAL GetThisButton(HWND hwnd)
{
    LPHELPBUTTON Button  = ButtonList;

    while( Button ) /* While unsearched nodes remain */
    if( Button->ThisButton == hwnd )
        return Button;
    else
        Button = Button->NextButton;
    
    return NULL;  /* Button failed setup */
}

/**********************************************************\
  LDLLHandler() Receives WinHelp notifications from each
                new task.
\**********************************************************/
LONG CALLBACK __export LDLLHandler( WORD wMsg,
                                    LONG lParam1, LONG lParam2 )
{
    switch( wMsg ) {
        case DW_WHATMSGS:
            return DC_CALLBACKS;
        
        case DW_CALLBACKS:
            GetTaskInfo( (FARPROC FAR *)lParam1 );
            return TRUE;
    }
    return FALSE;
}

/**********************************************************\
  GetTaskInfo() Sets pointer to internal WinHelp function
                for the help file that registered the DLL.
                This occurs every time a new help file is
                opened that needs button services. We keep
                each task in a SLL that holds the pointer
                to the helpfile internal API.
\**********************************************************/
BOOL PASCAL GetTaskInfo( FARPROC FAR *VPtr )
{
    LPHELPTASK Task;
    
    /* Allocate a new TASKLIST node and load it */
    Task = (LPHELPTASK) malloc( sizeof(HELPTASK) );
    if ( Task ) {
        Task->ThisTask	= GetCurrentTask();
        Task->lpfnWinHelp = (LPFN_WINHELP)VPtr[HE_API];
        /* If TaskList started, we insert ourselves into
           position one, otherwise we start the task list */
        Task->NextTask	= TaskList;
        TaskList        = Task;
    }
    return TRUE;
}
/**********************************************************\
  GetThisTask() Gets pointer to current tasks internal API
                function to allow a macro command to be
                passed directly to the button help file.
\**********************************************************/
LPHELPTASK PASCAL GetThisTask( void )
{
    HTASK      hCurrentTask = GetCurrentTask();
    LPHELPTASK Task         = TaskList;
        
    while( Task ) /* While unsearched nodes remain */
    if( Task->ThisTask == hCurrentTask )
        return Task;
    else
        Task = Task->NextTask;
    
    return NULL;  /* Button didn't register */
}

/**********************************************************\
  InitiateControls() Dummy entry point to cause WinHelp to
                     start passing us notifications. This
                     function never gets called.
\**********************************************************/
BOOL CALLBACK __export InitiateControls(LPSTR lpszHelpFile)
{
    return TRUE;
}

/**********************************************************\
  WEP()    	Standard DLL unload function
\**********************************************************/
int CALLBACK __export WEP (int bSystemExit)
{
    return TRUE;
}
