
//  ------------------------------------------------------------------
//  Test program for the Goldware Sound API.
//  Released to the Public Domain by Odinn Sorensen.
//  ------------------------------------------------------------------
//  This source has been tested and compiles cleanly with Borland C++
//  3.1 and Watcom C++ 10.0a.
//  ------------------------------------------------------------------
//  For 16-bit compilers, compile in a large memory model.
//  Compile with -DDEBUG for diagnostics.
//  ------------------------------------------------------------------
//  See GOLDSAPI.DOC for a specification of the Goldware Sound API.
//  ------------------------------------------------------------------


//  ------------------------------------------------------------------

#include <dos.h>
#include <stdio.h>
#include <malloc.h>
#include <string.h>


//  ------------------------------------------------------------------
//  Some typedefs

typedef unsigned char  byte;
typedef unsigned short word;
typedef unsigned int   uint;


//  ------------------------------------------------------------------
//  Goldware Sound API function numbers

#define GSAPI_INSTALL_CHECK       0x00
#define GSAPI_OPEN_API            0x10
#define GSAPI_CLOSE_API           0x11
#define GSAPI_OPEN_AND_LOAD_FILE  0x12
#define GSAPI_CLOSE_FILE          0x13
#define GSAPI_PLAY                0x14
#define GSAPI_STOP                0x15
#define GSAPI_PAUSE               0x16
#define GSAPI_RESUME              0x17
#define GSAPI_BREAK_LOOP          0x18
#define GSAPI_SPEAKER_ON_OFF      0x19


//  ------------------------------------------------------------------
//  Goldware Sound API data structure

struct gsapidata {
  word driver_version;
  word dsp_version;
  word io_port;
  byte irq_number;
  byte dma_channel;
  word sample_rate;
  volatile word status;
  word buffer_segment;
  word buffer_offset;
  long buffer_length;
  char parameters[80];
};


//  ------------------------------------------------------------------
//  The AMIS signature string

struct amis_signature {
  char manufacturer[8];
  char product_name[8];
  char product_description[64];
};


//  ------------------------------------------------------------------
//  Compiler portability defines

#if defined(__WATCOMC__)
  #define x_ax w.ax
  #define x_bx w.bx
  #define x_cx w.cx
  #define x_dx w.dx
  #define x_di w.di
  #if defined(__386__)
    /* NOTE: This is for the DOS/4GW DOS extender! */
    #define MK_FP2(s,o) ((s << 4) + o)
    #define INT86 int386
  #else
    #define MK_FP2(s,o) MK_FP(s,o)
    #define INT86 int86
    #define HUGEMALLOC(n) halloc(((long)n/1024L)+1,1024)
  #endif
#else
  #define x_ax x.ax
  #define x_bx x.bx
  #define x_cx x.cx
  #define x_dx x.dx
  #define x_di x.di
  #define MK_FP2(s,o) MK_FP(s,o)
  #define INT86 int86
  #define HUGEMALLOC(n) farmalloc(n)
#endif


//  ------------------------------------------------------------------
//  The AMIS multiplex number found for the Goldware Sound API

int mpx;


//  ------------------------------------------------------------------
//  Call a function in the API

int call_api(uint al, uint bx) {

  union REGS cpu;
  cpu.h.al = al;
  cpu.h.ah = mpx;
  cpu.x_bx = bx;
  INT86(0x2D, &cpu, &cpu);
  #ifdef DEBUG
  if(al == GSAPI_OPEN_AND_LOAD_FILE) {
    if(cpu.x_ax == 0xFFFF)
      printf("DOS error code %u\n", cpu.x_bx);
  }
  #endif
  return cpu.x_ax;
}


//  ------------------------------------------------------------------

int main(int argc, char** argv) {

  union REGS cpu;
  int key_value, apiret;
  amis_signature* signature;
  gsapidata* data;
  #if defined(__WATCOMC__)
  long memory_needed;
  #if defined(__386__)
  word buffer;
  #else
  char* buffer;
  #endif
  #else
  char* buffer;
  #endif

  #ifdef DEBUG
  printf("Memory: %lu bytes\n", coreleft());
  #endif


  if(argc > 1) {

    // Installation check
    for(mpx=0; mpx<256; mpx++) {
      cpu.h.ah = mpx;
      cpu.h.al = GSAPI_INSTALL_CHECK;
      INT86(0x2D, &cpu, &cpu);
      if(cpu.h.al == 0xFF) {
        signature = (amis_signature*)MK_FP2(cpu.x_dx, cpu.x_di);
        #ifdef DEBUG
        printf("Found %8.8s version %04Xh\n", signature->product_name, cpu.x_cx);
        #endif
        if((cpu.x_cx >= 0x0100) && (cpu.x_cx < 0x0200))
          if(strnicmp(signature->product_name, "GoldSAPI", 8) == 0)
            break;
      }
    }

    // Return if not installed
    if(mpx >= 256) {
      #ifdef DEBUG
      printf("GSAPI is not installed.\n");
      #endif
      mpx = -1;
      return -1;
    }

    // Call the open api function
    cpu.h.al = GSAPI_OPEN_API;
    cpu.h.ah = mpx;
    INT86(0x2D, &cpu, &cpu);
    apiret = cpu.x_ax;
    if(apiret) {
      #ifdef DEBUG
      printf("Could not open GSAPI. Errorcode %u.\n", apiret);
      #endif
      return apiret;
    }

    // Successfully opened the api. Now get the data pointer and key value
    data = (gsapidata*)MK_FP2(cpu.x_dx, cpu.x_di);
    key_value = cpu.x_bx;

    // Copy the filename to the parameters field
    strcpy(data->parameters, argv[1]);

    // Now call the api to open and load the file
    apiret = call_api(GSAPI_OPEN_AND_LOAD_FILE, 0);
    if(apiret == 0x02) {
      #if defined(__WATCOMC__) && defined(__386__)
      // Determine how much is available
      cpu.x_ax = 0x100;
      cpu.x_bx = 65535u;
      INT86(0x31, &cpu, &cpu);
      memory_needed = (data->buffer_length + 16L) / 16L;
      if(memory_needed <= (long)cpu.x_bx*16L) {
        cpu.x_ax = 0x100;
        cpu.x_bx = memory_needed;
        INT86(0x31, &cpu, &cpu);
        data->buffer_segment = cpu.x_ax;
        data->buffer_offset = 0;
        buffer = cpu.x_dx;
        apiret = call_api(GSAPI_OPEN_AND_LOAD_FILE, 0);
      }
      else {
        #ifdef DEBUG
        printf("Not enough memory to load the file\n");
        #endif
      }
      #else
      buffer = (char*)HUGEMALLOC(data->buffer_length);
      if(buffer) {
        data->buffer_segment = FP_SEG(buffer);
        data->buffer_offset = FP_OFF(buffer);
        apiret = call_api(GSAPI_OPEN_AND_LOAD_FILE, 0);
      }
      else {
        #ifdef DEBUG
        printf("Not enough memory to load the file\n");
        #endif
      }
      #endif
      if(apiret == 0) {
        #ifdef DEBUG
        printf("Now playing %s via GSAPI\n", argv[1]);
        #endif
        call_api(GSAPI_PLAY, 0);
        while(data->status)
          /* Wait until play stops */;
        call_api(GSAPI_CLOSE_FILE, 0);
      }
      else {
        #ifdef DEBUG
        printf("GSAPI returned errorcode %u while opening the file second time\n", apiret);
        #endif
      }
    }
    else {
      #ifdef DEBUG
      printf("GSAPI returned errorcode %u while opening the file first time\n", apiret);
      #endif
    }

    call_api(GSAPI_CLOSE_API, key_value);
  }

  return 117;   /* Value to test errorlevel in GCTVSAPI */
}


//  ------------------------------------------------------------------

