{------------------------------------------------------------------------------}
unit FSpin;
{------------------------------------------------------------------------------}

interface

{------------------------------------------------------------------------------}
uses WinTypes, Classes, StdCtrls, ExtCtrls, Controls, Messages, SysUtils,
  Forms, Graphics, Menus, Buttons, Spin;

{------------------------------------------------------------------------------}
{ TFSpinEdit }
{------------------------------------------------------------------------------}
type
 TFSpinEdit = class(TCustomEdit)
  private
    FMinValue: Single;
    FMaxValue: Single;
    FCanvas: TCanvas;
    FIncrement: Single;
    FPrecision: byte;
    FButton: TSpinButton;
    FEditorEnabled: Boolean;
    function GetMinHeight: Integer;
    function GetValue: Single;
    function CheckValue (NewValue: Single): Single;
    procedure SetValue (NewValue: Single);
    procedure SetEditRect;
    procedure WMSize(var Message: TWMSize); message WM_SIZE;
    procedure CMEnter(var Message: TCMGotFocus); message CM_ENTER;
    procedure CMExit(var Message: TCMExit);   message CM_EXIT;
    procedure WMPaste(var Message: TWMPaste);   message WM_PASTE;
    procedure WMCut(var Message: TWMCut);   message WM_CUT;
  protected
    function IsValidChar(Key: Char): Boolean; virtual;
    procedure UpClick (Sender: TObject); virtual;
    procedure DownClick (Sender: TObject); virtual;
    procedure KeyDown(var Key: Word; Shift: TShiftState); override;
    procedure KeyPress(var Key: Char); override;
    procedure CreateParams(var Params: TCreateParams); override;
    procedure CreateWnd; override;
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    property Button: TSpinButton read FButton;
  published
    property AutoSelect;
    property AutoSize;
    property Color;
    property Ctl3D;
    property DragCursor;
    property DragMode;
    property EditorEnabled: Boolean read FEditorEnabled write FEditorEnabled default True;
    property Enabled;
    property Font;
    property Increment: Single read FIncrement write FIncrement;
    property MaxLength;
    property MaxValue: Single read FMaxValue write FMaxValue;
    property MinValue: Single read FMinValue write FMinValue;
    property Precision: byte read FPrecision write FPrecision default 2;
    property ParentColor;
    property ParentCtl3D;
    property ParentFont;
    property ParentShowHint;
    property PopupMenu;
    property ReadOnly;
    property ShowHint;
    property TabOrder;
    property TabStop;
    property Value: Single read GetValue write SetValue;
    property Visible;
    property OnChange;
    property OnClick;
    property OnDblClick;
    property OnDragDrop;
    property OnDragOver;
    property OnEndDrag;
    property OnEnter;
    property OnExit;
    property OnKeyDown;
    property OnKeyPress;
    property OnKeyUp;
    property OnMouseDown;
    property OnMouseMove;
    property OnMouseUp;
  end;

procedure Register;

{------------------------------------------------------------------------------}

implementation

{------------------------------------------------------------------------------}

uses WinProcs, DsgnIntf;

{------------------------------------------------------------------------------}
constructor TFSpinEdit.Create(AOwner: TComponent);
{------------------------------------------------------------------------------}
begin
 inherited Create(AOwner);
 FButton := TSpinButton.Create (Self);
 FButton.Width := 15;
 FButton.Height := 17;
 FButton.Visible := True;  
 FButton.Parent := Self;
 FButton.FocusControl := Self;
 FButton.OnUpClick := UpClick;
 FButton.OnDownClick := DownClick;
 Text := '0';
 ControlStyle := ControlStyle - [csSetCaption];
 FIncrement:= 1;
 FPrecision:= 2;
 FEditorEnabled := True;
end;

{------------------------------------------------------------------------------}
destructor TFSpinEdit.Destroy;
{------------------------------------------------------------------------------}
begin
 FButton := nil;
 inherited Destroy;
end;

{------------------------------------------------------------------------------}
procedure TFSpinEdit.KeyDown(var Key: Word; Shift: TShiftState);
{------------------------------------------------------------------------------}
begin
 if Key = VK_UP 
  then UpClick (Self)
  else if Key = VK_DOWN then DownClick (Self);
 inherited KeyDown(Key, Shift);
end;

{------------------------------------------------------------------------------}
procedure TFSpinEdit.KeyPress(var Key: Char);
{------------------------------------------------------------------------------}
begin
  if not IsValidChar(Key) then
  begin
    Key := #0;
    MessageBeep(0)
  end;
  if Key <> #0 then inherited KeyPress(Key);
end;

{------------------------------------------------------------------------------}
function TFSpinEdit.IsValidChar(Key: Char): Boolean;
{------------------------------------------------------------------------------}
begin
  Result := (Key in [DecimalSeparator, '+', '-', '0'..'9']) or
    ((Key < #32) and (Key <> Chr(VK_RETURN)));
  if not FEditorEnabled and Result and ((Key >= #32) or
      (Key = Char(VK_BACK)) or (Key = Char(VK_DELETE))) then
    Result := False;
end;

{------------------------------------------------------------------------------}
procedure TFSpinEdit.CreateParams(var Params: TCreateParams);
{------------------------------------------------------------------------------}
begin
  inherited CreateParams(Params);
{  Params.Style := Params.Style and not WS_BORDER;  }
  Params.Style := Params.Style or ES_MULTILINE or WS_CLIPCHILDREN;
end;

{------------------------------------------------------------------------------}
procedure TFSpinEdit.CreateWnd;
{------------------------------------------------------------------------------}
var
  Loc: TRect;
begin
  inherited CreateWnd;
  SetEditRect;
end;

{------------------------------------------------------------------------------}
procedure TFSpinEdit.SetEditRect;
{------------------------------------------------------------------------------}
var
  Loc: TRect;
begin
  SendMessage(Handle, EM_GETRECT, 0, LongInt(@Loc));
  Loc.Bottom := ClientHeight;
  Loc.Right := ClientWidth - FButton.Width - 2;
  Loc.Top := 0;  
  Loc.Left := 0;  
  SendMessage(Handle, EM_SETRECTNP, 0, LongInt(@Loc));
  SendMessage(Handle, EM_GETRECT, 0, LongInt(@Loc));  {debug}
end;

{------------------------------------------------------------------------------}
procedure TFSpinEdit.WMSize(var Message: TWMSize);
{------------------------------------------------------------------------------}
var
  Loc: TRect;
  MinHeight: Integer;
begin
  inherited;
  MinHeight := GetMinHeight;
    { text edit bug: if size to less than minheight, then edit ctrl does
      not display the text }
  if Height < MinHeight then   
    Height := MinHeight
  else if FButton <> nil then
  begin
    FButton.SetBounds (Width - FButton.Width, 0, FButton.Width, Height);  
    SetEditRect;
  end;
end;

{------------------------------------------------------------------------------}
function TFSpinEdit.GetMinHeight: Integer;
{------------------------------------------------------------------------------}
var
  DC: HDC;
  SaveFont: HFont;
  I: Integer;
  SysMetrics, Metrics: TTextMetric;
begin
  DC := GetDC(0);
  GetTextMetrics(DC, SysMetrics);
  SaveFont := SelectObject(DC, Font.Handle);
  GetTextMetrics(DC, Metrics);
  SelectObject(DC, SaveFont);
  ReleaseDC(0, DC);
  I := SysMetrics.tmHeight;
  if I > Metrics.tmHeight then I := Metrics.tmHeight;
  Result := Metrics.tmHeight + I div 4 + GetSystemMetrics(SM_CYBORDER) * 4 + 2;
end;

{------------------------------------------------------------------------------}
procedure TFSpinEdit.UpClick (Sender: TObject);
{------------------------------------------------------------------------------}
begin
  if ReadOnly then MessageBeep(0)
  else Value := Value + FIncrement;
end;

{------------------------------------------------------------------------------}
procedure TFSpinEdit.DownClick (Sender: TObject);
{------------------------------------------------------------------------------}
begin
  if ReadOnly then MessageBeep(0)
  else Value := Value - FIncrement;
end;

{------------------------------------------------------------------------------}
procedure TFSpinEdit.WMPaste(var Message: TWMPaste);   
{------------------------------------------------------------------------------}
begin
  if not FEditorEnabled or ReadOnly then Exit;
  inherited;
end;

{------------------------------------------------------------------------------}
procedure TFSpinEdit.WMCut(var Message: TWMPaste);   
{------------------------------------------------------------------------------}
begin
  if not FEditorEnabled or ReadOnly then Exit;
  inherited;
end;

{------------------------------------------------------------------------------}
procedure TFSpinEdit.CMExit(var Message: TCMExit);
{------------------------------------------------------------------------------}
begin
  inherited;
  if CheckValue (Value) <> Value then
    SetValue (Value);
end;

{------------------------------------------------------------------------------}
function TFSpinEdit.GetValue: Single;
{------------------------------------------------------------------------------}
begin
 try
  Result := StrToFloat(Text); {***********************}
 except
  Result := FMinValue;
 end;
end;

{------------------------------------------------------------------------------}
procedure TFSpinEdit.SetValue (NewValue: Single);
{------------------------------------------------------------------------------}
begin
 try
  Text:= FloatToStrF(CheckValue(NewValue), ffFixed, 7, FPrecision);
 except
 end;
end;

{------------------------------------------------------------------------------}
function TFSpinEdit.CheckValue (NewValue: Single): Single;
{------------------------------------------------------------------------------}
begin
 Result:= NewValue;
 if (FMaxValue <> FMinValue) then
  begin
   if NewValue < FMinValue
    then Result:= FMinValue
    else 
     if NewValue > FMaxValue then Result:= FMaxValue;
  end;
end;

{------------------------------------------------------------------------------}
procedure TFSpinEdit.CMEnter(var Message: TCMGotFocus);
{------------------------------------------------------------------------------}
begin
 if AutoSelect and not(csLButtonDown in ControlState) then SelectAll;
 inherited;
end;

{------------------------------------------------------------------------------}
procedure Register;
{------------------------------------------------------------------------------}
begin
 RegisterComponents('Exemples', [TFSpinEdit]);
end;

end.

