
/*
    IPX support library - kernel dependent functions

    Copyright (C) 1994, 1995  Ales Dryak <e-mail: A.Dryak@sh.cvut.cz>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/
#include <errno.h>
#include <unistd.h>
#include <sys/ioctl.h>
#include <sys/socket.h>
#include "ipxkern.h"

#define MAX_ROUTE (1024)

int ipx_kern_scan_rtable(IPXrtScanFunc f,void* data)
{
	int sock;
	struct ipx_route_def rt_table[MAX_ROUTE];
	struct ipx_route_info rt_info={MAX_ROUTE,rt_table};
	struct ipx_route_def* rt_def;
	
	sock=socket(AF_IPX,SOCK_DGRAM,PF_IPX);
	if (sock==-1)
	{
		sprintf(ipx_err_string,"open socket: %s",strerror(errno));
		return -1;
	}
	
	if(ioctl(sock,SIOCGIFCONF,(void *)&rt_info)==-1)
	{
                sprintf(ipx_err_string,"ioctl SIOGIFCONF: %s",strerror(errno));
		close(sock);
		return -1;
	}
	
	for(rt_def=rt_info.ipx_rt_list;
		rt_def<rt_info.ipx_rt_list+rt_info.ipx_rt_len;rt_def++)
	{
		int type=0;
		if (rt_def->ipx_router_network!=0) type|=IPX_KRT_ROUTE;
		if (strcmp(rt_def->ipx_device,"lo")==0) type|=IPX_KRT_INTERNAL;
		if (!f(rt_def->ipx_network,rt_def->ipx_router_network,
		       rt_def->ipx_router_node,type,data))
		{
			close(sock);
			return 0;
		}
  	}
  	close(sock);
  	return 1;
}

struct scan_ifc_info
{
	IPXifcScanFunc f;
	void* data;
};

static int ipx_kern_scan_iface(IPXNet net,IPXNet rt_net,IPXNode rt_node,int type,void* data)
{
	struct scan_ifc_info* p=(struct scan_ifc_info*) data;
	if ((type & IPX_KRT_ROUTE)==0)
		return (p->f)(net,rt_node,type,p->data);
	else
		return 1;
}

int ipx_kern_scan_ifaces(IPXifcScanFunc f,void* data)
{
	struct scan_ifc_info sii;
	sii.f=f;
	sii.data=data;
	return ipx_kern_scan_rtable(ipx_kern_scan_iface,(void*)&sii);
}

static int ipx_kern_scan_addr(IPXNet net,IPXNode node,int type,void* data)
{
	struct sockaddr_ipx* addr=(struct sockaddr_ipx*)data;
	
	addr->sipx_network=net;
	ipx_assign_node(addr->sipx_node,node);
	if (type & IPX_KRT_INTERNAL) return 0;
	return 1;
}

int ipx_kern_get_internet_addr(struct sockaddr_ipx* addr)
{
	addr->sipx_family=AF_IPX;
	addr->sipx_type=IPX_USER_PTYPE;
	addr->sipx_port=0;
	addr->sipx_network=IPX_THIS_NET;
	ipx_assign_node(addr->sipx_node,IPX_THIS_NODE);
	return ipx_kern_scan_ifaces(ipx_kern_scan_addr,(void*)addr)<0?-1:0;	
}

int ipx_kern_route_add(int sock,IPXNet net,IPXNet rt_net,IPXNode rt_node)
{
	struct ipx_route_def rt_def;
	rt_def.ipx_network=net;
	rt_def.ipx_router_network=rt_net;
	ipx_assign_node(rt_def.ipx_router_node,rt_node);
	rt_def.ipx_flags=0;

	if(ioctl(sock,SIOCADDRT,(void *)&rt_def)!=0)
	{
		sprintf(ipx_err_string,"ioctl SIOCADDRT: %s",strerror(errno));
		return -1;
	}
	return 0;
}

int ipx_kern_route_delete(int sock,IPXNet net)
{
	if(ioctl(sock,SIOCDELRT,(void *)&net)!=0)
	{
		sprintf(ipx_err_string,"ioctl SIOCDELRT: %s",strerror(errno));
		return -1;
	}
	return 0;
}

int ipx_kern_enable_broadcast(int sock)
{
	int opt=1;
	/* Permit broadcast output */
	if(setsockopt(sock,SOL_SOCKET,SO_BROADCAST, &opt,sizeof(opt))==-1)
	{
		sprintf(ipx_err_string,"setsockopt SO_BROADCAST: %s",strerror(errno));
		return -1;
	}
	return 0;
}

int ipx_kern_dont_route(int sock)
{
	int opt=1;
	/* Disable routing and set src address generation by iface */
	if(setsockopt(sock,SOL_SOCKET,SO_DONTROUTE, &opt,sizeof(opt))==-1)
	{
		sprintf(ipx_err_string,"setsockopt SO_DONTROUTE: %s",strerror(errno));
		return -1;
	}
	return 0;
}
