/* *******************************************************************

The Oracle SQL Class Library Version 1.0,
Author: Sergei Kuchin
Copyright (C) Mosakin Corp., 1995
This library is free software.  Permission to use, copy,
modify and redistribute the Oracle SQL class library for any
purpose is hereby granted without fee, provided that the
above copyright notice appear in all copies.


******************************************************************* */
#ifndef __OSCL_STR_H
#define __OSCL_STR_H

// ------------- SQL stream classes -----------------

/******************* CLASS HIERARCHY *********************
*                                                        *
*                      OraStream                         *
*                       /     \                          *
*		       /       \                         *
*                 SelStream   OStream                    *
*                    /           \                       *
*                   /             \                      *
*            SelectStream       OutStream                *
*                                   \                    *
*                                    \                   *
*                                 InOutStream            *
*                                                        *
*                SQLStream   SQL                         *
*                                                        *
**********************************************************/

// Generic class for Oracle streams
class OraStream{
public:

 OraStream();
 OraStream(char* StmText);
 virtual ~OraStream(){delete stm_text;};

 virtual int fail(void){return err::on();};
   // test if current stream operation failed

 virtual PStm GetStm(void){return 0;};


protected:

 PHostVarArray va; // internal vector of host variables
                   // the vector is actually internal buffer 
		   // attached to the stream
 int cur_x; // current horizontal index in the vector
 int cur_y; // current vertical index in the vector
 int va_len; // horizontal dimension of the vector
 int vect_size; // vertical dimension of the vector
 char* stm_text; // SQL statement text

 virtual int CheckType(int TypeCode){return 1;};
  // internal function. check if the current element to be fetched from the
  // internal buffer is of appropriate type for the current
  // stream operation.

};

// Select Stream class
// Example:
//
//  SelStream s("select State, Capital, State_code from state",10);
//  while(!s.eof()){
//    char  state_name[33];
//    char  state_capital[20];
//    short state_code;
//
//    s>>state_name>>state_capital>>state_code;
//    cout<<state_name<<state_capital<<state_code<<endl;
// }
//
// This class is for SELECT statements ONLY!!!
class SelStream:public OraStream{
public:

 SelStream(PConnect db,             // Connect object
	   char* StmText,           // SQL statement
	   short VectSize=1,        // Internal buffer size in rows
	   PHostVarArray InVar=0    // Array of host variables 
	                            // used in the SQL statement
        );
    // Create input stream

  SelStream(){};

 ~SelStream();

 PStm GetStm(void){return stm;};
   // get the pointer to the SQL statement attached to the stream

 virtual int eof(void){return rc==FetchEod;};
   // test if end of stream has been reached

 virtual int fail(void){return rc==FetchErr;};
   // test if current stream operation failed

 virtual void rewind();
   // rewind stream.
   // SQL statement is re-executed

 int is_null(void){return null_fetched;};
    // test if NULL has been fetched 
    // during last stream operation

// Read various objects from the stream

 SelStream& operator>>(char& c);
 SelStream& operator>>(unsigned char& c);
 SelStream& operator>>(char* s);
 SelStream& operator>>(unsigned char* s);
 SelStream& operator>>(int& n);
 SelStream& operator>>(unsigned& u);
 SelStream& operator>>(short& sh);
 SelStream& operator>>(long int& l);
 SelStream& operator>>(float& f);
 SelStream& operator>>(double& d);


// Internal stuff
protected:
 int rc;
 int null_fetched;

 void GetNext(void);

 PHostVarArray ___stmVarList(void){return va;}
 DEFINE_DEFAULT_SELECT_LIST(stm,vect_size);
 DEFINE_SELECT_STM(stm,stm_text,vect_size);

protected:

 int CheckType(int TypeCode);
 void LookAhead(void);

};

#define CONSTR_VAR_LIST(Lst) ___##Lst##VarList() 
  // Special macro, see example below

// Example:
//
// INT_VAR(v_code); // host variable definition -- integer
// STRING_VAR(v_state,33);// host variable -- string
//
// BEGIN_VAR_LIST(vlist1) // binding the host variables defined above
//                        // into a list
//   VAR(v_code),  
//   VAR(v_state),
// END_VAR_LIST;
//
// v_code=10; 
// strcpy(v_state,"M%");
// SelStream in(db,
//	      "select state, capital "
//	      "from state "
//            "where "
//            "  state_code <= :v_code and "
//            "  state like :v_state "
//	      "order by state_code",
//	      10,
//	      CONSTR_VAR_LIST(vlist1) // construct the host variable list and
//                                    // attach it to this  SELECT statement
//	     );
//
// while(!in.eof()){
//  char state[33];
//  capital char[20];
//
//  in>>state>>capital;
//  cout<<state<<capital<<endl;
// }
//

// This class is used to write NULLs into Oracle.
// see operator<<(onull n);
class onull{
public:
 onull(){};
 ~onull(){};
};

// Output stream class

// Example:
//
// OStream o(db,
//    	     20,
//	     "insert into state values(:state,:capital,:state_code)",
//	     ":state char[33]", 
//	     ":capital char[20]",
//           ":state_code short",
//           NO_NAME
//	    );
// char state[33];
// char capital[20];
// short state_code;
//
// strcpy(state,"Minnesota");
// strcpy(capital,"St.Paul");
// state_code=23;
//
// o<<state<<capital<<state_code;
//

class OStream:public OraStream{
public:
 OStream(PConnect db,    // Connect object
	 char* StmText,  // SQL statement text
	 short VectSize, // Internal buffer size (in rows)
	 char* hvd,      // host variable declaration list
	 ...             // (see the example above)
        );
  // Create output stream

// The following data types of host variables are allowed in the
// declarations:
//
//  :var char[<length>]
//  :var int
//  :var short
//  :var long
//  :var unsigned
//  :var float
//  :var double

 OStream(PConnect db,char* StmText,short VectSize, char* hvd[]);
// One more constructor. The host variable declaration list is 
// a null terminated array: char*[], similar to the second argument
// in main(int argc, char* argv[])

 OStream(){va_should_delete=0;};
 ~OStream();

  PStm GetStm(void){return stm;};
   // get the pointer to the SQL statement attached to the stream

// Write various objects into stream

 OStream& operator<<(char c);
 OStream& operator<<(unsigned char c);
 OStream& operator<<(const char* s);
 OStream& operator<<(const unsigned char* s);
 OStream& operator<<(int n);
 OStream& operator<<(unsigned u);
 OStream& operator<<(short sh);
 OStream& operator<<(long int l);
 OStream& operator<<(float f);
 OStream& operator<<(double d);
 OStream& operator<<(onull n); // write NULL into stream

 virtual void flush(void); 
  // Flush buffer. Actually the SQL statement is executed
  // and the internal buffer is cleaned.

 void SetCommit(int CommitOnFlush=1){commit_on_flush=CommitOnFlush;};
  // Set "commit-on-flush" flag. If the flag is set up then 
  // current transaction is commited. 
  // If you need to control transaction on your own, make the
  // following function call:
  //
  //    <stream>.SetCommit(0);


// Internal stuff
protected:
 int commit_on_flush;
 PConnect adb;
 int dirty;
 int va_should_delete;

 PHostVarArray ___stmVarList(void){return va;} 
 DEFINE_EMPTY_SELECT_LIST(stm);
 DEFINE_STM(stm,stm_text);

 void GetNext(void);
 int CheckType(int TypeCode);
 void CheckBuf(void);

};

typedef OStream* POStream;
typedef SelStream* PSelStream;


//One more output stream class 
//It is slightly different from its ancestor
// Example:
//
// OutStream o(db,
//     	       20,
//	       "insert into state values( "
//             "      :state<char[33]>, "
//             "      :capital<char[20]>, "
//             "      :state_code<short>)"
//	    );
// char state[33];
// char capital[20];
// short state_code;
//
// strcpy(state,"Minnesota");
// strcpy(capital,"St.Paul");
// state_code=23;
//
// o<<state<<capital<<state_code;
//

class OutStream:public OStream{
public:

 OutStream(PConnect db,     // Connect object
	   char* StmText,   // SQL statement text
	   short VectSize=1 // Internal buffer size (in rows)
          );
  // Create output stream

 OutStream(){};

};
typedef OutStream* POutStream;


//One more Select stream class
// Example:
//
// SelectStream in(db,
//	      "select state, capital "
//	      "from state "
//            "where "
//            "  state_code <= :v_code<int> and "
//            "  state like :v_state<char[33]> "
//	      "order by state_code",
//	      10
//	     );
//
// in<<10<<"M%";
// while(!in.eof()){
//  char state[33];
//  capital char[20];
//
//  in>>state>>capital;
//  cout<<state<<capital<<endl;
// }
//
// in<<20<<"C%";
// while(!in.eof()){
//  char state[33];
//  capital char[20];
//
//  in>>state>>capital;
//  cout<<state<<capital<<endl;
// }
//
// This class is for SELECT statements ONLY!!!
class SelectStream:public SelStream{
public:

 SelectStream(PConnect db,char* StmText,short VectSize=1);
   // Create Select stream. See the example above


// Operators to assign values to the input parameters of 
// the SELECT statement, one by one. See the example above

 SelectStream& operator<<(char c);
 SelectStream& operator<<(unsigned char c);
 SelectStream& operator<<(const char* s);
 SelectStream& operator<<(const unsigned char* s);
 SelectStream& operator<<(int n);
 SelectStream& operator<<(unsigned u);
 SelectStream& operator<<(short sh);
 SelectStream& operator<<(long int l);
 SelectStream& operator<<(float f);
 SelectStream& operator<<(double d);


// Read various objects from the stream

 SelectStream& operator>>(char& c);
 SelectStream& operator>>(unsigned char& c);
 SelectStream& operator>>(char* s);
 SelectStream& operator>>(unsigned char* s);
 SelectStream& operator>>(int& n);
 SelectStream& operator>>(unsigned& u);
 SelectStream& operator>>(short& sh);
 SelectStream& operator>>(long int& l);
 SelectStream& operator>>(float& f);
 SelectStream& operator>>(double& d);

// Internals
private:

 PHostVarArray in_var;
 int iv_len;
 short executed;
 int cur_in_x;

 void GetInNext(void);
 int CheckInType(int TypeCode);
};
typedef OutStream* POutStream;


// Input/output stream class
//
// Example:
//
// InOutStream s(db,
//               "begin :A<int,out> := :B<int,in>; end;",
//               3		 
//              );
// s<<1<<2<<3;
// while(!s.eof()){
//  int k;
//  s>>k;
//  cout<<k<<endl;
// }
//
// Streams of this class are intended primarily for PL/SQL
// blocks having input and output variables. It is also
// possible to use the InOutStream for the SQL and PL/SQL
// statements only with input variables or only with output
// variables. 
 
class InOutStream:public OutStream{
public:

 InOutStream(PConnect db,     // Connect object
	     char* StmText,   // SQL statement text
	     short VectSize=1 // Buffer size (in rows)
	    );
 // Create input/ouput stream

 ~InOutStream();

 int eof(void);
  // test if all data is already read from the stream

 void flush(void); 
  // "flush" output buffer. It actually means execute the
  // statement  attached to the stream as many times as
  // rows are entered to the buffers. The stream is
  // automatically flushed when the buffer is full, as in
  // the example above.

// Logically, the InOutStream has two buffers: for input
// variables and for output variables. The stream has
// operations of reading data from the input buffer and
// operations of writing data to the output buffer. These
// two buffers are not intersected. When the output buffer
// is flushed and the SQL statement is executed, the input
// buffer is filled with data.

 int is_null(void){return null_fetched;};
  // test if NULL was fetched from the stream


// Write various objects into streams.

 InOutStream& operator<<(char c);
 InOutStream& operator<<(unsigned char c);
 InOutStream& operator<<(const char* s);
 InOutStream& operator<<(const unsigned char* s);
 InOutStream& operator<<(int n);
 InOutStream& operator<<(unsigned u);
 InOutStream& operator<<(short sh);
 InOutStream& operator<<(long int l);
 InOutStream& operator<<(float f);
 InOutStream& operator<<(double d);
 InOutStream& operator<<(onull n); // write NULL into stream

// Read various objects from the stream

 InOutStream& operator>>(char& c);
 InOutStream& operator>>(unsigned char& c);
 InOutStream& operator>>(char* s);
 InOutStream& operator>>(unsigned char* s);
 InOutStream& operator>>(int& n);
 InOutStream& operator>>(unsigned& u);
 InOutStream& operator>>(short& sh);
 InOutStream& operator>>(long int& l);
 InOutStream& operator>>(float& f);
 InOutStream& operator>>(double& d);

// Internal stuff
protected:
 
 PHostVarArray in_va;
 int iv_len;
 int cur_in_x;
 int cur_in_y;
 int in_y_len;
 int null_fetched;

 void GetInNext();
 int CheckInType(int TypeCode);
 int is_null1(void);
};
typedef InOutStream* PInOutStream;


// ----------- Simplified interface to SQL ----------

class SQL{
public:

 SQL(char* Connect); // Connect to the Oracle Server
 ~SQL(); // Disconnect 

 static void  err_clear(void); // clear error status
 static char* err_msg(void);   // get error message
 static int   err_code(void);  // get error code
 
 void  commit(void); // commit transaction
 void  roll_back(void); // roll back transaction

 int   exec(char* Stm,int ignore_error=0); 
   // execute the Stm statement and ignore error if needed

private:
 TConnect* db;
 friend class SQLStream;
};

// ------------------- SQL Stream class -----------------
//
// The class is unified for using any SQL or PL/SQL statement

class SQLStream{
public:

 SQLStream(SQL* sql,char* Stm,short BufSize=1);
  // open a SQL stream using the "sql" connection 
  // for the "Stm" statement. Attach internal buffer of the 
  // "BufSize" size. BufSize defines the dimension of 
  //  internal host arrays to be used.

 ~SQLStream(); // close stream

 int  eof(void); // test if "eod-of-file" is reached
 int  fail(void); // test if the last stream operation failed
 int  is_null(void); // test if the last values fetched from the 
                     // stream is NULL or not
 void flush(void);  // "flush" the stream 
 void set_commit(int commit_when_flush=1); // set "auto-commit" flag
 void commit(void); // explicitly commit transaction
 void roll_back(void); // roll back transaction

// -------- Write various objects to stream -----------

 SQLStream& operator<<(char c);
 SQLStream& operator<<(unsigned char c);
 SQLStream& operator<<(const char* s);
 SQLStream& operator<<(const unsigned char* s);
 SQLStream& operator<<(int n);
 SQLStream& operator<<(unsigned u);
 SQLStream& operator<<(short sh);
 SQLStream& operator<<(long int l);
 SQLStream& operator<<(float f);
 SQLStream& operator<<(double d);
 SQLStream& operator<<(onull n); // write NULL into stream


// -------- Read various objects from  stream -----------

 SQLStream& operator>>(char& c);
 SQLStream& operator>>(unsigned char& c);
 SQLStream& operator>>(char* s);
 SQLStream& operator>>(unsigned char* s);
 SQLStream& operator>>(int& n);
 SQLStream& operator>>(unsigned& u);
 SQLStream& operator>>(short& sh);
 SQLStream& operator>>(long int& l);
 SQLStream& operator>>(float& f);
 SQLStream& operator>>(double& d);

private:

 InOutStream* io;
 SelectStream* ss;
 SQL* db;
};

#endif
