/* *******************************************************************

The Oracle SQL Class Library Version 1.0,
Author: Sergei Kuchin
Copyright (C) Mosakin Corp., 1995
This library is free software.  Permission to use, copy,
modify and redistribute the Oracle SQL class library for any
purpose is hereby granted without fee, provided that the
above copyright notice appear in all copies.


******************************************************************* */
#ifndef __OSCL_H
#define  __OSCL_H

#ifdef __BCPLUSPLUS__
 #define __TEMPLATE_ENABLED__
 #define __EXCEPTION_ENABLED__
 #define TEMPLATE_INITIALIZER(n) = n 

#else
 #define ora_text text
#endif

#ifdef __xlC__

 #define __TEMPLATE_ENABLED__
 #define __EXCEPTION_ENABLED__
 #define __32BIT__

 #if __xlC__>=0x0201
  #define TEMPLATE_INITIALIZER(n) = n 
 #else
  #define TEMPLATE_INITIALIZER(n)
 #endif

#endif

#ifdef __GNUC__
 #define __TEMPLATE_ENABLED__
 #define __32BIT__
 #define TEMPLATE_INITIALIZER(n)
#endif


#include <malloc.h>
#include <stdlib.h>
extern "C"{
#include <ociapr.h>
}

#ifdef __32BIT__
typedef sword long_ndx;
#else
typedef long int long_ndx;
#endif

#ifdef _MSC_VER
 #define farmalloc(p) _halloc(p,1)
 #define farfree(p) _hfree(p)
#else
 #ifdef __32BIT__
  #define _huge
  #define farmalloc(p) malloc(p)
  #define farfree(p) free(p)
 #endif
#endif

// **************************  List of type identifiers *********************
//
//  A type identifier is an integer used  by the function isA() (see below).
//  Every class in the OSCL can be dynamically identified by the isA()
//  call, e.g.:

//
//                     if(v->isA()==type_TDouble){
//                      ...
//                     }
//
// Identifiers in the range [1..100] are reserved by the OSCL for
// internal usage, that is
// in derived classes it is recommended to use the numbers greater than 100.

enum type_Identifier{

   type_TObj=1,
   type_inFldDesc=2,
   type_THostVar=3,
   type_GenericVar=4,
   type_TDouble=5,
   type_TFloat=6,
   type_TSignedChar=7,
   type_TShortInt=8,
   type_TInt=9,
   type_TLongInt=10,
   type_TUnsignedInt=11,
   type_RowId=12,
   type_TVarNum=13,
   type_TNumber=14,
   type_TVarChar2=15,
   type_TCChar=16,
   type_TLong=17,
   type_TVarChar=18,
   type_TVarRaw=19,
   type_TRaw=20,
   type_TLongRaw=21,
   type_TLongVarChar=22,
   type_TLongVarRaw=23,
   type_TChar=24,
   type_TCharZ=25,
   type_TLongString=26,
   type_TLongByteString=27,
   type_TDateTime=28,
   type_THostVarArray=30,
   type_inRowDesc=31,
   type_TConnect=32,
   type_TOCursor=33,
   type_TStm=34,
   type_TSelectStm=35,
   type_TSelectCursor=36
};

const int MAX_VAR_ARRAY_SIZE=512;
   // Maximum array size for an array of host variables

const sword MAX_LONG_SIZE=8000;
  //  Default maximum size for "LONG"  ORACLE data types
  // in the DefineSelectList function when a select
  // item list is formed as it is "described" by
  // the function DescribeSelectList

const int MAX_CURSOR_NUM=512;
  // Maximum number of cursors that can be simultaneously opened
  //  for one connection

// ***************** Generic class used in the OSCL hierarchy ***************

class TObj{
public:

 TObj(){owner=0; should_delete=1;}; // by default an object has no owner and
                                    // and should be deleted
 virtual ~TObj(){};

 void SetOwner(TObj* Owner){owner=Owner;};
 virtual int isA(void){ return type_TObj;}; //  gives an object identifier

protected:
 TObj* owner; //  Object may have an owner
 short should_delete; // Object potentially can be deleted by its owner
};

class TConnect;
class TOCursor;
class TStm;
class inFldDesc;
class THostVar;
class THostVarArray;
class err;
class TSelectStm;
class TSelectCursor;

typedef TOCursor* POCursor;
typedef TSelectCursor* PSelectCursor;
typedef TSelectStm* PSelectStm;
typedef TSelectCursor* PSelectCursor;
typedef TStm* PStm;
typedef TConnect* PConnect;
typedef TObj* PObj;
typedef inFldDesc* PinFldDesc;
typedef THostVar* PHostVar;
typedef THostVarArray* PHostVarArray;


  enum {FetchOk=0,FetchEod=1,FetchErr=2};
    //  list of return codes for a series of
    // "fetch" functions

// FetchOk --
//         operation terminates normally: a portion of rows has been
//         fetched and there is more data to be fetched

// FetchEod --

//        operation terminates normally, but the end of the data
//        selected has already been reached. If the operation is
//        a group fetch (more than one row at once) then use a
//        function RowFetched() to learn how many rows have been
//        actually fetched when last fetching.

// FetchErr --

//          an error occurred when fetching. Use the functions
//          err::code() and err::msg() to get respectively the
//          error code and error message. See "ORACLE  Error Messages"


//  ************** Internal ORACLE datatype code list ***************
//  for more info see Programmer's guide to the ORACLE Call Interface ([1]).

  enum InternalType{

    inVarChar2=1,
    inNumber=2,
    inLong=8,
    inRowId=11,
    inDate=12,
    inRaw=23,
    inLongRaw=24,
    inChar=96,
    inMslabel=106

  };

//  ************** External ORACLE datatype code list ***************
//  for more info see [1].

  enum ExternalType{

   extVarChar2=inVarChar2,
   extNumber=inNumber,
   extInt=3,
   extFloat=4,
   extCChar=5,
   extVarNum=6,
   extLong=inLong,
   extVarChar=9,
   extRowId=inRowId,
   extDate=inDate,
   extVarRaw=15,
   extRaw=inRaw,
   extLongRaw=inLongRaw,
   extUInt=68,
   extLongVarChar=94,
   extLongVarRaw=95,
   extChar=inChar,
   extCharZ=97,
   extMslabel=inMslabel
  };

class LDA_DEF{
public:
   Lda_Def lda; // Logon Descriptor Area (LDA)
private:
   ub1     hda[256]; // Hidden descriptor area used by the OCI
   friend class OCI;
};

class CDA_DEF{
public:
  Cda_Def cda; // Cursor Descriptor Area
  short   stm_code; // a SQL statement code after parsing the statement
  ub4 row_count; // row count
  short row_fetched; // row fetched
  LDA_DEF* lda; // pointer to the parent LDA

  CDA_DEF(LDA_DEF* Alda){stm_code=0;lda=Alda;row_count=0;row_fetched=0;};

};


//  Descriptor of internals of a field - an item of a select list in
//  a SELECT statement, e.g.:
//
//          SELECT state, capital FROM state where state like 'm%'
//
//  instances of this class contain descriptors of internals of
//  the fields:
//
//       state
//       capital

enum type_Number{
 type_notNumber=0,
 type_shortNumber=1,
 type_floatNumber=2,
 type_Number=3
}; 

class inFldDesc:public TObj{
public:

   inFldDesc(){name=0;};

   inFldDesc(CDA_DEF* cda,int pos);
     // construct the description of the field number "pos".
     // Fields in a SELECT statement are numbered from 1:
     //   state   - 1
     //   capital - 2

   ~inFldDesc();

  int isA(void){ return type_inFldDesc;};

  char* Name(void){return name;};
  sb4   Dbsize(void){return dbsize;};
  sb2   Dbtype(void){return dbtype;};
  sb2   Scale(void){return scale;};
  sb2   Prec(void){return prec;};
  sb4   Dsize(void){return dsize;};
  sb2   Nullok(void){return nullok;};
  
  short NumberOfType(void);
 
private:
  char* name;
   // name of the field
  sb4 dbsize;
   // size of the field as it is presented inside ORACLE
  sb2 dbtype;
   // internal datatype code, e.g.: inNumber, inVarChar2
  sb2 scale;
   // scale for the field of the inNumber type NUMBER(scale,precision)
  sb2 prec;
   // precision for the field of the inNumber type
  sb4 dsize;
   // maximum display size of the field
  sb2 nullok;
   // specifies if the field can have NULL (ORACLE NULL) value or not
};


//  ************ Host Variables **********************
// for more info about host variables see [1]
// This is a generic class from which  all other types of host
// variable are constructed via the mechanism of C++ templates.

class THostVar:public TObj{
public:

   THostVar();
   THostVar(char* Name,sword FType,sword ElemSize,
            sword MaxVectSize=1,void* Buf=0);

//  construct a host variable:
//       Name - variable name. Name=0 if the variable has no name.
//       FType -  variable type (see also "External datatypes" in [1])
//       ElemSize -  maximum size of the buffer for the variable.
//                   Actually host variables can be arrays (vectors).
//                   In this case ElemSize is a miximum size of
//                   a vector element
//       MaxVectSize - maximum size of a variable if it is a vector.
//       Buf - address of an external buffer. By default (Buf=0)
//             internal buffers
//             are created and used.

   THostVar(char* Name,sword FType,sword Size,void* Buf=0);
      //  construct a scalar (non-vector) host variable:
      //     Size is equivalent to ElemSize

   ~THostVar();

   int isA(void){return type_THostVar;};

   void SetBuf(void* Buf); // set a new external buffer
   void SetName(char* Name); // set a new name

   void* Buf(void){return (void*)buf;}; // get the buffer address
   char* Name(void){return name;}; // get the name
   sword ElemSize(void){return elem_size;}; // get ElemSize

   void* Val(int Ndx=0){return (void*)&buf[((unsigned)Ndx)*elem_size];};
      // get the address of  V[Ndx] if the variable is a vector
   void GetVal(int Ndx,void* Dst);
      // get the value of V[Ndx] and copy it to Dst
   void GetVal(void* Dst);
      // get the value of V[0] and copy it to Dst
   void SetVal(int Ndx,sword len,void* Src);
      // Set the value of V[Ndx] -- copy from Src to V[Ndx] Len bytes
   void SetVal(sword len,void* Src);
      // Set the value of V[0] -- copy from Src to V[Ndx] Len bytes

// For input variables:

   void SetNull(int Ndx=0){indp[Ndx]=-1;};
      //  set NULL to V[Ndx], indicate that the input variable has
      //  the NULL value
   void SetLen(int Ndx,sword Len){indp[Ndx]=Len;};
      // set the length of V[Ndx] as Len
   void SetLen(sword Len){indp[0]=Len;};
      // set the length of V[Ndx] as Len
   void SetNotNull(int Ndx=0){indp[Ndx]=elem_size;};
      // set not NULL status of V[Ndx]

// For output variables:

   int isNull(int Ndx=0){return indp[Ndx]==-1;};
      //  test if V[Ndx] is NULL or not
   int isSuccess(int Ndx=0){return indp[Ndx]==0;};
      // test if  an ORACLE operation has completed normally or not
   int isTruncated(int Ndx=0){return indp[Ndx]==-2||indp[Ndx]>0;};
      // test if a value fetched to V[Ndx] has been truncated or not

// Only for output variables, defined in "DefineSelectList":

   int isInvalidConversion(int Ndx=0){return rcode[Ndx]==1454;};
      //  test if V[Ndx] has a value that has been converted normally
   int isRealOverflow(int Ndx=0){return rcode[Ndx]==1456;};
      // test (when V is an ORACLE NUMBER) if for V[Ndx] real overflow
      // occurred or not
   int isUnsupportedDatatype(int Ndx=0){return rcode[Ndx]==3115;};
      // test if a returned value for V[Ndx] is of unsupported datatype.
   ub2 Len(int Ndx=0){return rlen[Ndx];};
      // get a returned  value length for V[Ndx]
   sword VectSize(void){return vect_size;};
      // get the V vector size
   sword Ftype(void){return ftype;};
      // get the V type
   int   IsExternalBuf(void){return buf_is_external;};
      // test if V  uses an external buffer

// low-level diagnostics:

   ub2   Rlen(int Ndx=0){return rlen[Ndx];};
    // returned length -- for a "DefineSelectList" variable
   ub2   Rcode(int Ndx=0){return rcode[Ndx];};
    // returned code --  for a "DefineSelectList" variable
   sword Rind(int Ndx=0){return indp[Ndx];};
    // returned indicator -- for output variables,
    //  defined both in "DefineSelectList"
    //  and in "BindVarList"

// For more details see the OCI documentation


// Convertion functions to convert values of the host-variable

   short      double2short(int Ndx=0);
   float      double2float(int Ndx=0);   
   int        double2int(int Ndx=0);
   unsigned   double2unsigned(int Ndx=0);
   long       double2long(int Ndx=0);
   
   char       cchar2char(int Ndx=0);
   
   
protected:

 char* name; // variable name
 ub1*  buf; //  pointer to the buffer
 ub1   buf_is_external; // use of an external buffer
 sword elem_size; // element size if the variable is a vector
 sword ftype; // variable type
 sb2*  indp; // indicator array
 ub2*  rlen; // returned length array
 ub2*  rcode; // returned code array
 sb2   vect_size; // vector size

 friend class OCI;
 friend class TSelectStm;

private:
 void InitVar(char* Name,sword FType,sword ElemSize,
              sword MaxVectSize,void* Buf);

};


#define NO_NAME           ((char*)0)
 //  use this macro, if a host variable has no name
#define EMPTY_LIST        ((PHostVarArray)0)
 // an array (list) of host variables is empty
#define END_OF_LIST       ((PHostVar)0)
 // end of variable list

const short SHOULD_DELETE=1; //  you "should delete"  an object
const short SHOULD_NOT_DELETE=0; // an object "should not been deleted"


typedef ub1 TDateIntern[7];
typedef ub1 TRowIdIntern[14];
typedef char TCCharRowId[19];
typedef ub1 TVarNumIntern[22];
typedef ub1 TNumberIntern[21];

//  C++ datatypes to contain the corresponding values
//  returned from ORACLE server

#ifdef __TEMPLATE_ENABLED__ //  ------ For ANSI C++ with templates -----

// Template class to construct numeric types of host variables, as well as
//  ORACLE ROWIDs, DATEs and VARNUMs

template <class T,
          const int TypeCode,
          const int type_Id
          TEMPLATE_INITIALIZER(type_GenericVar)
         >
class TGenericVar: public THostVar{
public:
   TGenericVar(char* Name,sword MaxVectSize,void* Buf=0):
      THostVar(Name,TypeCode,sizeof(T),MaxVectSize,Buf){};
   TGenericVar(char* Name,void* Buf):
      THostVar(Name,TypeCode,sizeof(T),Buf){};
   TGenericVar(sword MaxVectSize,void* Buf=0):
      THostVar(NO_NAME,TypeCode,sizeof(T),MaxVectSize,Buf){};
   TGenericVar(void* Buf):
      THostVar(NO_NAME,TypeCode,sizeof(T),Buf){};
   TGenericVar(void):
      THostVar(NO_NAME,TypeCode,sizeof(T),0){};
  int isA(void){ return type_Id;};

};


typedef TGenericVar<double,extFloat,type_TDouble> TDouble;
typedef TGenericVar<float,extFloat,type_TFloat> TFloat;
typedef TGenericVar<char,extInt,type_TSignedChar> TSignedChar;
typedef TGenericVar<short,extInt,type_TShortInt> TShortInt;
typedef TGenericVar<int,extInt,type_TInt> TOInt;
typedef TGenericVar<long int,extInt,type_TLongInt> TLongInt;
typedef TGenericVar<unsigned int,extInt,type_TUnsignedInt> TUnsignedInt;

// For more info on these types (above and below, also) see
// documenatation on Oracle Call Interface (OCI): an external data type table

typedef TGenericVar<TDateIntern,extDate,type_TDateTime> TDateTime;

// there's a little use of
// this type. As recommended
// in the OCI, it's better to convert an internal
// DATE type to strings.

typedef TGenericVar<TRowIdIntern,extRowId,type_RowId> TRowId;

// see comment for TDateTime.

typedef TGenericVar<TVarNumIntern,extVarNum,type_TVarNum> TVarNum;
typedef TGenericVar<TNumberIntern,extNumber,type_TNumber> TNumber;

// --------------------- Generic "String" classes ----------------------
// Template class to construct ORACLE string and byte string
// types statically (at compile time)

#define  STATIC_STRING_CLASS(ClassName,TypeCode,MaxStringSizeType)    \
   template <const MaxStringSizeType MaxStringSize,                   \
             const int type_Id                                        \
             TEMPLATE_INITIALIZER(type_##ClassName)                   \
            >                                                         \
   class ClassName: public THostVar{                                  \
   public:                                                            \
      ClassName(char* Name,sword MaxVectSize,void* Buf=0):            \
         THostVar(Name,TypeCode,MaxStringSize,MaxVectSize,Buf){};     \
      ClassName(char* Name,void* Buf):                                \
         THostVar(Name,TypeCode,MaxStringSize,Buf){};                 \
      ClassName(sword MaxVectSize,void* Buf=0):                       \
         THostVar(NO_NAME,TypeCode,MaxStringSize,MaxVectSize,Buf){};  \
               ClassName(void* Buf):                                  \
         THostVar(NO_NAME,TypeCode,MaxStringSize,Buf){};              \
      ClassName(void):                                                \
         THostVar(NO_NAME,TypeCode,MaxStringSize,0){};                \
      int isA(void){ return type_Id;};                                \
   }

   STATIC_STRING_CLASS(TVarChar2,extVarChar2,sword);
   STATIC_STRING_CLASS(TCChar,extCChar,sword);
      // CChar means a null terminated string
   STATIC_STRING_CLASS(TLong,extLong,sword);
   STATIC_STRING_CLASS(TVarChar,extVarChar,sword);
   STATIC_STRING_CLASS(TVarRaw,extVarRaw,sword);
   STATIC_STRING_CLASS(TRaw,extRaw,ub1);
   STATIC_STRING_CLASS(TLongRaw,extLongRaw,sword);
   STATIC_STRING_CLASS(TLongVarChar,extLongVarChar,sword);
   STATIC_STRING_CLASS(TLongVarRaw,extLongVarRaw,sword);
   STATIC_STRING_CLASS(TChar,extChar,ub1);
   STATIC_STRING_CLASS(TCharZ,extCharZ,sword);

// These are templates to generate concrete "String" classes:
// both character and
// byte strings (see OCI's external datatype table)

// ---------- Examples of the use of the generic string classes -----

// typedef TCChar<60> TEmployeeName;
// typedef TVarChar2<2000> TSummary;
// typedef TLong<16000> TDocumentText;


// If there's need to define your own type identifier for your string type,
// then use the generic templates with two parameters:

// const int type_TUserName=101; //  Use  integer values greater than 100,
//                               //  since the values  less than 100 are
//                               //  reserved
// typedef TCChar<60,type_TUserName> TUserName;

// ---------------------------------------------------------------------------
// Template class to construct ORACLE string and byte string types dynamically

#define  DYNAMIC_STRING_CLASS(ClassName,TypeCode,MaxStringSizeType,         \
                              type_ClassName)                               \
template <const int type_Id TEMPLATE_INITIALIZER(type_ClassName)>           \
class ClassName: public THostVar{                                           \
public:                                                                     \
   ClassName(char* Name,MaxStringSizeType MaxStringSize,sword MaxVectSize,  \
             void* Buf=0):                                                  \
      THostVar(Name,TypeCode,MaxStringSize,MaxVectSize,Buf){};              \
   ClassName(char* Name,MaxStringSizeType MaxStringSize,void* Buf):         \
      THostVar(Name,TypeCode,MaxStringSize,Buf){};                          \
   ClassName(MaxStringSizeType MaxStringSize,sword MaxVectSize,void* Buf=0):\
      THostVar(NO_NAME,TypeCode,MaxStringSize,MaxVectSize,Buf){};           \
   ClassName(MaxStringSizeType MaxStringSize,void* Buf):                    \
      THostVar(NO_NAME,TypeCode,MaxStringSize,Buf){};                       \
   ClassName(MaxStringSizeType MaxStringSize):                              \
      THostVar(NO_NAME,TypeCode,MaxStringSize,0){};                         \
   int isA(void){ return type_Id;};                                         \
}

   DYNAMIC_STRING_CLASS(yTVarChar2,extVarChar2,sword,type_TVarChar2);
   DYNAMIC_STRING_CLASS(yTCChar,extCChar,sword,type_TCChar);
      // CChar means a null terminated string
   DYNAMIC_STRING_CLASS(yTLong,extLong,sword,type_TLong);
   DYNAMIC_STRING_CLASS(yTVarChar,extVarChar,sword,type_TVarChar);
   DYNAMIC_STRING_CLASS(yTVarRaw,extVarRaw,sword,type_TVarRaw);
   DYNAMIC_STRING_CLASS(yTRaw,extRaw,ub1,type_TRaw);
   DYNAMIC_STRING_CLASS(yTLongRaw,extLongRaw,sword,type_TLongRaw);
   DYNAMIC_STRING_CLASS(yTLongVarChar,extLongVarChar,sword,type_TLongVarChar);
   DYNAMIC_STRING_CLASS(yTLongVarRaw,extLongVarRaw,sword,type_TLongVarRaw);
   DYNAMIC_STRING_CLASS(yTChar,extChar,ub1,type_TChar);
   DYNAMIC_STRING_CLASS(yTCharZ,extCharZ,sword,type_TCharZ);

// --------------- Examples of the use of the dynamic string classes --------

// typedef yTCChar<> TUserName;


// If there's need to define your own type identifier for your string type,
// then use the generic templates with two parameters:

// const int type_TUserName=101; //  Use  integer values greater than 100,
//                               // since the values  less than 100 are
//                               // reserved
// typedef yTCChar<type_TUserName> TUserName;


#else 

#define DEFINE_TGENERIC_VAR(T,TypeCode,type_Id,TGenericVar)        \
class TGenericVar: public THostVar{                                \
public:                                                            \
   TGenericVar(char* Name,sword MaxVectSize,void* Buf=0):          \
      THostVar(Name,TypeCode,sizeof(T),MaxVectSize,Buf){};         \
   TGenericVar(char* Name,void* Buf):                              \
      THostVar(Name,TypeCode,sizeof(T),Buf){};                     \
   TGenericVar(sword MaxVectSize,void* Buf=0):                     \
      THostVar(NO_NAME,TypeCode,sizeof(T),MaxVectSize,Buf){};      \
   TGenericVar(void* Buf):                                         \
      THostVar(NO_NAME,TypeCode,sizeof(T),Buf){};                  \
   TGenericVar(void):                                              \
      THostVar(NO_NAME,TypeCode,sizeof(T),0){};                    \
  int isA(void){ return type_Id;};                                 \
}


DEFINE_TGENERIC_VAR(double,extFloat,type_TDouble,TDouble);
DEFINE_TGENERIC_VAR(float,extFloat,type_TFloat,TFloat);
DEFINE_TGENERIC_VAR(char,extInt,type_TSignedChar,TSignedChar);
DEFINE_TGENERIC_VAR(short,extInt,type_TShortInt,TShortInt);
DEFINE_TGENERIC_VAR(int,extInt,type_TInt,TOInt);
DEFINE_TGENERIC_VAR(long int,extInt,type_TLongInt,TLongInt);
DEFINE_TGENERIC_VAR(unsigned int,extInt,type_TUnsignedInt,TUnsignedInt);

// For more info on these types (above and below, also) see
// documenatation on Oracle Call Interface (OCI): an external data type table

DEFINE_TGENERIC_VAR(TDateIntern,extDate,type_TDateTime,TDateTime);

// there's a little use of
// this type. As recommended
// in the OCI, it's better to convert an internal
// DATE type to strings.

DEFINE_TGENERIC_VAR(TRowIdIntern,extRowId,type_RowId,TRowId);

// see comment for TDateTime.

DEFINE_TGENERIC_VAR(TVarNumIntern,extVarNum,type_TVarNum,TVarNum);
DEFINE_TGENERIC_VAR(TNumberIntern,extNumber,type_TNumber,TNumber);

#define  DEFINE_STATIC_TCCHAR(ClassName,MaxStringSize)                \
   class ClassName: public THostVar{                                  \
   public:                                                            \
      ClassName(char* Name,sword MaxVectSize,void* Buf=0):            \
         THostVar(Name,extCChar,MaxStringSize,MaxVectSize,Buf){};     \
      ClassName(char* Name,void* Buf):                                \
         THostVar(Name,extCChar,MaxStringSize,Buf){};                 \
      ClassName(sword MaxVectSize,void* Buf=0):                       \
         THostVar(NO_NAME,extCChar,MaxStringSize,MaxVectSize,Buf){};  \
               ClassName(void* Buf):                                  \
         THostVar(NO_NAME,extCChar,MaxStringSize,Buf){};              \
      ClassName(void):                                                \
         THostVar(NO_NAME,extCChar,MaxStringSize,0){};                \
      int isA(void){ return type_TCChar;};                            \
   }

class TCChar: public THostVar{
public:
   TCChar(char* Name,sword MaxStringSize,sword MaxVectSize,
          void* Buf=0):
      THostVar(Name,extCChar,MaxStringSize,MaxVectSize,Buf){};
   TCChar(char* Name,sword MaxStringSize,void* Buf):
      THostVar(Name,extCChar,MaxStringSize,Buf){};
   TCChar(sword MaxStringSize,sword MaxVectSize,void* Buf=0):
      THostVar(NO_NAME,extCChar,MaxStringSize,MaxVectSize,Buf){};
   TCChar(sword MaxStringSize,void* Buf):
      THostVar(NO_NAME,extCChar,MaxStringSize,Buf){};
   TCChar(sword MaxStringSize):
      THostVar(NO_NAME,extCChar,MaxStringSize,0){};
   int isA(void){ return type_TCChar;};
};

#endif

#ifdef __BCPLUSPLUS__

// --------------------------------------------------------------------------
// Template classes to generate long (>64Kb) strings and byte strings

#define LONG_STRING_CLASS(TParentString,ByteType,type_TChildString,      \
                          TChildString,type_TParentString)               \
template <const long MaxStringSize,                                      \
          const int type_Id                                              \
          TEMPLATE_INITIALIZER(type_TChildString)                        \
         >                                                               \
class TChildString: public TParentString<-1,type_TParentString>{         \
public:                                                                  \
      TChildString(char* Name=0):                                        \
        TParentString<-1,type_TParentString>(Name,1,0)                   \
          {size=MaxStringSize;                                           \
           val=(ByteType _huge*)farmalloc(size);                         \
           buf=(ub1*)val;                                                \
           plen=(long_ndx*)val;                                          \
           SetLen(size);                                                 \
           real_val=(ByteType _huge*)(val+4);                            \
          };                                                             \
                                                                         \
      ~TChildString(){farfree(val);};                                    \
                                                                         \
      int isA(void){return type_TChildString;};                          \
                                                                         \
      long_ndx Len(void){return *plen;};                                 \
      void SetLen(long_ndx NewLen){*plen=NewLen;};                       \
      long_ndx Size(void){return size;};                                 \
      void SetSize(long_ndx NewSize){size=*plen=NewSize;};               \
                                                                         \
      ByteType _huge& operator[](long_ndx ndx){return real_val[ndx];};   \
                                                                         \
      TChildString& void  operator=(ByteType _huge* str)                 \
       {long_ndx i=0;while(((*this)[i]=str[i])!='\0')++i;SetLen(i);      \
        return this;};                                                   \
                                                                         \
      void SetVal(ByteType _huge* str,long_ndx strLen)                   \
       {for(long_ndx i=0;i<strLen;++i)(*this)[i]=str[i];SetLen(strLen);};\
                                                                         \
      void SetVal(ByteType _huge* str)                                   \
       {*this=str;};                                                     \
                                                                         \
      void GetVal(ByteType _huge* str,long_ndx& strLen)                  \
       {strLen=*plen;for(long_ndx i=0;i<strLen;++i)str[i]=(*this)[i];};  \
                                                                         \
      void GetVal(ByteType _huge* str)                                   \
      {long_ndx strLen=*plen;for(long i=0;i<strLen;++i)str[i]=(*this)[i];\
       str[strLen]='\0';};                                               \
                                                                         \
   char* Name(void){return name;};                                       \
   void SetName(char* Name){THostVar::SetName(Name);};                   \
   sword Ftype(void){return ftype;};                                     \
   void SetNull(void){indp[0]=-1;};                                      \
   int isNull(void){return indp[0]==-1;};                                \
   int isSuccess(void){return indp[0]==0;};                              \
   int isTruncated(void){return indp[0]==-2||indp[0]>0;};                \
   int isInvalidConversion(int Ndx){return rcode[Ndx]==1454;};           \
   int isUnsupportedDatatype(int Ndx){return rcode[Ndx]==3115;};         \
                                                                         \
   ub2   Rlen(void){return rlen[0];};                                    \
   ub2   Rcode(void){return rcode[0];};                                  \
   sword Rind(void){return indp[0];};                                    \
                                                                         \
/* This constructor is for internal use only */                          \
      TChildString(char* Name,sword MaxVectSize,TChildString*& Str):     \
        TParentString<-1,type_TParentString>(Name,1,0)                   \
          {size=MaxStringSize;                                           \
           val=(ByteType _huge*)farmalloc(size);                         \
           buf=(ub1*)val;                                                \
           plen=(long_ndx*)val;                                          \
           SetLen(size);                                                 \
           real_val=(ByteType _huge*)(val+4);                            \
           Str=this;                                                     \
          };                                                             \
private:                                                                 \
  ByteType _huge* val;                                                   \
  ByteType _huge* real_val;                                              \
  long_ndx*  plen;                                                       \
  long_ndx size;                                                         \
}

  LONG_STRING_CLASS(TLongVarChar,char,type_TLongString,
                    TLongString,type_TLongVarChar);
  LONG_STRING_CLASS(TLongVarRaw,unsigned char,type_TLongByteString,
                    TLongByteString,type_TLongVarRaw);

//
// typedef TLongString<100000L> TDocument;
// typedef TLongByteString<100000L> TImage;

#else
// ---------------------------------------------------------------------
// Long (>64Kb) strings

#define STATIC_LONG_STRING_CLASS(TString,MaxStringSize,ByteType,         \
                                 type_Id,TypeCode)                       \
class TString: public THostVar{                                          \
public:                                                                  \
      TString(char* Name=0):                                             \
        THostVar(Name,TypeCode,-1,1,0)                                   \
          {size=MaxStringSize;                                           \
           val=(ByteType _huge*)farmalloc(size);                         \
           buf=(ub1*)val;                                                \
           plen=(long_ndx*)val;                                          \
           SetLen(size);                                                 \
           real_val=(ByteType _huge*)(val+4);                            \
          };                                                             \
                                                                         \
      ~TString(){farfree(val);};                                         \
                                                                         \
      int isA(void){return type_Id;};                                    \
                                                                         \
      long_ndx Len(void){return *plen;};                                 \
      void SetLen(long_ndx NewLen){*plen=NewLen;};                       \
      long_ndx Size(void){return size;};                                 \
      void SetSize(long_ndx NewSize){size=*plen=NewSize;};               \
                                                                         \
      ByteType _huge& operator[](long_ndx ndx){return real_val[ndx];};   \
      TString& operator=(ByteType _huge* str)                            \
       {long_ndx i=0;while(((*this)[i]=str[i])!='\0')++i;SetLen(i);      \
        return *this;};                                                  \
                                                                         \
      void SetVal(ByteType _huge* str,long_ndx strLen)                   \
       {for(long_ndx i=0;i<strLen;++i)(*this)[i]=str[i];SetLen(strLen);};\
                                                                         \
      void SetVal(ByteType _huge* str)                                   \
       {*this=str;};                                                     \
                                                                         \
      void GetVal(ByteType _huge* str,long_ndx& strLen)                  \
       {strLen=*plen;for(long_ndx i=0;i<strLen;++i)str[i]=(*this)[i];};  \
                                                                         \
      void GetVal(ByteType _huge* str)                                   \
      {long_ndx strLen=*plen;for(long i=0;i<strLen;++i)str[i]=(*this)[i];\
       str[strLen]='\0';};                                               \
                                                                         \
   char* Name(void){return name;};                                       \
   void SetName(char* Name){THostVar::SetName(Name);};                   \
   sword Ftype(void){return ftype;};                                     \
   void SetNull(void){indp[0]=-1;};                                      \
   int isNull(void){return indp[0]==-1;};                                \
   int isSuccess(void){return indp[0]==0;};                              \
   int isTruncated(void){return indp[0]==-2||indp[0]>0;};                \
   int isInvalidConversion(int Ndx){return rcode[Ndx]==1454;};           \
   int isUnsupportedDatatype(int Ndx){return rcode[Ndx]==3115;};         \
                                                                         \
   ub2   Rlen(void){return rlen[0];};                                    \
   ub2   Rcode(void){return rcode[0];};                                  \
   sword Rind(void){return indp[0];};                                    \
                                                                         \
/* This constructor is for internal use only */                          \
      TString(char* Name,sword MaxVectSize,TString*& Str):               \
        THostVar(Name,TypeCode,-1,1,0)                                   \
          {size=MaxStringSize;                                           \
           val=(ByteType _huge*)farmalloc(size);                         \
           buf=(ub1*)val;                                                \
           plen=(long_ndx*)val;                                          \
           SetLen(size);                                                 \
           real_val=(ByteType _huge*)(val+4);                            \
           Str=this;                                                     \
          };                                                             \
private:                                                                 \
  ByteType _huge* val;                                                   \
  ByteType _huge* real_val;                                              \
  long_ndx*  plen;                                                       \
  long_ndx size;                                                         \
}

#define DYNAMIC_LONG_STRING(TString,ByteType,type_Id,TypeCode)           \
class TString: public THostVar{                                          \
public:                                                                  \
      TString(char* Name,long_ndx MaxStringSize):                        \
        THostVar(Name,TypeCode,-1,1,0)                                   \
          {size=MaxStringSize;                                           \
           val=(ByteType _huge*)farmalloc(size);                         \
           buf=(ub1*)val;                                                \
           plen=(long_ndx*)val;                                          \
           SetLen(size);                                                 \
           real_val=(ByteType _huge*)(val+4);                            \
          };                                                             \
                                                                         \
      ~TString(){farfree(val);};                                         \
                                                                         \
      int isA(void){return type_Id;};                                    \
                                                                         \
      long_ndx Len(void){return *plen;};                                 \
      void SetLen(long_ndx NewLen){*plen=NewLen;};                       \
      long_ndx Size(void){return size;};                                 \
      void SetSize(long_ndx NewSize){size=*plen=NewSize;};               \
                                                                         \
      ByteType _huge& operator[](long_ndx ndx){return real_val[ndx];};   \
      TString& operator=(ByteType _huge* str)                            \
       {long_ndx i=0;while(((*this)[i]=str[i])!='\0')++i;SetLen(i);      \
        return *this;};                                                  \
                                                                         \
      void SetVal(ByteType _huge* str,long_ndx strLen)                   \
       {for(long_ndx i=0;i<strLen;++i)(*this)[i]=str[i];SetLen(strLen);};\
                                                                         \
      void SetVal(ByteType _huge* str)                                   \
       {*this=str;};                                                     \
                                                                         \
      void GetVal(ByteType _huge* str,long_ndx& strLen)                  \
       {strLen=*plen;for(long_ndx i=0;i<strLen;++i)str[i]=(*this)[i];};  \
                                                                         \
      void GetVal(ByteType _huge* str)                                   \
      {long_ndx strLen=*plen;                                            \
       for(long_ndx i=0;i<strLen;++i)                                    \
        str[i]=(*this)[i];                                               \
       str[strLen]='\0';                                                 \
      };                                                                 \
                                                                         \
   char* Name(void){return name;};                                       \
   void SetName(char* Name){THostVar::SetName(Name);};                   \
   sword Ftype(void){return ftype;};                                     \
   void SetNull(void){indp[0]=-1;};                                      \
   int isNull(void){return indp[0]==-1;};                                \
   int isSuccess(void){return indp[0]==0;};                              \
   int isTruncated(void){return indp[0]==-2||indp[0]>0;};                \
   int isInvalidConversion(int Ndx){return rcode[Ndx]==1454;};           \
   int isUnsupportedDatatype(int Ndx){return rcode[Ndx]==3115;};         \
                                                                         \
   ub2   Rlen(void){return rlen[0];};                                    \
   ub2   Rcode(void){return rcode[0];};                                  \
   sword Rind(void){return indp[0];};                                    \
                                                                         \
private:                                                                 \
  ByteType _huge* val;                                                   \
  ByteType _huge* real_val;                                              \
  long_ndx*  plen;                                                       \
  long_ndx size;                                                         \
}

DYNAMIC_LONG_STRING(TLongString,char,type_TLongString,extLongVarChar);
DYNAMIC_LONG_STRING(TLongByteString,ub1,type_TLongByteString,extLongVarRaw);

#endif


// ************* Array of host variables *********************

class THostVarArray: public TObj{
public:

    THostVarArray(int Size=1,short ShouldDelete=1);
    THostVarArray(short ShouldDelete,PHostVar v,...);
    THostVarArray(PHostVar v,...); // for this constructor ShouldDelete==0
      // create array from the list of host variables.
      // Use END_OF_LIST  macro as a list terminator.

    THostVarArray(THostVarArray& v,short NewVectSize);
      // construct a new array by the array v, and make every host variable
      // in  the new array as vectors of the size NewVectSize.

    ~THostVarArray();

    int isA(void){ return type_THostVarArray;};

    THostVar& operator[](int ndx){return (THostVar&)*va[ndx];};
    THostVar& operator[](char* Name);
      // the overloaded operator [] for arrays returns reference to a host
      // variable by index [0..Size()-1] or by Name:
      //
      //   v[1]
      //   v["EMPNAME"], e.g. for a field from a select statement
      //   v[":ObjType"], e.g. for a placeholder

    int  Size(void){return valen;};
      // Size of array
    void SetHostVar(int ndx,PHostVar v){va[ndx]=v;};
      // set a host variable v to the array

   PHostVar GetHostVarPtr(int ndx){return va[ndx];};
      // Get Pointer to the HostVar


//  When pointers to the arrays of host variables are used as parameters
// in functions,
// use the EMPTY_LIST macro if you want to substitute an empty array.

private:
  PHostVar* va;
  int   valen;

  friend class OCI;
  friend class TStm;
  friend class TOCursor;
};

// Descriptor of output row in a SELECT statement. Contains
// an array of inFldDesc instances.

class inRowDesc:public TObj{
public:

  inRowDesc(){row_desc=0;row_len=0;};
  inRowDesc(CDA_DEF* cda);
  ~inRowDesc();
  int isA(void){ return type_inRowDesc;};

 int RowLen(void){return row_len;};
   // get row length
 PinFldDesc Describe(int FldPos){return row_desc[FldPos];};
   // "describe"  FldPos's field in the row
 inFldDesc& operator [](int FldPos){return (inFldDesc&)*row_desc[FldPos];};
   // same as the function "Describe"

private:
 PinFldDesc* row_desc;
 int row_len;
};

// NOTE: Generally, most of the functions below which have int as a return
// value,
// return 0 on success and nonzero values otherwise.

// Cover of the OCI functions

class OCI{
public:

  static LDA_DEF* RLogOn(char* UserID);
  static int RLogOn(LDA_DEF* lda,char* UserID);
  static LDA_DEF* LogOn(char* UserID);

  static int LogOut(LDA_DEF* lda);

  static int AutoCommitOff(LDA_DEF* lda);
  static int AutoCommitOn(LDA_DEF* lda);
  static int RollBack(LDA_DEF* lda);
  static int Commit(LDA_DEF* lda);
  static int Break(LDA_DEF* lda);

  static CDA_DEF* OpenCursor(LDA_DEF* lda);
  static int CloseCursor(CDA_DEF* cda);
  static int Option(CDA_DEF* cda,sword rbopt,sword waitopt);
  static int Cancel(CDA_DEF* cda);
  static int Exec(CDA_DEF* cda, short VectSize=1);
  static int Fetch(CDA_DEF* cda, short VectSize=1);
    // returns FetchOk,FetchEod,FetchErr
  static int Parse(CDA_DEF* cda,char* Stm,int& StmCode,int& ErrPos);
  static int Parse(CDA_DEF* cda,char* Stm);
  static inRowDesc* DescribeSelectList(CDA_DEF* cda)
         {return cda->stm_code==4?new inRowDesc(cda):0;};
      // Describe select item list. Pointer to a row descriptor is returned

  static THostVarArray* BindVarList(CDA_DEF* cda,short ShouldDelete,
                                    PHostVar v,...);
      //  Bind the list of host variables and group them into an array.
  static int RebindVar(CDA_DEF* cda,PHostVar v);
      //  rebind variable if needed.

  static THostVarArray* DefineSelectList(CDA_DEF* cda,short ShouldDelete,
                                         PHostVar v,...);
      // define select item list using a list of host variables and group
      // the list
      // to an array, that can be further used in a program.

  static int DefineSelectList(CDA_DEF* cda,PHostVarArray v);
      // define select item list using a host variable array
  static THostVarArray* DefineSelectList(CDA_DEF*cda,inRowDesc* row_desc,
                                         short VectSize=1);
      // define select item list using row_desc that has been obtained by the
      // DescribeSelectList function. That is define a select itme list,
      // using default
      // type conversions.

  static void  ClearErr(void);
      // clear error
  static int ErrCode(void){return err_code;};
      //  get error code
  static char* ErrMsg(void){return (char*)msg_buf;};
      // get error message
  static void SetErr(char* Msg,int Code);
      // simulate error
  static unsigned long RowCount(CDA_DEF* cda)
   {return (unsigned long)cda->row_count;};
      // get the "row count"  field in the Cda_Def structure
  static short RowFetched(CDA_DEF* cda)
   {return cda->row_fetched;};
      // get a number of fetched rows

private:

  friend class inFldDesc;

  static void get_error(CDA_DEF* cur);
  static void get_error(LDA_DEF* lda);

  static ora_text msg_buf[1000];
  static uword err_code;
};


// -----------------------------------------------------------------------
// Class "Connect"

class TConnect: public TObj{
public:

    TConnect();
    TConnect(char* UserId);
    ~TConnect();

    int LogOn(char* UserId);
    int LogOut(void);
    int isConnected(void){return is_connected;};

    int isA(void){return type_TConnect;};

    virtual void Commit(void){OCI::Commit(lda);};
    virtual void RollBack(void){OCI::RollBack(lda);};
    virtual void AutoCommitOn(void){OCI::AutoCommitOn(lda);};
    virtual void AutoCommitOff(void){OCI::AutoCommitOff(lda);};

    void Break(void){OCI::Break(lda);};

    int CursorsOpen(void){return cda_len;};
      // number of cursors opened for this Connect
    POCursor Cursor(int Ndx){return cda[Ndx];};
      // get the pointer to the cursor

    int ExecDirect(char* Stm); // directly execute a statement

protected:

  LDA_DEF* lda;
  POCursor cda[MAX_CURSOR_NUM];
  int cda_len;
  short is_connected;

  friend class TOCursor;
};

//  Class "Cursor"

class TOCursor: public TObj{
public:

    TOCursor(PConnect db,short ShouldDelete=1);
      // open a cursor in the "connect" object db.
    ~TOCursor();

    int isA(void){return type_TOCursor;};

    virtual int  Cancel(void){return OCI::Cancel(cda);};
    virtual int  Exec(short VectSize=1);
    virtual int  Parse(void);
    virtual int  Fetch(short VectSize=1);
      // returns FetchOk,FetchEod,FetchErr

    void Option(sword rbopt,sword waitopt){OCI::Option(cda,rbopt,waitopt);};

    virtual ub4   RowCount(void){return row_count;};
    virtual short RowFetched(void){return row_fetched;};
    PStm  Stm(void){return stm;};
      // returns pointer to a SQL statement attached to the cursor
    int   ErrPos(void){return err_pos;};
      // get error position a SQL statement has errors
    int   StmCode(void){return cda->stm_code;};
    // get an attached SQL statement code after
    //  the Parse function call.

    inRowDesc* RowDesc(void){return row_desc;};
      //  get a select item list descriptor

protected:

  CDA_DEF* cda;
  ub4 row_count;
  short row_fetched;
  int err_pos;
  PStm stm;
  inRowDesc* row_desc;

  friend class TConnect;
  friend class TStm;
};

// Class "SQL statement"

class TStm: public TObj{
public:

    TStm(POCursor cur,char* Stm,short ShouldDelete,
         PHostVarArray SelectList=0,
         PHostVarArray BoundVarList=0);
// creates the SQL statement "Stm" ,attaches it to the cursor "cur".
// defines the select item list "SelectList" if the statement
// is of type SELECT.
// binds the host variable list "BoundVarList" if it is defined.

    TStm(POCursor cur,char* Stm,
         short SelectListVectSize,
         short ShouldDelete=1,
         PHostVarArray BoundVarList=0);

// creates the SQL statement "Stm", attaches it to the cursor "cur",
// defines the default item list if the statement is of type SELECT,
//  binds the host variable list "BoundVarList" if it is defined.
// (!) Use this constructor if you want to use a default select item list


    TStm(POCursor cur,char* Stm);

    TStm(PConnect db,char* Stm,
         short ShouldDelete,
         PHostVarArray SelectList=0,
         PHostVarArray BoundVarList=0);
//  opens a cursor in the "db" object,
// creates the SQL statement "Stm" ,attaches it to the cursor,
// defines the select item list "SelectList" if the statement
// is of type SELECT.
// binds the host variable list "BoundVarList" if it is defined.

    TStm(PConnect db,char* Stm,
         short SelectListVectSize,
         short ShouldDelete=1,
         PHostVarArray BoundVarList=0);
//  opens a cursor in the "db" object,
// creates the SQL statement "Stm", attaches it to the cursor,
// defines the default item list if the statement is of type SELECT,
//  binds the host variable list "BoundVarList" if it is defined.
// (!) Use this constructor if you want to use a default select item list

    TStm(PConnect db,char* Stm);
//  opens a cursor in "db" , creates the SQL statement "Stm"
// attaches it to the cursor.

    TStm(){};

    ~TStm();

    int isA(void){return type_TStm;};

    void AttachCursor(POCursor cur);
      // attaches the statement to the cursor "cur"

    virtual int  Cancel(void)
      {return ((POCursor)owner)->Cancel();};
    virtual int  Exec(short VectSize=1)
      {return ((POCursor)owner)->Exec(VectSize);};
    virtual int  Parse(void)
      {return ((POCursor)owner)->Parse();};
    virtual int  Fetch(short VectSize=1)
      {return ((POCursor)owner)->Fetch(VectSize);};

    void Option(sword rbopt,sword waitopt)
      {((POCursor)owner)->Option(rbopt,waitopt);};

    virtual ub4   RowCount(void)
      {return ((POCursor)owner)->RowCount();};
    virtual short RowFetched(void)
      {return ((POCursor)owner)->RowFetched();};

    int   ErrPos(void){return ((POCursor)owner)->ErrPos();};
    int   StmCode(void){return ((POCursor)owner)->StmCode();};

    inRowDesc* RowDesc(void){return ((POCursor)owner)->RowDesc();};


    PHostVarArray BoundVarList(void){return bound_var_list;};
    PHostVarArray VarList(void){return bound_var_list;};
      //  returns a list of host variables bound to the statement
    PHostVarArray SelectList(void){return select_list;};
      // returns a select item list defined for the statement
    char* Stm(void){return stm;};
      // returns the pointer to the statement string

protected:

  char* stm;
  PHostVarArray select_list;
  PHostVarArray bound_var_list;
  short default_select_list;
  short select_list_vect_size;
  short cur_is_child;

  friend class TOCursor;
};

// Class "Cursor" specialized for SELECT statements

class TSelectCursor: public TOCursor{
public:

    TSelectCursor(PConnect db,short ShouldDelete=1);

    int isA(void){return type_TSelectCursor;};

    int  Exec(short VectSize=1){return 0;};
      //  Use functions GetFirst and GetNext instead of
      // Exec and Fetch for this class
    int  Fetch(short VectSize=1){return 0;};
    int  Parse(void);

    ub4   RowCount(void)
      {return real_row_count;};
    short RowFetched(void)
      {return real_row_fetched;};

    int  GetFirst(void);
      // execute the statement and  fetch the first row in the selection

    int  GetNext(void);
      // get next row

protected:

  short row_num;
  int last_fetch_rc;
  ub4 real_row_count;
  short real_row_fetched;

  friend class TSelectStm;
};

// Class "SQL statement" specialized for SELECT statements

class TSelectStm: public TStm{
public:

   TSelectStm(PSelectCursor cur,
              char* Stm,
              short ShouldDelete,
              short InternalVectSize,
              PHostVarArray SelectList,
              PHostVarArray BoundVarList
             );
      // see the class TStm. For a SELECT statement you should
      // define (in "SelectList" ) a select item list consisting of
      // scalar host variables
      // and specify an internal vector size by "InternalVectorSize"

   TSelectStm(PSelectCursor cur,
              char* Stm,
              short ShouldDelete=1,
              short InternalVectSize=1,
              PHostVarArray BoundVarList=0
             );
      //  see the class TStm.
      // (!) Use this constructor if you want to use
      // a default select item list


   TSelectStm(PConnect db,
              char* Stm,
              short ShouldDelete,
              short InternalVectSize,
              PHostVarArray SelectList,
              PHostVarArray BoundVarList
             );
      //  see the class TStm.

   TSelectStm(PConnect db,
              char* Stm,
              short ShouldDelete=1,
              short InternalVectSize=1,
              PHostVarArray BoundVarList=0
             );

    //  see the class TStm.
    // (!) Use this constructor if you want to use a default select item list


   ~TSelectStm();

  int isA(void){return type_TSelectStm;};


  int  Exec(short VectSize=1){return 0;}
  int  Parse(void){return ((PSelectCursor)owner)->Parse();};
  int  Fetch(short VectSize=1){return 0;};

  int  GetFirst(void){return ((PSelectCursor)owner)->GetFirst();};
  int  GetNext(void){return ((PSelectCursor)owner)->GetNext();};

  ub4   RowCount(void){return ((PSelectCursor)owner)->RowCount();};
  short RowFetched(void){return ((PSelectCursor)owner)->RowFetched();};


private:

  PHostVarArray internal_select_list;
  short     internal_vect_size;

  void CopyFrom(int Ndx);

  friend class TSelectCursor;

};

//  Class "Error"

class err{
public:
  static void clear(void){OCI::ClearErr();}; // clear error
  static int code(void){return OCI::ErrCode();}; // get error code
  static int on(void){return code()!=0;}; // test if an error occurred
  static char* msg(void){return OCI::ErrMsg();}; // get error message

  static void set(char* Msg,int Code){OCI::SetErr(Msg,Code);};
};

#ifdef __EXCEPTION_ENABLED__

// Oracle exception class

class oraException{
public:

  oraException(){};
  ~oraException(){};

  static void ora_enable(void){raise_mode=1;}; // make exceptions enabled
  static void ora_disable(void){raise_mode=0;}; // make exceptions disabled

  static void raise(void) // raise exception
    { if(nesting_lvl==1&&raise_mode&&err::on()){
      nesting_lvl=-100;
      throw oraException();
     }
    };

  static void raise1(void) // raise exception
    {if(raise_mode&&err::on()){
      nesting_lvl=-100;
      throw oraException();
     }
    };

// ---------- functions for internal use -------------

  static void  inc_lvl(void){++nesting_lvl;};
  static void  dec_lvl(void){--nesting_lvl;};
  static short get_raise_mode(void){return raise_mode;};

private:

  static short nesting_lvl;
  static short raise_mode;
};

// Macros for global enabling and disabling Oracle exceptions

#define EXCEPTION_ENABLE oraException::ora_enable();
#define EXCEPTION_DISABLE oraException::ora_disable();

#define EX_DISABLE  EXCEPTION_DISABLE
#define EX_ENABLE   EXCEPTION_ENABLE

// Raise exception
#define RAISE oraException::raise();
#define RAISE1 oraException::raise1();

// Macros to define catch block

#define BEGIN_CATCH  catch(oraException){
#define CATCH_ALL catch(...){
#define END_CATCH   err::clear();EXCEPTION_ENABLE;}

// "Epilog" Macro. It is used to begin an "epilog" section in
// a catch block. In the "epilog" section you can delete objects which
// you allocated in dynamic memory in the try block.

#define EPILOG EXCEPTION_DISABLE;


// Macros to define try block

#define BEGIN_TRY try{
#define END_TRY   }

#else
#include <setjmp.h>
extern jmp_buf jb;
void exception_enable(void);
void exception_disable(void);
void raise(void);
void raise2(void);

#define EXCEPTION_ENABLE  exception_enable();
#define EXCEPTION_DISABLE exception_disable(); 

#define EX_DISABLE  EXCEPTION_DISABLE
#define EX_ENABLE   EXCEPTION_ENABLE


#define BEGIN_CATCH  else{
#define CATCH_ALL   {
#define END_CATCH   err::clear();EXCEPTION_ENABLE;}

#define EPILOG EXCEPTION_DISABLE;

#define BEGIN_TRY if(setjmp(jb)==0){
#define END_TRY   }

#define RAISE  raise(); 
#define RAISE1 raise();

#endif


#define DELETE(p) delete p;p=0;
// delete p and assign zero to p

#define NEW(p)  p=0;p=new \
                          \
// assign zero to p then call "new"

#define DP(TP,p) TP p=0;
#define DEFINE_PTR(TP,p) DP(TP,p)
// define a pointer and initialize the pointer with zero

// Collection of macros

#include <oscl_def.h>

// Oracle stream classes

#include <oscl_str.h>

#endif
