;;; emx-test.el --- test program name handlers

;; Copyright (C) 1994-1995 Eberhard Mattes

;; Author: Eberhard Mattes <mattes@azu.informatik.uni-stuttgart.de>
;; Keywords: emx

;; This file is part of GNU Emacs.

;; GNU Emacs is free software; you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 2, or (at your option)
;; any later version.

;; GNU Emacs is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with GNU Emacs; see the file COPYING.  If not, write to
;; the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

;;; Code:

(defvar program-name-handler-test-cases
  nil
  "List of test cases for test-program-name-handlers.
Each element of the list is a list of two elements:
A list of arguments for pnh-test, and the expected vector to be returned
by pnh-test.")

(setq program-name-handler-test-cases
  '(
    ;;
    ;; 1. pnh-cmdproc, no arguments
    ;;
    (("cmd" nil)
     [(pnh-test) "cmd" nil pnh-cmdproc nil nil])
    ;;
    ;; 2. pnh-cmdproc, no -c
    ;;
    (("cmd" ("ab/cd"))
     [(pnh-test) "cmd" ("ab/cd") pnh-cmdproc nil nil])
    ;;
    ;; 3. pnh-cmdproc, -c, but no other arguments
    ;;
    (("cmd" ("-c"))
     [(pnh-test) "cmd" ("-c") pnh-cmdproc nil nil])
    ;;
    ;; 4. pnh-cmdproc, check translation of forward slashes
    ;;
    (("cmd" ("-c" "/abc/def ghi/jkl 123/456/789"))
     [(pnh-test) "cmd" ("/c" "\\abc\\def ghi/jkl 123/456/789")
      pnh-cmdproc nil nil])
    ;;
    ;; 5. Ditto, with exec
    ;;
    (("cmd" ("-c" "exec /abc/def ghi/jkl 123/456/789"))
     [(pnh-test) "cmd" ("/c" "\\abc\\def ghi/jkl 123/456/789")
      pnh-cmdproc nil nil])
    ;;
    ;; 6. pnh-cmdproc, "gunzip" -> "gzip -d", binary input
    ;;
    (("cmd" ("-c" "exec gunzip"))
     [(pnh-test) "cmd" ("/c" "gzip.exe -d") pnh-cmdproc t nil])
    ;;
    ;; 7. pnh-cmdproc, "gunzip" -> "gzip -d", full path name
    ;;
    (("cmd" ("-c" "exec c:/bin/gunzip.exe abc"))
     [(pnh-test) "cmd" ("/c" "c:\\bin\\gzip.exe -d abc") pnh-cmdproc t nil])
    ;;
    ;; 8. pnh-cmdproc, "gzip -d", option not at beginning, binary input
    ;;
    (("cmd" ("-c" "exec gzip -f -d abc"))
     [(pnh-test) "cmd" ("/c" "gzip -f -d abc") pnh-cmdproc t nil])
    ;;
    ;; 9. pnh-cmdproc, -d option after first file name, binary *output*
    ;;
    (("cmd" ("-c" "exec gzip -f abc -d"))
     [(pnh-test) "cmd" ("/c" "gzip -f abc -d") pnh-cmdproc nil t])
    ;;
    ;; 10. pnh-cmdproc, "exec" not dropped if /c (instead of -c) is used
    ;;
    (("cmd" ("/c" "exec gzip"))
     [(pnh-test) "cmd" ("/c" "exec gzip") pnh-cmdproc nil nil])
    ;;
    ;; 11. pnh-shell, "/bin/sh" -> "sh.exe", binary input
    ;;
    (("/bin/sh" ("-c" "exec gunzip abc"))
     [(pnh-test) "sh.exe" ("-c" "exec gzip.exe -d abc") pnh-shell t nil])
    ;;
    ;; 12. Ditto, binary output
    ;;
    (("/bin/sh" ("-c" "gzip abc"))
     [(pnh-test) "sh.exe" ("-c" "gzip abc") pnh-shell nil t])
    ;;
    ;; 13. pnh-shell, bash
    ;;
    (("bash.exe" ("-c" "gzip -d abc"))
     [(pnh-test) "bash.exe" ("-c" "gzip -d abc") pnh-shell t nil])
    ;;
    ;; 14. pnh-shell, bash, "gunzip" used as argument
    ;;
    (("bash.exe" ("-c" "test /gunzip"))
     [(pnh-test) "bash.exe" ("-c" "test /gunzip") pnh-shell nil nil])
    ;;
    ;; 15. pnh-shell, no arguments
    ;;
    (("bash.exe" nil)
     [(pnh-test) "bash.exe" nil pnh-shell nil nil])
    ;;
    ;; 16. pnh-gzip, binary output
    ;;
    (("gzip" ("file1"))
     [(pnh-test) "gzip" ("file1") pnh-gzip nil t])
    ;;
    ;; 17. pnh-gzip, binary input
    ;;
    (("gzip" ("-f" "-d" "file1"))
     [(pnh-test) "gzip" ("-f" "-d" "file1") pnh-gzip t nil])
    ;;
    ;; 18. pnh-gzip, -d after file name
    ;;
    (("gzip" ("file1" "-d"))
     [(pnh-test) "gzip" ("file1" "-d") pnh-gzip nil t])
    ;;
    ;; 19. pnh-gunzip
    ;;
    (("gunzip" ("file1"))
     [(pnh-test) "gzip.exe" ("-d" "file1") pnh-gzip t nil])
    ;;
    ;; 20. pnh-gunzip, with directory
    ;;
    (("c:/bin/gunzip" ("file1"))
     [(pnh-test) "c:/bin/gzip.exe" ("-d" "file1") pnh-gzip t nil])
    ;;
    ;; 21. sendmail
    ;;
    (("sendmail" ("-oi" "-t" "-f" "myname" "-oA" "alias" "-oem"))
     [(pnh-test) "sendmail" ("-t" "-f" "myname") pnh-sendmail nil nil])
    ;;
    ;; 22. tcp, without service (defaults to "nntp")
    ;;
    (("tcp" ("host"))
     [(pnh-test) "tcp" ("host") pnh-tcp t t])
    ;;
    ;; 23. tcp, with service name
    ;;
    (("tcp" ("host" "nntp"))
     [(pnh-test) "tcp" ("host" "nntp") pnh-tcp t t])
    ;;
    ;; 24. tcp, with port number
    ;;
    (("tcp" ("host" "119"))
     [(pnh-test) "tcp" ("host" "119") pnh-tcp t t])
    ;;
    ;; 25. tcp, with unknown service name
    ;;
    (("tcp" ("host" "unknown-service"))
     [(pnh-test) "tcp" ("host" "unknown-service") pnh-tcp nil nil])
    ))


(defun pnh-test (operation program args &optional handler)
  (let ((tail program-name-handler-alist)
	elt next-handler)
    (while (and (consp tail) (null next-handler))
      (setq elt (car tail))
      (if (and (string-match (car elt) program)
	       (not (memq (cdr elt) inhibit-program-name-handlers)))
	  (setq next-handler (cdr elt))
	(setq tail (cdr tail))))
    (if next-handler
	(funcall next-handler operation program args)
      (vector operation program args handler
	      binary-process-input binary-process-output))))


(defun test-program-name-handlers ()
  "Test program name handlers."
  (interactive)
  (let ((tail program-name-handler-test-cases)
        (testno 0)
	(binary-process-input nil)
	(binary-process-output nil)
        test ret)
    (with-output-to-temp-buffer
        "*Test results*"
      (while (consp tail)
        (setq test (car tail))
        (setq tail (cdr tail))
        (setq testno (1+ testno))
        (setq ret (apply 'pnh-test '(pnh-test) (nth 0 test)))
	(if (equal ret (nth 1 test))
            (princ (format "Test %d passed\n" testno))
          (princ (format
		  "Test %d failed:\n    result=%s\n  expected=%s\n"
		  testno ret (nth 1 test))))))))

;;; emx-test.el ends here
