/*
 * %W% %E% %U%  [EXTREL_1.2]
 *
 * VersaTrack orbit calculations are based on those that appear in Dr. Manfred
 * Bester's sattrack program (the Unix(tm) version, circa 1992).
 *
 * The data from which the maps where generated come from "xsat", an
 * X-Windows program by David A. Curry (N9MSW).
 *
 * Site coordinates come from various sources, including a couple of
 * World Almanacs, and also from both of the programs mentioned above.
 *
 * The following are authors' applicable copyright notices:
 *
 *                                                                               
 * Copyright (c) 1992, 1993, 1994 Manfred Bester. All Rights Reserved.        
 *                                                                           
 * Permission to use, copy, modify, and distribute this software and its      
 * documentation for educational, research and non-profit purposes, without   
 * fee, and without a written agreement is hereby granted, provided that the  
 * above copyright notice and the following three paragraphs appear in all    
 * copies.                                                                    
 *                                                                              
 * Permission to incorporate this software into commercial products may be    
 * obtained from the author, Dr. Manfred Bester, 1636 M. L. King Jr. Way,     
 * Berkeley, CA 94709, USA.                                                   
 *                                                                             
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE TO ANY PARTY FOR DIRECT, INDIRECT,  
 * SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF THE USE OF    
 * THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF THE AUTHOR HAS BEEN ADVISED   
 * OF THE POSSIBILITY OF SUCH DAMAGE.                                         
 *                                                                             
 * THE AUTHOR SPECIFICALLY DISCLAIMS ANY WARRANTIES, INCLUDING, BUT NOT       
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A    
 * PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS ON AN "AS IS"       
 * BASIS, AND THE AUTHOR HAS NO OBLIGATIONS TO PROVIDE MAINTENANCE, SUPPORT,  
 * UPDATES, ENHANCEMENTS, OR MODIFICATIONS.                                   
 *                                                                             
 *                                                                             
 * Copyright 1992 by David A. Curry                                            
 *                                                                             
 * Permission to use, copy, modify, distribute, and sell this software and its 
 * documentation for any purpose is hereby granted without fee, provided that  
 * the above copyright notice appear in all copies and that both that copyright
 * notice and this permission notice appear in supporting documentation.  The  
 * author makes no representations about the suitability of this software for  
 * any purpose.  It is provided "as is" without express or implied warranty.   
 *                                                                             
 * David A. Curry, N9MSW                                                       
 * Purdue University                                                           
 * Engineering Computer Network                                                
 * 1285 Electrical Engineering Building                                        
 * West Lafayette, IN 47907                                                    
 * davy@ecn.purdue.edu                                                         
 *                                                                             
 * VersaTrack Copyright (c) 1993, 1994 Siamack Navabpour. All Rights Reserved.
 *
 * Permission is hereby granted to copy, modify and distribute VersaTrack
 * in whole, or in part, for educational, non-profit and non-commercial use
 * only, free of charge or obligation, and without agreement, provided that
 * all copyrights and restrictions noted herein are observed and followed, and
 * additionally, that this and all other copyright notices listed herein
 * appear unaltered in all copies and in all derived work.
 *
 * This notice shall not in any way void or superceed any of the other authors
 * rights or privilages.
 *
 * VersaTrack IS PRESENTED FREE AND "AS IS", WITHOUT ANY WARRANTY OR SUPPORT.
 * YOU USE IT AT YOUR OWN RISK. The author(s) shall not be liable for any
 * direct, indirect, incidental, or consequential damage, loss of profits or
 * other tangible or intangible losses or benefits, arising out of or related
 * to its use. VersaTrack carries no warranty, explicit or implied, including
 * but not limited to those of merchantability and fitness for a particular
 * purpose.
 *
 * Siamack Navabpour, 12342 Hunter's Chase Dr. Apt. 2114, Austin, TX 78729.
 * sia@bga.com or sia@realtime.com.
 */


#include <windows.h>
#include <stdio.h>
#include <math.h>

#include "vstdefs.h"
#include "constant.h"
#include "vsttype.h"
#include "vstextrn.h"
#include "vstprop.h"

/*
 * changeNoradUnits: changes units of fundamental elements
 *      for use with NORAD model
 */

static void
changeNoradUnits(sp, p)
select_t *sp;
register satprop_t *p;
{
    register track_t *tp = sp->sl_tp;

    p->epoch0     = tp->epochday;
    p->ecc0       = tp->eccentricity;

    p->inc0       = tp->inclination;                         /* [rad]         */
    p->cosInc     = cos(tp->inclination);
    p->sinInc     = sin(tp->inclination);

    p->meanMot0   = tp->epochmeanmotion * TWOPI / MPD;       /* [rad/min]     */
    p->meanAn0    = tp->epochmeananomaly;                    /* [rad]         */
    p->argPer0    = tp->epochargperigee;                     /* [rad]         */
    p->raan0      = tp->epochRAAN;                           /* [rad]         */

    p->decRate    = tp->decayrate * TWOPI / MPD2;            /* [rad/min^2]   */
    p->decRateDot = tp->satp->s_dratedot * TWOPI / MPD3;     /* [rad/min^3]   */
    p->bStar      = tp->satp->s_bstarcoeff / AE;
}

/*
 * getSemiMajorAxis: recover original mean motion (meanMotDeep)
 *        and semimajor axis (sma0Deep) from input elements
 */

static void
getSemiMajorAxis(sp, p)
select_t *sp;
register satprop_t *p;
{    
    double rDelta0, rDelta1, rDelta12, rDelta13, rEccSQ, rEccCB;
    double rSma0, rSma1;
    
    changeNoradUnits(sp, p);

    p->cos2Inc       = SQR(p->cosInc);
    p->cos4Inc       = SQR(p->cos2Inc);

    p->x3thm1        = 3.0 * p->cos2Inc - 1.0;
    p->x7thm1        = 7.0 * p->cos2Inc - 1.0;
    p->x1m1th        = 1.0 - p->cos2Inc;
    p->x1m5th        = 1.0 - 5.0 * p->cos2Inc;

    rEccSQ           = SQR(p->ecc0);
    rEccCB           = p->ecc0 * rEccSQ;

    p->beta02        = 1.0 - rEccSQ;                     /* needed elsewhere */
    p->beta0         = sqrt(p->beta02);                  /* needed elsewhere */
    p->beta03        = p->beta0 * p->beta02;             /* needed elsewhere */

    rSma1            = pow((KE/p->meanMot0), TWOTHIRDS);

    rDelta1          = 1.5 * CK2 * p->x3thm1 / (SQR(rSma1) * p->beta03);
    rDelta12         = SQR(rDelta1);
    rDelta13         = rDelta1 * rDelta12;

    rSma0            = rSma1 * (1.0 - ONETHIRD * rDelta1 - rDelta12 
                                - 134.0/81.0 * rDelta13);

    rDelta0          = 1.5 * CK2 * p->x3thm1 / (SQR(rSma0) * p->beta03);

    p->meanMotDeep   = p->meanMot0 / (1.0 + rDelta0);
    p->sma0Deep      = rSma0 / (1.0 - rDelta0);
    p->semiMajorAxis = p->sma0Deep * EARTHRADIUS;

    p->deepSpaceFlag = (TWOPI / p->meanMotDeep >= 225.0) ? TRUE : FALSE;
}

/*
 * getTimeArgs: gets time arguments for the SGP4/SDP4 model calculations
 */

static void
getTimeArgs(tmArg, p)
double tmArg;
register satprop_t *p;
{
    p->tFP = (tmArg - p->epoch0) * MPD;                            /* [min] */
    p->tSQ = p->tFP * p->tFP;
    p->tCB = p->tSQ * p->tFP;
    p->tQD = p->tCB * p->tFP;
    p->tQN = p->tQD * p->tFP;
}

/*
 * checkPerigeeHeight: checks perigee height and make modifications
 *                     accordingly
 */

static void
checkPerigeeHeight(p)
register satprop_t *p;
{
    p->perigeeHeight = (p->sma0Deep * (1.0 - p->ecc0) - AE) * EARTHRADIUS;

    p->truncFlag     = (p->perigeeHeight < 220.0) ? TRUE : FALSE;
    p->satCrashFlag  = (p->perigeeHeight <   0.0) ? TRUE : FALSE;

    p->sStar = S;
    p->qmst4 = Q0MS0T4;

    if (p->perigeeHeight < 156.0) {
        p->sStar = p->perigeeHeight - S0;

        if (p->perigeeHeight <= 98.0)
            p->sStar = 20.0;

        p->qmst4  = pow(((Q0 - p->sStar) * AE / EARTHRADIUS),4.0);
        p->sStar /= EARTHRADIUS;
        p->sStar += AE;
    }
}

/*
 * calcCommonPerturb: calculates quantities that are common to the SGP4 and
 *                    SDP4 models
 */

static void
calcCommonPerturb(p)
register satprop_t *p;
{
    double rC2, rCoef1, rEtaSQ, rPInvSQ, rPsiSQ;
    
    rPInvSQ   = 1.0 / (SQR(p->sma0Deep) * SQR(p->beta02));
    p->xi     = 1.0 / (p->sma0Deep - p->sStar);
    p->eta    = p->sma0Deep * p->ecc0 * p->xi;
    rEtaSQ    = SQR(p->eta);
    p->eccEta = p->ecc0 * p->eta;
    rPsiSQ    = fabs(1.0 - rEtaSQ);
    p->coef0  = p->qmst4 * pow(p->xi, 4.0);
    rCoef1    = p->coef0 / pow(rPsiSQ, 3.5);

    rC2       = rCoef1 * p->meanMotDeep * 
                (p->sma0Deep * (1.0 + 1.5 * rEtaSQ + p->eccEta *
                (4.0 + rEtaSQ)) + 0.75 * CK2 * p->xi / rPsiSQ * p->x3thm1 *
                (8.0 + 3.0 * rEtaSQ * (8.0 + rEtaSQ)));

    p->c1     = p->bStar * rC2;

    p->c3     = p->coef0 * p->xi * A30CK2 * p->meanMotDeep * AE *
                p->sinInc / p->ecc0;

    p->c4     = 2.0 * p->meanMotDeep * rCoef1 * p->sma0Deep * p->beta02 * 
                (p->eta * (2.0 + 0.5 * rEtaSQ) + p->ecc0 *
                (0.5 + 2.0 * rEtaSQ) - 2.0 * CK2 * p->xi /
                (p->sma0Deep * rPsiSQ) * (-3.0 * p->x3thm1 *
                (1.0 - 2.0 * p->eccEta + rEtaSQ * 
                (1.5 - 0.5 * p->eccEta)) + 0.75 * p->x1m1th * 
                (2.0 * rEtaSQ - p->eccEta * (1.0 + rEtaSQ))
                * cos(2.0 * p->argPer0)));

    p->c5     = 2.0 * rCoef1 * p->sma0Deep * p->beta02 * 
                (1.0 + 2.75 * (rEtaSQ + p->eccEta) + p->eccEta * rEtaSQ);

    p->clc1   = 3.0  * CK2 * rPInvSQ * p->meanMotDeep;
    p->clc2   = p->clc1 * CK2 * rPInvSQ;
    p->clc3   = 1.25 * CK4 * SQR(rPInvSQ) * p->meanMotDeep;
}

/*
 * updateGravityAndDrag: updates quantities for secular gravity and
 *                       atmospheric drag
 */

static void
updateGravityAndDrag(p)
register satprop_t *p;
{
    double rClc0, rDelA, rDelAM, rDelArgPer, rDelE, rDelMeanAn;
    double rDelL, rD2, rD3, rD4, rT3Cof, rT4Cof, rT5Cof;
    double rMeanAnP, rSinMeanAnP, rSinMeanAn0;
    
    p->meanAnDot = p->meanMotDeep + 0.5 * p->clc1 * p->beta0 * p->x3thm1 +
                   0.0625 * p->clc2 * p->beta0 *
                   (13.0 - 78.0 * p->cos2Inc + 137.0 * p->cos4Inc);

    p->argPerDot = -0.5 * p->clc1 * p->x1m5th + 0.0625 * p->clc2 * 
                   (7.0 - 114.0 * p->cos2Inc + 395.0 * p->cos4Inc) + 
                   p->clc3 * (3.0 - 36.0 * p->cos2Inc + 49.0 * p->cos4Inc);

    p->raanDot   = -p->clc1 * p->cosInc + (0.5 * p->clc2 * (4.0 - 19.0 *
                   p->cos2Inc) + 2.0 * p->clc3 * (3.0 -  7.0 * p->cos2Inc)) *
                   p->cosInc;

    p->meanAnDF  = p->meanAn0 + p->meanAnDot * p->tFP;
    p->argPerDF  = p->argPer0 + p->argPerDot * p->tFP;
    p->raanDF    = p->raan0   + p->raanDot   * p->tFP;

    rDelA        = 1.0 - p->c1 * p->tFP;
    rDelE        = p->bStar * p->c4 * p->tFP;
    rDelL        = 1.5 * p->c1 * p->tSQ;

    if (!p->truncFlag) {
        p->c1SQ       = SQR(p->c1);
        rD2           = 4.0 * p->sma0Deep * p->xi * p->c1SQ;
        rClc0         = rD2 * p->xi * p->c1 / 3.0;
        rD3           = (17.0 * p->sma0Deep + p->sStar) * rClc0;
        rD4           = 0.5 * rClc0 * p->sma0Deep * p->xi * 
                        (221.0 * p->sma0Deep + 31.0 * p->sStar) * p->c1;

        rT3Cof        = rD2 + 2.0 * p->c1SQ;
        rT4Cof        = 0.25 * (3.0 * rD3 + p->c1 * (12.0 * rD2 + 10.0 *
                        p->c1SQ));
        rT5Cof        = 0.2 * (3.0 * rD4 + 12.0 * p->c1 * rD3 + 
                        6.0 * SQR(rD2) + 15.0 * p->c1SQ *
                        (2.0 * rD2 + p->c1SQ));

        p->argPerCof  = p->bStar * p->c3 * cos(p->argPer0);
        p->meanAnCof  = -TWOTHIRDS * p->coef0 * p->bStar * AE / p->eccEta;
        p->raanCof    = -3.5 * p->beta02 * p->clc1 * p->cosInc * p->c1;

        rDelArgPer    = p->argPerCof * p->tFP;
        rDelMeanAn    = p->meanAnCof *
                        ( pow(1.0 + p->eta * cos(p->meanAnDF),3.0) - 
                          pow(1.0 + p->eta * cos(p->meanAn0),3.0) );

        rDelAM        = rDelArgPer + rDelMeanAn;
        rMeanAnP      = p->meanAnDF + rDelAM;
        p->argPer     = p->argPerDF - rDelAM;

        rSinMeanAn0   = sin(p->meanAn0);
        rSinMeanAnP   = sin(rMeanAnP);

        rDelA        -= rD2 * p->tSQ - rD3 * p->tCB - rD4 * p->tQD;
        rDelE        += p->bStar * p->c5  *
                        (rSinMeanAnP - rSinMeanAn0);
        rDelL        += rT3Cof * p->tCB + rT4Cof * p->tQD +
                        rT5Cof * p->tQN;
    }
    else {
        rMeanAnP    = p->meanAnDF;
        p->argPer     = p->argPerDF;
    }

    p->raan    = p->raanDF - p->raanCof * p->tSQ;

    p->smallE  = p->ecc0 - rDelE;
    p->smallA  = p->sma0Deep * SQR(rDelA);

    p->xL      = rMeanAnP + p->argPer + p->raan + p->meanMotDeep * rDelL;
    p->beta    = sqrt(1.0 - SQR(p->smallE));
    p->meanMot = KE / pow(p->smallA, 1.5);

    p->perigeeHeight  = (p->smallA * (1.0 - p->ecc0) - AE) * EARTHRADIUS; /*km*/

    p->curArgPerigeeX = p->argPer;                              /* [rad]   */
    p->curRaanX       = p->raan;                                /* [rad]   */
    p->curMotionX     = p->meanMot * MPD / TWOPI;               /* [rev/d] */
}

/*
 * updateLongPeridics: updates long periodic quantities
 */

static void
updateLongPeriodics(p)
register satprop_t *p;
{
    double rUpd6, rAyCof, rXLCoef, rXLL;
    
    rUpd6    = 1.0 / (p->smallA * SQR(p->beta));

    p->axN   = p->smallE * cos(p->argPer);

    rXLCoef  = 0.125 * A30CK2 * p->sinInc * (3.0 + 5.0 * p->cosInc) /
               (1.0 + p->cosInc);
    rXLL     = rUpd6 * rXLCoef * p->axN;
    p->xLT   = p->xL + rXLL;

    rAyCof   = 0.250 * A30CK2 * p->sinInc;
    p->ayNL  = rUpd6 * rAyCof;
    p->ayN   = p->smallE * sin(p->argPer) + p->ayNL;
}

/*
 * updateShortPeridics: gets osculating quantities by adding delta terms
 */

static void
updateShortPeriodics(p)
register satprop_t *p;
{
    double rCosU, rCos2U, rDelInc, rDelR, rDelRDot, rDelRFDot, rDelRaan, rDelU;
    double rESinE, rECosE, rEL2, rPL, rSinU, rSin2U, rSmallR, rSmallU;
    double rUpd1, rUpd2, rUpd3, rUpd4, rUpd5;
    
    rECosE      = p->kep4 + p->kep5;
    rESinE      = p->kep2 - p->kep3;

    rEL2        = SQR(p->axN) + SQR(p->ayN);
    rUpd1       = 1.0 - rEL2;
    rPL         = p->smallA * rUpd1;
    p->betaL    = sqrt(rUpd1);

    rSmallR     = p->smallA * (1.0 - rECosE);

    p->rDot     = KE * sqrt(p->smallA) / rSmallR * rESinE;
    p->rfDot    = KE * sqrt(rPL) / rSmallR;

    rUpd2       = p->smallA / rSmallR;
    rUpd3       = 1.0 / (1.0 + p->betaL);

    rCosU       = rUpd2 * (p->cosEpAP - p->axN + p->ayN * rESinE * rUpd3);
    rSinU       = rUpd2 * (p->sinEpAP - p->ayN - p->axN * rESinE * rUpd3);

    rSin2U      = 2.0 * rSinU * rCosU;
    rCos2U      = 2.0 * rCosU * rCosU - 1.0;

    rSmallU     = atan2(rSinU, rCosU);
    rSmallU     = reduce(rSmallU,ZERO,TWOPI);

    rUpd4       = CK2 / SQR(rPL);
    rUpd5       = CK2 / rPL;

    rDelR       = 0.5 * rUpd5 * p->x1m1th * rCos2U;
    rDelU       = -0.25 * rUpd4 * p->x7thm1 * rSin2U;

    rDelRaan    = 1.5 * rUpd4 * p->cosInc * rSin2U;
    rDelInc     = 1.5 * rUpd4 * p->cosInc * p->sinInc * rCos2U;

    rDelRDot    = -p->meanMot * rUpd5 * p->x1m1th * rSin2U;
    rDelRFDot   = p->meanMot * rUpd5 * (p->x1m1th * rCos2U + 1.5
                  * p->x3thm1);

    p->rk       = rSmallR * (1.0 - 1.5 * rUpd4 * p->betaL * p->x3thm1) +
                  rDelR;

    p->uk       = rSmallU + rDelU;
    p->raanK    = p->raan   + rDelRaan;
    p->incK     = p->inc0   + rDelInc;
    p->rkDot    = p->rDot   + rDelRDot;
    p->rfkDot   = p->rfDot  + rDelRFDot;

    p->curArgNodeX = p->uk;
}

/*
 * keplerX: solve Kepler's equation
 */

static void
keplerX(p)
register satprop_t *p;
{
    double rCapU, rCapEpAP, rSinEpAP, rCosEpAP, diffAnom, eccAnom;
    double rKep1, rKep2, rKep3, rKep4, rKep5;

    rCapU    = reduce(p->xLT - p->raan, ZERO, TWOPI);
    rCapEpAP = rCapU;

    do {
        rKep1    = rCapEpAP;
        rSinEpAP = sin(rKep1);
        rCosEpAP = cos(rKep1);

        rKep2    = p->axN * rSinEpAP;
        rKep3    = p->ayN * rCosEpAP;
        rKep4    = p->axN * rCosEpAP;
        rKep5    = p->ayN * rSinEpAP;

        rCapEpAP = rKep1 + (rCapU - rKep1 + rKep2 - rKep3) / 
                           (1.0 - rKep4 - rKep5);

        diffAnom = rCapEpAP - rKep1;

    } while (fabs(diffAnom) >= ONEPPM);               /* 0.2 arcsec precision */

    p->sinEpAP = rSinEpAP;
    p->cosEpAP = rCosEpAP;
#if 0
    p->kep1    = rKep1;
#endif
    p->kep2    = rKep2;
    p->kep3    = rKep3;
    p->kep4    = rKep4;
    p->kep5    = rKep5;

    eccAnom = rCapEpAP - p->argPer;

    if (fabs(eccAnom) < ONEPPM)
        p->trueAnomalyX = PI;
    else
        p->trueAnomalyX = 2.0 * atan(sqrt((1.0 + p->ecc0) /
                            (1.0 - p->ecc0)) * tan(eccAnom / 2.0));

    p->trueAnomalyX = reduce(p->trueAnomalyX, ZERO, TWOPI);
}

#if 0
/*
 * getOrbitNumberX: gets orbit number and mean anomaly
 */

static void
getOrbitNumberX(tArg, sp, p)
double tArg;
select_t *sp;
satprop_t *p;
{
    double dT, of, curOrbit, fdummy;
    track_t *tp = sp->sl_tp;

    dT             = tArg - p->epoch0;

    tp->reforbit   = (double) tp->epochorbitnum + tp->epochmeananomaly
                     / TWOPI;                                       /* [rev] */
    curOrbit       = tp->reforbit + p->curMotionX * dT -
                     (p->curMotionX - tp->epochmeanmotion) * dT;

    tp->orbitnum   = (int) curOrbit + 1L;
    of             = modf(curOrbit, &fdummy);
    tp->orbitfrac  = (int) (of * 100.0);
    tp->meananomaly= of * TWOPI;
}

#endif


/*
 * getStateVectorX: calculates position and velocity components
 */

static void
satposX(sp, p, ttime, X, Y, Z, Radius, vX, vY, vZ)
select_t *sp;
satprop_t *p;
double ttime, *X, *Y, *Z, *Radius, *vX, *vY, *vZ;
{
    double mVec[3], nVec[3], uVec[3], vVec[3], satVelS[3], satPosS[3];
    double cosIK, cosRK, cosUK, sinIK, sinRK, sinUK;
    double dT, curOrbit, fdummy, of;
    track_t *tp = sp->sl_tp;
    int    i;
    extern SquintAngle(select_t *, double, double, double, double);

    cosUK   = cos(p->uk);
    sinUK   = sin(p->uk);
    cosIK   = cos(p->incK);
    sinIK   = sin(p->incK);
    cosRK   = cos(p->raanK);
    sinRK   = sin(p->raanK);

    mVec[0] = -sinRK * cosIK;
    mVec[1] = cosRK * cosIK;
    mVec[2] = sinIK;

    nVec[0] = cosRK;
    nVec[1] = sinRK;
    nVec[2] = 0.0;

    for (i = 0; i <= 2; i++) {
        uVec[i]     = mVec[i] * sinUK + nVec[i] * cosUK;
        vVec[i]     = mVec[i] * cosUK - nVec[i] * sinUK;
        satPosS[i]  = (p->rk * uVec[i]) * EARTHRADIUS;
        satVelS[i]  = (p->rkDot * uVec[i] + p->rfkDot * vVec[i]) * (EARTHRADIUS / 60.0);
    }

    *X = satPosS[0];
    *Y = satPosS[1];
    *Z = satPosS[2];

    *vX = satVelS[0];
    *vY = satVelS[1];
    *vZ = satVelS[2];

    dT              = ttime - p->epoch0;
    tp->reforbit    = (double) tp->epochorbitnum + tp->epochmeananomaly
                      / TWOPI;                                       /* [rev] */

    curOrbit        = tp->reforbit + p->curMotionX * dT -
                      (p->curMotionX - tp->epochmeanmotion) * dT;

    tp->orbitnum    = (int) curOrbit + 1L;
    of              = modf(curOrbit, &fdummy);
    tp->orbitfrac   = (int) (of * 100.0);
    tp->meananomaly = of * TWOPI;
    tp->trueanomaly = p->trueAnomalyX;

    *Radius = tp->semimajoraxis * (1.0 - SQR(tp->eccentricity))
              / (1.0 + (tp->eccentricity * cos(tp->trueanomaly)));

    SquintAngle(sp, tp->trueanomaly, p->curRaanX, p->curArgPerigeeX, *Radius);
}

/*
 * calcSGP4: calculates SGP4 propagation model
 *
 * For perigee less than 220 km, the truncFlag is set and the equations for
 * 'smallA' and 'xL' are truncated after the 'c1' term, and the terms
 *  involving 'c5', 'deltaArgPer' and 'deltaMeanAn' are dropped.
 *
 * For a perigee below 156 km, the values of S and Q0MS0T4 are altered.
 * (see 'checkPerigeeHeight()')
 *
 */

void
satposSGP4(sp, ttime, X, Y, Z, Radius, vX, vY, vZ)
select_t *sp;
double ttime, *X, *Y, *Z, *Radius, *vX, *vY, *vZ;
{
    /*
     * HUEMONGUS STRUCT WARNING - BEWARE OF STACK OVERFLOW!
     */
    satprop_t prop_params;

    memset(&prop_params, 0, sizeof(prop_params));
    getSemiMajorAxis(sp, &prop_params);
        
    checkPerigeeHeight(&prop_params);

    if (prop_params.satCrashFlag)
        return;

    getTimeArgs(ttime, &prop_params);
    calcCommonPerturb(&prop_params);
    updateGravityAndDrag(&prop_params);
    updateLongPeriodics(&prop_params);
    keplerX(&prop_params);
    updateShortPeriodics(&prop_params);
    satposX(sp, &prop_params, ttime, X, Y, Z, Radius, vX, vY, vZ);
}
