//-------------------------------------------------------------------------
// The following is part of the VIOWIN user-interface library source code.
// The code is copyright (c) 1994-1995 by Larry Salomon, Jr. and may not
// be used in a for-profit application without the expressed, written
// consent of Larry Salomon, Jr.  All rights reserved.
//-------------------------------------------------------------------------
#define INCL_DOSDATETIME
#define INCL_DOSERRORS
#define INCL_DOSPROCESS
#define INCL_DOSSEMAPHORES
#include "vwdefs.h"
#include <stdlib.h>
#include <time.h>

typedef VOID (* _Optlink PFNTIMER)(PVOID);

VOID _Optlink timerThread(PVWTIMERINFO ptiInfo)
//-------------------------------------------------------------------------
// This is the function which does "the hard work" of the timer
// processing.  It creates a public event semaphore and then an
// asynchronous timer whose timeout is the value specified on the
// vwStartTimer() call.  The reason that a one-shot asynchronous
// timer is used instead of a repeating timer is because the user
// can call vwStartTimer() again with the same timer ID to change
// the timeout value.  So, after each timeout, we need to recreate
// the timer with the possibly changed timeout value.
//
// Input:  ptiInfo - points to a VWTIMERINFO structure containing
//                   information about the timer.
//-------------------------------------------------------------------------
{
   BOOL bError;
   CHAR achSemName[256];
   HEV hsmTimeout;
   HTIMER htTimeout;
   APIRET arRc;
   ULONG ulPost;
   BOOL bNeedMsg;
   ULONG ulIndex;
   PVWQMSG pqmMsg;

   bError=TRUE;

   //----------------------------------------------------------------------
   // Generate a unique semaphore name and create a public event semaphore
   //----------------------------------------------------------------------
   sprintf(achSemName,"\\SEM32\\VIOWIN\\%08lX",(ULONG)time(NULL));

   if (DosCreateEventSem(achSemName,&hsmTimeout,DC_SEM_SHARED,FALSE)) {
      goto EXIT_PROC;
   } /* endif */

   //----------------------------------------------------------------------
   // Start the timer.
   //----------------------------------------------------------------------
   if (DosAsyncTimer(ptiInfo->ulTimeout,(HSEM)hsmTimeout,&htTimeout)) {
      DosCloseEventSem(hsmTimeout);
      goto EXIT_PROC;
   } /* endif */

   //----------------------------------------------------------------------
   // Loop until we are told to stop by vwStopTimer().
   //----------------------------------------------------------------------
   while (!ptiInfo->bStop) {
      arRc=DosWaitEventSem(hsmTimeout,SEM_IMMEDIATE_RETURN);

      //-------------------------------------------------------------------
      // If the event semaphore was posted, we might need to post a
      // WM_TIMER message.
      //-------------------------------------------------------------------
      if (arRc==0) {
         DosTimerStop(htTimeout);
         DosResetEventSem(hsmTimeout,&ulPost);

         //----------------------------------------------------------------
         // Assume innocent until proven guilty:  check the message queue
         // for a previous WM_TIMER message.  If one was not found, post
         // one, otherwise do nothing.
         //----------------------------------------------------------------
         bNeedMsg=TRUE;
         ulIndex=hmqQueue->ulHead;

         while (ulIndex!=hmqQueue->ulTail) {
            pqmMsg=&hmqQueue->aqmMsgs[ulIndex];

            if ((pqmMsg->hwndWnd==ptiInfo->hwndWnd) &&
                (pqmMsg->ulMsg==WM_TIMER)) {
               bNeedMsg=FALSE;
            } /* endif */

            ulIndex++;
            if (ulIndex==VW_SIZEQUEUE) {
               ulIndex=0;
            } /* endif */
         } /* endwhile */

         if (bNeedMsg) {
            vwPostMsg(ptiInfo->hwndWnd,WM_TIMER,MPFROMLONG(ptiInfo->ulId),0);
         } /* endif */

         //----------------------------------------------------------------
         // Start the timer again.
         //----------------------------------------------------------------
         if (DosAsyncTimer(ptiInfo->ulTimeout,(HSEM)hsmTimeout,&htTimeout)) {
            DosCloseEventSem(hsmTimeout);
            goto EXIT_PROC;
         } /* endif */
      } else
      if (arRc!=ERROR_TIMEOUT) {
         DosTimerStop(htTimeout);
         DosCloseEventSem(hsmTimeout);
         goto EXIT_PROC;
      } /* endif */

      DosSleep(0);
   } /* endwhile */

   DosTimerStop(htTimeout);
   DosCloseEventSem(hsmTimeout);

   bError=FALSE;

   EXIT_PROC:
   if (bError) {
      CmnLstDeleteRecord(habAnchor->hclTimers,ptiInfo);
   } /* endif */

   DosEnterCritSec();

   if (!bError) {
      ptiInfo->tidThread=-1;
   } /* endif */
}

SHORT EXPENTRY findTimer(PVWTIMERINFO ptiFirst,PVWTIMERINFO ptiSecond)
//-------------------------------------------------------------------------
// This is a callback for CmnLstSearchRecord() to find the timer with
// the associated window handle and identifier.
//
// Input:  ptiFirst - points to a record in the timer linked list
//         ptiSecond - points to the search information
// Returns:  0 if matched, 1 otherwise.
//-------------------------------------------------------------------------
{
   if ((ptiFirst->hwndWnd==ptiSecond->hwndWnd) &&
       (ptiFirst->ulId==ptiSecond->ulId)) {
      return 0;
   } else {
      return 1;
   } /* endif */
}

BOOL EXPENTRY vwStartTimer(HVWWND hwndWnd,ULONG ulId,ULONG ulTimeout)
//-------------------------------------------------------------------------
// This function starts a new timer, or changes the timeout of an
// existing timer.  See WinStartTimer() for more information.
//
// Input:  hwndWnd - handle of the window to which the timer will belong.
//         ulId - identifier of the timer
//         ulTimeout - timeout in milliseconds between WM_TIMER messages.
// Returns:  TRUE if successful, FALSE otherwise
//
// If a timer with the same hwndWnd and ulId exists, this function simply
// changes the timeout value.
//-------------------------------------------------------------------------
{
   VWTIMERINFO tiNew;
   PVWTIMERINFO ptiFound;
   ULONG ulNumTimers;

   if (!vwIsWindow(hwndWnd)) {
      return FALSE;
   } /* endif */

   tiNew.hwndWnd=hwndWnd;
   tiNew.ulId=ulId;
   tiNew.ulTimeout=ulTimeout;
   tiNew.bStop=FALSE;
   tiNew.tidThread=-1;

   //----------------------------------------------------------------------
   // Search for an already existing timer
   //----------------------------------------------------------------------
   ptiFound=CmnLstQueryRecord(habAnchor->hclTimers,0);
   ptiFound=(PVWTIMERINFO)CmnLstSearchRecord(ptiFound,
                                             &tiNew,
                                             (PFNRECCOMP)findTimer);
   if (ptiFound!=NULL) {
      ptiFound->ulTimeout=ulTimeout;
      return TRUE;
   } /* endif */

   //----------------------------------------------------------------------
   // See if we've exceeded the timer limit.  Since the linked-lists
   // do not have a fixed-size, this is purely for backward compatibility.
   //----------------------------------------------------------------------
   ulNumTimers=CmnLstQueryRecordCount(habAnchor->hclTimers);
   if (ulNumTimers>=vwQuerySysValue(VWSV_CTIMERS)) {
      return FALSE;
   } /* endif */

   //----------------------------------------------------------------------
   // Add the record first, and store the pointer to the new record in
   // the list.  This will be passed to the timer thread, which gets
   // created next.
   //----------------------------------------------------------------------
   if (!CmnLstAddRecord(habAnchor->hclTimers,
                        &tiNew,
                        LAR_TAIL,
                        (PPVOID)&ptiFound)) {
      return FALSE;
   } /* endif */

   ptiFound->tidThread=_beginthread((PFNTIMER)timerThread,
                                    NULL,
                                    0x4000,
                                    ptiFound);
   if (ptiFound->tidThread==(TID)-1) {
      CmnLstDeleteRecord(habAnchor->hclTimers,ptiFound);
      return FALSE;
   } /* endif */

   return TRUE;
}

BOOL EXPENTRY vwStopTimer(HVWWND hwndWnd,ULONG ulId)
//-------------------------------------------------------------------------
// This function stops an already started timer.
//
// Input:  hwndWnd - handle of the window to which the timer will belong.
//         ulId - identifier of the timer
// Returns:  TRUE if successful, FALSE otherwise
//-------------------------------------------------------------------------
{
   VWTIMERINFO tiSearch;
   PVWTIMERINFO ptiFound;

   if (!vwIsWindow(hwndWnd)) {
      return FALSE;
   } /* endif */

   tiSearch.hwndWnd=hwndWnd;
   tiSearch.ulId=ulId;
   tiSearch.ulTimeout=0;
   tiSearch.bStop=FALSE;
   tiSearch.tidThread=-1;

   //----------------------------------------------------------------------
   // Search for the timer and return an error if not found.
   //----------------------------------------------------------------------
   ptiFound=CmnLstQueryRecord(habAnchor->hclTimers,0);
   ptiFound=(PVWTIMERINFO)CmnLstSearchRecord(ptiFound,
                                             &tiSearch,
                                             (PFNRECCOMP)findTimer);
   if (ptiFound==NULL) {
      return FALSE;
   } /* endif */

   //----------------------------------------------------------------------
   // Tell the thread to stop and wait for it to comply.
   //----------------------------------------------------------------------
   ptiFound->bStop=TRUE;

   while (ptiFound->tidThread!=(TID)-1) {
      DosSleep(0);
   } /* endwhile */

   CmnLstDeleteRecord(habAnchor->hclTimers,ptiFound);
   return TRUE;
}
