Listing One

/*  EGGCARTN.C - generates a sample PCX depth map file called "eggcartn.pcx" 
for testing 3D.C stereogram program. Command-line param 0-15 generates 
different surfaces */

#include <stdio.h>
#include <graphics.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include <conio.h>

#define MAX_X   640
#define MAX_Y   480
#define BPP     80      /* bytes per plane */
#define NPLANES 4       /* number of color planes */

void main(int argc,char **argv);
void doodle(void);
void dump_screen(void);
void put_line(unsigned char *data,int cnt);
void out_run(int byte,int rep_cnt);
void open_pcx(char *name);
void init_graphics(void);
double func(int val);

int mode;
double eggsz = 200.0;

/*      main        */
void main(int argc,char **argv)
{
    if(argc == 2)
        mode = atoi(argv[1]);
    open_pcx("eggcartn.pcx");
    init_graphics();
    doodle();
    dump_screen();
    closegraph();
    printf("Output in 'eggcartn.pcx'.\n");
    exit(0);
}  
/*      doodle      */
void doodle(void)
{
    int x,y,color;
    double xf,yf;

    if(mode & 4) eggsz *= 2.0;
    for(y = 0; y < MAX_Y; y++) {
        if(kbhit()) {
            getch();
            closegraph();
            exit(0);
        }
        yf = func(y);
        for(x = 0; x < MAX_X; x++) {
            xf = func(x);
            color = (int)(7.99 * (1 + xf * yf));
            putpixel(x,y,color);
        }
    }
}
/*      func    */
double func(int val)
{
    double res;
    
    res = mode & 1 ? val % (int) eggsz : val;
    if(mode & 2) {
        res /= eggsz / 2.0;
        res -= 1.0;
    }
    else {
        res *= M_PI / eggsz;
        res = cos(res);
    }
    return(mode & 8 ? res * 1.666 : res);
}
/* stuff for PCX dump routines */
struct pcx_header {
    char manufacturer, version, encoding, bits_per_pixel;
    int xmin, ymin, xmax, ymax, hres, vres;
    char colormap[16 * 3];
    char reserved, num_planes;
    int bytes_per_line, palette_code;
    char filler[58];
} header = {
    10,             /* manu */      5,              /* version */
    1,              /* encoding */  1,              /* bits per pixel */
    0,              /* xmin */      0,              /* ymin */
    639,            /* xmax */      479,            /* ymax */
    800,            /* src hres */  600,            /* src vres */
    0,0,0,          /* color 0 */   0x80,0,0,       /* color 1 */
    0,0x80,0,       /* color 2 */   0x80,0x80,0,    /* color 3 */
    0,0,0x80,       /* color 4 */   0x80,0,0x80,    /* color 5 */
    0,0x80,0x80,    /* color 6 */   0x80,0x80,0x80, /* color 7 */
    0xc0,0xc0,0xc0, /* color 8 */   0xff,0,0,       /* color 9 */
    0,0xff,0,       /* color 10 */  0xff,0xff,0,    /* color 11 */
    0,0,0xff,       /* color 12 */  0xff,0,0xff,    /* color 13 */
    0,0xff,0xff,    /* color 14 */  0xff,0xff,0xff, /* color 15 */
    0,              /* reserved */  4,              /* # planes */
    80,             /* bytes per line */
};
unsigned char planes[BPP * NPLANES];
FILE *pcx_fp;
#define pcx_putc(c) fputc(c,pcx_fp);

/*      dump_screen     */
void dump_screen(void)
{
    int x,y,color,mask;
    unsigned char *p;
    static masktab[] = {0x80,0x40,0x20,0x10,0x8,0x4,0x2,0x1};
    /* write PCX header */
    for(x = 0, p = (unsigned char *)&header; x < sizeof(header); x++)
        pcx_putc(*p++);
    /* write out the screen */  
    for(y = 0; y < MAX_Y; y++) {
        memset(planes,0,sizeof(planes)); /* clear planes */
        for(x = 0; x < MAX_X; x++) {     /* break color into sep planes */
            color = getpixel(x,y);
            mask = masktab[x & 7];
            p = &planes[x >> 3];
            if(color & 1) *p |= mask;
            p += BPP;
            if(color & 2) *p |= mask;
            p += BPP;
            if(color & 4) *p |= mask;
            p += BPP;
            if(color & 8) *p |= mask;
        }
        put_line(planes,BPP * NPLANES);
    }                   
}
/*      put_line    */
void put_line(unsigned char *data,int cnt)
{
    int i,byte,rep_cnt;
    for(i = rep_cnt = 0; i < cnt; i++) {
        if(rep_cnt == 0) {      /* no "current byte" */
            byte = data[i];
            rep_cnt = 1;
            continue;
        }
        if(data[i] == byte) {   /* same as previous, inc run length */
            rep_cnt++;
            /* full run then output */
            if(rep_cnt == 0x3f) {
                out_run(byte,rep_cnt);
                rep_cnt = 0;
            }
            continue;
        }
        out_run(byte,rep_cnt);  /* not equal to previous */
        byte = data[i];
        rep_cnt = 1;
    }
    if(rep_cnt)                 /* shove out any stragglers */
        out_run(byte,rep_cnt);
}
/*      out_run     */
void out_run(int byte,int rep_cnt)
{
    if((byte & 0xc0) == 0xc0 || rep_cnt > 1)
        pcx_putc(0xc0 | rep_cnt);
    pcx_putc(byte);
}
/*      open_pcx    */
void open_pcx(char *name)
{
    pcx_fp = fopen(name,"wb");
    if(pcx_fp == NULL) {
        printf("Can't open output PCX file '%s'.\n",name);
        exit(1);
    }
}
/*      init_graphics   */
void init_graphics(void)
{
    int graphdriver,graphmode,grapherr;

    /* set VGA 640x480 graphics mode */
    graphdriver = VGA;
    graphmode = VGAHI;
    initgraph(&graphdriver,&graphmode,"" );
    if((grapherr = graphresult()) != grOk) {
        printf("Graphics init error: %s\n",grapherrormsg(grapherr));
        exit(1);
    }
    /* make double sure we hit 640x480 mode */
    if(getmaxx() != MAX_X - 1 || getmaxy() != MAX_Y - 1) {
        closegraph();
        printf("Wrong screen size, x=%u y=%u.\n",getmaxx(),getmaxy());
        exit(1);
    }
}