// File I/O Functions

#include <mem.h>
#include <dir.h>
#include "errors.h"
#include "fileio.hpp"

// Class Constructor
AFile::AFile(void) {

  Access = NOTOPEN;
  memset(FileName, 0, sizeof(FileName));
  ErrorCode = NoError;
  nIndex = 0;
  iDataBytes = 0;
}


// Class Destructor
AFile::~AFile(void) {

  if (Access != NOTOPEN) {
    if (Access == OPENFORWRITE)
      FlushWriteFile();

    CloseFile();
  }
}


// Open a file for reading
BOOL AFile::OpenReadFile(LPSTR TheFileName) {

  ErrorCode = NoError;

  if (Access != NOTOPEN) {        // If object already has active file
    ErrorCode = EAccessError;
    return FALSE;
  }
  // Copy filename to local storage
  lstrcpyn(FileName, TheFileName, MAXPATH);

  // Attempt to open the specified
  hFile = _lopen(FileName, READ);
  if (hFile == HFILE_ERROR) {
    ErrorCode = EFileNotFound;
    return FALSE;
  }
  Access = OPENFORREAD;
  return TRUE;
}

BOOL AFile::OpenWriteFile(LPSTR TheFileName) {

  ErrorCode = NoError;

  if (Access != NOTOPEN) {        // If object already has active file
    ErrorCode = EAccessError;
    return FALSE;
  }
  // Copy filename to local storage
  lstrcpyn(FileName, TheFileName, MAXPATH);

  // Attempt to open the specified
  hFile = _lcreat(TheFileName, 0);
  if (hFile == HFILE_ERROR) {
    ErrorCode = EOpeningFile;
    return FALSE;
  }
  Access = OPENFORWRITE;
  return TRUE;
}

BOOL AFile::CloseFile(void) {

  ErrorCode = NoError;

  if (Access == NOTOPEN) {
    ErrorCode = EAccessError;
    return FALSE;
  }

  if (Access == OPENFORWRITE)
    FlushWriteFile();

  if (_lclose(hFile) == HFILE_ERROR) {
    ErrorCode = EClosingFile;
    return FALSE;
  }
  Access = NOTOPEN;
  return TRUE;
}

// Read the next byte from the file buffer. Read file if necessary
int AFile::ReadByte(void) {

  ErrorCode = NoError;

  if (Access != OPENFORREAD) {
    ErrorCode = EAccessError;
    return HFILE_ERROR;
  }

  if (iDataBytes == 0) {
    if ((iDataBytes = _lread(hFile, FileBuf, sizeof(FileBuf))) <= 0) {
      ErrorCode = ECorrupt;
      return HFILE_ERROR;
    }

    nIndex = 0;
  }
  --iDataBytes;
  return(*(FileBuf+(nIndex++)));
}

// Read multiple byte entities from the input file
BOOL AFile::ReadMBytes(BYTE huge *Buffer, register DWORD Number) {
  register DWORD Index;

  ErrorCode = NoError;

  if (Access != OPENFORREAD) {
    ErrorCode = EAccessError;
    return FALSE;
  }

  Index = 0L;

  while (Number--) {
    if (iDataBytes == 0) {
      if ((iDataBytes = _lread(hFile, FileBuf, sizeof(FileBuf))) <= 0) {
        ErrorCode = ECorrupt;
        return FALSE;
      }
      nIndex = 0;
    }
    --iDataBytes;
    Buffer[Index++] = *(FileBuf+(nIndex++));
  }
  return TRUE;
}


BOOL AFile::WriteByte(BYTE TheByte) {

  ErrorCode = NoError;

  if (Access != OPENFORWRITE) {
    ErrorCode = EAccessError;
    return FALSE;
  }

  if (nIndex == sizeof(FileBuf)) {
    if (_lwrite(hFile, FileBuf, sizeof(FileBuf)) <= 0) {
      ErrorCode = EWrtOutFile;
      return FALSE;
    }
    nIndex = 0;
  }
  *(FileBuf+(nIndex++)) = TheByte;
  return TRUE;
}

// Write multiple byte entities to the output file
BOOL AFile::WriteMBytes(BYTE huge *Buffer, register DWORD Number) {
  register DWORD Index;

  ErrorCode = NoError;

  if (Access != OPENFORWRITE) {
    ErrorCode = EAccessError;
    return FALSE;
  }

  Index = 0L;

  while (Number--)
    if (!WriteByte(Buffer[Index++]))
      return FALSE;
  return TRUE;
}

// Flush the output write file
BOOL AFile::FlushWriteFile(void) {

  ErrorCode = NoError;

  if (Access != OPENFORWRITE) {
    ErrorCode = EAccessError;
    return FALSE;
  }

  if (_lwrite(hFile, FileBuf, nIndex) <= 0) {
    ErrorCode = EWrtOutFile;
    return FALSE;
  }
  return TRUE;
}

int AFile::GetError(void) {

  return ErrorCode;
}

