// Discrete Cosine Class Member Functions

#include <math.h>
#include <mem.h>
#include "dct.hpp"
#include "tables.hpp"

#define SCALE4BITS          4
#define SCALE6BITS          6
#define DIVIDEBYFOUR        2

#define SCALEBYBITS        10     // Scale by this number of bits
#define SCALEFACTOR        (1 << SCALEBYBITS)
#define SCALEBITSANDDIVIDE (SCALEBYBITS + DIVIDEBYFOUR + SCALE6BITS)

// Class Constructor for DCT class. Creates an array of weighting
// coefficients used for the discrete cosine transforms. Entries in
// Coefficients array are scaled up by SCALEFACTOR to remove the need
// for floating point arithematic. Coefficients array has dimensions of
// [MAXRESULTS][MAXSAMPLES].

DCT::DCT(void) {

  // Create the coefficient array
  double const PI = 4 * atan(1.0);
  double Constant = PI/16.0;

  for (register int k = 0; k < MAXRESULTS; k++) {
	for (register int m = 0; m < MAXSAMPLES; m++) {
	  if (k == 0)
		Coefficients[k][m] = ((1.0 / sqrt(2.0)) * (double) SCALEFACTOR) + 0.5;
	  else
		Coefficients[k][m] = (cos(Constant * k * (2.0 * m + 1)) *
                                 (double) SCALEFACTOR) + 0.5;
	}
  }
}


// Forward 2D discrete cosine transform. Transforms block of pixels in
// InBlock into a block of frequency components in OutBlock.
void DCT::FDCT(BYTEBLOCKPTR InBlock, INTBLOCKPTR OutBlock) {
  long TempBlock[8][8];

  memset(TempBlock, 0, sizeof(TempBlock));

  long *plDest;
  // Do a one dimensional row FDCT
  for (register int Row = 0; Row < 8; Row++) {
	 for (register int K = 0; K < 8; K++) {
		plDest = &TempBlock[Row][K];
		for (register int Col = 0; Col < 8; Col++) 
		  *plDest += (*InBlock)[Row][Col] * Coefficients[K][Col];

		*plDest >>= SCALE4BITS;   // Limited scaling to keep as much
	 }                            // precision as possible
  }

  long TempLong;
  // Do a one dimensional column FDCT
  for (register int Col = 0; Col < 8; Col++) {
	 for (register int K = 0; K < 8; K++) {
		TempLong = 0;
		for (register int Row = 0; Row < 8; Row++) {
		  TempLong += TempBlock[Row][Col] * Coefficients[K][Row];
		}
		(*OutBlock)[K][Col] = (int)(TempLong >> SCALEBITSANDDIVIDE);
	 }
  }
}

// Inverse 2D discrete cosine transform
void DCT::IDCT(INTBLOCKPTR InBlock, BYTEBLOCKPTR OutBlock) {
  long TempBlock[8][8];

  memset(TempBlock, 0, sizeof(TempBlock));

  long *plDest;
  // Do a one dimensional column FDCT
  for (register int Col = 0; Col < 8; Col++) {
	 for (register int K = 0; K < 8; K++) {
		plDest = (long *) &TempBlock[K][Col];
		for (register int Row = 0; Row < 8; Row++)
		  *plDest += (*InBlock)[Row][Col] * Coefficients[Row][K];
		
		*plDest >>= SCALE4BITS;   // Limited scaling to keep as
	 }                            // much precision as possible
  }

  long TempLong;
  // Do a one dimensional row FDCT
  for (register int Row = 0; Row < 8; Row++) {
	 for (register int K = 0; K < 8; K++) {
		TempLong = 0;
		for (register int Col = 0; Col < 8; Col++)
			TempLong += TempBlock[Row][Col] * Coefficients[Col][K];
		
		TempLong >>= SCALEBITSANDDIVIDE;
		// Clamp pixel value into valid range 0..255
		TempLong = (TempLong > 255) ? 255:TempLong;
		TempLong = (TempLong <   0) ?   0:TempLong;
		// Store pixel value into output block
		(*OutBlock)[Row][K] = (BYTE) TempLong;
	 }
  }
}

// Reorder the coefficients between pixel format and frequency content
// format. ZigZag table in file tables.cpp.
void DCT::ZigZagReorder(int *InBlock, int *OutBlock, BOOL ForwardReorder) {

  if (ForwardReorder)             // Convert from pixels to frequencies
    for (register int Index = 0; Index < 64; Index++)
      OutBlock[Index] = InBlock[ZigZagTable[Index]];
  else                            // Convert from frequencies to pixels
    for (register int Index = 0; Index < 64; Index++)
      OutBlock[ZigZagTable[Index]] = InBlock[Index];
}


