// Color Conversion Class Member Functions


#include <mem.h>
#include <windows.h>
#include "color.hpp"

#define SCALEBYBITS  15           // Scale by this number of bits
#define SCALEFACTOR  (1L << SCALEBYBITS)
#define CHROMAOFFSET 127          

/*
This file contains the member functions of the ColorConvert class which
perform RGB to YCbCr and YCbCr to RGB color conversions. YCbCr is as
defined in CCIR 601-1, except that Cb and Cr are normalized to the range
0..255 instead of -0.5 .. 0.5 to remove the need for floating point math.
The equations shown below are used for the conversions.

For RGB to YCbCr conversion:

  Y  =  0.29900 * R + 0.58700 * G + 0.11400 * B
  Cb = -0.16874 * R - 0.33126 * G + 0.50000 * B  + CHROMAOFFSET
  Cr =  0.50000 * R - 0.41869 * G - 0.08131 * B  + CHROMAOFFSET

For YCbCr to RGB conversion:

  R = Y                + 1.40200 * Cr
  G = Y - 0.34414 * Cb - 0.71414 * Cr
  B = Y + 1.77200 * Cb
  where Cb and Cr represent the incoming values less CHROMAOFFSET.

For speed, all of the values are precomputed and stored in arrays. For
this reason only array lookups and additions are required during the
conversions.
*/

// Class Constructor. Register all info about the image and the
// appropriate buffers used for storage and converson.
ColorConvert::ColorConvert(IMAGETYPE Type, WORD Width, WORD Height,
                           BYTE huge *lpImageData,
                           BYTE far *lpY1BufferData,
                           BYTE far *lpY2BufferData,
                           BYTE far *lpCbBufferData,
                           BYTE far *lpCrBufferData) {

  ImageType = Type;               // Save incoming parameters
  WidthInPixels = Width;          // Image dimensions
  HeightInPixels = Height;
  lpImage = lpImageData;          // Ptr to raster data for I/O
  lpY1Buffer = lpY1BufferData;    // Ptrs to four buffers
  lpY2Buffer = lpY2BufferData;
  lpCbBuffer = lpCbBufferData;
  lpCrBuffer = lpCrBufferData;

  // Allocate tables for color conversions if required
  if (ImageType == TRUECOLORTYPE) {
    // For RGB to YCbCr conversion
    WORD ArraySize = 256 * sizeof(long);
    pEncodeYRed    = (long *) MyAlloc(ArraySize);
    pEncodeYGreen  = (long *) MyAlloc(ArraySize);
    pEncodeYBlue   = (long *) MyAlloc(ArraySize);
    pEncodeCbRed   = (long *) MyAlloc(ArraySize);
    pEncodeCbGreen = (long *) MyAlloc(ArraySize);
    pEncodeCbBlue  = (long *) MyAlloc(ArraySize);
    pEncodeCrRed   = (long *) MyAlloc(ArraySize);
    pEncodeCrGreen = (long *) MyAlloc(ArraySize);
    pEncodeCrBlue  = (long *) MyAlloc(ArraySize);

    // For YCbCr to RGB conversion
    pDecodeRCr     = (long *) MyAlloc(ArraySize);
    pDecodeGCb     = (long *) MyAlloc(ArraySize);
    pDecodeGCr     = (long *) MyAlloc(ArraySize);
    pDecodeBCb     = (long *) MyAlloc(ArraySize);

    // Now proceed to fill the tables.
    for (register int Index = 0; Index < 256; Index++) {
      pEncodeYRed[Index]   = (Index *  0.29900 * SCALEFACTOR) + 0.5;
      pEncodeYGreen[Index] = (Index *  0.58700 * SCALEFACTOR) + 0.5;
      pEncodeYBlue[Index]  = (Index *  0.11400 * SCALEFACTOR) + 0.5;
      pEncodeCbRed[Index]  = (Index * -0.16874 * SCALEFACTOR) + 0.5;
      pEncodeCbGreen[Index]= (Index * -0.33126 * SCALEFACTOR) + 0.5;
      pEncodeCbBlue[Index] = (Index *  0.50000 * SCALEFACTOR) + 0.5;
      pEncodeCrRed[Index]  = (Index *  0.50000 * SCALEFACTOR) + 0.5;
      pEncodeCrGreen[Index]= (Index * -0.41869 * SCALEFACTOR) + 0.5;
      pEncodeCrBlue[Index] = (Index * -0.08131 * SCALEFACTOR) + 0.5;

      pDecodeRCr[Index]    = ((Index - CHROMAOFFSET) *  1.40200 * SCALEFACTOR) + 0.5;
      pDecodeGCb[Index]    = ((Index - CHROMAOFFSET) * -0.34414 * SCALEFACTOR) + 0.5;
      pDecodeGCr[Index]    = ((Index - CHROMAOFFSET) * -0.71414 * SCALEFACTOR) + 0.5;
      pDecodeBCb[Index]    = ((Index - CHROMAOFFSET) *  1.77200 * SCALEFACTOR) + 0.5;
    }
    ChromaRangeOffset = CHROMAOFFSET * SCALEFACTOR;
  }

  // Calculate bytes/line of DIB image and buffers
  WORD NumberHorzBlocks = ((Width + 15) / 16) * 2;    // Num of horz blocks
  YBufferBytesPerLine = NumberHorzBlocks * 8;  // Bytes/line in those blocks
  CBufferBytesPerLine = YBufferBytesPerLine / 2;
  SizeOfCBuffers = CBufferBytesPerLine * 8;
  DIBBytesPerLine = (Type == TRUECOLORTYPE) ? Width * 3:Width;
  DIBBytesPerLine = ALIGN_DWORD(DIBBytesPerLine);
}


// Class Destructor
ColorConvert::~ColorConvert(void) {

  // Free up all of the arrays used for conversions if required
  if (ImageType == TRUECOLORTYPE) {
    if (pEncodeYRed)    MyFree(pEncodeYRed);
    if (pEncodeYGreen)  MyFree(pEncodeYGreen);
    if (pEncodeYBlue)   MyFree(pEncodeYBlue);
    if (pEncodeCbRed)   MyFree(pEncodeCbRed);
    if (pEncodeCbGreen) MyFree(pEncodeCbGreen);
    if (pEncodeCbBlue)  MyFree(pEncodeCbBlue);
    if (pEncodeCrRed)   MyFree(pEncodeCrRed);
    if (pEncodeCrGreen) MyFree(pEncodeCrGreen);
    if (pEncodeCrBlue)  MyFree(pEncodeCrBlue);
    if (pDecodeRCr)     MyFree(pDecodeRCr);
    if (pDecodeGCb)     MyFree(pDecodeGCb);
    if (pDecodeGCr)     MyFree(pDecodeGCr);
    if (pDecodeBCb)     MyFree(pDecodeBCb);
  }
}


// This function fills the Y, Cb and Cr buffers with data from the image
// specified to the constructor of this class. If the DIB contains RGB
// image data the data is converted on the fly. If the DIB contains
// grayscale data, the data is copied into the Y buffers without
// modification. Sixteen rows of DIB image data are fetched starting
// at the specified row number each time this function is called.
void ColorConvert::GetDIBData(WORD FromRowNumber) {

  BYTE huge *lpSRow;
  BYTE huge *lpSPixel;
  BYTE far  *lpYRow;
  BYTE far  *lpCbRow;
  BYTE far  *lpCrRow;
  BYTE Red, Green, Blue;
  long Y, Cb, Cr;
  register int Row;
  register int OutputRow;
  register int Col;

  register int FromRowNumber16 = FromRowNumber + 16;

  switch(ImageType) {
    case TRUECOLORTYPE:           // DIB image is truecolor format
      for (Row = FromRowNumber, OutputRow = 0;
           Row < FromRowNumber16; Row++, OutputRow++) {
        lpSRow  = lpImage + (DIBBytesPerLine * (DWORD) Row);
        if (OutputRow < 8)
          lpYRow = lpY1Buffer + (YBufferBytesPerLine * OutputRow);
        else
          lpYRow = lpY2Buffer + (YBufferBytesPerLine * (OutputRow - 8));
        lpCbRow = lpCbBuffer + (CBufferBytesPerLine * (OutputRow / 2));
        lpCrRow = lpCrBuffer + (CBufferBytesPerLine * (OutputRow / 2));

        if (Row < HeightInPixels) {    // Only fetch real data
          for (Col = 0; Col < WidthInPixels; Col++) {
            // Fetch RGB pixel from DIB
            lpSPixel = lpSRow + (Col * 3L);  // Ptr to source pixel
            Blue  = *lpSPixel++;
            Green = *lpSPixel++;
            Red   = *lpSPixel;

            // Now perform color conversion to YCbCr.
            // NOTE: Cb and Cr in range -127 to +127, Y is always positive.
            Y  = pEncodeYRed [Red] + pEncodeYGreen [Green] + pEncodeYBlue [Blue];
            Cb = pEncodeCbRed[Red] + pEncodeCbGreen[Green] + pEncodeCbBlue[Blue];
            Cr = pEncodeCrRed[Red] + pEncodeCrGreen[Green] + pEncodeCrBlue[Blue];

            // Store the converted RGB value into YCbCr buffers
			*(lpYRow  + Col) = (BYTE)(Y >> SCALEBYBITS);
			*(lpCbRow + Col / 2) = (BYTE)((Cb + ChromaRangeOffset) >> SCALEBYBITS);
			*(lpCrRow + Col / 2) = (BYTE)((Cr + ChromaRangeOffset) >> SCALEBYBITS);
          }
        } else {                  // Fill buffers will zero as no image exists
          for (Col = 0; Col < WidthInPixels; Col++) {
            // Store the converted RGB value into YCbCr buffers
            *(lpYRow  + Col) = 0;
            *(lpCbRow + Col / 2) = 0;
            *(lpCrRow + Col / 2) = 0;
          }
        }
      }
      break;

    case GRAYSCALETYPE:           // DIB image is grayscale format
      for (Row = FromRowNumber, OutputRow = 0;
           Row < FromRowNumber16; Row++, OutputRow++) {
        if (Row < HeightInPixels) {    //
          lpSRow = lpImage + (DIBBytesPerLine * (DWORD) Row);
          if (OutputRow < 8)
            lpYRow = lpY1Buffer + (YBufferBytesPerLine * OutputRow);
          else
            lpYRow = lpY2Buffer + (YBufferBytesPerLine * (OutputRow - 8));

          // Now copy the BYTE sized data
          hmemcpy(lpYRow, lpSRow, WidthInPixels);
        } else {
          memset(lpYRow, 0, WidthInPixels);
        }
      }
      break;

    default:
      // Add other images types here
      break;
  }
}


// For truecolor images, this function converts the Y, Cb and Cr image
// data in the buffers to RGB format and stores the data into a DIB.
// For grayscale images, data is moved from the Y buffer to the DIB
// without manipulation. Sixteen rows of DIB image data are stored starting
// at the specified row number each time this function is called.
void ColorConvert::PutDIBData(WORD ToRowNumber) {

  BYTE huge *lpDRow;
  BYTE huge *lpDPixel;
  BYTE far  *lpYRow;
  BYTE far  *lpCbRow;
  BYTE far  *lpCrRow;
  long Red, Green, Blue;
  long Y;
  BYTE Cb, Cr;
  register int Row;
  register int OutputRow;
  register int Col;

  register int ToRowNumber16 = ToRowNumber + 16;

  switch(ImageType) {
    case TRUECOLORTYPE:           // DIB image is truecolor format
      for (Row = ToRowNumber, OutputRow = 0;
           Row < ToRowNumber16; Row++, OutputRow++) {
        lpDRow = lpImage + (DIBBytesPerLine * (DWORD) Row);
        if (OutputRow < 8)
          lpYRow = lpY1Buffer + (YBufferBytesPerLine * OutputRow);
        else
          lpYRow = lpY2Buffer + (YBufferBytesPerLine * (OutputRow - 8));
        lpCbRow = lpCbBuffer + (CBufferBytesPerLine * (OutputRow / 2));
        lpCrRow = lpCrBuffer + (CBufferBytesPerLine * (OutputRow / 2));

        if (Row < HeightInPixels) {    //
          for (Col = 0; Col < WidthInPixels; Col++) {
            // Fetch YCbCr values from buffers
            Y  = (DWORD)(*(lpYRow + Col)) << SCALEBYBITS;
            Cb = *(lpCbRow + Col / 2);
            Cr = *(lpCrRow + Col / 2);

            // Now perform color conversion to RGB
            Red   = (Y + pDecodeRCr[Cr]) >> SCALEBYBITS;
            Green = (Y + pDecodeGCb[Cb] + pDecodeGCr[Cr]) >> SCALEBYBITS;
            Blue  = (Y + pDecodeBCb[Cb]) >> SCALEBYBITS;

            // Clamp the values into 0..255 range
            Red   = (Red   <   0) ?   0:Red;
            Red   = (Red   > 255) ? 255:Red;
            Green = (Green <   0) ?   0:Green;
            Green = (Green > 255) ? 255:Green;
            Blue  = (Blue  <   0) ?   0:Blue;
            Blue  = (Blue  > 255) ? 255:Blue;

            // Store the converted RGB pixel value into DIB
            lpDPixel = lpDRow + (Col * 3L);  // Ptr to destination pixel
			*lpDPixel++ = (BYTE) Blue;
			*lpDPixel++ = (BYTE) Green;
			*lpDPixel   = (BYTE) Red;
          }
        }
      }
      break;

    case GRAYSCALETYPE:           // DIB image is grayscale format
      for (Row = ToRowNumber, OutputRow = 0;
           Row < ToRowNumber16; Row++, OutputRow++) {
        if (Row < HeightInPixels) {    //
          lpDRow = lpImage + (DIBBytesPerLine * (DWORD) Row);
          if (OutputRow < 8)
            lpYRow = lpY1Buffer + (YBufferBytesPerLine * OutputRow);
          else
            lpYRow = lpY2Buffer + (YBufferBytesPerLine * (OutputRow - 8));

          // Now copy the BYTE sized data
          hmemcpy(lpDRow, lpYRow, WidthInPixels);
        }
      }
      break;

    default:
      break;
  }
}


