// Buffer Manager Class Member Functions

#include <mem.h>
#include "bufman.hpp"


BufferManager::BufferManager(IMAGETYPE Type, WORD Width, WORD Height,
                             BYTE huge *lpImageData) {

  ImageType = Type;            // Save incoming parameters
  WidthInPixels  = Width;
  lpDIBImageData = lpImageData;

  Y1Offset = 0;                   // Set initial offsets to zero. An offset
  Y2Offset = 0;                   // for each of four buffers
  CbOffset = 0;
  CrOffset = 0;

  // NOTE: rounding up so all of image is included
  NumberOfHorzBlocks = ((Width  + 15) / 16) * 2; // Blocks in width

  YBufferBytesPerLine = NumberOfHorzBlocks * 8;  // Bytes/line in Y buffers
  CBufferBytesPerLine = YBufferBytesPerLine / 2; // Bytes/line in Cb Cr buffers

  // Calculate how large the Y buffers must be
  BufferSize = YBufferBytesPerLine * 8;     // Total size of each buffer

  // Initialize object data
  FromRow = 0;
  ToRow = 0;
  HorzBlocksLeft = -1;            // Set to illegal number to indicate 
                                  // initialization required.
  // Allocate Y buffers. Needed for both TRUECOLOR and GRAYSCALE images
  lpY1Buffer = (BYTE far *) MyAlloc(BufferSize);
  lpY2Buffer = (BYTE far *) MyAlloc(BufferSize);

  // Now allocate the buffers required for manipulation of the image data.
  // NOTE: buffers for chroma are half as big as luma buffers because of
  // color subsampling.
  if (ImageType == TRUECOLORTYPE) {
	lpCbBuffer = (BYTE far *) MyAlloc(BufferSize / 2);
	lpCrBuffer = (BYTE far *) MyAlloc(BufferSize / 2);
  }

  // Instantiate a ColorConvert object as required
  ColorConvertObject = new ColorConvert(ImageType,
                                        WidthInPixels, Height,
                                        lpDIBImageData,
										lpY1Buffer, lpY2Buffer,
										lpCbBuffer, lpCrBuffer);
}


// Class Destructor
BufferManager::~BufferManager(void) {

  // Free up the various buffers
  if (lpY1Buffer) MyFree(lpY1Buffer);
  if (lpY2Buffer) MyFree(lpY2Buffer);
  if (ImageType == TRUECOLORTYPE) {
	if (lpCbBuffer) MyFree(lpCbBuffer);
	if (lpCrBuffer) MyFree(lpCrBuffer);
  }
  // Delete the ColorConvert object
  delete ColorConvertObject;
}

// Get the next block of pixels from the image. This function will be
// called by compression code as many times as necessary to fetch all
// blocks of an image. This code does not need to determine if any
// image blocks remain. The code which calls this function will do that.
void BufferManager::GetNextBlock(BYTE *lpBlock, int BlockTypeNumber) {
  WORD Index, RowOffset, RowOffsetCol;
  WORD Y1Offset8, Y2Offset8, CbOffset8, CrOffset8;
  register WORD Row;
  register WORD Col;
  
  // Block fetched depends upon block number
  switch(BlockTypeNumber) {
	case 0:                       // Block 0 - Fetch luma sample 1

	  // First check to see if buffer need filling. This is the only
	  // place where this check is necessary.
	  if (HorzBlocksLeft <= 0) {      // No blocks left in strip left

		// Fill the buffers with next strip of image data. 16 rows of
        // data are read each time GetDIBData is called.
		ColorConvertObject->GetDIBData(FromRow);
        FromRow += 16;

		// Reset offsets so fetches are at beginning of buffers
		Y1Offset = 0;
		Y2Offset = 0;
		CbOffset = 0;
		CrOffset = 0;

		// Reset management variables
		HorzBlocksLeft = NumberOfHorzBlocks;
	  }
	  // Fetch luma sample from buffer Y1
	  Y1Offset8 = Y1Offset + 8;

	  // Fill the blocks from the appropriate buffer
	  Index = 0;
	  for (Row = 0; Row < 8; Row++) {
		RowOffset = Row * YBufferBytesPerLine;
		for (Col = Y1Offset; Col < Y1Offset8; Col++) {
		  RowOffsetCol = RowOffset + Col;
		  lpBlock[Index++] = *(lpY1Buffer + RowOffsetCol);
		}
	  }
	  // Do house cleaning
	  HorzBlocksLeft--;
	  Y1Offset = Y1Offset8;
	  break;

	case 1:                       // Block 1 - Fetch luma sample 2

	  // Fetch luma sample from buffer Y1
	  Y1Offset8 = Y1Offset + 8;

	  // Fill the blocks from the appropriate buffer
	  Index = 0;
	  for (Row = 0; Row < 8; Row++) {
		RowOffset = Row * YBufferBytesPerLine;
		for (Col = Y1Offset; Col < Y1Offset8; Col++) {
		  RowOffsetCol = RowOffset + Col;
		  lpBlock[Index++] = *(lpY1Buffer + RowOffsetCol);
		}
	  }
	  // Do house cleaning
	  HorzBlocksLeft--;
	  Y1Offset = Y1Offset8;
	  break;

	case 2:                       // Block 2 - Fetch luma sample 3

	  // Fetch the luma sample from buffer Y2
	  Y2Offset8 = Y2Offset + 8;

	  // Fill the blocks from the appropriate buffer
	  Index = 0;
	  for (Row = 0; Row < 8; Row++) {
		RowOffset = Row * YBufferBytesPerLine;
		for (Col = Y2Offset; Col < Y2Offset8; Col++) {
		  RowOffsetCol = RowOffset + Col;
		  lpBlock[Index++] = *(lpY2Buffer + RowOffsetCol);
		}
	  }
	  // Do house cleaning
	  Y2Offset = Y2Offset8;
	  break;

	case 3:                       // Block 3 - Fetch luma sample 4

	  // Fetch luma sample from buffer Y2
	  Y2Offset8 = Y2Offset + 8;

	  // Fill the blocks from the appropriate buffer
	  Index = 0;
	  for (Row = 0; Row < 8; Row++) {
		RowOffset = Row * YBufferBytesPerLine;
		for (Col = Y2Offset; Col < Y2Offset8; Col++) {
		  RowOffsetCol = RowOffset + Col;
		  lpBlock[Index++] = *(lpY2Buffer + RowOffsetCol);
		}
	  }
	  // Do house cleaning
	  Y2Offset = Y2Offset8;
	  break;

	case 4:                       // Block 4 - Fetch Cb chroma sample 

	  // Fetch the chroma sample from buffer Cb
	  CbOffset8 = CbOffset + 8;

	  // Fill the blocks from the appropriate buffer
	  Index = 0;
	  for (Row = 0; Row < 8; Row++) {
		RowOffset = Row * CBufferBytesPerLine;
		for (Col = CbOffset; Col < CbOffset8; Col++) {
		  RowOffsetCol = RowOffset + Col;
		  lpBlock[Index++] = *(lpCbBuffer + RowOffsetCol);
		}
	  }
	  // Do house cleaning
	  CbOffset = CbOffset8;
	  break;

	case 5:                       // Block 5 - Fetch Cr chroma sample

	  // Fetch chroma sample from buffer Cr
	  CrOffset8 = CrOffset + 8;

	  // Fill the blocks from the appropriate buffer
	  Index = 0;
	  for (Row = 0; Row < 8; Row++) {
		RowOffset = Row * CBufferBytesPerLine;
		for (Col = CrOffset; Col < CrOffset8; Col++) {
		  RowOffsetCol = RowOffset + Col;
		  lpBlock[Index++] = *(lpCrBuffer + RowOffsetCol);
		}
	  }
	  // Do house cleaning
	  CrOffset = CrOffset8;
	  break;
  }
}

// Put the next block of pixels to the image.
void BufferManager::PutNextBlock(BYTE *lpBlock, int BlockTypeNumber) {
  WORD Index, RowOffset, RowOffsetCol;
  WORD Y1Offset8, Y2Offset8, CbOffset8, CrOffset8;
  register WORD Row;
  register WORD Col;

  // There may not be any room in current strip of image data when
  // this function is called. The check will be made everytime
  // block 0 storage is requested. If no room is available, the strip
  // of image data will be flushed to the image buffer to make room
  // available.

  // Where block is stored depends upon BlockTypeNumber
  switch(BlockTypeNumber) {

	case 0:                       // Block 0 - Store luma sample 1
      // Prevent write of DIB data during first call of this function
      if (HorzBlocksLeft == -1)
        HorzBlocksLeft = NumberOfHorzBlocks;

	  // First check to see if buffer need filling. This is the only
	  // place where this check is necessary.
	  if (HorzBlocksLeft == 0) {      // No blocks left in strip left

		// Flush image data to image buffer. 16 rows of data are written
        // each time PutDIBData is called.
		ColorConvertObject->PutDIBData(ToRow);
		ToRow += 16;
		// Reset offsets so fetches are at beginning of buffers
		Y1Offset = 0;
		Y2Offset = 0;
		CbOffset = 0;
		CrOffset = 0;

		// Reset management variables
		HorzBlocksLeft = NumberOfHorzBlocks;
	  }

	  // Store luma sample into buffer Y1
	  Y1Offset8 = Y1Offset + 8;

	  // Fill the buffer from the block
	  Index = 0;
	  for (Row = 0; Row < 8; Row++) {
		RowOffset = Row * YBufferBytesPerLine;
		for (Col = Y1Offset; Col < Y1Offset8; Col++) {
		  RowOffsetCol = RowOffset + Col;
		  *(lpY1Buffer + RowOffsetCol) = lpBlock[Index++];
		}
	  }
	  // Do house cleaning
	  HorzBlocksLeft--;
	  Y1Offset = Y1Offset8;
	  break;

	case 1:                       // Block 1 - Store luma sample 2

	  // Store luma sample into buffer Y1
	  Y1Offset8 = Y1Offset + 8;

	  // Fill the buffer from the block
	  Index = 0;
	  for (Row = 0; Row < 8; Row++) {
		RowOffset = Row * YBufferBytesPerLine;
		for (Col = Y1Offset; Col < Y1Offset8; Col++) {
		  RowOffsetCol = RowOffset + Col;
		  *(lpY1Buffer + RowOffsetCol) = lpBlock[Index++];
		}
	  }
	  // Do house cleaning
	  HorzBlocksLeft--;
	  Y1Offset = Y1Offset8;
	  break;

	case 2:                       // Block 2 - Store luma sample 3

	  // Store luma sample into buffer Y2
	  Y2Offset8 = Y2Offset + 8;

	  // Fill the buffer from the block
	  Index = 0;
	  for (Row = 0; Row < 8; Row++) {
		RowOffset = Row * YBufferBytesPerLine;
		for (Col = Y2Offset; Col < Y2Offset8; Col++) {
		  RowOffsetCol = RowOffset + Col;
		  *(lpY2Buffer + RowOffsetCol) = lpBlock[Index++];
		}
	  }
	  // Do house cleaning
	  Y2Offset = Y2Offset8;
	  break;

	case 3:                       // Block 3 - Store luma sample 4

	  // Store luma sample into buffer Y2
	  Y2Offset8 = Y2Offset + 8;

	  // Fill the buffer from the block
	  Index = 0;
	  for (Row = 0; Row < 8; Row++) {
		RowOffset = Row * YBufferBytesPerLine;
		for (Col = Y2Offset; Col < Y2Offset8; Col++) {
		  RowOffsetCol = RowOffset + Col;
		  *(lpY2Buffer + RowOffsetCol) = lpBlock[Index++];
		}
	  }
	  // Do house cleaning
	  Y2Offset = Y2Offset8;
	  break;

	case 4:                       // Block 4 - Store Cb chroma sample 

	  // Store chroma sample into buffer Cb
	  CbOffset8 = CbOffset + 8;

	  // Fill the buffer from the block
	  Index = 0;
	  for (Row = 0; Row < 8; Row++) {
		RowOffset = Row * CBufferBytesPerLine;
		for (Col = CbOffset; Col < CbOffset8; Col++) {
		  RowOffsetCol = RowOffset + Col;
		  *(lpCbBuffer + RowOffsetCol) = lpBlock[Index++];
		}
	  }
	  // Do house cleaning
	  CbOffset = CbOffset8;
	  break;

	case 5:                       // Block 5 - Store Cr chroma sample

	  // Store chroma sample into buffer Cr
	  CrOffset8 = CrOffset + 8;

	  // Fill the buffer from the block
	  Index = 0;
	  for (Row = 0; Row < 8; Row++) {
		RowOffset = Row * CBufferBytesPerLine;
		for (Col = CrOffset; Col < CrOffset8; Col++) {
		  RowOffsetCol = RowOffset + Col;
		  *(lpCrBuffer + RowOffsetCol) = lpBlock[Index++];
		}
	  }
	  // Do house cleaning
	  CrOffset = CrOffset8;
	  break;
  }
}

// Make sure all of the DIB image data is flushed to the image buffer
void BufferManager::FlushDIBData(void) {

  // Flush image data to image buffer
  ColorConvertObject->PutDIBData(ToRow);
}



