/***********************************************************************\
 *                                PC2.c                                *
 *           Copyright (C) by Stangl Roman, 1993, 1994, 1995           *
 * This Code may be freely distributed, provided the Copyright isn't   *
 * removed, under the conditions indicated in the documentation.       *
 *                                                                     *
 * PC/2 - Program Commander/2 is a configurable program starter for    *
 * OS/2 2.x PM. If the user clicks button 1 on the DESKTOP, a user     *
 * modifyable popup menu is displayed. The user then selects a program *
 * to be started, or configuration of PC/2 or dismisses it.            *
 * You can define Hotkeys to switch to a program, or start it if it is *
 * not already running.                                                *
 * PC/2 is an alternative method of starting programs compared to      *
 * icons and uses no space on DESKTOP, and no folder must be opended   *
 * to start a program. For frequently used programs, this reduces the  *
 * time to start an application.                                       *
 * PC/2 also implements an optional virtual Desktop and sliding focus. *
 *                                                                     *
\***********************************************************************/

static char RCSID[]="@(#) $Header: PC2.c/PC2.h Version 1.90 05,1995 $ (LBL)";

#define         _FILE_  "PC/2 - PC2.c V1.90"

#include        "PC2.h"                 /* User include files */
#include        "Error.h"

HEV             hevPc2;                 /* Handle of PC/2 semaphore */
UCHAR           *pucFilenameProfile;    /* The buffer holding the filename of the profile */
UCHAR           *pucFilenameINI;        /* Path and filename of PC2.INI */
UCHAR           *pucFilenameHLP;        /* The buffer holding the filename of the HLP file */
UCHAR           *pucPathDLL;            /* Path to directory PC/2 was started from and DLLs reside */
TID             tidThread=0;            /* Working thread ID */
HMODULE         hDLLPc2Hook=NULLHANDLE; /* PC2HOOK.DLL module handle */
                                        /* PC2SPOOL.DLL module handle */
HMODULE         hDLLPc2Spooler=NULLHANDLE;
PFNWP           pfnMenuWindowProc;      /* PC/2 smarticons menu window procedure saved for subclassing */
PFNWP           pfnListboxWindowProc;   /* PC/2's setup dialog CDLB_MENUPROGRAM listbox window procedure
                                           saved for subclassing */
HDC             hdcMemory;              /* In memory Overview window DC */

/*                                                                                      *\
 * Function pointer of functions loaded dynamically.                                    *
\*                                                                                      */
HOOKPARAMETERS  *pHP=NULL;              /* Pointer to PC/2 control structure loaded from PC2HOOK.DLL */
PFFUNCPTR1      *pSetParameters;        /* Initialize PC2HOOK.DLL */
PFFUNCPTR2      *pInputHook;            /* Input hook procedure */
PFFUNCPTR3      *pWinSendMsgHook;       /* WinSendMsg() hook procedure */
PFFUNCPTR5      *pSpoolerInitialize;    /* Initialize PC2SPOOL.DLL */

/*--------------------------------------------------------------------------------------*\
 * The main procedure.                                                                  *
 * Req:                                                                                 *
 * Returns:                                                                             *
 *      int ........... Exitcode (0, or errorlevel)                                     *
\*--------------------------------------------------------------------------------------*/
int main(void)
{
HAB     habPc2;                         /* Anchor block handle */
HMQ     hmqPc2;                         /* Message queue handle */
HWND    hwndFrame;                      /* Frame window handle, used until HOOKPARAMETERS got
                                           loaded from PC2HOOK.DLL */
HWND    hwndClient;                     /* Client window handle, same as hwndFrame */
HWND    hwndHelp;                       /* Help window handle */
ULONG   ulOverviewFCF;                  /* Frame creation flags */
QMSG    qmsg;                           /* Message queue */
UCHAR   ucModuleName[CCHMAXPATH];       /* PC/2's full qualified path name, PC/2 was started
                                           from, obtained from PC/2's PIB */
TIB     *ptib;                          /* PC/2's thread information block */
PIB     *ppib;                          /* PC/2's process information block */
APIRET  Rc;                             /* Dos_* calls return code */

/*                                                                                      *\
 * Bring up an anchor block as soon as possible, to be able to display message boxes.   *
\*                                                                                      */
do
{
                                        /* Initialize anchor block and message queue */
    if(WinStartUp(&habPc2, &hmqPc2)==FALSE)
        break;
    if(!WinRegisterClass(               /* Register window class */
        habPc2,                         /* Handle of anchor block */
        (PSZ)PC2_CLASSNAME,             /* Window class name */
        (PFNWP)PC2_MainWindowProc,      /* Address of window procedure */
                                        /* Class style */
        CS_SIZEREDRAW | CS_SAVEBITS | CS_MOVENOTIFY,
        0))                             /* Extra window words */
        {
        PM_ERR(habPc2, HWND_DESKTOP, HELP_CREATEWINDOW, MB_ERROR|MB_OK|MB_MOVEABLE|MB_DEFBUTTON1,
            "Initializing the PM environment failed, PC/2 can't continue. You may have run out "\
            "of resources, close some windows or applications and retry. Exiting...");
        break;
        }
/*                                                                                      *\
 * Start frame window, which creates window(s) and WM_CREATE message. Then associate    *
 * a help instance with this window, to be able to display online help for message      *
 * boxes.                                                                               *
\*                                                                                      */
    ulOverviewFCF=FCF_OVERVIEWWINDOW;
    hwndFrame=WinCreateStdWindow(
        HWND_DESKTOP,                   /* DESKTOP is parent */
        0,                              /* Standard window styles */
        &ulOverviewFCF,                 /* Frame control flags */
        (PSZ)PC2_CLASSNAME,             /* Client window class name */
        "",                             /* No window text */
        0,                              /* No special class style */
        (HMODULE)0,                     /* Ressource is in .EXE file */
        ID_PC2MAINWINDOW,               /* Frame window identifier */
        &hwndClient);                   /* Client window handle */
    if(hwndFrame==NULLHANDLE)
        {
        PM_ERR(habPc2, HWND_DESKTOP, HELP_CREATEWINDOW, MB_ERROR|MB_OK|MB_MOVEABLE|MB_DEFBUTTON1,
            "Creation of PC/2's Overview Window, PC/2 can't continue. You may have run out "\
            "of resources, close some windows or applications and retry. Exiting...");
        break;
        }
/*                                                                                      *\
 * The only portable way to access the path PC/2 was invoked from, to access the        *
 * command-line parameters and the environment is to query the process information      *
 * block. Portable means that this is expected to work for OS/2 for PowerPC too.        *
\*                                                                                      */
    DosGetInfoBlocks(&ptib, &ppib);
    DosQueryModuleName(ppib->pib_hmte, sizeof(ucModuleName), ucModuleName);
/*                                                                                      *\
 * Get the full path and filename of the running copy of PC/2 and change the extension  *
 * .EXE into .cfg to open the configuration file under this name. If the user supplies  *
 * [-,/Profile filename.ext] then use this filename as the Profile. Also change .EXE    *
 * into .HLP, .INI and PC/2 directory as the current directory to access .DLL. All path *
 * names contain the drive letter + directory PC/2 was started from.                    *
\*                                                                                      */
                                        /* Long enough to hold user Profile name */
    pucFilenameProfile=calloc(CCHMAXPATH+1, sizeof(BYTE));
    pucFilenameINI=calloc(CCHMAXPATH+1, sizeof(BYTE));
    pucFilenameHLP=calloc(CCHMAXPATH+1, sizeof(BYTE));
    pucPathDLL=calloc(CCHMAXPATH+1, sizeof(BYTE));
                                        /* If no drive letter is available, e.g. using
                                           SET RUNWORKPLACE=\PMAPPS\PC2.EXE, add current
                                           drive, which is the boot drive */
    if(ucModuleName[0]=='\\')
        {
        ULONG   ulDriveNumber;          /* Current logical drive */
        ULONG   ulLogicalDriveMap;      /* Logical drive bitmapped flag */
        UCHAR   ucCurrentDrive[3]="C:"; /* Current drive */

                                        /* Get current logical drive 1...A, 2...B, 3...C, ... */
        if(!DosQueryCurrentDisk(&ulDriveNumber, &ulLogicalDriveMap))
                                        /* On no error use current drive, otherwise assume C: */
            ucCurrentDrive[0]=(CHAR)(--ulDriveNumber)+'A';
                                        /* Add drive letter */
        strcpy(pucFilenameProfile, ucCurrentDrive);
        strcpy(pucFilenameINI, ucCurrentDrive);
        strcpy(pucFilenameHLP, ucCurrentDrive);
        strcpy(pucPathDLL, ucCurrentDrive);
        }
                                        /* Add the full qualified path PC/2 was started from,
                                           which we queried from the process information block */
    strcat(pucFilenameProfile, ucModuleName);
    strcat(pucFilenameINI, ucModuleName);
    strcat(pucFilenameHLP, ucModuleName);
    strcat(pucPathDLL, ucModuleName);
    strcpy(strchr(pucFilenameProfile, '.'), ".cfg");
    strcpy(strchr(pucFilenameINI, '.'), ".ini");
    strcpy(strchr(pucFilenameHLP, '.'), ".hlp");
    strcpy(strrchr(pucPathDLL, '\\'), "");
/*                                                                                      *\
 * Now initilize Help, if it can't be initialized the we get no help but that's no      *
 * reason to terminate.                                                                 *
\*                                                                                      */
    if(WinStartHelp(habPc2, pucFilenameHLP, &hwndHelp, hwndFrame)==FALSE)
        PM_ERR(habPc2, hwndFrame, HELP_PC2HELP, MB_INFORMATION|MB_OK|MB_MOVEABLE|MB_DEFBUTTON1,
            "Can't find or load from PC2.HLP, please check that PC2.HLP is in the directory you "\
            "started PC/2 from. All help help requests will be ignored - continuing...");
/*                                                                                      *\
 * Check if we are allready loaded before by querying a semaphore that is defined the   *
 * first time PC/2 runs.                                                                *
\*                                                                                      */
    if(DosCreateEventSem(               /* Create a semaphore */
        PC2_SEM,                        /* Name */
        &hevPc2,                        /* Handle */
        (ULONG)0,                       /* Named semaphores are allways shared */
        (BOOL32)FALSE))                 /* Initially set */
        {                               /* If an error occurs, either we can't create
                                           the semaphore or it allready exists. We assume
                                           that it exists, meaning PC/2 allready loaded.
                                           Parameters NULL force an immediate exit(1). */
        USR_ERR(hwndFrame, HELP_PC2LOADED, MB_ERROR|MB_OK|MB_HELP|MB_MOVEABLE|MB_DEFBUTTON1,
            "One instance of PC/2 is already loaded - exiting...");
        break;
        }
/*                                                                                      *\
 * Load the Pc2Hook DLL either from the current directory or a LIBPATH path and         *
 * obtain the addresses of the entrypoints. There seems to be a little bug?, when the   *
 * library name contains a .DLL extension - the DLL is loaded sucessfully but not       *
 * always correct initialized? and calling functions in the DLL tend to fail. Extension *
 * .DLL therefore not appended to library name.                                         *
\*                                                                                      */
    {
    UCHAR       ucBuffer[80];           /* Buffer for possible return codes from DLL loading */
    UCHAR       ucDrive;
    UCHAR       *pucRunWorkPlace;       /* Scan environment for active WPS process */
    HMODULE     hDLLDOSCALL1;           /* Handle of DOSCALL1.DLL to get APIs for dynamically
                                           changing the LIBPATH for WARP+ */
    KEYDATA     *pKD;                   /* Used to clear KEYDATA table in PC2HOOL.DLL */
    ULONG       ulIndex;                /* KEYDATA table index */

                                        /* Get drive of PC/2's startup directory
                                           1=A, 2=B, 3=C,.... and direcotry itself */
    ucDrive=tolower(pucPathDLL[0]);
    DosSetDefaultDisk(++ucDrive-'a');
    DosSetCurrentDir(pucPathDLL+2);
    Rc=DosLoadModule(                   /* Load the DLL of PC/2 */
        ucBuffer,                       /* Save failure there */
        sizeof(ucBuffer)-1,             /* Length of save area */
        "PC2Hook",                      /* Fully qualified Library name */
        &hDLLPc2Hook);                   /* DLL module handle */
    if(Rc!=NO_ERROR)
        {                               /* DLL couldn't be found in the current PC/2
                                           directory or via the LIBPATH path */
        DOS_ERR(Rc, hwndFrame, HELP_DLLLOADED, MB_ERROR|MB_OK|MB_HELP|MB_MOVEABLE|MB_DEFBUTTON1,
            "PC/2 can't find PC2HOOK.DLL, please check that the DLL resides in the directory "\
            "PC/2 was started from - exiting");
        break;
        }
    Rc=DosQueryProcAddr(                /* Now get the address of the functions within the DLL */
        hDLLPc2Hook,                    /* DLL module handle */
        4,                              /* Ordinal number of procedure whose address is desired */
        "HookParameters",               /* Procedure name being referenced */
                                        /* Procedure address returned */
        (PFN *)(&pHP));
    if(Rc!=NO_ERROR)
        {                               /* An error occured */
        DosFreeModule(hDLLPc2Hook);     /* Free DLL reference */
        DOS_ERR(Rc, hwndFrame, HELP_DLLLOADED, MB_ERROR|MB_OK|MB_HELP|MB_MOVEABLE|MB_DEFBUTTON1,
            "PC/2 can't load from PC2HOOK.DLL, maybe you are using an outdated version - exiting");
        break;
        }
    memset(pHP, 0, sizeof(HOOKPARAMETERS));
    pHP->hDLLPc2Hook=hDLLPc2Hook;
    Rc=DosQueryProcAddr(hDLLPc2Hook, 1, "PC2DLL_SetParameters", (PFN *)(&pSetParameters));
    if(Rc!=NO_ERROR)
        {                               /* An error occured */
        DosFreeModule(hDLLPc2Hook);     /* Free DLL reference */
        DOS_ERR(Rc, hwndFrame, HELP_DLLLOADED, MB_ERROR|MB_OK|MB_HELP|MB_MOVEABLE|MB_DEFBUTTON1,
            "PC/2 can't load from PC2HOOK.DLL, maybe you are using an outdated version - exiting");
        break;
        }
    Rc=DosQueryProcAddr(hDLLPc2Hook, 2, "PC2DLL_InputHook", (PFN *)(&pInputHook));
    if(Rc!=NO_ERROR)
        {                               /* An error occured */
        DosFreeModule(hDLLPc2Hook);     /* Free DLL reference */
        DOS_ERR(Rc, hwndFrame, HELP_DLLLOADED, MB_ERROR|MB_OK|MB_HELP|MB_MOVEABLE|MB_DEFBUTTON1,
            "PC/2 can't load from PC2HOOK.DLL, maybe you are using an outdated version - exiting");
        break;
        }
    Rc=DosQueryProcAddr(hDLLPc2Hook, 3, "PC2DLL_WinSendMsgHook", (PFN *)(&pWinSendMsgHook));
    if(Rc!=NO_ERROR)
        {
        DosFreeModule(hDLLPc2Hook);
        DOS_ERR(Rc, hwndFrame, HELP_DLLLOADED, MB_ERROR|MB_OK|MB_HELP|MB_MOVEABLE|MB_DEFBUTTON1,
            "PC/2 can't load from PC2HOOK.DLL, maybe you are using an outdated version - exiting");
        break;
        }
    Rc=DosQueryProcAddr(hDLLPc2Hook, 5, "KeyData", (PFN *)(&pHP->pKeyData));
    if(Rc!=NO_ERROR)
        {
        DosFreeModule(hDLLPc2Hook);
        DOS_ERR(Rc, hwndFrame, HELP_DLLLOADED, MB_ERROR|MB_OK|MB_HELP|MB_MOVEABLE|MB_DEFBUTTON1,
            "PC/2 can't load from PC2HOOK.DLL, maybe you are using an outdated version - exiting");
        break;
        }
/*                                                                                      *\
 * Initialize HookParameters structure.                                                 *
\*                                                                                      */
                                        /* Because PC2HOOK.DLL may already be loaded in the system,
                                           we have to clear the Hotkeys already used from the previous
                                           invokation of PC/2. E.g. IBM CM/2 does lock PC2HOOK.DLL,
                                           for unknown reasons, even if PC/2 gets terminated, so
                                           the next invokation may get uninitialized Hotkey data */
    for(pKD=pHP->pKeyData, ulIndex=0; ulIndex<KEYDATACOUNT; ulIndex++, pKD++)
        pKD->bUsed=FALSE;
    pHP->pTib=ptib;                     /* Save thread and process information block of PC/2 */
    pHP->pPib=ppib;
    pHP->ProcessId=ppib->pib_ulpid;
    pHP->PriorityClass=(USHORT)((UCHAR)(ptib->tib_ptib2->tib2_ulpri & (~0xFF)) >> 8);
    pHP->PriorityDelta=(SHORT)((CHAR)(ptib->tib_ptib2->tib2_ulpri & 0xFF));
    pHP->MenuDataId=ID_POPUPMENU;
    pHP->habPc2=habPc2;
    pHP->hmqPc2=hmqPc2;
    pHP->hwndFrame=hwndFrame;
    pHP->hwndClient=hwndClient;
    pHP->hwndHelp=hwndHelp;
    pHP->ulOverviewFCF=FCF_OVERVIEWWINDOW;
    pHP->ulSpoolerFCF=FCF_SPOOLERWINDOW;
    strcpy(pHP->ucPathDLL, pucPathDLL);
    free(pucPathDLL);
    strcpy(pHP->ucFilenameINI, pucFilenameINI);
    free(pucFilenameINI);
    pHP->DosSetExtLIBPATH=NULL;         /* First assume this new APIs are not available */
    pHP->DosQueryExtLIBPATH=NULL;
    pHP->hbmMemory=NULLHANDLE;          /* Clear in memory bitmp and presentation space handle */
    pHP->hpsMemory=NULLHANDLE;
/*                                                                                      *\
 * Load DOSCALL1.998 DOSSETEXTLIBPATH and DOSCALL1.999 DOSQUERYEXTLIBPATH OS/2 APIs,    *
 * which were newly introduced by OS/2 3.00 WARP. These APIs are use to set and query   *
 * the EXTENDED LIBPATH, allowing to change the search path for dynamically loading     *
 * DLLs. These settings are bound to the process's information block. Applications      *
 * started by PC/2 may specify values for these variables, which are inherited from     *
 * PC/2 after setting them before starting the application.                             *
\*                                                                                      */
    if(!DosLoadModule(ucBuffer, sizeof(ucBuffer)-1, "DOSCALL1", &hDLLDOSCALL1)!=NO_ERROR)
        {
        DosQueryProcAddr(hDLLDOSCALL1, 998, "DOSSETEXTLIBPATH", (PFN *)(&pHP->DosSetExtLIBPATH));
        DosQueryProcAddr(hDLLDOSCALL1, 999, "DOSQUERYEXTLIBPATH", (PFN *)(&pHP->DosQueryExtLIBPATH));
        }
    DosFreeModule(hDLLDOSCALL1);
                                        /* Clear extended LIBPATH to prevent inheritance
                                           of unwanted data from a previous launch that set
                                           the extended LIBPATH */
    if(pHP->DosSetExtLIBPATH)
        pHP->DosSetExtLIBPATH("", BEGINLIBPATH);
    if(pHP->DosSetExtLIBPATH)
        pHP->DosSetExtLIBPATH("", ENDLIBPATH);
/*                                                                                      *\
 * Load from PC2.INI and test for PC/2 running as a WPS replacement.                    *
\*                                                                                      */
                                        /* Get data from PC2.INI into HookParameters. Commandline
                                           parameters may overload the settings in PC2.INI. */
    if(!(INIAccess(pHP->ucFilenameINI, TRUE)))
        USR_ERR(hwndFrame, HELP_PC2INI, MB_INFORMATION|MB_OK|MB_HELP|MB_MOVEABLE|MB_DEFBUTTON1,
            "Can't load configuration data from PC2.INI. "\
            "Either you have started PC/2 for the first time, "\
            "or you upgraded from a previous version of PC/2, or PC2.INI "\
            "is defective. For unreadable data default values are assumed - continuing...");
    pucRunWorkPlace=NULL;
                                        /* Query environment for RUNWORKPLACE=[d:]path\PC2.EXE */
    DosScanEnv("RUNWORKPLACE", (char **)&pucRunWorkPlace);
                                        /* Find P of PC2.EXE (or PMSHELL.EXE) */
    for( ; *pucRunWorkPlace; pucRunWorkPlace++)
        {
        if((*pucRunWorkPlace!='P') && (*pucRunWorkPlace!='p')) continue;
                                        /* We found a P or p now look for PC2.EXE */
        if(!strnicmp(pucRunWorkPlace, "PC2.EXE", 8))
            {
                                        /* If PC/2 is the WPS process save fact */
            pHP->ulStatusFlag|=PC2RUNNINGASWPS;
                                        /* PC/2 as WPS process should appear in Window List,
                                           to enable the user to unhide PC/2 after hiding,
                                           but can't be closed from Window List, Exit smart-
                                           icon or Exit PC/2 menuentry. The door using the
                                           accelerator F3 will stay open, but undokumented. */
            }
        }
/*                                                                                      *\
 * If PC/2 is running as the WPS process, get PC2SPOOL.DLL module handle.               *
\*                                                                                      */
    if((pHP->ulStatusFlag & PC2RUNNINGASWPS) || (pHP->ulStatusFlag & SHOWSPOOLERWINDOW))
        {
                                        /* Fully qualified path of PC2SPOOL.DLL */
        Rc=DosLoadModule(ucBuffer, sizeof(ucBuffer)-1, "PC2Spool", &hDLLPc2Spooler);
        if(Rc!=NO_ERROR)
            {                           /* DLL couldn't be found in the current PC/2
                                           directory or via the LIBPATH path */
            DOS_ERR(Rc, hwndFrame, HELP_PC2SPOOLDLL, MB_INFORMATION|MB_OK|MB_HELP|MB_MOVEABLE|MB_DEFBUTTON1,
                "PC/2 can't find PC2SPOOL.DLL, please check that the DLL resides in the directory "\
                "PC/2 was started from. Spooler window services will not be available - continuing...");
                pHP->ulStatusFlag&=(~PC2RUNNINGASWPS);
            }
        else
            {
            if(DosQueryProcAddr(hDLLPc2Spooler, 1, "SpoolerInitialize", (PFN *)(&pSpoolerInitialize))!=NO_ERROR)
                DosFreeModule(hDLLPc2Spooler);
                DOS_ERR(Rc, hwndFrame, HELP_PC2SPOOLDLL, MB_ERROR|MB_OK|MB_HELP|MB_MOVEABLE|MB_DEFBUTTON1,
                    "PC/2 can't load from PC2SPOOL.DLL, maybe you are using an outdated version - continuing...");
            }
        }
    pHP->hDLLPc2Spooler=hDLLPc2Spooler;
    }
/*                                                                                      *\
 * Now check possible commandline parameters, but read PC2.INI first to overwrite       *
 * values read by commandline parameters. Again, the portable way to access them is via *
 * the process information block.                                                       *
\*                                                                                      */
    {
    UCHAR   *pucCommandline;            /* Pointer to the commandline passed during invokation */
    UCHAR   *pucNextParameter;          /* Pointer to the current commandline parameter */
                                        /* Default the Popup-Menu is displayed after a
                                           WM_BUTTON1DBLCLK on the Desktop, but users may
                                           prefer WM_BUTTON1CLICK to popup the menu, which
                                           is set by default during reading PC2.INI, but
                                           we never know if there is a chance to miss it,
                                           so we explicitely test this */
    if(pHP->ulClickFlag==0) pHP->ulClickFlag=WM_BUTTON1DBLCLK;
                                        /* The commandline immediately follows the full qualified
                                           path PC/2 was invoked  from */
    pucCommandline=pHP->pPib->pib_pchcmd+strlen(pHP->pPib->pib_pchcmd)+1;
    strupr(pucCommandline);             /* Uppercase commandline parameters, which are separated
                                           by (one or more) spaces */
                                        /* Loop as long commandline parameters are available */
    for(pucNextParameter=pucCommandline;
        (pucNextParameter && *pucNextParameter);
        pucNextParameter=strchr(++pucNextParameter, ' '))
        {
                                        /* Skip spaces */
        while(*pucNextParameter==' ') pucNextParameter++;
                                        /* Test for /PROFILE or -PROFILE to get a
                                           profile name */
        if((strncmp(pucNextParameter, "/PROFILE", sizeof("/PROFILE")-1)==0) ||
            (strncmp(pucNextParameter, "-PROFILE", sizeof("-PROFILE")-1)==0))
            {
            UCHAR   *pucProfile;
            ULONG   ulCounter;

                                        /* Find next parameter */
            pucNextParameter=strchr(pucNextParameter, ' ');
            while(*pucNextParameter==' ')
                pucNextParameter++;
            ulCounter=0;
                                        /* For HPFS. the filename may be enclosed in quotes (") */
            if(*pucNextParameter=='"')
                {
                pucNextParameter++;
                while((*(pucNextParameter+ulCounter)!='"') && (*(pucNextParameter+ulCounter)!='\0'))
                    ulCounter++;
                }
            else
                while((*(pucNextParameter+ulCounter)!=' ') && (*(pucNextParameter+ulCounter)!='\0'))
                    ulCounter++;
            pucProfile=pucFilenameProfile+strlen(pucFilenameProfile)-7;
                                        /* Get the filename */
            strncpy(pucProfile, pucNextParameter, ulCounter);
            *(pucProfile+ulCounter)='\0';
            pucNextParameter+=ulCounter;
            continue;
            }
                                        /* Test for /INSTALL or -INSTALL to start the help
                                           panels during initialization */
        if((strncmp(pucNextParameter, "/INSTALL", sizeof("/INSTALL")-1)==0) ||
            (strncmp(pucNextParameter, "-INSTALL", sizeof("-INSTALL")-1)==0))
            pHP->ulStatusFlag |= INSTALLATIONHELP;
                                        /* Test for /NOAUTOSTART or -NOAUTOSTART to start the help
                                           panels during initialization */
        if((strncmp(pucNextParameter, "/NOAUTOSTART", sizeof("/NOAUTOSTART")-1)==0) ||
            (strncmp(pucNextParameter, "-NOAUTOSTART", sizeof("-NOAUTOSTART")-1)==0))
            pHP->ulStatusFlag |= IGNOREAUTOSTART;
                                        /* Test for /DOUBLECLICK or -DOUBLECLICK to display
                                           the Popup-Menu after a double-click instead of
                                           a single click */
        if((strncmp(pucNextParameter, "/DOUBLECLICK", sizeof("/DOUBLECLICK")-1)==0) ||
            (strncmp(pucNextParameter, "-DOUBLECLICK", sizeof("-DOUBLECLICK")-1)==0))
            pHP->ulClickFlag=WM_BUTTON1DBLCLK;
                                        /* Test for /SINGLECLICK or -SINGLECLICK to display
                                           the Popup-Menu after a double-click instead of
                                           a single click */
        if((strncmp(pucNextParameter, "/SINGLECLICK", sizeof("/SINGLECLICK")-1)==0) ||
            (strncmp(pucNextParameter, "-SINGLECLICK", sizeof("-SINGLECLICK")-1)==0))
            pHP->ulClickFlag=WM_BUTTON1CLICK;
        }
    }
/*                                                                                      *\
 * Initialize PC2HOOK.DLL.                                                              *
\*                                                                                      */
    pSetParameters();                   /* Initialize data in PC2Hook DLL */
/*                                                                                      *\
 * Subclass smart icon menu bar to catch changes of presentation parameters.            *
\*                                                                                      */
                                        /* Get smarticon menu window handle */
    pHP->hwndMenu=WinWindowFromID(pHP->hwndFrame, FID_MENU);
                                        /* Subclass menu window procedure to get hold of
                                           presentation parameter changes */
    pfnMenuWindowProc=WinSubclassWindow(pHP->hwndMenu, SubclassedMenuWindowProc);
/*                                                                                      *\
 * Now setup the Popup-Menu by loading the data from the profile and install the hook   *
 * into the system input queue, load the Popup-Menu while PC/2 startup.                 *
\*                                                                                      */
                                        /* Load popup menu */
    pHP->hwndPopupMenu=WinLoadMenu(
        pHP->hwndClient,                /* Owner window handle */
        (HMODULE)0,                     /* Ressource in .EXE file */
        ID_PC2POPUPMENU);               /* Menu identifier in ressource file */
                                        /* Load the data from the profile */
                                        /* Set this font also into the Popup-Menu */
    WinSetPresParam(pHP->hwndPopupMenu, PP_FONTNAMESIZE,
        sizeof(pHP->ucPopupMenuFont), pHP->ucPopupMenuFont);
    WinPostMsg(pHP->hwndClient, WM_PC2STARTUP, NULL, NULL);
/*                                                                                      *\
 * Here we loop dispatching the messages...                                             *
\*                                                                                      */
    while(TRUE)
        {
        while(WinGetMsg(pHP->habPc2, &qmsg, 0, 0, 0))
                                        /* Dispatch messages to window procedure */
            WinDispatchMsg(pHP->habPc2, &qmsg);
                                        /* Break only when one WM_QUIT occurred before, because
                                           only after WM_CLOSE was processed we are really ready
                                           to shutdown (and WM_CLOSE posts WM_QUIT again) */
        if(pHP->ulStatusFlag & QUITFROMWINDOWLIST) break;
                                        /* Dispatch WM_QUIT */
        WinDispatchMsg(pHP->habPc2, &qmsg);
                                        /* If PC/2 is running as a WPS replacement, tell the user
                                           that the workplace process is always restarted */
        if(pHP->ulStatusFlag&PC2RUNNINGASWPS)
            USR_ERR(pHP->hwndFrame, HELP_PC2RUNNINGASWPS, MB_ERROR|MB_OK|MB_HELP|MB_MOVEABLE|MB_DEFBUTTON1,
                "PC/2 is running as the WPS (WorkPlace Shell) process. The WorkPlace process mustn't be "\
                "terminated. PC/2 will ignore your request - continuing...");
                                        /* If PC/2 is not running as a WPS replacement, or an
                                           unrecoverable error occured, accept the shutdown request
                                           but use WM_CLOSE to terminate in an orderly manner */
        if((!(pHP->ulStatusFlag&PC2RUNNINGASWPS)) ||
            (pHP->ulStatusFlag&PC2EMERGENCYEXIT))
            {
                                        /* Flag that we are using WM_CLOSE to make a clean shutdown */
            pHP->ulStatusFlag|=QUITFROMWINDOWLIST;
                                        /* Post WM_CLOSE to shutdown PC/2 in a orderly manner,
                                           which posts WM_QUIT again afterwards PC/2 cleaned up */
            WinPostMsg(pHP->hwndClient, WM_CLOSE, NULL, NULL);
            }
        }
                                        /* Close window */
    WinDestroyWindow(pHP->hwndFrame);
} while (FALSE);
free(pucFilenameProfile);               /* Free path references */
free(pucFilenameHLP);
                                        /* Wait for working thread to close */
if(tidThread)
    DosWaitThread(&tidThread, DCWW_WAIT);
if(pHP!=NULL)
    {
                                        /* Destroy in memory bitmap */
    if(pHP->hbmMemory) GpiDeleteBitmap(pHP->hbmMemory);
                                        /* Destroy in memory presenation space */
    if(pHP->hpsMemory) GpiDestroyPS(pHP->hpsMemory);
                                        /* Destroy in memory device context */
    if(hdcMemory) DevCloseDC(hdcMemory);
    }
                                        /* Free DLL references */
if(hDLLPc2Spooler)
    DosFreeModule(hDLLPc2Spooler);
if(hDLLPc2Hook)
    DosFreeModule(hDLLPc2Hook);
if(WinCloseDown(&hwndHelp, &habPc2, &hmqPc2)==FALSE)
    return(1);
else
    return(0);
}

/*--------------------------------------------------------------------------------------*\
 * This procedure is the PC/2 window procedure.                                         *
\*--------------------------------------------------------------------------------------*/
MRESULT EXPENTRY PC2_MainWindowProc(HWND hwnd, ULONG msg, MPARAM mp1, MPARAM mp2)
{
static ULONG    ulThreadReady=0;        /* Working thread ready indicator. It may be seen as (or
                                           replaced by) a semaphore that is set (THREADBUSY) whenever the
                                           working thread is doing his (time-consuming) work. Requests
                                           for further work are ignored until the thread gets ready
                                           again, to avoid queuing of work (which may be invalidated
                                           by the last job) */
static USHORT   usMenuCommand;          /* Last message from smarticons */
static HPOINTER hPointerMove;           /* Pointer when moving windows on overview window */
static HPOINTER hPointerAction;         /* Pointer when performing the other actions on windows */
static USHORT   usTimer;                /* Timer identity used to update spooler regularily */
static ULONG    ulTimerTicks=0;         /* Timer ticks (about every 5 seconds) */

switch(msg)
{
case WM_CREATE:                         /* Create window by WinCreateStdWindow() */
                                        /* First call default window procedure */
    WinDefWindowProc(hwnd, msg, mp1, mp2);
    break;

case WM_PAINT:
    {
    HPS     hpsClient;
    RECTL   rectlClient;

                                        /* Get a cached presentation space */
    hpsClient=WinBeginPaint(hwnd, NULLHANDLE, &rectlClient);
    if(pHP!=NULL)
        WinPostMsg(pHP->hwndThread, WM_PAINT, NULL, NULL);
    else
        WinFillRect(hpsClient, &rectlClient, CLR_WHITE);
    WinEndPaint(hpsClient);
    }
    break;

case WM_PRESPARAMCHANGED:
    if((ULONG)mp1==PP_FONTNAMESIZE)
        {
        ULONG       ulAttrFound;
        HPS         hpsClient;
        FONTMETRICS fmPC2Font;
        FATTRS      *pfatPC2Font;

                                        /* Get font selected for PC/2's overview window */
        if(WinQueryPresParam(hwnd, PP_FONTNAMESIZE, 0, &ulAttrFound,
            sizeof(pHP->ucPC2WindowFont), pHP->ucPC2WindowFont, 0))
            {
                                        /* Allocate FATTRS structure for overview window */
            pfatPC2Font=(FATTRS *)malloc(sizeof(FATTRS));
                                        /* Get new font */
            hpsClient=WinGetPS(pHP->hwndClient);
            GpiQueryFontMetrics(hpsClient, sizeof(FONTMETRICS), &fmPC2Font);
            WinReleasePS(hpsClient);
            pfatPC2Font->usRecordLength=sizeof(FATTRS);
            pfatPC2Font->fsSelection=0;
            pfatPC2Font->lMatch=0;
            strcpy(pfatPC2Font->szFacename, fmPC2Font.szFacename);
            pfatPC2Font->idRegistry=fmPC2Font.idRegistry;
            pfatPC2Font->usCodePage=fmPC2Font.usCodePage;
            if(fmPC2Font.fsDefn&FM_DEFN_OUTLINE)
                {
                pfatPC2Font->lMaxBaselineExt=fmPC2Font.lMaxBaselineExt;
                pfatPC2Font->lAveCharWidth=0;
                pfatPC2Font->fsType=0;
                pfatPC2Font->fsFontUse=FATTR_FONTUSE_OUTLINE|FATTR_FONTUSE_TRANSFORMABLE;
                }
            else
                {
                pfatPC2Font->lMaxBaselineExt=fmPC2Font.lMaxBaselineExt;
                pfatPC2Font->lAveCharWidth=fmPC2Font.lAveCharWidth;
                pfatPC2Font->fsType=0;
                pfatPC2Font->fsFontUse=FATTR_FONTUSE_NOMIX;
                }
                                        /* Post to working thread to change overview window font */
            WinPostMsg(pHP->hwndThread, WM_PRESPARAMCHANGED,
                MPFROMLONG(PP_FONTNAMESIZE), MPFROMP(pfatPC2Font));
            }
        }
    break;

/*                                                                                      *\
 * Syntax: WM_PC2STARTUP, NULL, NULL                                                    *
\*                                                                                      */
case WM_PC2STARTUP:
/*                                                                                      *\
 * This message is posted during main window initialization, after the access to the    *
 * HOOKPARAMETERS, loaded from PC2HOOK.DLL, is possible, to complete startup.           *
\*                                                                                      */
    {
    SWP     swp;                        /* According to a tip from members of OS/2 development in
                                           an article in OS/2 Developer magazine, it is more
                                           efficient to call WinSetMultWindowPos(), because
                                           WinSetWindowPost() internally calls WinSetMultWindowPos()
                                           (even if just one window is moved) */
    ULONG   ulChangedFCF=0;             /* Frame flags to adjust */

                                        /* Get the current screen resolution settings. We call
                                           to avoid WM_SIZE/WM_MOVE messages to be processed in
                                           between, because they rely on valid screen resolution data */
    WinSendMsg(pHP->hwndClient, WM_PC2STARTUPSCREEN, NULL, NULL);
                                        /* Setup a memory device contect to draw into, when
                                           using a "fast" video subsystem, e.g. XGA-2 */
    WinSendMsg(pHP->hwndClient, WM_PC2STARTUPMEMORYDC, NULL, NULL);
                                        /* During initial creation, the (invisible) overview
                                           window was created with a titlebar and smarticonbar.
                                           If this assumption was false correct it */
    if(pHP->ulStatusFlag&HIDETITLEBAR)
        ulChangedFCF|=FCF_TITLEBAR|FCF_HIDEBUTTON;
    if(pHP->ulStatusFlag&HIDESMARTICONBAR)
        ulChangedFCF|=FCF_MENU;
    if(ulChangedFCF)
        WinSendMsg(pHP->hwndClient, WM_OVERVIEWFCF, MPFROMLONG(ulChangedFCF), NULL);
                                        /* Load pointers from resource */
    hPointerMove=WinLoadPointer(HWND_DESKTOP, NULLHANDLE, ID_MOVEPOINTER);
    hPointerAction=WinLoadPointer(HWND_DESKTOP, NULLHANDLE, ID_ACTIONPOINTER);
    usMenuCommand=ID_ICONMOVE;          /* Setup smarticon's control */
    pHP->Windows.ulDesktop=(ULONG)-1;   /* Set to -1 because currently we don't know Desktop's name */
                                        /* Set to -1 because currently we don't know Window List's name */
    pHP->Windows.ulWindowList=(ULONG)-1;
                                        /* For installation display help panels */
    if(pHP->ulStatusFlag & INSTALLATIONHELP)
        WinPostMsg(hwnd, WM_COMMAND, MPFROMSHORT(ID_HELP), NULL);
/*                                                                                      *\
 * Set text to titlebar. Size & position and ensurance that position is limited is      *
 * done in WM_SIZE and WM_MOVE during window creation.                                  *
\*                                                                                      */

    WinSetWindowText(pHP->hwndFrame, PC2_OVERVIEW_WINDOW);
    swp.fl=SWP_HIDE|SWP_MOVE|SWP_SIZE|SWP_DEACTIVATE;
    swp.x=pHP->swpPC2.x;
    swp.y=pHP->swpPC2.y;
    swp.cx=pHP->swpPC2.cx;
    swp.cy=pHP->swpPC2.cy;
    swp.hwndInsertBehind=NULLHANDLE;
    swp.hwnd=pHP->hwndFrame;
    WinSetMultWindowPos(pHP->habPc2, &swp, 1);
                                        /* Create the working thread */
    WinPostMsg(pHP->hwndClient, WM_PC2STARTUPTHREAD, NULL, NULL);
                                        /* Now load the Popup-Menu from the profile */
    WinPostMsg(pHP->hwndClient, WM_SETPOPUPMENU, NULL, NULL);
                                        /* Now install the hook */
    WinPostMsg(pHP->hwndClient, WM_LOADHOOK, NULL, NULL);
    }
    break;

/*                                                                                      *\
 * Syntax: WM_PC2STARTUPSCREEN, NULL, NULL                                              *
\*                                                                                      */
case WM_PC2STARTUPSCREEN:
/*                                                                                      *\
 * Now get device resolution and save it and other initilization data to the structure  *
 * used to communicate with PC2HOOK.DLL.                                                *
\*                                                                                      */
    {
    POINTL  DesktopCountMin, DesktopCountMax;

                                        /* Query and save the device resolution, f.e.
                                           1024 * 768 */
    pHP->DesktopSize.x=pHP->swpScreen.cx=WinQuerySysValue(HWND_DESKTOP, SV_CXSCREEN);
    pHP->DesktopSize.y=pHP->swpScreen.cy=WinQuerySysValue(HWND_DESKTOP, SV_CYSCREEN);
                                        /* Get number of horizontal and vertical Virtual Desktops */
    DesktopCountMin.x=(pHP->ulHorizontalDesktops-1)>>1;
    DesktopCountMin.y=(pHP->ulVerticalDesktops-1)>>1;
    DesktopCountMax.x=pHP->ulHorizontalDesktops-DesktopCountMin.x;
    DesktopCountMax.y=pHP->ulVerticalDesktops-DesktopCountMin.y;
                                        /* Now initialized the virtual Desktop data */
    pHP->LLHotBorder.x=pHP->DesktopSize.x*0.15;
    pHP->LLHotBorder.y=pHP->DesktopSize.y*0.15;
    pHP->URHotBorder.x=pHP->DesktopSize.x*0.85;
    pHP->URHotBorder.y=pHP->DesktopSize.y*0.85;
    pHP->VirtualDesktopPos.x=0;
    pHP->VirtualDesktopPos.y=0;
                                        /* Calculate absolute lower left and upper right
                                           corners of the Virtual Desktop */
    pHP->VirtualDesktopMin.x=(-pHP->DesktopSize.x)*DesktopCountMin.x;
    pHP->VirtualDesktopMin.y=(-pHP->DesktopSize.y)*DesktopCountMin.y;
    pHP->VirtualDesktopMax.x=pHP->DesktopSize.x*DesktopCountMax.x;
    pHP->VirtualDesktopMax.y=pHP->DesktopSize.y*DesktopCountMax.y;
    pHP->SlidingXFactor=(pHP->ulScrollPercentage*
        pHP->DesktopSize.x)/100;
    pHP->SlidingYFactor=(pHP->ulScrollPercentage*
        pHP->DesktopSize.y)/100;
    pHP->hwndDesktop=0;     /* Initialize to a value not used by PM (hopefully) */
    pHP->hwndWPS=0;
    }
    break;

/*                                                                                      *\
 * Syntax: WM_PC2STARTUPMEMORYDC, NULL, NULL                                            *
\*                                                                                      */
case WM_PC2STARTUPMEMORYDC:
/*                                                                                      *\
 * Now setup a memory device context and associate a presentation space into. Into this *
 * presentation sapce we draw the overview window and clip it into the client area      *
 * presenation space after all drawing was finished. This avoids the flickering in the  *
 * Overview window caused that we draw all windows from the bottom to the top of the    *
 * Z-order which causes windows to be drawn and overlayed immediately afterwards by     *
 * another window.                                                                      *
\*                                                                                      */
    {
    SIZEL   sizelClient;

                                            /* I'm not sure, why we can create a zero sized
                                               presentation space, but draw any sized graphics
                                               into, possibly because we associate a bitmap
                                               to the presentation space, which is sized
                                               according to the client area size. */
    sizelClient.cx=0;
    sizelClient.cy=0;
                                            /* Get memory device context */
    hdcMemory=DevOpenDC(pHP->habPc2, OD_MEMORY, "*", 0L, NULL, 0L);
                                            /* Create a presentation space into the memory device context */
    pHP->hpsMemory=GpiCreatePS(pHP->habPc2, hdcMemory, &sizelClient, PU_PELS|GPIA_ASSOC);
                                            /* Change color table into RGB mode */
    GpiCreateLogColorTable(pHP->hpsMemory, 0L, LCOLF_RGB, 0L, 0L, NULL);
    }
    break;

/*                                                                                      *\
 * Syntax: WM_PC2STARTUPTHREAD, NULL, NULL                                              *
\*                                                                                      */
case WM_PC2STARTUPTHREAD:
/*                                                                                      *\
 * Now that frame and client windows are created (of course invisible), start working   *
 * thread that opens a presentation space on the client area and performs all timely    *
 * intensive tasks.                                                                     *
\*                                                                                      */
                                        /* Start working thread */
    tidThread=_beginthread(PC2_Thread, NULL, 65536, NULL);
    if(tidThread==(TID)-1)
        {
        USR_ERR(pHP->hwndFrame, HELP_PC2THREAD, MB_ERROR|MB_OK|MB_HELP|MB_MOVEABLE|MB_DEFBUTTON1,
            "PC/2 can't start its working thread - exiting...");
                                        /* Post exit message */
        pHP->ulStatusFlag|=PC2EMERGENCYEXIT;
        WinPostMsg(pHP->hwndFrame, WM_QUIT, NULL, NULL);
        }
    break;

/*                                                                                      *\
 * Syntax: WM_OVERVIEWFCF, ULONG ulChangedFCF, NULL                                     *
\*                                                                                      */
case WM_OVERVIEWFCF:
/*                                                                                      *\
 * This message is posted by Desktop configuration dialog to change the titlebar and    *
 * menubar of PC/2 according to the user selection.                                     *
\*                                                                                      */
    {
    ULONG   ulChangedFCF;

    ulChangedFCF=(ULONG)LONGFROMMP(mp1);
                                        /* Update titlebar + hide button only if changed */
    if(ulChangedFCF&(FCF_TITLEBAR|FCF_HIDEBUTTON))
        {
        if(pHP->ulOverviewFCF&(FCF_TITLEBAR|FCF_HIDEBUTTON))
            {
            FRAMECDATA  framecdataOverview;

            if(pHP->ulDebug>=DEBUG_FULL)
                printf("PC2: WM_OVERVIEWFCF creating titlebar\n");
                                        /* Setup frame control data to create a titlebar and a
                                           hide button */
            framecdataOverview.cb=sizeof(FRAMECDATA);
            framecdataOverview.flCreateFlags=FCF_TITLEBAR|FCF_HIDEBUTTON;
            framecdataOverview.hmodResources=NULLHANDLE;
            framecdataOverview.idResources=ID_PC2MAINWINDOW;
            WinCreateFrameControls(pHP->hwndFrame, &framecdataOverview, NULL);
            }
        else
            {                           /* If titlebar and hide button are no longer requested,
                                           destroy them */
            if(pHP->ulDebug>=DEBUG_FULL)
                printf("PC2: WM_OVERVIEWFCF destroying titlebar\n");
            WinDestroyWindow(WinWindowFromID(pHP->hwndFrame, FID_TITLEBAR));
            WinDestroyWindow(WinWindowFromID(pHP->hwndFrame, FID_MINMAX));
            }
        }
                                        /* Update Smart Icon menubar only if changed */
    if(ulChangedFCF&(FCF_MENU))
        {
        if(pHP->ulOverviewFCF&FCF_MENU)
            {
            if(pHP->ulDebug>=DEBUG_FULL)
                printf("PC2: WM_OVERVIEWFCF creating smarticonbar\n");
            pHP->hwndMenu=WinLoadMenu(pHP->hwndFrame, NULLHANDLE, ID_PC2MAINWINDOW);
            }
        else
            {                           /* If menubar is no longer requested, destroy it */
            if(pHP->ulDebug>=DEBUG_FULL)
                printf("PC2: WM_OVERVIEWFCF destroying smarticonbar\n");
            WinDestroyWindow(WinWindowFromID(pHP->hwndFrame, FID_MENU));
            }
        }
                                        /* Force a redraw */
    if(ulChangedFCF)
        {
        WinSendMsg(pHP->hwndFrame, WM_UPDATEFRAME, MPFROMLONG(ulChangedFCF), NULL);
        WinSetWindowText(pHP->hwndFrame, PC2_OVERVIEW_WINDOW);
        }
    }
    break;

/*                                                                                      *\
 * Syntax: WM_THREADSTARTUPREADY, NULL, NULL                                            *
\*                                                                                      */
case WM_THREADSTARTUPREADY:
/*                                                                                      *\
 * Working thread is now ready, make overview window visible when requested.            *
\*                                                                                      */
    if(pHP->ulDebug>=DEBUG_FULL)
        printf("PC2: WM_THREADSTARTUPREADY\n");
                                        /* Set overview window font */
    WinSetPresParam(pHP->hwndClient, PP_FONTNAMESIZE,
        sizeof(pHP->ucPC2WindowFont), pHP->ucPC2WindowFont);
                                        /* Display overview window if requested */
    if(pHP->ulStatusFlag & OVERVIEW)
        {
        SWP     swp;

        swp.fl=SWP_SHOW|SWP_DEACTIVATE|SWP_ZORDER;
        swp.x=swp.y=swp.cx=swp.cy=0;
        swp.hwndInsertBehind=HWND_TOP;
        swp.hwnd=pHP->hwndFrame;
        WinSetMultWindowPos(pHP->habPc2, &swp, 1);
        WinPostMsg(pHP->hwndThread, WM_PAINT, NULL, NULL);
        }
                                        /* Scan for autostart applications, however if the user
                                           has for some reason disabled the general setting,
                                           don't autostart the sessions, even when autostart
                                           was selected */
    if(!(pHP->ulStatusFlag&IGNOREAUTOSTART))
        WinPostMsg(pHP->hwndThread, WM_AUTOSTART, NULL, NULL);
    if(pHP->ulDebug>=DEBUG_FULL)
        printf("PC2: Sending WM_AUTOSTART\n");
                                        /* Create a timer to update the spooler container regularily */
    for(usTimer=0; usTimer<=TID_USERMAX; usTimer++)
        if(WinStartTimer(pHP->habPc2, hwnd, usTimer, 1000*10)) break;
    break;

/*                                                                                      *\
 * Syntax: WM_SETPOPUPMENU, NULL, NULL                                                  *
\*                                                                                      */
case WM_SETPOPUPMENU:
/*                                                                                      *\
 * Open the profile for reading the linked list containing the popup menu data. If the  *
 * profile can't be opened, the file is assumed to be empty so the popup menu is empty. *
\*                                                                                      */
    {
    ULONG   ulSpoolerError;

                                        /* Assume the Configuration Dialog isn't part
                                           of the profile */
    pHP->ulStatusFlag|=DISPLAYCONFIGDIALOG;
    if((pHP->Pc2Profile=fopen(pucFilenameProfile, "r"))==NULL)
        {
                                        /* Allocate an empty MENUDATA structure used as
                                           the first element of linked list */
        pHP->pPopupMenu=AllocateMenuData();
        USR_ERR(pHP->hwndFrame, HELP_PC2CFG, MB_INFORMATION|MB_OK|MB_HELP|MB_MOVEABLE|MB_DEFBUTTON1,
            "Cannot open confguration file - assuming empty file");
        }
    else
        {
        static UCHAR    Buffer[256];

                                        /* Allocate an empty MENUDATA structure used as
                                           the first element of linked list */
        pHP->pPopupMenu=AllocateMenuData();
        fgets(Buffer, sizeof(Buffer), pHP->Pc2Profile);
        while(!feof(pHP->Pc2Profile))
            {
            if(strstr(Buffer, "PROFILE START"))
                {
                                        /* Load the rest by calling a recursive procedure */
                LoadMenu(pHP->pPopupMenu);
                break;
                }
            fgets(Buffer, sizeof(Buffer), pHP->Pc2Profile);
            }
        fclose(pHP->Pc2Profile);
        if(pHP->pPopupMenu->Item==ENTRYEMPTY)
            USR_ERR(pHP->hwndFrame, HELP_PC2CFG, MB_INFORMATION|MB_OK|MB_HELP|MB_MOVEABLE|MB_DEFBUTTON1,
                "Configuration file does not contain valid data - assuming empty file");
        }
                                        /* Initialize *MENUDATA for Configuration dialog
                                           procedure to a known value */
    pHP->pMenuData=pHP->pPopupMenu;
                                        /* If PC/2 is the WPS process or user requested
                                           to have the spooler window always, initialize spooler */
    if(pSpoolerInitialize==0)
        pHP->ulStatusFlag|=PC2SPOOLERERROR;
    if(((pHP->ulStatusFlag & PC2RUNNINGASWPS) || (pHP->ulStatusFlag & SHOWSPOOLERWINDOW)) &&
        !(pHP->ulStatusFlag & PC2SPOOLERERROR))
                                        /* Initialize the spooler */
        if((ulSpoolerError=pSpoolerInitialize(pHP->hDLLPc2Hook))!=NO_ERROR)
            {
            UCHAR   ucSpoolerError[256];

            sprintf(ucSpoolerError, "Initialisation of the OS/2 Spooler failed. Because the "\
                "Spooler PMSPL.DLL is part of any OS/2 installation, your installation may be"\
                "defective. The error code is %04X. - Continuing without PC/2 Spooler Control "\
                "Window...", (int)ulSpoolerError);
            USR_ERR(pHP->hwndFrame, HELP_PC2SPOOLDLL, MB_INFORMATION|MB_OK|MB_HELP|MB_MOVEABLE|MB_DEFBUTTON1,
                ucSpoolerError);
            pHP->ulStatusFlag|=PC2SPOOLERERROR;
                                        /* Release spooler DLL */
            DosFreeModule(hDLLPc2Spooler);
            hDLLPc2Spooler=NULLHANDLE;
            }
                                        /* If PC/2 is not the WPS process and Spooler is not enabled,
                                           or available, disable spooler menu item in Popup Menu */
    if(((!(pHP->ulStatusFlag & PC2RUNNINGASWPS)) && (!(pHP->ulStatusFlag & SHOWSPOOLERWINDOW))) ||
        (pHP->ulStatusFlag & PC2SPOOLERERROR))
        WinSendMsg(pHP->hwndPopupMenu, MM_SETITEMATTR,
            MPFROM2SHORT(ID_SPOOLER, TRUE), MPFROM2SHORT(MIA_DISABLED, MIA_DISABLED));
                                        /* If PC/2 is running as the WPS, then disable the Exit
                                           smarticon, and the Exit OS/2 menuentry */
    if(pHP->ulStatusFlag&PC2RUNNINGASWPS)
        {
        WinSendMsg(pHP->hwndPopupMenu, MM_SETITEMATTR,
            MPFROM2SHORT(ID_EXIT, TRUE), MPFROM2SHORT(MIA_DISABLED, MIA_DISABLED));
        WinSendMsg(pHP->hwndMenu, MM_SETITEMATTR,
            MPFROM2SHORT(ID_ICONEXIT, TRUE), MPFROM2SHORT(MIA_DISABLED, MIA_DISABLED));
        }
    }
    break;

/*                                                                                      *\
 * Syntax: WM_LOADHOOK, NULL, NULL                                                      *
\*                                                                                      */
case WM_LOADHOOK:
/*                                                                                      *\
 * Install the hook into the system input queue pointing to the PC2DLL_Hook() procedure *
 * in the DLL PC2HOOK.DLL. If we can't do this we exit after an error message box.      *
\*                                                                                      */
    if(WinSetHook(                      /* Set a hook */
        pHP->habPc2,                    /* Handle of anchor block */
        NULLHANDLE,                     /* Hook into system message queue */
        HK_INPUT,                       /* Hook of system input queue */
        (PFN)pInputHook,                /* Pointer to hook procedure */
        pHP->hDLLPc2Hook)==FALSE)
        {
        USR_ERR(pHP->hwndFrame, HELP_PC2HOOK, MB_INFORMATION|MB_OK|MB_HELP|MB_MOVEABLE|MB_DEFBUTTON1,
            "Hooking the system input queue failed - exiting...");
        pHP->ulStatusFlag|=PC2EMERGENCYEXIT;
        WinPostMsg(hwnd, WM_QUIT, NULL, NULL);
        }
                                        /* Hook of WinSendMsg() function when calling an
                                           application's window procedure */
    if(WinSetHook(pHP->habPc2, NULLHANDLE, HK_SENDMSG, (PFN)pWinSendMsgHook,
        pHP->hDLLPc2Hook)==FALSE)
        {
        USR_ERR(pHP->hwndFrame, HELP_PC2HOOK, MB_INFORMATION|MB_OK|MB_HELP|MB_MOVEABLE|MB_DEFBUTTON1,
            "Hooking the system sendmessage dispatcher failed - exiting...");
        pHP->ulStatusFlag|=PC2EMERGENCYEXIT;
        WinPostMsg(hwnd, WM_QUIT, NULL, NULL);
        }
    break;

case WM_SIZE:
case WM_MOVE:
/*                                                                                      *\
 * One fundamental assumption is, that when getting here, the HOOKPARAMETERS structure  *
 * already contains the screen resolution data. Its therefore most important to call    *
 * WM_PC2STARTUPSCREEN using the synchronous WinSendMsg() in WM_PC2STARTUP, otherwise   *
 * we get a 0 sized Overview window.                                                    *
\*                                                                                      */
    {
    LONG    lXBorder=WinQuerySysValue(HWND_DESKTOP, SV_CXSIZEBORDER);
    LONG    lYBorder=WinQuerySysValue(HWND_DESKTOP, SV_CYSIZEBORDER);
    SWP     swpPC2;

                                        /* Get the frame area size */
    WinQueryWindowPos(pHP->hwndFrame, &swpPC2);
    if((swpPC2.cx>pHP->DesktopSize.x) || (pHP->swpPC2.cx<=0))
        pHP->swpPC2.cx=pHP->DesktopSize.x/3;
    else
        pHP->swpPC2.cx=swpPC2.cx;
    if((swpPC2.cy>pHP->DesktopSize.y) || (pHP->swpPC2.cy<=0))
        pHP->swpPC2.cy=pHP->DesktopSize.y/4;
    else
        pHP->swpPC2.cy=swpPC2.cy;
    if((swpPC2.x<(0-lXBorder)) ||
        ((swpPC2.x+pHP->swpPC2.cx)>(pHP->DesktopSize.x+lXBorder)))
        pHP->swpPC2.x=0;
    else
        pHP->swpPC2.x=swpPC2.x;
    if((swpPC2.y<(0-lYBorder)) ||
        ((swpPC2.y+pHP->swpPC2.cy)>(pHP->DesktopSize.y+lYBorder)))
        pHP->swpPC2.y=0;
    else
        pHP->swpPC2.y=swpPC2.y;
    pHP->swpPC2.fl=SWP_SIZE|SWP_MOVE;
    pHP->swpPC2.hwndInsertBehind=NULLHANDLE;
    pHP->swpPC2.hwnd=pHP->hwndFrame;
    WinSetMultWindowPos(pHP->habPc2, &pHP->swpPC2, 1);
                                        /* Get the client area size */
    WinQueryWindowPos(pHP->hwndClient, &pHP->swpPC2Client);
                                        /* Request repaint from working thread when resized */
    if(msg==WM_SIZE)
        {

        BITMAPINFOHEADER2   bmihMemory; /* In memory Overview window bitmap info header */
                                        /* Bitmapformat of current display & driver combination */
        LONG                lBitmapFormat[2];

                                        /* Get the number of planes and the bitcount of the current
                                           display & driver combination */
        GpiQueryDeviceBitmapFormats(pHP->hpsMemory, 2L, lBitmapFormat);
                                        /* Create a bitmap into the memory presentation space of same
                                           size as client area */
        memset(&bmihMemory, 0, sizeof(BITMAPINFOHEADER2));
        bmihMemory.cbFix=sizeof(BITMAPINFOHEADER2);
        bmihMemory.cx=pHP->swpPC2Client.cx;
        bmihMemory.cy=pHP->swpPC2Client.cy;
        bmihMemory.cPlanes=lBitmapFormat[0];
        bmihMemory.cBitCount=lBitmapFormat[1];
                                        /* Delete previous bitmap if it exists */
        if(pHP->hbmMemory)
            GpiDeleteBitmap(pHP->hbmMemory);
        pHP->hbmMemory=GpiCreateBitmap(pHP->hpsMemory, &bmihMemory, 0L, NULL, NULL);
        GpiSetBitmap(pHP->hpsMemory, pHP->hbmMemory);
                                        /* Now repaint Overview window */
        WinPostMsg(pHP->hwndThread, WM_PAINT, NULL, NULL);
        }
    }
    break;

/*                                                                                      *\
 * Syntax: WM_MOUSEMOVE, NULL, NULL                                                     *
\*                                                                                      */
case WM_MOUSEMOVE:                      /* Set mouse pointer accordingly to last smarticon pressed */
    if(usMenuCommand==ID_ICONMOVE) WinSetPointer(HWND_DESKTOP, hPointerMove);
    else WinSetPointer(HWND_DESKTOP, hPointerAction);
    break;

/*                                                                                      *\
 * Syntax: WM_HOTKEY, (USHORT usFlags, USHORT usCh), ULONG ulKeyDataIndex               *
\*                                                                                      */
case WM_HOTKEY:
                                        /* Get the currently active top level window */
    pHP->hwndActiveWindow=WinQueryActiveWindow(HWND_DESKTOP);
                                        /* Just pass this message to the working thread */
    WinPostMsg(pHP->hwndThread, WM_HOTKEY, MPFROMLONG(mp1), MPFROMLONG(mp2));
    break;

case WM_TIMER:
/*                                                                                      *\
 * The regular timer is used to refresh the container window, to have a real time       *
 * status for the spooler option and to query the WPS window handle regularily.         *
\*                                                                                      */
    {
    if(pHP->ulDebug>=DEBUG_LOW)
        {
        DosBeep(1000,20);
        DosBeep(250,20);
        DosBeep(1000,20);
        }
                                        /* Post message to query Desktop's/PM window handle */
    WinPostMsg(pHP->hwndThread, WM_SETDESKTOPHANDLE, NULL, NULL);
    if(++ulTimerTicks>=6)
        {
                                        /* Post message to refresh spooler */
        WinPostMsg(pHP->hwndSpooler, WM_REFRESHSPOOLER, NULL, NULL);
        ulTimerTicks=0;
        }
    }
    break;

case WM_BUTTON1DOWN:
    return((MRESULT)TRUE);              /* Avoid default window procedure which sets the
                                           focus to PC/2 */

/*                                                                                      *\
 * Syntax: WM_BUTTON2DOWN, (LONG lClickX), (LONG lClickY)                               *
\*                                                                                      */
case WM_BUTTON2DOWN:
/*                                                                                      *\
 * This message detected and passed from the PC/2 window procedure is used to track or  *
 * perform certain actions on windows on the Virtual Desktops.                          *
\*                                                                                      */
    {
    LONG        lClickX, lClickY;       /* Pointer position during click */
    ULONG       ulWindowIndex;          /* Index in Windows.wdWindow[] */
    BOOL        bFound=FALSE;           /* TRUE if one window was found */
    TRACKINFO   tiWindow;               /* Trackinfo structure for one window */

                                        /* Get position of mouse button 1 down */
    lClickX=(ULONG)(SHORT1FROMMP(mp1));
    lClickY=(ULONG)(SHORT2FROMMP(mp1));
    lClickX=((float)(lClickX-pHP->ptlOrigin.x))/pHP->fScaleX;
    lClickY=((float)(lClickY-pHP->ptlOrigin.y))/pHP->fScaleY;
                                        /* Loop for all windows, from topmost to bottommost, on
                                           overview window */
    for(ulWindowIndex=0;
        pHP->Windows.ulWindowLast!=(ULONG)-1 && ulWindowIndex<=pHP->Windows.ulWindowLast;
        ulWindowIndex++)
        {
                                        /* Ignore invisible windows */
        if(!(pHP->Windows.wdWindow[ulWindowIndex].ulStatus & VISIBLE)) continue;
        if((pHP->Windows.wdWindow[ulWindowIndex].swpWindow.x<=lClickX) &&
            (pHP->Windows.wdWindow[ulWindowIndex].swpWindow.x+pHP->Windows.wdWindow[ulWindowIndex].swpWindow.cx>=lClickX) &&
            (pHP->Windows.wdWindow[ulWindowIndex].swpWindow.y<=lClickY) &&
            (pHP->Windows.wdWindow[ulWindowIndex].swpWindow.y+pHP->Windows.wdWindow[ulWindowIndex].swpWindow.cy>=lClickY))
            {                           /* We have found one */
            bFound=TRUE;
            break;
            }
        }
    if(bFound)
        {
        SWP     swpWindow;              /* Copy of Window's position, because working thread may update it */

        memcpy(&swpWindow, &pHP->Windows.wdWindow[ulWindowIndex].swpWindow, sizeof(SWP));
        switch(usMenuCommand)
        {
        case ID_ICONMOVE:
                                        /* Calculate and draw starting tracking rectangle */
            tiWindow.rclTrack.xLeft=pHP->ptlOrigin.x+
                (float)swpWindow.x*pHP->fScaleX;
            tiWindow.rclTrack.yBottom=pHP->ptlOrigin.y+
                (float)swpWindow.y*pHP->fScaleY;
            tiWindow.rclTrack.xRight=tiWindow.rclTrack.xLeft+
                (float)swpWindow.cx*pHP->fScaleX;
            tiWindow.rclTrack.yTop=tiWindow.rclTrack.yBottom+
                (float)swpWindow.cy*pHP->fScaleY;
            tiWindow.cxBorder=1;        /* Border width */
            tiWindow.cyBorder=1;
            tiWindow.cxGrid=1;          /* Grid width */
            tiWindow.cyGrid=1;
            tiWindow.cxKeyboard=1;      /* Movement in pixel for keyboard (keyboard currently can't
                                           start tracking) */
            tiWindow.cyKeyboard=1;
                                        /* Minimum tracking size */
            tiWindow.ptlMinTrackSize.x=2;
            tiWindow.ptlMinTrackSize.y=2;
                                        /* Maximum tracking size */
            tiWindow.ptlMaxTrackSize.x=pHP->swpPC2Client.cx;
            tiWindow.ptlMaxTrackSize.y=pHP->swpPC2Client.cy;
                                        /* Boundary rectangle */
            tiWindow.rclBoundary.xLeft=0;
            tiWindow.rclBoundary.yBottom=0;
            tiWindow.rclBoundary.xRight=pHP->swpPC2Client.cx;
            tiWindow.rclBoundary.yTop=pHP->swpPC2Client.cy;
                                        /* Tracking options */
            tiWindow.fs=TF_SETPOINTERPOS | TF_MOVE | TF_ALLINBOUNDARY;
            if(WinTrackRect(hwnd, NULLHANDLE, &tiWindow))
                {                       /* If tracking was successful reposition tracked window */
                swpWindow.fl=SWP_MOVE|SWP_NOADJUST;
                swpWindow.x=((float)(tiWindow.rclTrack.xLeft-pHP->ptlOrigin.x))/pHP->fScaleX;
                swpWindow.y=((float)(tiWindow.rclTrack.yBottom-pHP->ptlOrigin.y))/pHP->fScaleY;
                swpWindow.cx=swpWindow.cy=0;
                swpWindow.hwndInsertBehind=NULLHANDLE;
                WinSetMultWindowPos(pHP->habPc2, &swpWindow, 1);
                                        /* Force a reread of windows on Desktop */
                WinPostMsg(hwnd, WM_SETUPSIZEPOSITION, MPFROMLONG(WM_MOVE), NULL);
                }
            break;

        case ID_ICONHIDE:               /* Hide selected window */
            swpWindow.fl=SWP_HIDE;
            swpWindow.x=swpWindow.y=swpWindow.cx=swpWindow.cy=0;
            swpWindow.hwndInsertBehind=NULLHANDLE;
            WinSetMultWindowPos(pHP->habPc2, &swpWindow, 1);
                                        /* Force a reread of windows on Desktop */
            WinPostMsg(hwnd, WM_SETUPSIZEPOSITION, MPFROMLONG(WM_MOVE), NULL);
            break;

        case ID_ICONZORDERTOP:          /* Set selected window to top of Desktop */
            swpWindow.fl=SWP_ZORDER;
            swpWindow.x=swpWindow.y=swpWindow.cx=swpWindow.cy=0;
            swpWindow.hwndInsertBehind=HWND_TOP;
            WinSetMultWindowPos(pHP->habPc2, &swpWindow, 1);
            break;

        case ID_ICONZORDERBOTTOM:       /* Set selected window to bottom of Desktop */
            swpWindow.fl=SWP_ZORDER;
            swpWindow.x=swpWindow.y=swpWindow.cx=swpWindow.cy=0;
            swpWindow.hwndInsertBehind=HWND_BOTTOM;
            WinSetMultWindowPos(pHP->habPc2, &swpWindow, 1);
            break;

        case ID_ICONCLOSE:              /* Close selected window by simulating the selection of
                                           the "close" menuitem of the applications's system menu */
            WinPostMsg(swpWindow.hwnd, WM_SYSCOMMAND, MPFROMLONG(SC_CLOSE), MPFROM2SHORT(CMDSRC_MENU, FALSE));
            break;

        case ID_ICONMAXIMIZE:           /* Maximize selected window */
            swpWindow.fl=SWP_MAXIMIZE;
            swpWindow.x=swpWindow.y=swpWindow.cx=swpWindow.cy=0;
            swpWindow.hwndInsertBehind=NULLHANDLE;
            WinSetMultWindowPos(pHP->habPc2, &swpWindow, 1);
            break;

        case ID_ICONMINIMIZE:           /* Maximize selected window */
            swpWindow.fl=SWP_MINIMIZE;
            swpWindow.x=swpWindow.y=swpWindow.cx=swpWindow.cy=0;
            swpWindow.hwndInsertBehind=NULLHANDLE;
            WinSetMultWindowPos(pHP->habPc2, &swpWindow, 1);
            break;

        case ID_ICONRESTORE:            /* Maximize selected window */
            swpWindow.fl=SWP_RESTORE;
            swpWindow.x=swpWindow.y=swpWindow.cx=swpWindow.cy=0;
            swpWindow.hwndInsertBehind=NULLHANDLE;
            WinSetMultWindowPos(pHP->habPc2, &swpWindow, 1);
            break;
        }
        }
    }
    break;

case WM_BUTTON1DBLCLK:
    if(pHP->ulDebug>=DEBUG_LOW)
        {
        KEYDATA *pKD=pHP->pKeyData;
        ULONG   ulIndex;
     
        for(ulIndex=0; ulIndex<KEYDATACOUNT; ulIndex++, pKD++)
            printf("%s+%c : %c\n",
                (pKD->usFlags==KC_CTRL ? "CTRL" : "ALT"), (char)pKD->usCh, (pKD->bUsed==FALSE ? ' ' : '<'));
        }
    if(pHP->ulDebug>=DEBUG_FULL)
        printf("PC2: WM_BUTTON1DBLCLK ");
    if(!(ulThreadReady & THREADMOVEBUSY))
        {
        if(pHP->ulDebug>=DEBUG_FULL)
            printf("thread not busy\n");

                                        /* Set that working thread possibly has to
                                           move windows on the Virtual Desktop */
        ulThreadReady|=THREADMOVEBUSY;
        WinPostMsg(pHP->hwndThread, WM_BUTTON1DBLCLK, MPFROMLONG(mp1), MPFROMLONG(mp2));
        }
    else
        if(pHP->ulDebug>=DEBUG_FULL)
            printf("thread busy\n");
    break;

/*                                                                                      *\
 * Syntax: WM_WINDOWLIST, (USHORT x, USHORT y), NULL                                    *
\*                                                                                      */
case WM_WINDOWLIST:
                                        /* Pass this message to the working thread */
    WinPostMsg(pHP->hwndThread, WM_WINDOWLIST, MPFROMLONG(mp1), MPFROMLONG(mp2));
    break;

/*                                                                                      *\
 * Syntax: WM_POPUPMENU, (SHORT x, SHORT y), HWND hwndPopup                             *
\*                                                                                      */
case WM_POPUPMENU:
/*                                                                                      *\
 * The hook found that button 1 was clicked on the Desktop and sent us this message. It *
 * is either a WM_BUTTON1CLICK or WM_BUTTON1DBLCLK. First we obtain the focus, to be    *
 * able to start our programs in the foreground. The coordinates passed are relative to *
 * the screen, that is, they need not to be translated from window coordinates to       *
 * Desktop coordinates. Currently we don't differentiate if button 1 was clicked on PM  *
 * or WPS, which can be determined by hwndPopup. The top level window active before the *
 * Popup-Menu got called was saved by the input hook.                                   *
\*                                                                                      */
    {
    POINTL      ptlPopupPosition;
    USHORT      fsOptions=PU_NONE | PU_KEYBOARD | PU_MOUSEBUTTON1 |
                          PU_HCONSTRAIN | PU_VCONSTRAIN;

                                        /* Display Configuration dialog if the Popup-Menu
                                           doesn't contain one */
    if(pHP->ulStatusFlag & DISPLAYCONFIGDIALOG)
                                        /* Post message to load Configure PC/2 dialog box.
                                           We post it because send is synchronous and will
                                           block the message queue which however must
                                           not be blocked in order to handle the
                                           Configuration dialog window procedure */
        WinPostMsg(hwnd, WM_COMMAND, MPFROMSHORT(ID_CONFIGDIALOG), MPFROMSHORT(CMDSRC_OTHER));
    else
        {                               /* If there is one menuentry in the Popup-Menu
                                           position on the first */
        if((pHP->pPopupMenu->Item!=ENTRYEMPTY) &&
            (pHP->ulStatusFlag & SELECTFIRSTITEM))
            fsOptions|=PU_POSITIONONITEM;
                                        /* Get the position and window, where the user
                                           clicked to get the Popup-Menu */
        ptlPopupPosition.x=(ULONG)SHORT1FROMMP(mp1);
        ptlPopupPosition.y=(ULONG)SHORT2FROMMP(mp1);
        if(!WinPopupMenu(               /* Pop up the popup menu */
            HWND_DESKTOP,               /* Parent window handle */
            hwnd,                       /* Owner window handle that receives all the
                                           notification messages generated by the pop-up
                                           menu */
                                        /* Popup menu window handle */
            pHP->hwndPopupMenu,
            ptlPopupPosition.x,         /* x-coordinate of mouse pointer for popup menu */
            ptlPopupPosition.y,         /* y-coordinate of mouse pointer for popup menu */
                                        /* Input item identity, if PU_POSITIONONITEM or
                                           PU_SELECTITEM is set */
            pHP->pPopupMenu->id,
            fsOptions)                  /* Input options */
        ) PM_ERR(pHP->habPc2, pHP->hwndFrame, HELP_CREATEWINDOW, MB_ERROR|MB_OK|MB_HELP|MB_MOVEABLE|MB_DEFBUTTON1,
            "Activation of PC/2's Popup-Menu failed - continuing...");
        }
    break;
    }

/*                                                                                      *\
 * Syntax: WM_MOVEREQUEST, (USHORT usMouseXPos, USHORT usMouseYPos), (ULONG ulMoveFlag) *
\*                                                                                      */
case WM_MOVEREQUEST:
/*                                                                                      *\
 * This message is sent by the PC/2 Input Hook as a result of clicking mouse button 1   *
 * onto any border row or column of the Desktop, requesting to move to another Virtual  *
 * Desktop.                                                                             *
\*                                                                                      */
    if(pHP->ulDebug>=DEBUG_FULL)
        printf("PC2: WM_MOVEREQUEST ");
    if(!(ulThreadReady & THREADMOVEBUSY))
        {
        if(pHP->ulDebug>=DEBUG_FULL)
            printf("Thread not busy\n");
        ulThreadReady|=THREADMOVEBUSY;  /* Set that working thread gets something to do */
        WinPostMsg(pHP->hwndThread, WM_MOVEREQUEST, MPFROMLONG(mp1), MPFROMLONG(mp2));
        }
    else
        if(pHP->ulDebug>=DEBUG_FULL)
            printf("Thread busy\n");
    break;

/*                                                                                      *\
 * Syntax: WM_ZORDER, (USHORT usMouseXPos, USHORT usMouseYPos), (HWND hwndTitlebar)     *
\*                                                                                      */
case WM_ZORDER:
/*                                                                                      *\
 * The hook found that button 2 was clicked on a window's titlebar. After setting this  *
 * window to bottom, get the next window immediately below the mouse pointer and        *
 * activate it. The same code moved into the message queue hook does not work, I        *
 * assume it has something to do that the message queue hook is blocked until the hook  *
 * returns, and some calls here require a message queue the same time.                  *
\*                                                                                      */
    {
                                        /* Window handle of the titlebar user clicked on */
    HWND    hwndTitlebar;
                                        /* Window immediately below (z-order) the mouse pointer */
    HWND    hwndNextFrame;
    POINTL  ptlMouse;
                                        /* Get mouse position relative to the titlebar clicked on */
    ptlMouse.x=(ULONG)SHORT1FROMMP(mp1);
    ptlMouse.y=(ULONG)SHORT2FROMMP(mp1);
    hwndTitlebar=HWNDFROMMP(mp2);
    WinMapWindowPoints(                 /* Map coordinates from titlebar clicked on to screen */
        hwndTitlebar,                   /* Window to map from */
        HWND_DESKTOP,                   /* Window to map to */
        &ptlMouse,                      /* Point to remap */
        1);                             /* Map 1 point */
                                        /* Get the frame window handle under the mouse */
    hwndNextFrame=WinWindowFromPoint(
        HWND_DESKTOP,                   /* Child windows to be tested */
        &ptlMouse,                      /* Point */
        FALSE);                         /* Test only immediate child windows */
    if(hwndNextFrame!=WinQueryWindow(hwndTitlebar, QW_PARENT))
                                        /* Now switch to the new frame window, causing a new
                                           Z-order, but only when it is not the same window we
                                           set to bottom. It will generate all messages
                                           of deactivating old and activating the
                                           new window. */
        WinFocusChange(HWND_DESKTOP, WinWindowFromID(hwndNextFrame, FID_CLIENT), 0);
    }
    break;

/*                                                                                      *\
 * Syntax: WM_SETUPSIZEPOSITION, ULONG ulMsg, NULL                                      *
\*                                                                                      */
case WM_SETUPSIZEPOSITION:
/*                                                                                      *\
 * This message is sent by the PC/2 WinSendMsg()-Hook as a result of creating, sizing,  *
 * moving, destroying a frame window or switching to another frame window. This         *
 * requires that one or more windows need to be redrawn on the overview window.         *
\*                                                                                      */
    if(pHP->ulDebug>=DEBUG_FULL)
        printf("PC2: WM_SETUPSIZEPOSITION ");
    if(!(ulThreadReady & THREADWINDOWBUSY))
        {
        if(pHP->ulDebug>=DEBUG_FULL)
            printf("Thread not busy\n");
                                        /* Set that working thread gets something to do */
        ulThreadReady|=THREADWINDOWBUSY;
        WinPostMsg(pHP->hwndThread, WM_SETUPSIZEPOSITION, MPFROMLONG(mp1), NULL);
        }
    else
        {
        if(pHP->ulDebug>=DEBUG_FULL)
            printf("Thread busy, request queued\n");
                                        /* Save additional work until thread is not busy again */
        ulThreadReady|=THREADWINDOWQUEUED;
        }
    break;

/*                                                                                      *\
 * Syntax: WM_THREADREADY, ULONG ulFlag, NULL                                           *
\*                                                                                      */
case WM_THREADREADY:
/*                                                                                      *\
 * This message is sent by the PC/2 working thread when he finished handling a          *
 * WM_MOVEREQUEST or WM_SETUPSIZEPOSITION message. This ensures that pending messages   *
 * possibly caused by the same user action are not queued by WinPostMsg() calls.        *
\*                                                                                      */
    if(pHP->ulDebug>=DEBUG_ENTRY)
        {
        DosBeep(250,250);
        }
    if(pHP->ulDebug>=DEBUG_FULL)
        printf("PC2: WM_THREADREADY ");
    if(LONGFROMMP(mp1)==THREADMOVEBUSY)
        {
        if(pHP->ulDebug>=DEBUG_FULL)
            printf("return from move request\n");
        ulThreadReady&=(~THREADMOVEBUSY);
        }
    else
        {
        if(pHP->ulDebug>=DEBUG_FULL)
            printf("return from size request");
                                        /* Set that working thread finished his work of
                                           enumerating windows */
        ulThreadReady&=(~THREADWINDOWBUSY);
                                        /* If some work was queued do work once */
        if(ulThreadReady & THREADWINDOWQUEUED)
            {
            if(pHP->ulDebug>=DEBUG_FULL)
                printf(", doing queued size request\n");
                                        /* We are doing one queued enumerating windows request */
            ulThreadReady&=(~THREADWINDOWQUEUED);
                                        /* Set that working thread gets something to do */
            ulThreadReady|=THREADWINDOWBUSY;
            WinPostMsg(pHP->hwndThread, WM_SETUPSIZEPOSITION, NULL, NULL);
            }
        else
            if(pHP->ulDebug>=DEBUG_FULL)
                printf("\n");
        }
    break;

case WM_CLOSE:
                                        /* When an internal processiong failure occured, that
                                           requires immediate shutdown of PC/2, the flag
                                           PC2EMERGENCY is set, and a WM_QUIT message is posted,
                                           which breaks us out of the message loop */
    if(!(pHP->ulStatusFlag&PC2EMERGENCYEXIT))
        {
                                        /* When running as WPS, PC/2 has already disabled the
                                           menuentries to exit PC/2, so we don't need to touch
                                           them. */
        if(!(pHP->ulStatusFlag&PC2RUNNINGASWPS))
            {
                                        /* Disable menuitem during dialog display */
            WinSendMsg(pHP->hwndPopupMenu, MM_SETITEMATTR,
                MPFROM2SHORT(ID_EXIT, TRUE), MPFROM2SHORT(MIA_DISABLED, MIA_DISABLED));
            WinSendMsg(pHP->hwndMenu, MM_SETITEMATTR,
                MPFROM2SHORT(ID_ICONEXIT, TRUE), MPFROM2SHORT(MIA_DISABLED, MIA_DISABLED));
            }
        if(WinMessageBox(               /* Ask the user if he really wants to exit */
            HWND_DESKTOP, HWND_DESKTOP,
            "Are you sure you want to close PC/2?",
            "PC/2 - Program Commander/2",
            ID_PC2MAINWINDOW,
            MB_YESNO | MB_ICONQUESTION | MB_DEFBUTTON1 | MB_MOVEABLE)!=MBID_YES)
            {
            if(!(pHP->ulStatusFlag&PC2RUNNINGASWPS))
                {
                                        /* Enable menuitem again */
                WinSendMsg(pHP->hwndPopupMenu, MM_SETITEMATTR,
                    MPFROM2SHORT(ID_EXIT, TRUE), MPFROM2SHORT(MIA_DISABLED, 0));
                WinSendMsg(pHP->hwndMenu, MM_SETITEMATTR,
                    MPFROM2SHORT(ID_ICONEXIT, TRUE), MPFROM2SHORT(MIA_DISABLED, 0));
                }
                                        /* Reset */
            pHP->ulStatusFlag&=(~QUITFROMWINDOWLIST);
            return((MRESULT)TRUE);      /* Only exit if OK is pressed */
            }
        }
                                        /* Reduce WPS to original size */
    WinSendMsg(pHP->hwndThread, WM_EXPANDWPS, (MPARAM)FALSE, NULL);
                                        /* Allow the spooler container window to terminate itself */
    WinSendMsg(pHP->hwndSpooler, WM_DESTROY, NULL, NULL);
                                        /* Send WM_QUIT to thread window, because this message
                                           terminates the message loop */
    WinPostMsg(pHP->hwndThread, WM_QUIT, NULL, NULL);
                                        /* Write changes to PC2.INI */
    if(!INIAccess(pHP->ucFilenameINI, FALSE))
        USR_ERR(pHP->hwndFrame, HELP_PC2INI, MB_INFORMATION|MB_OK|MB_HELP|MB_MOVEABLE|MB_DEFBUTTON1,
            "Error writing to PC2.INI - continuing...");
    if(WinReleaseHook(                  /* Release hook */
        pHP->habPc2,                    /* Handle  of anchor block */
        NULLHANDLE,                     /* Release from system hook chain */
        HK_INPUT,                       /* Hook of system input queue */
        (PFN)pInputHook,                /* Pointer to hook procedure */
        pHP->hDLLPc2Hook)==FALSE)
        USR_ERR(pHP->hwndFrame, HELP_PC2HOOK, MB_INFORMATION|MB_OK|MB_HELP|MB_MOVEABLE|MB_DEFBUTTON1,
            "Unhooking the system input queue failed, System ShutDown suggested - exiting...");
                                        /* Release hook of system sendmessage */
    if(WinReleaseHook(pHP->habPc2, NULLHANDLE, HK_SENDMSG, (PFN)pWinSendMsgHook,
        pHP->hDLLPc2Hook)==FALSE)
        USR_ERR(pHP->hwndFrame, HELP_PC2HOOK, MB_INFORMATION|MB_OK|MB_HELP|MB_MOVEABLE|MB_DEFBUTTON1,
            "Unhooking the sendmessage dispatcher failed, System ShutDown suggested - exiting...");
                                        /* Post exit message */
    WinPostMsg(hwnd, WM_QUIT, NULL, NULL);
    break;

case HM_ERROR:
    {
    PM_ERR(pHP->habPc2, pHP->hwndFrame, HELP_CREATEWINDOW, MB_ERROR|MB_OK|MB_HELP|MB_MOVEABLE|MB_DEFBUTTON1,
        "Help request failed - continuing...");
    break;
    }

case WM_COMMAND:
    {
    USHORT      command;

    command=SHORT1FROMMP(mp1);          /* Extract the command value */
/*                                                                                      *\
 * Filter the IDs of the user defined items of the Popup-Menu. If one is found, call    *
 * SearchItem() to search for the corresponding MENUDATA structure, copy it to a        *
 * SESSIONDATA structure and start the session.                                         *
\*                                                                                      */
    if((command>=USERITEMFIRST) && (command<=USERITEMLAST))
        {
        SESSIONDATA     SessionData;    /* Used by Menu Installation dialog and by
                                           Program Installation dialog to store menu or
                                           program data, to be filled from the user or
                                           to be presented to the user. */
        ULONG           id=(ULONG)command;
        MENUDATA        *pMD=NULL;

                                        /* Search in the linked list for this entry */
        if((pMD=SearchItem(pHP->pPopupMenu, &id))!=NULL)
            if(pMD->Item==ENTRYMENUITEM)
                {
                                        /* Load SessionData with MENUDATA structure */
                LoadMenuData2SessionData(pMD, &SessionData);
                                        /* If user defined size and position defined
                                           set SWP_MOVEWINDOW flag, which invokes a
                                           reposition of the window, the first time the
                                           started application's window is found. The flag
                                           will be reset afterwards */
                if(pMD->PgmControl & SSF_CONTROL_SETPOS)
                    {                   /* If have to move the window create it invisible
                                           first to avoid drawing of window before movement.
                                           If the window should be not invisible set flag
                                           SWP_MOVEWINDOWVISIBLE to show window after movement.
                                           The flag will be reset afterwards. */
                    if(pMD->PgmControl & SSF_CONTROL_INVISIBLE)
                        pMD->SwpFlag|=SWP_MOVEWINDOW;
                    else
                        pMD->SwpFlag|=(SWP_MOVEWINDOW | SWP_MOVEWINDOWVISIBLE);
                                        /* Create window invisible before movement */
                    SessionData.PgmControl|=SSF_CONTROL_INVISIBLE;
                    }
                                        /* If the session should be started not in background
                                           switch PC/2 into foreground, because only the foreground
                                           application can start another session in foreground */
                if(!(pMD->FgBg&SSF_FGBG_BACK))
                    WinSwitchToProgram(WinQuerySwitchHandle(pHP->hwndFrame, pHP->pPib->pib_ulpid));
                                        /* Start the session */
                StartSession(&SessionData);
                                        /* If the session was started in background, try
                                           to activate the top level window that was active
                                           before PC/2 got active */
                if(pMD->FgBg&SSF_FGBG_BACK)
                    {
                    WinFocusChange(HWND_DESKTOP, WinWindowFromID(pHP->hwndActiveWindow, FID_CLIENT),
                        FC_NOSETACTIVE);
                    WinPostMsg(pHP->hwndActiveWindow, WM_ACTIVATE,
                        MPFROMSHORT(TRUE), MPFROMHWND(WinWindowFromID(pHP->hwndActiveWindow, FID_TITLEBAR)));
                    }
                                        /* Because a working directory may have been set change
                                           to root directory of all local drives */
                WinPostMsg(pHP->hwndThread, WM_SETDRIVEROOT, NULL, NULL);
                }
        break;                          /* We don't need further testing for this command */
        }
    pHP->hwndMenu=WinWindowFromID(pHP->hwndFrame, FID_MENU);
    switch(command)
    {
                                        /* Test for messages sent from smarticons */
    case ID_ICONEXIT:
        if(usMenuCommand!=ID_ICONMOVE)
            {
            WinSendMsg(pHP->hwndMenu, MM_SETITEMATTR,
                MPFROM2SHORT(usMenuCommand, TRUE), MPFROM2SHORT(MIA_FRAMED|MIA_HILITED, 0));
            WinSendMsg(pHP->hwndMenu, MM_SETITEMATTR,
                MPFROM2SHORT(ID_ICONMOVE, TRUE), MPFROM2SHORT(MIA_FRAMED|MIA_HILITED, MIA_FRAMED|MIA_HILITED));
            }
        usMenuCommand=ID_ICONMOVE;
                                        /* Post exit to PC/2 */
        WinPostMsg(hwnd, WM_COMMAND, MPFROMSHORT(ID_EXIT), MPFROMSHORT(CMDSRC_MENU));
        break;

    case ID_ICONMOVE:
        if(usMenuCommand!=ID_ICONMOVE)
            {                           /* Remove frame attribute from previous smarticon to move smarticon */
            WinSendMsg(pHP->hwndMenu, MM_SETITEMATTR,
                MPFROM2SHORT(usMenuCommand, TRUE), MPFROM2SHORT(MIA_FRAMED|MIA_HILITED, 0));
            WinSendMsg(pHP->hwndMenu, MM_SETITEMATTR,
                MPFROM2SHORT(ID_ICONMOVE, TRUE), MPFROM2SHORT(MIA_FRAMED|MIA_HILITED, MIA_FRAMED|MIA_HILITED));
            }
        usMenuCommand=ID_ICONMOVE;
        break;

    case ID_ICONHIDE:
        if(usMenuCommand!=ID_ICONHIDE)
            {                           /* Remove frame attribute from previous smarticon to move smarticon */
            WinSendMsg(pHP->hwndMenu, MM_SETITEMATTR,
                MPFROM2SHORT(usMenuCommand, TRUE), MPFROM2SHORT(MIA_FRAMED|MIA_HILITED, 0));
            WinSendMsg(pHP->hwndMenu, MM_SETITEMATTR,
                MPFROM2SHORT(ID_ICONHIDE, TRUE), MPFROM2SHORT(MIA_FRAMED|MIA_HILITED, MIA_FRAMED|MIA_HILITED));
            }
        usMenuCommand=ID_ICONHIDE;
        break;

    case ID_ICONZORDERTOP:
        if(usMenuCommand!=ID_ICONZORDERTOP)
            {
            WinSendMsg(pHP->hwndMenu, MM_SETITEMATTR,
                MPFROM2SHORT(usMenuCommand, TRUE), MPFROM2SHORT(MIA_FRAMED|MIA_HILITED, 0));
            WinSendMsg(pHP->hwndMenu, MM_SETITEMATTR,
                MPFROM2SHORT(ID_ICONZORDERTOP, TRUE), MPFROM2SHORT(MIA_FRAMED|MIA_HILITED, MIA_FRAMED|MIA_HILITED));
            }
        usMenuCommand=ID_ICONZORDERTOP;
        break;

    case ID_ICONZORDERBOTTOM:
        if(usMenuCommand!=ID_ICONZORDERBOTTOM)
            {
            WinSendMsg(pHP->hwndMenu, MM_SETITEMATTR,
                MPFROM2SHORT(usMenuCommand, TRUE), MPFROM2SHORT(MIA_FRAMED|MIA_HILITED, 0));
            WinSendMsg(pHP->hwndMenu, MM_SETITEMATTR,
                MPFROM2SHORT(ID_ICONZORDERBOTTOM, TRUE), MPFROM2SHORT(MIA_FRAMED|MIA_HILITED, MIA_FRAMED|MIA_HILITED));
            }
        usMenuCommand=ID_ICONZORDERBOTTOM;
        break;

    case ID_ICONCLOSE:
        if(usMenuCommand!=ID_ICONCLOSE)
            {
            WinSendMsg(pHP->hwndMenu, MM_SETITEMATTR,
                MPFROM2SHORT(usMenuCommand, TRUE), MPFROM2SHORT(MIA_FRAMED|MIA_HILITED, 0));
            WinSendMsg(pHP->hwndMenu, MM_SETITEMATTR,
                MPFROM2SHORT(ID_ICONCLOSE, TRUE), MPFROM2SHORT(MIA_FRAMED|MIA_HILITED, MIA_FRAMED|MIA_HILITED));
            }
        usMenuCommand=ID_ICONCLOSE;
        break;

    case ID_ICONMAXIMIZE:
        if(usMenuCommand!=ID_ICONMAXIMIZE)
            {
            WinSendMsg(pHP->hwndMenu, MM_SETITEMATTR,
                MPFROM2SHORT(usMenuCommand, TRUE), MPFROM2SHORT(MIA_FRAMED|MIA_HILITED, 0));
            WinSendMsg(pHP->hwndMenu, MM_SETITEMATTR,
                MPFROM2SHORT(ID_ICONMAXIMIZE, TRUE), MPFROM2SHORT(MIA_FRAMED|MIA_HILITED, MIA_FRAMED|MIA_HILITED));
            }
        usMenuCommand=ID_ICONMAXIMIZE;
        break;

    case ID_ICONMINIMIZE:
        if(usMenuCommand!=ID_ICONMINIMIZE)
            {
            WinSendMsg(pHP->hwndMenu, MM_SETITEMATTR,
                MPFROM2SHORT(usMenuCommand, TRUE), MPFROM2SHORT(MIA_FRAMED|MIA_HILITED, 0));
            WinSendMsg(pHP->hwndMenu, MM_SETITEMATTR,
                MPFROM2SHORT(ID_ICONMINIMIZE, TRUE), MPFROM2SHORT(MIA_FRAMED|MIA_HILITED, MIA_FRAMED|MIA_HILITED));
            }
        usMenuCommand=ID_ICONMINIMIZE;
        break;

    case ID_ICONRESTORE:
        if(usMenuCommand!=ID_ICONRESTORE)
            {
            WinSendMsg(pHP->hwndMenu, MM_SETITEMATTR,
                MPFROM2SHORT(usMenuCommand, TRUE), MPFROM2SHORT(MIA_FRAMED|MIA_HILITED, 0));
            WinSendMsg(pHP->hwndMenu, MM_SETITEMATTR,
                MPFROM2SHORT(ID_ICONRESTORE, TRUE), MPFROM2SHORT(MIA_FRAMED|MIA_HILITED, MIA_FRAMED|MIA_HILITED));
            }
        usMenuCommand=ID_ICONRESTORE;
        break;

    case ID_ICONSHUTDOWN:
        if(usMenuCommand!=ID_ICONMOVE)
            {
            WinSendMsg(pHP->hwndMenu, MM_SETITEMATTR,
                MPFROM2SHORT(usMenuCommand, TRUE), MPFROM2SHORT(MIA_FRAMED|MIA_HILITED, 0));
            WinSendMsg(pHP->hwndMenu, MM_SETITEMATTR,
                MPFROM2SHORT(ID_ICONMOVE, TRUE), MPFROM2SHORT(MIA_FRAMED|MIA_HILITED, MIA_FRAMED|MIA_HILITED));
            }
        usMenuCommand=ID_ICONMOVE;
                                        /* Write changes to PC2.INI */
        INIAccess(pHP->ucFilenameINI, FALSE);
                                        /* Post ShutDowm OS/2 to PC/2 */
        WinPostMsg(hwnd, WM_COMMAND, MPFROMSHORT(ID_SHUTDOWN), MPFROMSHORT(CMDSRC_MENU));
        break;

    case ID_ICONHELP:
        if(usMenuCommand!=ID_ICONMOVE)
            {
            WinSendMsg(pHP->hwndMenu, MM_SETITEMATTR,
                MPFROM2SHORT(usMenuCommand, TRUE), MPFROM2SHORT(MIA_FRAMED|MIA_HILITED, 0));
            WinSendMsg(pHP->hwndMenu, MM_SETITEMATTR,
                MPFROM2SHORT(ID_ICONMOVE, TRUE), MPFROM2SHORT(MIA_FRAMED|MIA_HILITED, MIA_FRAMED|MIA_HILITED));
            }
        usMenuCommand=ID_ICONMOVE;
                                        /* Post help to PC/2 */
        WinPostMsg(hwnd, WM_COMMAND, MPFROMSHORT(ID_HELP), MPFROMSHORT(CMDSRC_MENU));
        break;

    case ID_HELP:                       /* Display general help panel */
        if(pHP->hwndHelp!=NULLHANDLE) WinSendMsg(
            pHP->hwndHelp,  /* Help window */
            HM_DISPLAY_HELP,            /* Display a help panel */
            MPFROMSHORT(ID_HELP),       /* Panel ID in ressource file */
            HM_RESOURCEID);             /* MP1 points to the help window identity */
        break;

    case ID_CONFIGDIALOG:               /* Popup menuitem Configure Menu selected */
                                        /* Disable menuitem during dialog display */
        WinSendMsg(pHP->hwndPopupMenu, MM_SETITEMATTR,
            MPFROM2SHORT(ID_CONFIGDIALOG, TRUE), MPFROM2SHORT(MIA_DISABLED, MIA_DISABLED));
        if(!WinDlgBox(                  /* Start Configure PC/2 dialog box */
            HWND_DESKTOP,               /* DESKTOP is parent */
            HWND_DESKTOP,               /* DESKTOP is owner */
            CD_DialogProcedure,         /* Dialog procedure of Program Installation
                                           dialog */
            0,                          /* Ressource is .EXE file */
            CDID_CONFIGDIALOG,          /* ID of Configure PC/2 dialog */
            0))                         /* No initialization data */
        PM_ERR(pHP->habPc2, pHP->hwndFrame, HELP_CREATEDIALOG, MB_ERROR|MB_OK|MB_HELP|MB_MOVEABLE|MB_DEFBUTTON1,
            "Creation of a dialog box failed - continuing...");
                                        /* Enable menuitem again */
        WinSendMsg(pHP->hwndPopupMenu, MM_SETITEMATTR,
            MPFROM2SHORT(ID_CONFIGDIALOG, TRUE), MPFROM2SHORT(MIA_DISABLED, 0));
        break;

    case ID_DESKTOPDIALOG:              /* Popup menuitem Configure Desktop selected */
                                        /* Disable menuitem during dialog display */
        WinSendMsg(pHP->hwndPopupMenu, MM_SETITEMATTR,
            MPFROM2SHORT(ID_DESKTOPDIALOG, TRUE), MPFROM2SHORT(MIA_DISABLED, MIA_DISABLED));
        if(!WinDlgBox(HWND_DESKTOP, HWND_DESKTOP, DD_DialogProcedure,
            0, DDID_DESKTOPDIALOG, 0))
        PM_ERR(pHP->habPc2, pHP->hwndFrame, HELP_CREATEDIALOG, MB_ERROR|MB_OK|MB_HELP|MB_MOVEABLE|MB_DEFBUTTON1,
            "Creation of a dialog box failed - continuing...");
                                        /* Enable menuitem again */
        WinSendMsg(pHP->hwndPopupMenu, MM_SETITEMATTR,
            MPFROM2SHORT(ID_DESKTOPDIALOG, TRUE), MPFROM2SHORT(MIA_DISABLED, 0));
        break;

    case ID_SHUTDOWN:                   /* ShutDown OS/2 menuitem selected */
        if(pHP->ulStatusFlag&NORMALSHUTDOWN)
                                        /* If the user requested the shutdown he knows from the
                                           WPS's context menu, do what is requested */
            WinShutdownSystem(pHP->habPc2, pHP->hmqPc2);
        else
            {
            if(WinMessageBox(           /* Ask the user if he really wants to shut down OS/2 */
            HWND_DESKTOP, HWND_DESKTOP,
            "Are you really sure you want to ShutDown OS/2?",
            "PC/2 - Program Commander/2",
            ID_PC2MAINWINDOW,
            MB_OKCANCEL | MB_ICONQUESTION | MB_DEFBUTTON1)!=MBID_OK)
            return((MRESULT)TRUE);      /* Only shut down if OK is pressed */
            if(!WinDlgBox(              /* Start ShutDown OS/2 dialog box */
                HWND_DESKTOP, HWND_DESKTOP, SD_DialogProcedure, 0,
                SDID_SHUTDOWNDIALOG, 0))
            PM_ERR(pHP->habPc2, pHP->hwndFrame, HELP_CREATEDIALOG, MB_ERROR|MB_OK|MB_HELP|MB_MOVEABLE|MB_DEFBUTTON1,
                "Creation of a dialog box failed - continuing...");
            }
        break;

    case ID_EXIT:                       /* User selected F3 to shutdown PC/2, selected Exit PC/2
                                           from the Popup-Menu or selected the Exit smarticon. */

                                        /* If running as the WPS replacement, don't exit */
        if(!(pHP->ulStatusFlag&PC2RUNNINGASWPS))
            {
                                        /* Because WM_CLOSE is called, PC/2 is shutdown in
                                           an orderly manner, so we can accept the next WM_QUIT
                                           (a WM_QUIT terminates the message loop) message,
                                           posted during WM_CLOSE immediately when the user really
                                           wants to quit, to terminate PC/2 without any way to
                                           return then, by setting QUITFROMWINDOWLIST */
            pHP->ulStatusFlag|=QUITFROMWINDOWLIST;
            WinPostMsg(hwnd, WM_CLOSE, 0, 0);
            }
        break;

    case ID_EMERGENCYEXIT:              /* User selected CTRL+ALT+F3 to exit in an undocumented way,
                                           which is useful for testing or debugging when running
                                           as WPS replacement */
        pHP->ulStatusFlag|=QUITFROMWINDOWLIST;
        WinPostMsg(hwnd, WM_CLOSE, 0, 0);
        break;

    case ID_ABOUTDIALOG:                /* User selected About PC/2 dialog */
                                        /* Disable menuitem during dialog display */
        WinSendMsg(pHP->hwndPopupMenu, MM_SETITEMATTR,
            MPFROM2SHORT(ID_ABOUTDIALOG, TRUE), MPFROM2SHORT(MIA_DISABLED, MIA_DISABLED));
        if(!WinDlgBox(                  /* Start About PC/2 dialog box */
            HWND_DESKTOP, HWND_DESKTOP, AD_DialogProcedure, 0,
            ADID_ABOUTDIALOG, 0))
            PM_ERR(pHP->habPc2, pHP->hwndFrame, HELP_CREATEDIALOG, MB_ERROR|MB_OK|MB_HELP|MB_MOVEABLE|MB_DEFBUTTON1,
                "Creation of a dialog box failed - continuing...");
                                        /* Enable menuitem again */
        WinSendMsg(pHP->hwndPopupMenu, MM_SETITEMATTR,
            MPFROM2SHORT(ID_ABOUTDIALOG, TRUE), MPFROM2SHORT(MIA_DISABLED, 0));
        break;

    case ID_SPOOLER:                    /* User selected PC/2 spooler dialog */
        WinPostMsg(pHP->hwndSpooler, WM_SHOWSPOOLER, NULL, NULL);
        break;

    case ID_DEBUG:                      /* User pressed SysRq key to toggle debug status */
        pHP->ulDebug++;
        if(pHP->ulDebug>DEBUG_FULL) pHP->ulDebug=DEBUG_NO;
                                        /* Write changes to PC2.INI */
        INIAccess(pHP->ucFilenameINI, FALSE);
        break;
    }
    break;
    }

default:                                /* Default window procedure must be called */
    return((MRESULT)WinDefWindowProc(hwnd, msg, mp1, mp2));
}
return((MRESULT)FALSE);                 /* We have handled the message */
}

/*--------------------------------------------------------------------------------------*\
 * This dialog procedure handles the PC/2 - Configuration (Setup) dialog.               *
 * Req: none                                                                            *
\*--------------------------------------------------------------------------------------*/
MRESULT  EXPENTRY CD_DialogProcedure(HWND hwndDlg, ULONG msg, MPARAM mp1, MPARAM mp2)
{
static ULONG    ulListboxCursor;        /* Index of the listbox item that has the cursor selection */

switch(msg)
{
case WM_INITDLG:
    {
    SWP         swp;

    WinQueryWindowPos(                  /* Query position of dialog window */
        hwndDlg,                        /* Handle of dialog window */
        &swp);                          /* Fill with position */
    swp.fl=SWP_MOVE;                    /* Center dialog window */
    swp.x=(pHP->swpScreen.cx-swp.cx)>>1;
    swp.y=(pHP->swpScreen.cy-swp.cy)>>1;
    swp.cx=swp.cy=0;
    swp.hwndInsertBehind=NULLHANDLE;
    WinSetMultWindowPos(pHP->habPc2, &swp, 1);
                                        /* Subclass CDLB_MENUPROGRAM listbox to allow dropping into */
    pfnListboxWindowProc=WinSubclassWindow(WinWindowFromID(hwndDlg, CDLB_MENUPROGRAM),
        SubclassedListboxWindowProc);
    ulListboxCursor=0;                  /* Set cursor selection to first entry */
                                        /* Initialize the listbox */
    WinSendMsg(hwndDlg, WM_LOADPOPUPMENU, MPFROMP(pHP->pMenuData), NULL);
    break;
    }

/*                                                                                      *\
 * Syntax: WM_LOADPOPUPMENU, *MENUDATA, NULL                                            *
\*                                                                                      */
case WM_LOADPOPUPMENU:                  /* Load the current level of the Popup-Menu in
                                           the listbox after removing the old items */
    {
    MENUDATA    *pMD;

    pMD=PVOIDFROMMP(mp1);               /* Get the pointer to the first MENUDATA of the
                                           current level */
    WinSendDlgItemMsg(                  /* Send message to listbox */
        hwndDlg,                        /* Handle of dialog window */
        CDLB_MENUPROGRAM,               /* Submenu & Program listbox */
        LM_DELETEALL,                   /* Delete all list box items */
        (MPARAM)NULL,
        (MPARAM)NULL);
    if(pMD==NULL) break;                /* If linked list is empty break out */
    do
    {
        if(pMD->Item==ENTRYSUBMENU)     /* It is a Submenu */
            {
            UCHAR       Buffer[MAXNAMEL+4];
                                        /* Add >> for a Submenu */
            sprintf(Buffer, "%s >>", pMD->PgmTitle);
            WinSendDlgItemMsg(
                hwndDlg,
                CDLB_MENUPROGRAM,
                LM_INSERTITEM,          /* Insert Submenu Title at the end */
                MPFROMSHORT(LIT_END),
                MPFROMP(Buffer));
            }
        if(pMD->Item==ENTRYMENUITEM)    /* It's a Menuitem */
            WinSendDlgItemMsg(
                hwndDlg,
                CDLB_MENUPROGRAM,
                LM_INSERTITEM,          /* Insert Menuitem Title at the end */
                MPFROMSHORT(LIT_END),
                MPFROMP(pMD->PgmTitle));
        if(pMD->Item==ENTRYCONTROL)     /* It's a Control */
            {
            WinSendDlgItemMsg(
                hwndDlg,
                CDLB_MENUPROGRAM,
                LM_INSERTITEM,          /* Insert Control Title at the end */
                MPFROMSHORT(LIT_END),
                MPFROMP(pMD->PgmTitle));
            }
                                        /* It may also be an empty entry, but then we
                                           ignore it, because it must be filled with
                                           Menuitem or Submenu data first */
        if(pMD->Next!=NULL)             /* Get through linked list without diving into
                                           Submenus */
                pMD=pMD->Next;
        else break;                     /* We're at the end of the linked list */
    }while(TRUE);
                                        /* Set cursor selection to first entry */
    WinSendDlgItemMsg(hwndDlg, CDLB_MENUPROGRAM, LM_SETTOPINDEX,
        MPFROMSHORT((USHORT)(ulListboxCursor>7 ? (ulListboxCursor-7) : ulListboxCursor)), MPFROMCHAR(TRUE));
    WinSendDlgItemMsg(hwndDlg, CDLB_MENUPROGRAM, LM_SELECTITEM,
        MPFROMSHORT((USHORT)ulListboxCursor), MPFROMCHAR(TRUE));
    break;
    }

/*                                                                                      *\
 * Syntax: WM_SAVEPOPUPMENU, NULL, NULL                                                 *
\*                                                                                      */
case WM_SAVEPOPUPMENU:                  /* Save the Popup-Menu to the configuraion file */
    if((pHP->Pc2Profile=fopen(pucFilenameProfile, "w"))==NULL)
        USR_ERR(pHP->hwndFrame, HELP_PC2CFG, MB_INFORMATION|MB_OK|MB_HELP|MB_MOVEABLE|MB_DEFBUTTON1,
            "Cannot open confguration file - changes won't be saved");
    else
        {
        fprintf(pHP->Pc2Profile, "PROFILE START\n");
                                        /* Save the menu linked list */
        SaveMenu(pHP->pPopupMenu);
        fprintf(pHP->Pc2Profile, "PROFILE END\n");
        fclose(pHP->Pc2Profile);
        }
    break;

case WM_HELP:                           /* Help pressed */
    WinSendMsg(
        pHP->hwndHelp,                  /* Help window */
        HM_DISPLAY_HELP,                /* Display a help panel */
        MPFROMSHORT(ID_CONFIGDIALOG),   /* Panel ID in resource file */
        HM_RESOURCEID);                 /* MP1 points to the help window identity */
    break;

case WM_CONTROL:
                                        /* If user doubleclicked on an listbox item process */
    if(SHORT1FROMMP(mp1)==CDLB_MENUPROGRAM)
        switch(SHORT2FROMMP(mp1))
        {
        case LN_ENTER:
            {
            MENUDATA        *pMD;
            SHORT           sCount;

                                        /* Point to the first element of the linked list
                                           at the current level */
            pMD=pHP->pMenuData;
                                        /* Send message to listbox */
            sCount=(SHORT)WinSendDlgItemMsg(
                hwndDlg,                /* Handle of dialog window */
                CDLB_MENUPROGRAM,       /* Submenu & Program listbox */
                LM_QUERYSELECTION,      /* Query first selected list box item */
                MPFROMSHORT(LIT_FIRST),
                (MPARAM)NULL);
                                        /* Save cursor selection */
            ulListboxCursor=(ULONG)sCount;
            if(sCount==LIT_NONE)        /* If no item selected ignore this button */
                return((MRESULT)FALSE);
            for( ;sCount>0; sCount--)   /* Walk through the linked list to the selected
                                           item */
                pMD=pMD->Next;
                                        /* For a Menuitem simulate change entry, for a Submenu
                                           simulate level down */
            if(pMD->Item==ENTRYSUBMENU)
                WinPostMsg(hwndDlg, WM_COMMAND, MPFROM2SHORT(CDID_LEVELDOWN, CMDSRC_OTHER), (MPARAM)TRUE);
            if(pMD->Item==ENTRYMENUITEM)
                WinPostMsg(hwndDlg, WM_COMMAND, MPFROM2SHORT(CDID_CHANGEENTRY, CMDSRC_OTHER), (MPARAM)TRUE);
            }
            break;
        }
        break;


case WM_COMMAND:                        /* Button pressed */
    switch(SHORT1FROMMP(mp1))
    {
/*                                                                                      *\
 * Chain up the linked list until we find the node, where this part-list comes from or  *
 * the beginning of the complete list. The pointer pMenuData is adjusted.               *
\*                                                                                      */
    case CDID_LEVELUP:                  /* Get up one level in the linked list */
        {
        MENUDATA        *pMD;           /* Pointer to a MENUDATA structure to find the
                                           Submenu where this part-list starts */

        pMD=pHP->pMenuData; /* Point to the first element of the linked list
                                           at the current level */
        if(pMD->Back==NULL)             /* If we're at the beginning of the complete linked
                                           list ignore button */
            return((MRESULT)FALSE);
        else pMD=pMD->Back;             /* Submenu which started current level */
                                        /* Now chain back through the linked list and find
                                           the element, where the pointer to a Submenu
                                           equals the back pointer of the first element
                                           in this Submenu. Then we've found the node */
        while(TRUE)
            {
            if(pMD->Back==NULL)         /* If we're now at the beginning break */
                break;
            if((pMD->Back)->Submenu==pMD)
                break;
            else pMD=pMD->Back;
            }
        pHP->pMenuData=pMD; /* Load as the top element of the current item */
                                        /* Set cursor selection */
        ulListboxCursor=0;
                                        /* Now redraw items in listbox */
        WinSendMsg(hwndDlg, WM_LOADPOPUPMENU, MPFROMP(pHP->pMenuData), NULL);
        return((MRESULT)FALSE);         /* We handled this button */
        }

/*                                                                                      *\
 * Test the user selection for being a Submenu. If one found chain into this submenu    *
 * and adjust the pointer pMenuData.                                                    *
\*                                                                                      */
    case CDID_LEVELDOWN:                /* Get down one level in the linked list */
        {
        MENUDATA        *pMD;           /* Pointer to a MENUDATA structure to find the
                                           Submenu to chain into */
        SHORT           sCount;

        pMD=pHP->pMenuData; /* Point to the first element of the linked list
                                           at the current level */
                                        /* Send message to listbox */
        sCount=(SHORT)WinSendDlgItemMsg(
            hwndDlg,                    /* Handle of dialog window */
            CDLB_MENUPROGRAM,           /* Submenu & Program listbox */
            LM_QUERYSELECTION,          /* Query first selected list box item */
            MPFROMSHORT(LIT_FIRST),
            (MPARAM)NULL);
                                        /* If no item selected, ignore this button */
        if(sCount==LIT_NONE)
            return((MRESULT)FALSE);
        for( ;sCount>0; sCount--)       /* Walk through the linked list to the selected
                                           item */
            pMD=pMD->Next;
        if(pMD->Item!=ENTRYSUBMENU)     /* It's not a Submenu that's selected, ignore */
            return((MRESULT)FALSE);
                                        /* Otherwise chain into this part-list */
        pHP->pMenuData=pMD->Submenu;
                                        /* Set cursor selection */
        ulListboxCursor=0;
                                        /* Now redraw items in listbox */
        WinSendMsg(hwndDlg, WM_LOADPOPUPMENU, MPFROMP(pHP->pMenuData), NULL);
        return((MRESULT)FALSE);         /* We handled this button */
        }

/*                                                                                      *\
 * The user selected to add a (Sub)Menu. Thus dismiss the PC/2 Configuration dialog and *
 * load the (Sub)Menu Installation dialog. The new (Sub)Menu is entered in a            *
 * STARTSESSION structure named StartSession. Save the changes and reload the PC/2      *
 * Configuration dialog again.                                                          *
\*                                                                                      */
    case CDID_ADDMENU:                  /* Add a Menu to PC/2 Configuration selected */
/*                                                                                      *\
 * The user selected to add a Program. Thus dismiss the PC/2 Configuration dialog and   *
 * load the Menucontrol Addition dialog. The new control style is selected via          *
 * autoradiobuttons and is fille STARTSESSION structure named StartSession. Save the    *
 * changes and reload the PC/2 Configuration dialog again.                              *
\*                                                                                      */
    case CDID_ADDPROGRAM:               /* Add a Program to PC/2 Configuration selected */
/*                                                                                      *\
 * The user selected to add a Control Style. Thus dismiss the PC/2 Configuration dialog *
 * and load the Program Installation dialog. The new session data is entered in a       *
 * STARTSESSION structure named StartSession. Save the changes and reload the PC/2      *
 * Configuration dialog again.                                                          *
\*                                                                                      */
    case CDID_ADDCONTROL:               /* Add a Control Entry to PC/2 Configuration selected */
        {
        SESSIONDATA     SessionData;    /* Used by Menu Installation dialog and by
                                           Program Installation dialog to store menu or
                                           program data, to be filled from the user or
                                           to be presented to the user. */
        UCHAR           *pU;            /* Temporary character pointer */
        MENUDATA        *pMD;           /* Pointer to a MENUDATA structure to insert a
                                           new MENUDATA stucture after */
        MENUDATA        *pMDNew;        /* Temporary pointer for the new item to be inserted
                                           after pMD */
        ULONG           ulResult;       /* Each dialog procedure returns the result with
                                           WinDismissDlg() */
        SHORT           sCount;

        pMD=pHP->pMenuData; /* Point to the first element of the linked list
                                           at the current level */
                                        /* Send message to listbox */
        sCount=(SHORT)WinSendDlgItemMsg(
            hwndDlg,                    /* Handle of dialog window */
            CDLB_MENUPROGRAM,           /* Submenu & Program listbox */
            LM_QUERYSELECTION,          /* Query first selected list box item */
            MPFROMSHORT(LIT_FIRST),
            (MPARAM)NULL);
                                        /* Save cursor selection */
        if(sCount!=LIT_NONE)
            ulListboxCursor=(ULONG)sCount;
                                        /* If no item selected, and there exists one,
                                           add the new Menuitem after the last available
                                           Menuitem by querying the number from the listbox.
                                           Subtract 0 because we use 0-based instead 1-based. */
        if((sCount==LIT_NONE) && (pHP->pMenuData->Item!=ENTRYEMPTY))
            sCount=(SHORT)WinSendDlgItemMsg(hwndDlg, CDLB_MENUPROGRAM, LM_QUERYITEMCOUNT,
                MPFROM2SHORT(NULL, NULL), (MPARAM)NULL)-1;

        for( ;sCount>0; sCount--)       /* Walk through the linked list to the selected
                                           item */
            pMD=pMD->Next;
                                        /* Allocate a new item */
        pMDNew=AllocateMenuData();
        if(SHORT1FROMMP(mp1)!=CDID_ADDCONTROL)
            {                           /* Don't modify MenuData structure if a control entry
                                           is added */
            strcpy(pU=malloc(strlen("Insert here please")+1), "Insert here please");
            free(pMDNew->PgmTitle);
            pMDNew->PgmTitle=pU;
                                        /* Increment ID only for non control entries, because
                                           control entries get their own unique ID */
            pMDNew->id=pHP->MenuDataId++;
            }
        LoadMenuData2SessionData(pMDNew, &SessionData);
        if(SHORT1FROMMP(mp1)==CDID_ADDMENU)
            {
            if(!(ulResult=WinDlgBox(    /* Start Addmenu PC/2 dialog box */
                HWND_DESKTOP,           /* DESKTOP is parent */
                HWND_DESKTOP,           /* DESKTOP is owner */
                MI_DialogProcedure,     /* Dialog procedure of Program Installation
                                           dialog */
                0,                      /* Ressource is .EXE file */
                MIID_MENUDIALOG,        /* ID of Addmenu PC/2 dialog */
                &SessionData)))         /* Initialization data */
                PM_ERR(pHP->habPc2, pHP->hwndFrame, HELP_CREATEDIALOG, MB_ERROR|MB_OK|MB_HELP|MB_MOVEABLE|MB_DEFBUTTON1,
                    "Creation of a dialog box failed - continuing...");
            }
        if(SHORT1FROMMP(mp1)==CDID_ADDPROGRAM)
            {
            if(!(ulResult=WinDlgBox(    /* Start Program Installation dialog box */
                HWND_DESKTOP,           /* DESKTOP is parent */
                hwndDlg,                /* This dialog is owner */
                PI_DialogProcedure,     /* Dialog procedure of Program Installation
                                           dialog */
                0,                      /* Ressource is .EXE file */
                PIID_PROGRAMDIALOG,     /* ID of Addprogram PC/2 dialog */
                &SessionData)))         /* Initialization data */
                PM_ERR(pHP->habPc2, pHP->hwndFrame, HELP_CREATEDIALOG, MB_ERROR|MB_OK|MB_HELP|MB_MOVEABLE|MB_DEFBUTTON1,
                    "Creation of a dialog box failed - continuing...");
            }
        if(SHORT1FROMMP(mp1)==CDID_ADDCONTROL)
            {
            if(!(ulResult=WinDlgBox(    /* Start Menucontrol Addition dialog box */
                HWND_DESKTOP,           /* DESKTOP is parent */
                hwndDlg,                /* This dialog is owner */
                MD_DialogProcedure,     /* Dialog procedure of Menucontrol Addition
                                           dialog */
                0,                      /* Ressource is .EXE file */
                MDID_CONTROLDIALOG,     /* ID of Addmenucontrol PC/2 dialog */
                &SessionData)))         /* Initialization data */
                PM_ERR(pHP->habPc2, pHP->hwndFrame, HELP_CREATEDIALOG, MB_ERROR|MB_OK|MB_HELP|MB_MOVEABLE|MB_DEFBUTTON1,
                    "Creation of a dialog box failed - continuing...");
            }
                                        /* Check for break separators, because they can only
                                           be applied to functional menuitems as a change of
                                           their style. Break separators also can't be applied
                                           an break saparators. */
        if(!strcmp(SessionData.PgmTitle, CTRL_BREAKSEPARATOR))
            if((pMD->Back==NULL) || (pMD->Back->Submenu==pMD) ||
                (!strcmp(pMD->PgmTitle, CTRL_BREAKSEPARATOR)))
                ulResult=DID_CANCEL;    /* If user wants to apply a break separator style to
                                           first entry of a submenu, ignore request */

        if(ulResult==DID_OK)            /* If manipulation is done successfully, then load
                                           the SESSIONDATA structure back to the MENUDATA
                                           structure and save the changes */
            {
            LoadSessionData2MenuData(pMDNew, &SessionData);
            if(pMD->Item!=ENTRYEMPTY)   /* Add new entry, if the current entry isn't empty */
                {
                if(SHORT1FROMMP(mp1)==CDID_ADDMENU)
                    {                   /* It it is a Submenu, we also must add an empty
                                           first item for it */
                    MENUDATA    *pMDTemp;

                    pMDTemp=AllocateMenuData();
                    pMDNew->Submenu=pMDTemp;
                    pMDTemp->Back=pMDNew;
                    pMDNew->Item=ENTRYSUBMENU;
                    }
                if(SHORT1FROMMP(mp1)==CDID_ADDPROGRAM) pMDNew->Item=ENTRYMENUITEM;
                if(SHORT1FROMMP(mp1)==CDID_ADDCONTROL)
                    {                   /* For controls also add the ID of the control. These
                                           controls have an predefined ID because the action
                                           they should start is allways the same */
                    pMDNew->Item=ENTRYCONTROL;
                    if(!strcmp(pMDNew->PgmTitle, CTRL_CONFIGMENU))
                        {               /* If the Configuration Dialog is found reset flag
                                           to false */
                        pHP->ulStatusFlag&=(~DISPLAYCONFIGDIALOG);
                        pMDNew->id=ID_CONFIGDIALOG;
                        }
                    if(!strcmp(pMDNew->PgmTitle, CTRL_CONFIGDESKTOP)) pMDNew->id=ID_DESKTOPDIALOG;
                    if(!strcmp(pMDNew->PgmTitle, CTRL_ABOUT)) pMDNew->id=ID_ABOUTDIALOG;
                    if(!strcmp(pMDNew->PgmTitle, CTRL_SHUTDOWN)) pMDNew->id=ID_SHUTDOWN;
                    if(!strcmp(pMDNew->PgmTitle, CTRL_HELP)) pMDNew->id=ID_HELP;
                    if(!strcmp(pMDNew->PgmTitle, CTRL_EXIT)) pMDNew->id=ID_EXIT;
                    if(!strcmp(pMDNew->PgmTitle, CTRL_SPOOLER)) pMDNew->id=ID_SPOOLER;
                                        /* We assume that more than 1 separator may occur
                                           so to be able add, modify or delete on menuitems
                                           define a unique one */
                    if(!strcmp(pMDNew->PgmTitle, CTRL_BREAKSEPARATOR)) pMDNew->id=pHP->MenuDataId++;;
                    if(!strcmp(pMDNew->PgmTitle, CTRL_SEPARATOR)) pMDNew->id=pHP->MenuDataId++;;
                    }
                if(pMD->Next!=NULL) (pMD->Next)->Back=pMDNew;
                pMDNew->Next=pMD->Next;
                pMDNew->Back=pMD;
                pMD->Next=pMDNew;
                                        /* Insert item after the existing item */
                SetPopupMenu(MM_INSERTITEMMENUITEM, MPFROMP(pMDNew), MPFROMP(pMD));
                }
            else                        /* If it is an empty entry fill it with user data */
                {
                UCHAR   *pC;            /* Temporary character pointer */

                pMD->id=pMDNew->id;
                if(SHORT1FROMMP(mp1)==CDID_ADDMENU)
                    {                   /* It it is a Submenu, we also must add an empty
                                           first item for it */
                    MENUDATA    *pMDTemp;

                    pMDTemp=AllocateMenuData();
                    pMD->Submenu=pMDTemp;
                    pMDTemp->Back=pMD;
                    pMD->Item=ENTRYSUBMENU;
                    }
                if(SHORT1FROMMP(mp1)==CDID_ADDPROGRAM) pMD->Item=ENTRYMENUITEM;
                if(SHORT1FROMMP(mp1)==CDID_ADDCONTROL)
                    {                   /* For controls also add the ID of the control. These
                                           controls have an predefined ID because the action
                                           they should start is allways the same */
                    pMD->Item=ENTRYCONTROL;
                    if(!strcmp(pMDNew->PgmTitle, CTRL_CONFIGMENU))
                        {               /* If the Configuration Dialog is found reset flag
                                           to false */
                        pHP->ulStatusFlag&=(~DISPLAYCONFIGDIALOG);
                        pMD->id=ID_CONFIGDIALOG;
                        }
                    if(!strcmp(pMDNew->PgmTitle, CTRL_CONFIGDESKTOP)) pMD->id=ID_DESKTOPDIALOG;
                    if(!strcmp(pMDNew->PgmTitle, CTRL_ABOUT)) pMD->id=ID_ABOUTDIALOG;
                    if(!strcmp(pMDNew->PgmTitle, CTRL_SHUTDOWN)) pMD->id=ID_SHUTDOWN;
                    if(!strcmp(pMDNew->PgmTitle, CTRL_HELP)) pMD->id=ID_HELP;
                    if(!strcmp(pMDNew->PgmTitle, CTRL_EXIT)) pMD->id=ID_EXIT;
                    if(!strcmp(pMDNew->PgmTitle, CTRL_SPOOLER)) pMD->id=ID_SPOOLER;
                                        /* We assume that more than 1 separator may occur
                                           so to be able add, modify or delete on menuitems
                                           define a unique one */
                    if(!strcmp(pMDNew->PgmTitle, CTRL_BREAKSEPARATOR)) pMD->id=pHP->MenuDataId++;;
                    if(!strcmp(pMDNew->PgmTitle, CTRL_SEPARATOR)) pMD->id=pHP->MenuDataId++;;
                    }
                strcpy(pC=malloc(strlen(pMDNew->PgmTitle)+1), pMDNew->PgmTitle);
                free(pMD->PgmTitle);
                pMD->PgmTitle=pC;
                strcpy(pC=malloc(strlen(pMDNew->WindowTitle)+1), pMDNew->WindowTitle);
                free(pMD->WindowTitle);
                pMD->WindowTitle=pC;
                strcpy(pC=malloc(strlen(pMDNew->PgmName)+1), pMDNew->PgmName);
                free(pMD->PgmName);
                pMD->PgmName=pC;
                strcpy(pC=malloc(strlen(pMDNew->PgmDirectory)+1), pMDNew->PgmDirectory);
                free(pMD->PgmDirectory);
                pMD->PgmDirectory=pC;
                strcpy(pC=malloc(strlen(pMDNew->PgmInputs)+1), pMDNew->PgmInputs);
                free(pMD->PgmInputs);
                pMD->PgmInputs=pC;
                strcpy(pC=malloc(strlen(pMDNew->PgmDosSettings)+1), pMDNew->PgmDosSettings);
                free(pMD->PgmDosSettings);
                pMD->PgmDosSettings=pC;
                pMD->SessionType=pMDNew->SessionType;
                pMD->PgmControl=pMDNew->PgmControl;
                pMD->FgBg=pMDNew->FgBg;
                pMD->InitXPos=pMDNew->InitXPos;
                pMD->InitYPos=pMDNew->InitYPos;
                pMD->InitXSize=pMDNew->InitXSize;
                pMD->InitYSize=pMDNew->InitYSize;
                (pMD->KeyData).usFlags=(pMDNew->KeyData).usFlags;
                (pMD->KeyData).usCh=(pMDNew->KeyData).usCh;
                pMD->PriorityClass=pMDNew->PriorityClass;
                pMD->SwpFlag=pMDNew->SwpFlag;
                pMD->PriorityDelta=pMDNew->PriorityDelta;
                if(pMD->Back!=NULL)     /* This is the first item of a Submenu, then
                                           insert it there */
                    SetPopupMenu(MM_INSERTITEMSUBMENU, MPFROMP(pMD), MPFROMP(pMD->Back));
                else                    /* This is the complete first item of the linked
                                           list, so insert at the end */
                    SetPopupMenu(MM_INSERTITEMMENUITEM, MPFROMP(pMD), MPFROMP(NULL));
                free(pMDNew->PgmTitle); /* Free temporary used structure */
                free(pMDNew->WindowTitle);
                free(pMDNew->PgmName);
                free(pMDNew->PgmDirectory);
                free(pMDNew->PgmInputs);
                free(pMDNew->PgmDosSettings);
                free(pMDNew);
                }
            }
        else
            {
            free(pMDNew->PgmTitle);     /* Free temporary MENUDATA structure */
            free(pMDNew->WindowTitle);
            free(pMDNew->PgmName);
            free(pMDNew->PgmDirectory);
            free(pMDNew->PgmInputs);
            free(pMDNew->PgmDosSettings);
            free(pMDNew);
            }
                                        /* Initialize the listbox */
        WinSendMsg(hwndDlg, WM_LOADPOPUPMENU, MPFROMP(pHP->pMenuData), NULL);
                                        /* If PC/2 is not the WPS process and not enabled,
                                           disable spooler menu item in Popup Menu */
        if((!(pHP->ulStatusFlag & PC2RUNNINGASWPS)) && (!(pHP->ulStatusFlag & SHOWSPOOLERWINDOW)))
            WinSendMsg(pHP->hwndPopupMenu, MM_SETITEMATTR,
                MPFROM2SHORT(ID_SPOOLER, TRUE), MPFROM2SHORT(MIA_DISABLED, MIA_DISABLED));
                                        /* Disable Exit OS/2 menuitem when running as the
                                           WPS replacement */
        if(pHP->ulStatusFlag&PC2RUNNINGASWPS)
            {
            WinSendMsg(pHP->hwndPopupMenu, MM_SETITEMATTR,
                MPFROM2SHORT(ID_EXIT, TRUE), MPFROM2SHORT(MIA_DISABLED, MIA_DISABLED));
            }
        break;
        }

/*                                                                                      *\
 * The user selected to change an item. Thus dismiss the PC/2 Configuration dialog and  *
 * load the Menu or Program Installation dialog. The new session data is entered in a   *
 * STARTSESSION structure named StartSession.                                           *
 * Then reload the PC/2 Configuration dialog again.                                     *
\*                                                                                      */
    case CDID_CHANGEENTRY:              /* Change a Menu or Program configuration selected */
        {
        SESSIONDATA     SessionData;    /* Used by Menu Installation dialog and by
                                           Program Installation dialog to store menu or
                                           program data, to be filled from the user or
                                           to be presented to the user. */
        ULONG           ulResult;       /* Each dialog procedure returns the result with
                                           WinDismissDlg() */
        MENUDATA        *pMD;
        SHORT           sCount;

        pMD=pHP->pMenuData; /* Point to the first element of the linked list
                                           at the current level */
                                        /* Send message to listbox */
        sCount=(SHORT)WinSendDlgItemMsg(
            hwndDlg,                    /* Handle of dialog window */
            CDLB_MENUPROGRAM,           /* Submenu & Program listbox */
            LM_QUERYSELECTION,          /* Query first selected list box item */
            MPFROMSHORT(LIT_FIRST),
            (MPARAM)NULL);
        if(sCount==LIT_NONE)            /* If no item selected ignore this button */
            return((MRESULT)FALSE);
                                        /* Save cursor selection */
        ulListboxCursor=(ULONG)sCount;
        for( ;sCount>0; sCount--)       /* Walk through the linked list to the selected
                                           item */
            pMD=pMD->Next;
                                        /* Controls can't be changed, they must be
                                           removed and newly inserted */
        if(pMD->Item==ENTRYCONTROL) return((MRESULT)FALSE);
                                        /* Now load the MENUDATA to SESSIONDATA structure
                                           where the manipulations will take effect */
        LoadMenuData2SessionData(pMD, &SessionData);
        if(pMD->Item==ENTRYMENUITEM)
                                        /* It's a Menuitem  so call the Program
                                           Installation dialog box */
            if(!(ulResult=WinDlgBox(    /* Start Program Installation dialog box */
                HWND_DESKTOP,           /* DESKTOP is parent */
                hwndDlg,                /* This dialog is owner */
                PI_DialogProcedure,     /* Dialog procedure of Program Installation
                                           dialog */
                0,                      /* Ressource is .EXE file */
                PIID_PROGRAMDIALOG,     /* ID of Program Installation PC/2 dialog */
                &SessionData)))         /* Initialization data */
                PM_ERR(pHP->habPc2, pHP->hwndFrame, HELP_CREATEDIALOG, MB_ERROR|MB_OK|MB_HELP|MB_MOVEABLE|MB_DEFBUTTON1,
                    "Creation of a dialog box failed - continuing...");
        if(pMD->Item==ENTRYSUBMENU)
                                        /* It's a Submenu so call the Menu Installation
                                           dialog box */
            if(!(ulResult=WinDlgBox(    /* Start Addmenu PC/2 dialog box */
                HWND_DESKTOP,           /* DESKTOP is parent */
                hwndDlg,                /* This dialog is owner */
                MI_DialogProcedure,     /* Dialog procedure of Program Installation
                                           dialog */
                0,                      /* Ressource is .EXE file */
                MIID_MENUDIALOG,        /* ID of Addmenu PC/2 dialog */
                &SessionData)))         /* Initialization data */
                PM_ERR(pHP->habPc2, pHP->hwndFrame, HELP_CREATEDIALOG, MB_ERROR|MB_OK|MB_HELP|MB_MOVEABLE|MB_DEFBUTTON1,
                    "Creation of a dialog box failed - continuing...");
        if(ulResult==DID_OK)            /* If manipulation is done successfully, then load
                                           the SESSIONDATA structure back to the MENUDATA
                                           structure and save the changes */
            {
            LoadSessionData2MenuData(pMD, &SessionData);
                                        /* Now change the menuitem text to the new one */
            SetPopupMenu(MM_SETITEMTEXT, MPFROMP(pMD), MPFROMLONG(pMD->id));
                                        /* Initialize the listbox */
            WinSendMsg(hwndDlg, WM_LOADPOPUPMENU, MPFROMP(pHP->pMenuData), NULL);
            }
        break;
        }

/*                                                                                      *\
 * The user selected to remove an item. If thist item is the only one in the linked     *
 * list or the first item of a submenu set it to empty otherwise free it's ressources   *
 * and remove the entry.                                                                *
\*                                                                                      */
    case CDID_REMOVEENTRY:              /* Remove a item of the PC/2 Configuration selected */
        {
        UCHAR           *pU;
        MENUDATA        *pMD;
        SHORT           sCount;

        pMD=pHP->pMenuData;             /* Point to the first element of the linked list
                                           at the current level */
                                        /* Send message to listbox */
        sCount=(SHORT)WinSendDlgItemMsg(
            hwndDlg,                    /* Handle of dialog window */
            CDLB_MENUPROGRAM,           /* Submenu & Program listbox */
            LM_QUERYSELECTION,          /* Query first selected list box item */
            MPFROMSHORT(LIT_FIRST),
            (MPARAM)NULL);
        if(sCount==LIT_NONE)            /* If no item selected ignore this button */
            return((MRESULT)FALSE);
                                        /* Save cursor selection */
        ulListboxCursor=(ULONG)sCount;
                                        /* Save cursor selection */
        ulListboxCursor=(ULONG)sCount;
        for( ;sCount>0; sCount--)       /* Walk through the linked list to the selected
                                           item */
            pMD=pMD->Next;
        while(TRUE)
            {
                                        /* If the selected menu entry is followed by a
                                           break separator, which is just a style of the
                                           menu entry itself, disallow removal before the
                                           break separator is deleted, because this avoids
                                           very complex logic to correctly remove the
                                           break separator / append it to previous menu entry */
            if(pMD->Next!=NULL)
                if(!strcmp(pMD->Next->PgmTitle, CTRL_BREAKSEPARATOR))
                    break;
            if((pMD->Back==NULL) && (pMD->Next!=NULL))
                {                       /* Remove the first item of the complete linked list */
                if(pMD->Item==ENTRYSUBMENU)
                if((pMD->Submenu)->Item==ENTRYEMPTY)
                    {                   /* If it is an empty Submenu remove it completely */
                                        /* Remove the Submenu and the empty first item
                                           from the Popup-Menu */
                    SetPopupMenu(MM_DELETEITEM, MPFROMP(pMD), MPFROMP(pMD));
                    free((pMD->Submenu)->PgmTitle);
                    free((pMD->Submenu)->WindowTitle);
                    free((pMD->Submenu)->PgmName);
                    free((pMD->Submenu)->PgmDirectory);
                    free((pMD->Submenu)->PgmInputs);
                    free((pMD->Submenu)->PgmDosSettings);
                    free(pMD->Submenu);
                    (pMD->Next)->Back=NULL;
                                        /* Now next element is the first one */
                    pHP->pPopupMenu=pMD->Next;
                    pHP->pMenuData=pMD->Next;
                    free(pMD->PgmTitle);
                    free(pMD->WindowTitle);
                    free(pMD->PgmName);
                    free(pMD->PgmDirectory);
                    free(pMD->PgmInputs);
                    free(pMD->PgmDosSettings);
                    free(pMD);
                    break;              /* Ensure we only test once, because each routine
                                           changes the pointers */
                    }
                if((pMD->Item==ENTRYMENUITEM) || (pMD->Item==ENTRYCONTROL))
                    {                   /* If it is an empty Menuitem or Control remove it completly */
                                        /* If the Configuration Dialog is found set flag
                                           to false */
                    if(!strcmp(pMD->PgmTitle, CTRL_CONFIGDESKTOP)) pHP->ulStatusFlag &= (~DISPLAYCONFIGDIALOG);
                    (pMD->Next)->Back=NULL;
                                        /* Now next element is the first one */
                    pHP->pPopupMenu=pMD->Next;
                    pHP->pMenuData=pMD->Next;
                                        /* Remove the item from the Popup-Menu */
                    SetPopupMenu(MM_DELETEITEM, MPFROMP(pMD), MPFROMP(pMD));
                    free(pMD->PgmTitle);
                    free(pMD->WindowTitle);
                    free(pMD->PgmName);
                    free(pMD->PgmDirectory);
                    free(pMD->PgmInputs);
                    free(pMD->PgmDosSettings);
                    free(pMD);
                    break;              /* Ensure we only test once, because each routine
                                           changes the pointers */
                    }
                }
            if((pMD->Back==NULL) && (pMD->Next==NULL))
                {                       /* If it is the one and only item of the linked list
                                           set it to empty */
                if(pMD->Item==ENTRYSUBMENU)
                if((pMD->Submenu)->Item==ENTRYEMPTY)
                    {                   /* If it is an empty Submenu remove the empty
                                           item completely */
                                        /* Remove the item from the Popup-Menu */
                    SetPopupMenu(MM_DELETEITEM, MPFROMP(pMD), MPFROMP(pMD));
                    free((pMD->Submenu)->PgmTitle);
                    free((pMD->Submenu)->WindowTitle);
                    free((pMD->Submenu)->PgmName);
                    free((pMD->Submenu)->PgmDirectory);
                    free((pMD->Submenu)->PgmInputs);
                    free((pMD->Submenu)->PgmDosSettings);
                    free(pMD->Submenu);
                    free(pMD->PgmTitle);
                    strcpy(pU=malloc(strlen("")+1), "");
                    pMD->PgmTitle=pU;
                    free(pMD->WindowTitle);
                    strcpy(pU=malloc(strlen("")+1), "");
                    pMD->WindowTitle=pU;
                    free(pMD->PgmName);
                    strcpy(pU=malloc(strlen("")+1), "");
                    pMD->PgmName=pU;
                    free(pMD->PgmDirectory);
                    strcpy(pU=malloc(strlen("")+1), "");
                    pMD->PgmDirectory=pU;
                    free(pMD->PgmInputs);
                    strcpy(pU=malloc(strlen("")+1), "");
                    pMD->PgmInputs=pU;
                    free(pMD->PgmDosSettings);
                    strcpy(pU=malloc(strlen("")+1), "");
                    pMD->PgmDosSettings=pU;
                    pMD->Item=ENTRYEMPTY;
                    pMD->Back=NULL;
                    pMD->Submenu=NULL;
                    pMD->Next=NULL;
                    break;              /* Ensure we only test once, because each routine
                                           changes the pointers */
                    }
                if((pMD->Item==ENTRYMENUITEM) || (pMD->Item==ENTRYCONTROL))
                    {                   /* If it is a Menuitem or Control set it to empty */
                                        /* Remove the item from the Popup-Menu */
                    SetPopupMenu(MM_DELETEITEM, MPFROMP(pMD), MPFROMP(pMD));
                    free(pMD->PgmTitle);
                    strcpy(pU=malloc(strlen("")+1), "");
                    pMD->PgmTitle=pU;
                    free(pMD->WindowTitle);
                    strcpy(pU=malloc(strlen("")+1), "");
                    pMD->WindowTitle=pU;
                    free(pMD->PgmName);
                    strcpy(pU=malloc(strlen("")+1), "");
                    pMD->PgmName=pU;
                    free(pMD->PgmDirectory);
                    strcpy(pU=malloc(strlen("")+1), "");
                    pMD->PgmDirectory=pU;
                    free(pMD->PgmInputs);
                    strcpy(pU=malloc(strlen("")+1), "");
                    pMD->PgmInputs=pU;
                    free(pMD->PgmDosSettings);
                    strcpy(pU=malloc(strlen("")+1), "");
                    pMD->PgmDosSettings=pU;
                    pMD->Item=ENTRYEMPTY;
                    pMD->Back=NULL;
                    pMD->Submenu=NULL;
                    pMD->Next=NULL;
                    break;              /* Ensure we only test once, because each routine
                                           changes the pointers */
                    }
                }
            if(pMD->Back!=NULL)
                {                       /* It is any item of more than one item and not
                                           the first one */
                if(((pMD->Back)->Submenu==pMD) && (pMD->Submenu==NULL) && (pMD->Next==NULL))
                {                       /* If it is the first item of a Submenu not followed
                                           by any item, set it to empty */
                                        /* Remove the item from the Popup-Menu */
                    SetPopupMenu(MM_DELETEITEM, MPFROMP(pMD), MPFROMP(pMD));
                    free(pMD->PgmTitle);
                    strcpy(pU=malloc(strlen("")+1), "");
                    pMD->PgmTitle=pU;
                    free(pMD->WindowTitle);
                    strcpy(pU=malloc(strlen("")+1), "");
                    pMD->WindowTitle=pU;
                    free(pMD->PgmName);
                    strcpy(pU=malloc(strlen("")+1), "");
                    pMD->PgmName=pU;
                    free(pMD->PgmDirectory);
                    strcpy(pU=malloc(strlen("")+1), "");
                    pMD->PgmDirectory=pU;
                    free(pMD->PgmInputs);
                    strcpy(pU=malloc(strlen("")+1), "");
                    pMD->PgmInputs=pU;
                    free(pMD->PgmDosSettings);
                    strcpy(pU=malloc(strlen("")+1), "");
                    pMD->PgmDosSettings=pU;
                    pMD->Item=ENTRYEMPTY;
                    break;              /* Ensure we only test once, because each routine
                                           changes the pointers */
                    }
                if(pMD->Item==ENTRYSUBMENU)
                if((pMD->Submenu)->Item==ENTRYEMPTY)
                    {                   /* If it is an empty Submenu so also remove the
                                           first item in the Submenu */
                                        /* Remove the Submenu and the empty first item
                                           from the Popup-Menu */
                    SetPopupMenu(MM_DELETEITEM, MPFROMP(pMD), MPFROMP(pMD));
                    free((pMD->Submenu)->PgmTitle);
                    free((pMD->Submenu)->WindowTitle);
                    free((pMD->Submenu)->PgmName);
                    free((pMD->Submenu)->PgmDirectory);
                    free((pMD->Submenu)->PgmInputs);
                    free((pMD->Submenu)->PgmDosSettings);
                    free(pMD->Submenu);
                    if(((pMD->Back)->Submenu==pMD) && (pMD->Next==NULL))
                        {               /* If the previous item is a Submenu, this item is
                                           the first item of it. If none item follows, set
                                           this item to empty */
                        free(pMD->PgmTitle);
                        strcpy(pU=malloc(strlen("")+1), "");
                        pMD->PgmTitle=pU;
                        free(pMD->WindowTitle);
                        strcpy(pU=malloc(strlen("")+1), "");
                        pMD->WindowTitle=pU;
                        free(pMD->PgmName);
                        strcpy(pU=malloc(strlen("")+1), "");
                        pMD->PgmName=pU;
                        free(pMD->PgmDirectory);
                        strcpy(pU=malloc(strlen("")+1), "");
                        pMD->PgmDirectory=pU;
                        free(pMD->PgmInputs);
                        strcpy(pU=malloc(strlen("")+1), "");
                        pMD->PgmInputs=pU;
                        free(pMD->PgmDosSettings);
                        strcpy(pU=malloc(strlen("")+1), "");
                        pMD->PgmDosSettings=pU;
                        pMD->Item=ENTRYEMPTY;
                        pMD->Submenu=NULL;
                        pMD->Next=NULL;
                        break;          /* Ensure we only test once, because each routine
                                           changes the pointers */
                        }
                    if(((pMD->Back)->Submenu==pMD) && (pMD->Next!=NULL))
                        {               /* If the previous item is a Submenu, this item ist
                                           the first item of it. If one item follows adjust
                                           the pointer to the current level of items */
                        pHP->pMenuData=pMD->Next;
                        (pMD->Back)->Submenu=pMD->Next;
                        if(pMD->Next!=NULL) (pMD->Next)->Back=pMD->Back;
                        free(pMD->PgmTitle);
                        free(pMD->WindowTitle);
                        free(pMD->PgmName);
                        free(pMD->PgmDirectory);
                        free(pMD->PgmInputs);
                        free(pMD->PgmDosSettings);
                        free(pMD);
                        break;          /* Ensure we only test once, because each routine
                                           changes the pointers */
                        }
                    if((pMD->Back)->Submenu!=pMD)
                        {               /* If this item isn't the first item of a Submenu */
                        (pMD->Back)->Next=pMD->Next;
                        if(pMD->Next!=NULL) (pMD->Next)->Back=pMD->Back;
                        free(pMD->PgmTitle);
                        free(pMD->WindowTitle);
                        free(pMD->PgmName);
                        free(pMD->PgmDirectory);
                        free(pMD->PgmInputs);
                        free(pMD->PgmDosSettings);
                        free(pMD);
                        break;          /* Ensure we only test once, because each routine
                                           changes the pointers */
                        }
                    }
                if((pMD->Item==ENTRYMENUITEM) || pMD->Item==ENTRYCONTROL)
                    {                   /* If it is a Menuitem or Control just remove it completly */
                                        /* Remove the item from the Popup-Menu */
                    SetPopupMenu(MM_DELETEITEM, MPFROMP(pMD), MPFROMP(pMD));
                    if(((pMD->Back)->Submenu==pMD) && (pMD->Next==NULL))
                        {               /* If the previous item is a Submenu, this item is
                                           the first item of it. If none item follows, set
                                           this item to empty */
                        free(pMD->PgmTitle);
                        strcpy(pU=malloc(strlen("")+1), "");
                        pMD->PgmTitle=pU;
                        free(pMD->WindowTitle);
                        strcpy(pU=malloc(strlen("")+1), "");
                        pMD->WindowTitle=pU;
                        free(pMD->PgmName);
                        strcpy(pU=malloc(strlen("")+1), "");
                        pMD->PgmName=pU;
                        free(pMD->PgmDirectory);
                        strcpy(pU=malloc(strlen("")+1), "");
                        pMD->PgmDirectory=pU;
                        free(pMD->PgmInputs);
                        strcpy(pU=malloc(strlen("")+1), "");
                        pMD->PgmInputs=pU;
                        free(pMD->PgmDosSettings);
                        strcpy(pU=malloc(strlen("")+1), "");
                        pMD->PgmDosSettings=pU;
                        pMD->Item=ENTRYEMPTY;
                        pMD->Submenu=NULL;
                        pMD->Next=NULL;
                        break;          /* Ensure we only test once, because each routine
                                           changes the pointers */
                        }
                    if(((pMD->Back)->Submenu==pMD) && (pMD->Next!=NULL))
                        {               /* If the previous item is a Submenu, this item ist
                                           the first item of it. If one item follows adjust
                                           the pointer to the current level of items */
                        pHP->pMenuData=pMD->Next;
                        (pMD->Back)->Submenu=pMD->Next;
                        if(pMD->Next!=NULL) (pMD->Next)->Back=pMD->Back;
                        free(pMD->PgmTitle);
                        free(pMD->WindowTitle);
                        free(pMD->PgmName);
                        free(pMD->PgmDirectory);
                        free(pMD->PgmInputs);
                        free(pMD->PgmDosSettings);
                        free(pMD);
                        break;          /* Ensure we only test once, because each routine
                                           changes the pointers */
                        }
                    if((pMD->Back)->Submenu!=pMD)
                        {               /* If this item isn't the first item of a Submenu */
                        (pMD->Back)->Next=pMD->Next;
                        if(pMD->Next!=NULL) (pMD->Next)->Back=pMD->Back;
                        free(pMD->PgmTitle);
                        free(pMD->WindowTitle);
                        free(pMD->PgmName);
                        free(pMD->PgmDirectory);
                        free(pMD->PgmInputs);
                        free(pMD->PgmDosSettings);
                        free(pMD);
                        break;          /* Ensure we only test once, because each routine
                                           changes the pointers */
                        }
                    }
                }
            break;                      /* If we come here, we're trying to remove an not
                                           empty Submenu, but we also must exit the
                                           endless loop */
            }
                                        /* Initialize the listbox */
        WinSendMsg(hwndDlg, WM_LOADPOPUPMENU, MPFROMP(pHP->pMenuData), NULL);
        return((MRESULT)FALSE);         /* We have done everything */
        }

/*                                                                                      *\
 * The user selected to resort the current level of the menuentries. Load the dialog    *
 * and let the user resort the linked list of menues pointed to by pMenuData and to     *
 * resort the menuentries of the Popup-Menu.                                            *
\*                                                                                      */
    case CDID_RESORT:                   /* Load the resort dialog */
        if(!WinDlgBox(                  /* Start Resort dialog box */
            HWND_DESKTOP,               /* DESKTOP is parent */
            hwndDlg,                    /* This dialog is owner */
            RD_DialogProcedure,         /* Dialog procedure of Program Installation
                                           dialog */
            0,                          /* Ressource is .EXE file */
            RDID_RESORTDIALOG,          /* ID of Program Installation PC/2 dialog */
            0))                         /* No initialization data */
            PM_ERR(pHP->habPc2, pHP->hwndFrame, HELP_CREATEDIALOG, MB_ERROR|MB_OK|MB_HELP|MB_MOVEABLE|MB_DEFBUTTON1,
                "Creation of a dialog box failed - continuing...");
                                        /* Initialize the listbox */
        WinSendMsg(hwndDlg, WM_LOADPOPUPMENU, MPFROMP(pHP->pMenuData), NULL);
        break;

    case DID_OK:                        /* Enter key pressed */
                                        /* Save the changes */
        WinSendMsg(hwndDlg, WM_SAVEPOPUPMENU, NULL, NULL);
                                        /* Dialog terminated with DID_OK */
        WinDismissDlg(hwndDlg, DID_OK);
        break;

    case DID_CANCEL:                    /* Escape or Cancel pressed */
                                        /* Dialog terminated with DID_CANCEL */
        WinDismissDlg(hwndDlg, DID_CANCEL);
        break;

    default:
        return(WinDefDlgProc(hwndDlg, msg, mp1, mp2));
    }
    break;

default:                                /* Default window procedure must be called */
    return(WinDefDlgProc(hwndDlg, msg, mp1, mp2));
}
return((MRESULT)FALSE);                 /* We have handled the message */
}

/*--------------------------------------------------------------------------------------*\
 * This subclassed window procedure handles the PC/2's setup dialog CDLB_MENUPROGRAM    *
 * listbox to allow dropping into.                                                      *
 * Req: none                                                                            *
\*--------------------------------------------------------------------------------------*/
MRESULT  EXPENTRY SubclassedListboxWindowProc(HWND hwndListbox, ULONG msg, MPARAM mp1, MPARAM mp2)
{
PDRAGINFO       pDraginfo;              /* Pointer to DRAGINFO structure */
PDRAGITEM       pDragitem;              /* Pointer to DRAGITEM structure */

switch(msg)
{
/*                                                                                      *\
 * Allow known objects to be dropped.                                                   *
\*                                                                                      */
case DM_DRAGOVER:
    pDraginfo = (PDRAGINFO)mp1;         /* Get the pointer to the DRAGINFO structure */
                                        /* Access the structure */
    if(DrgAccessDraginfo(pDraginfo)==FALSE) break;
                                        /* Get the first itemp of the item(s) dragged
                                           onto dialog window */
    pDragitem = DrgQueryDragitemPtr(pDraginfo, 0);
                                        /* Allow only one object, coming from only WPS 2 PC/2 */
    if((pDraginfo->cditem!=1) || !(DrgVerifyRMF(pDragitem, "DRM_WPS2PC2", "DRF_WPS2PC2")))
                                        /* Don't allow dropping of more than one item */
        return(MPFROM2SHORT(DOR_NODROP, DO_UNKNOWN));
                                        /* Allow drop of undefined operation onto dialog window */
    return(MPFROM2SHORT(DOR_DROP, DO_UNKNOWN));

case DM_DROP:
    {
    UCHAR       ucObjectType[256];
    PBYTE       pbSharedMem;

    pDraginfo = (PDRAGINFO)mp1;         /* Get the pointer to the DRAGINFO structure */
                                        /* Access the structure */
    if(DrgAccessDraginfo(pDraginfo)==FALSE) break;
                                        /* Get the first itemp of the item(s) dragged
                                           onto dialog window */
    pDragitem = DrgQueryDragitemPtr(pDraginfo, 0);
                                        /* Query the rendering format */
    DrgQueryStrName(pDragitem->hstrRMF, sizeof(ucObjectType), ucObjectType);
                                        /* Scan for an file object */
    if(DrgVerifyRMF(pDragitem, DRM_WPS2PC2, DRF_WPS2PC2))
        {                               /* It was dragged from WPS 2 PC/2, so we know how to
                                           access the shared memory we expect */
        pbSharedMem=(PBYTE)pDragitem->ulItemID;
        if(DosGetSharedMem(pbSharedMem, PAG_READ | PAG_WRITE)==NO_ERROR)
            {
                                        /* Insert the LIST of WPSOBJECTLISTs into
                                           Popup-Menu */
            ConvertWPSObjectList2MenuData(pHP->pMenuData, (WPSOBJECTLIST *)pbSharedMem);
                                        /* Give memory back to OS/2 */
            DosFreeMem(pbSharedMem);
                                        /* Initialize the listbox */
            WinSendMsg(WinQueryWindow(hwndListbox, QW_OWNER), WM_LOADPOPUPMENU,
                MPFROMP(pHP->pMenuData), NULL);
            }
        break;
        }
    }
    return((MRESULT)FALSE);             /* We have handled the message */
}
                                        /* Call default window procedure */
return(pfnListboxWindowProc(hwndListbox, msg, mp1, mp2));
}

/*--------------------------------------------------------------------------------------*\
 * This subclassed window procedure handles the PC/2's smart icon menu.                 *
 * Req: none                                                                            *
\*--------------------------------------------------------------------------------------*/
MRESULT  EXPENTRY SubclassedMenuWindowProc(HWND hwndMenu, ULONG msg, MPARAM mp1, MPARAM mp2)
{
switch(msg)
{
case WM_PRESPARAMCHANGED:
    switch((ULONG)mp1)
    {
    case PP_FONTNAMESIZE:
        {
        ULONG       ulAttrFound;

                                        /* Get font selected for PC/2's smart icon menu */
        if(WinQueryPresParam(hwndMenu, PP_FONTNAMESIZE, 0, &ulAttrFound,
            sizeof(pHP->ucPopupMenuFont), pHP->ucPopupMenuFont, 0))
                                        /* Set this font also into the Popup-Menu */
            WinSetPresParam(pHP->hwndPopupMenu, PP_FONTNAMESIZE,
                sizeof(pHP->ucPopupMenuFont), pHP->ucPopupMenuFont);
        }
        break;
    }
}
                                        /* Call default window procedure */
return(pfnMenuWindowProc(hwndMenu, msg, mp1, mp2));
}

