/*
 * This OS/2 port was hacked by Harald Kipp from the
 *
 *      Network News Transfer Protocol server
 *
 *      Phil Lapsley
 *      University of California, Berkeley
 *      Stan Barber
 *      Baylor College of Medicine
 *
 * Bug reports related to THIS modified version should be sent to
 *
 *  harald@os2point.ping.de
 *  harald@haport.sesam.com
 *  Fido: 2:2448/434
 *
 */

#define OS2
#include <os2.h>

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <ctype.h>

#include <sys/types.h>
#include <sys/stat.h>

#include <tcpconn.h>

#include "nntp.h"
#include "config.h"
#include "globals.h"
#include "changi.h"


/* forward declarations */
void get_id(FILE * art_fp, char *id);


/*
 * open_valid_art -- determine if a given article name is valid;
 *              if it is, return a file pointer to the open article,
 *              along with a unique id of the article.
 *
 *      Parameters:     "artname" is a string containing the
 *                      name of the article.
 *                      "id" is space for us to put the article
 *                      id in.
 *
 *      Returns:        File pointer to the open article if the
 *                      article is valid; NULL otherwise
 *
 *      Side effects:   None.
 */

FILE *open_valid_art(PNEWSCLIENT pnc, char *artname, char *id)
{
    if (pnc -> art_fp != NULL) {
        if (pnc -> art_num == atol(artname)) {
	    if (fseek(pnc -> art_fp, 0L, SEEK_SET) < 0) {
		fclose(pnc -> art_fp);
		pnc -> art_fp = NULL;
	    }
	    else {
		strcpy(id, pnc -> art_id);
		return (pnc -> art_fp);
	    }
	}
	else {
	    fclose(pnc -> art_fp);
	    pnc -> art_fp = NULL;
	}
    }

    pnc -> art_fp = fopen(artname, "r");

    if (pnc -> art_fp == NULL)
	return (NULL);

    get_id(pnc -> art_fp, id);
    strcpy(pnc -> art_id, id);
    pnc -> art_num = atol(artname);
    return (pnc -> art_fp);
}


/*
 * gethistent -- return the path name of an article if it's
 * in the history file.
 *
 *      Parameters:     "msg_id" is the message ID of the
 *                      article, enclosed in <>'s.
 *                      "lookup", only check if article exists
 *
 *      Returns:        A char pointer to a static data area
 *                      containing the full pathname of the
 *                      article, or NULL if the message-id is not
 *                      in thef history file.
 *
 *      Side effects:   opens dbm database
 *                      (only once, keeps it open after that).
 *                      If running Bnews, converts "msg_id" to lower case.
 *                      If running Cnews, converts "msg_id" per rfc822.
 *
 */

char *gethistent(PNEWSCLIENT pnc, char *msg_id, char *path)
{
    char line[MAXBUFLEN];
    char *tmp;
    char *cp;

#ifdef NEWDBM
    datum key, val;

    if(pnc->db == NULL) {
        pnc->db = dbm_open(historyfile, O_RDONLY, 0);
        if (pnc->db == NULL)
            return (NULL);
    }

    key.dptr = msg_id;
    key.dsize = strlen(msg_id) + 1;
    val = dbm_fetch(pnc->db, key);
    if (val.dptr == NULL)
        return (NULL);
    strncpy(line, val.dptr, sizeof(line) - 1);
    line[sizeof(line) - 1] = '\0';
#if 0
    dbm_close(pnc->db);
#endif
    cp = line;

#else
    int len;
    FILE *hfp;

    if ((hfp = fopen(cfg.historyfile, "r")) == NULL) {
        lperror(cfg.historyfile);
	return (NULL);
    }
    len = strlen(msg_id);
    while (fgets(line, sizeof(line), hfp))
	if (!strnicmp(msg_id, line, len))
	    break;

    if (feof(hfp)) {
	fclose(hfp);
	return (NULL);
    }
    fclose(hfp);
    if ((cp = strchr(line, '\n')) != NULL)
	*cp = '\0';
    cp = strchr(line, ' ');

#endif

    if (cp != NULL)
	cp = strchr(cp + 1, ' ');
    if (cp != NULL)
	cp = strchr(cp + 1, ' ');
    if (cp == NULL) {
	return (NULL);			/* this article has expired */
    }
    cp++;

    strcpy(path, cfg.newsdir);
    tmp = path + strlen(path);
    *tmp++ = '/';
    while (*cp && *cp != ' ' && *cp != ',') {
	if (*cp == '.' || *cp == ':')
	    *tmp++ = '/';
	else
	    *tmp++ = *cp;
	cp++;
    }
    *tmp = '\0';

    return (path);
}

/*
 * openartbyid -- open an article by message-id.
 *
 *      Arguments:      "msg_id" is the message-id of the article
 *                      to open.
 *
 *      Returns:        File pointer to opened article, or NULL if
 *                      the article was not in the history file or
 *                      could not be opened.
 *
 *      Side effects:   Opens article.
 */

FILE *openartbyid(PNEWSCLIENT pnc, char *msg_id)
{
    char path[255];

    if (gethistent(pnc, msg_id, path))
	return (fopen(path, "r"));
    else
	return (NULL);
}


/*
 * check_ngperm -- check to see if they're allowed to see this
 * article by matching Newsgroups: and Distribution: line.
 *
 *      Parameters:     "fp" is the file pointer of this article.
 *
 *      Returns:        0 if they're not allowed to see it.
 *                      1 if they are.
 *
 *      Side effects:   None.
 */

int check_ngperm(FILE * fp)
{
    char buf[MAXBUFLEN];
    register char *cp;
    static char **ngarray;
    int ngcount = 0;

    if (ngpermcount == 0) {
	return (1);
    }

    while (fgets(buf, sizeof(buf), fp) != NULL) {
	if (buf[0] == '\n')		/* End of header */
	    break;
	if (buf[0] != 'N' && buf[0] != 'n')
	    continue;
	cp = strchr(buf, '\n');
	if (cp)
	    *cp = '\0';
	cp = strchr(buf, ':');
	if (cp == NULL)
	    continue;
	*cp = '\0';
	if (!stricmp(buf, "newsgroups")) {
	    ngcount = get_nglist(&ngarray, cp + 2);
	    break;
	}
    }

#ifndef USG
    rewind(fp);
#else
    rewind(fp);
#endif

    if (ngcount == 0)			/* Either no newgroups or null entry */
	return (1);

    return (ngmatch(s1strneql, 1, ngpermlist, ngpermcount,
		    ngarray, ngcount));
}


/*
 * spew -- spew out the contents of a file to stdout, doing
 * the necessary cr-lf additions at the end.  Finish with
 * a "." on a line by itself, and an fflush(stdout).
 *
 *      Parameters:     "how" tells what part of the file we
 *                      want spewed:
 *                              ARTICLE   The entire thing.
 *                              HEAD      Just the first part.
 *                              BODY      Just the second part.
 *                      "fp" is the open file to spew from.
 *
 *      Returns:        Nothing.
 *
 *      Side effects:   Changes current position in file.
 */

void spew(int s, FILE * fp, int how)
{
    char line[NNTP_STRLEN];
    register char *cp;

    if (how == STAT)
	return;

    while (fgets(line, sizeof(line) - 6, fp) != NULL && *line != '\n') {
	if (how == BODY)		/* We need to skip this anyway */
	    continue;
	cp = strchr(line, '\n');
	if (cp != NULL)
	    *cp = '\0';
	if (*line == '.')
            if (so_puts(s, ".") == -1)
		return;
        if (so_puts(s, line) == -1)
	    return;
        if (so_puts(s, "\r\n") == -1)
	    return;
	if (cp == NULL) {
	    for (;;) {
		if ((fgets(line, sizeof(line) - 6, fp) == NULL)
			|| (strchr(line, '\n') != NULL))
		    break;
	    }
	}
    }

    if (how == HEAD) {
        so_puts(s, ".\r\n");
	return;
    }
    else if (how == ARTICLE) {
        if (so_puts(s, "\r\n") == -1)
	    return;
    }

    while (fgets(line, sizeof(line) - 6, fp) != NULL) {
	cp = strchr(line, '\n');
	if (cp != NULL)
	    *cp = '\0';
	if (*line == '.')
            if (so_puts(s, ".") == -1)
		return;
        if (so_puts(s, line) == -1)
	    return;
        if (so_puts(s, "\r\n") == -1)
	    return;

	if (cp == NULL) {
	    for (;;) {
		if ((fgets(line, sizeof(line) - 6, fp) == NULL)
			|| (strchr(line, '\n') != NULL))
		    break;
	    }
	}
    }
    so_puts(s, ".\r\n");
}


/*
 * get_id -- get the message id of the current article.
 *
 *      Parameters:     "art_fp" is a pointer to the open file.
 *                      "id" is space for the message ID.
 *
 *      Returns:        Nothing.
 *
 *      Side effects:   Seeks and rewinds on "art_fp".
 *                      Changes space pointed to by "id".
 */

void get_id(FILE * art_fp, char *id)
{
    char line[MAXBUFLEN];
    char *cp;

    while (fgets(line, sizeof(line), art_fp) != NULL) {
	if (*line == '\n')
	    break;
	if (*line == 'M' || *line == 'm') {	/* "Message-ID" */
	    if ((cp = strchr(line, ' ')) != NULL) {
		*cp = '\0';
		if (!stricmp(line, "Message-ID:")) {
		    strcpy(id, cp + 1);
		    if ((cp = strchr(id, '\n')) != NULL)
			*cp = '\0';
		    rewind(art_fp);
		    return;
		}
	    }
	}
    }
    rewind(art_fp);
    strcpy(id, "<0>");
}


/*
 * findart -- find an article number in the article array.
 *
 *      Parameters:     "artname" is a string containing
 *                      the name of the article.
 *
 *      Returns:        An index into "pnc->art_array",
 *                      or -1 if "artname" isn't in "art_array".
 *
 *      Side effects:   None.
 *
 *      Improvement:    Replace this linear search with a binary one.
 */

int findart(char *artname, long art_array[], int num_arts)
{
    register int i;
    long artnum;

    artnum = atol(artname);

    for (i = 0; i < num_arts; ++i)
	if (art_array[i] == artnum)
	    return (i);

    return (-1);
}


/*
 * get_distlist -- return a nicely set up array of distribution groups
 * along with a count, when given an NNTP-spec distribution list
 * in the form <dist1,dist2,...,distn>.
 *
 *      Parameters:             "array" is storage for our array,
 *                              set to point at some static data.
 *                              "list" is the NNTP distribution list.
 *
 *      Returns:                Number of distributions found.
 *                              -1 on error.
 *
 *      Side effects:           Changes static data area.
 */

int get_distlist(char ***array, char *list)
{
    char *cp;
    int distcount;
    static char **dist_list = (char **)NULL;

    if (list[0] != '<')
	return (-1);

    cp = strchr(list + 1, '>');
    if (cp != NULL)
	*cp = '\0';
    else
	return (-1);

    for (cp = list + 1; *cp != '\0'; ++cp)
	if (*cp == ',')
	    *cp = ' ';
    distcount = parsit(list + 1, &dist_list);
    *array = dist_list;
    return (distcount);
}


/*
 * lower -- convert a character to lower case, if it's upper case.
 *
 *      Parameters:     "c" is the character to be
 *                      converted.
 *
 *      Returns:        "c" if the character is not
 *                      upper case, otherwise the lower
 *                      case equivalent of "c".
 *
 *      Side effects:   None.
 */

char lower(char c)
{
    if (isascii(c) && isupper(c))
	c = LOUCHAR(c - 'A' + 'a');
    return (c);
}

/*
 * The following is a mish-mosh of code submitted to the net
 * by Stan Barber <sob@bcm.tmc.edu>, Tad Guy <tadguy@cs.odu.edu>,
 * Chris Jepeway <jepeway@utkcs2.cs.utk.edu>, and Tom Lane <tgl@cs.cmu.edu>.
 */

#ifdef OS2PATCH
/*
 * returns 1 if there are lots of free blocks for the nntp server to use;
 * a zero value is the small number of blocks remaining (more or less).
 */
#define DFREE_OK        0
#define DFREE_INODES    1
#define DFREE_BLOCKS    2
#define DFREE_ERR       3

int dfree(char *spool, int free_space);

int space(int min_free)
{
    int result;

    result = dfree(cfg.newsdir, min_free);
    if (result == DFREE_OK)
	return (1);

#ifdef SYSLOG
    switch (result) {
    case DFREE_ERR:
	syslog(LOG_ERR, "dfree failed due to syscall error");
	break;

#ifdef LOG
    case DFREE_INODES:
        syslog(LOG_INFO, "no inodes on %s", cfg.newsdir);
	break;
    case DFREE_BLOCKS:
        syslog(LOG_INFO, "no space on %s", cfg.newsdir);
	break;
#endif
    }
#endif

    return (0);
}


/*
 * Now we define the dfree() routine, which returns the free space
 * on the file system containing the specified directory.
 * Space is measured in kilobytes.
 * A negative value is returned on error.
 */

#ifndef READ_SUPER

#if defined(sun) || defined(hpux) || defined(pyr) || defined(hp300) || defined(NeXT)
#include <sys/vfs.h>
#define statfilesys     statfs		/* routine to call when trying to  */
 /* stat a file system to get the # */
 /* of free blocks available        */
typedef struct statfs statfs_type;	/* the data type into which statfs() */

 /* wants to return useful information */
#define bombed(call)    ((call) == -1)	/* boolean expression returning 1 if */
 /* a call to statfs() fails          */
#define blkfree(fs)     ((fs).f_bfree)	/* given a statfs_type, return total */
 /* # of free blocks                  */
#define blkavail(fs)    ((fs).f_bavail)	/* given a statfs_type called fs,  */
 /* return # of blocks available to */
 /* a non-privileged user           */
#define filfree(fs)     ((fs).f_ffree)	/* given a statfs_type called fs,  */
 /* return number of free inodes    */
#endif

#if defined(apollo)
#include <sys/types.h>
#include <sys/statfs.h>
#define statfilesys(a,b)        statfs(a,b, sizeof(struct statfs), 0)	/* routine to call when
									 * trying to  */
 /* stat a file system to get the # */
 /* of free blocks available        */
typedef struct statfs statfs_type;	/* the data type into which statfs() */

 /* wants to return useful information */
#define bombed(call)    ((call) == -1)	/* boolean expression returning 1 if */
 /* a call to statfs() fails          */
#define blkfree(fs)     ((fs).f_bfree)	/* given a statfs_type, return total */
 /* # of free blocks                  */
#define blkavail(fs)    ((fs).f_bfree)	/* given a statfs_type called fs,  */
 /* return # of blocks available to */
 /* a non-privileged user           */
#define filfree(fs)     ((fs).f_ffree)	/* given a statfs_type called fs,  */
 /* return number of free inodes    */
#endif				/* apollo */

#ifdef ultrix
#include <sys/mount.h>
typedef struct fs_data statfs_type;

#define statfilesys     statfs
#define bombed(call)    ((call) <= 0)
#define blkfree(fs)     ((int)((fs).fd_req.bfree))
#define blkavail(fs)    ((int)((fs).fd_req.bfreen))
#define filfree(fs)     ((int)((fs).fd_req.gfree))
#endif

#if defined(USG) && !defined(hpux)
#include <ustat.h>
typedef struct ustat statfs_type;

/*
 * You've got to make calls to 2 functions to get
 * free blocks on a USG system, so statfilesys can't just be a macro.
 * written by Stan Barber <sob@watson.bcm.tmc.edu>
 */
int statfilesys(dir, fs)
char *dir;
statfs_type *fs;
{
    struct stat file;

    if (stat(dir, &file))
	return (-1);
    if (ustat(file.st_dev, fs))
	return (-2);
    return (0);
}

#define bombed(call)    (call != 0)
#define blkfree(fs)     ((fs).f_tfree)
#define blkavail(fs)    ((fs).f_tfree)
 /* USG doesn't reserve blocks for root */
#define filfree(fs)     ((fs).f_tinode)
#endif				/* USG && !hpux */

#ifdef CMU_MACH
/* This code supplied by Tom Lane <tgl@cs.cmu.edu> */
#include <sys/ioctl.h>
typedef struct fsparam statfs_type;
int statfilesys(dir, fs)
char *dir;
statfs_type *fs;
{
    int fd;

    fd = open(dir, O_RDONLY);
    if (fd < 0)
	return (-1);
    if (ioctl(fd, FIOCFSPARAM, fs) < 0) {
	close(fd);
	return (-2);
    }
    close(fd);
    return (0);
}

#define bombed(call)    ((call) < 0)
#define blkfree(fs)     ((fs).fsp_free-((fs).fsp_size*(fs).fsp_minfree+99)/100)
#define blkavail(fs)    (-1)
#endif				/* CMU_MACH */

int dfree(char *spool, int free_space)
{
    statfs_type fsys;
    int err;

    if (bombed(err = statfilesys(cfg.newsdir, &fsys)))
	return (DFREE_ERR);		/* can't get file system info */

#if defined(filfree) && defined(MINFILES)
    if (filfree(fsys) < MINFILES)
	return (DFREE_INODES);
#endif

    if (blkavail(fsys) < 0L) {
	/* the bavail field doesn't apply to this file system */
	if (blkfree(fsys) < free_space)
	    return (DFREE_BLOCKS);
    }
    else {
	if (blkavail(fsys) < free_space)
	    return (DFREE_BLOCKS);
    }
    return (DFREE_OK);
}

#else				/* READ_SUPER */
/*
 * This code is used if you've got to directly read the superblock
 * to determine how much space you've got left.  It's copied from
 * patches posted by Tad Guy <tadguy@cs.odu.edu>
 */

#include <sys/fs.h>
#include <fstab.h>

/*
 * return the number of free kilobytes remaining on the filesystem where
 * the named file resides.  returns -1 on error.
 */

off_t lseek();

dfree(name, free_space)
char *name;
int free_space;
{
    struct stat namest, fsst;
    struct fstab *fsp;
    char lname[MAXPATHLEN];
    int fd;
    union {
	struct fs u_fs;
	char dummy[SBSIZE];
    } sb;

#define sblock sb.u_fs

    strcpy(lname, name);
    do {
	if (stat(lname, &namest)) {	/* if stat fails, die */

#ifdef SYSLOG
	    syslog(LOG_ERR, "dfree stat(%s) failed: %m", lname);
#endif

	    return DFREE_ERR;
	}
	if ((namest.st_mode & S_IFMT) == S_IFLNK) {	/* if symlink */
	    if ((fd = readlink(lname, lname, sizeof(lname))) < 0) {

#ifdef SYSLOG
		syslog(LOG_ERR, "dfree readlink() failed: %m");
#endif

		return DFREE_ERR;
	    }
	    lname[fd] = '\0';
	}
    } while ((namest.st_mode & S_IFMT) == S_IFLNK);

    setfsent();

    while (fsp = getfsent()){
	if (stat(fsp -> fs_spec, &fsst))
	    continue;
	if (fsst.st_rdev == namest.st_dev)
	    break;
    }

    if (!fsp || (fd = open(fsp -> fs_spec, O_RDONLY)) < 0) {
	endfsent();

#ifdef SYSLOG
	syslog(LOG_ERR, "dfree open(%s,O_RDONLY) failed: %m", fsp -> fs_spec);
#endif

	return DFREE_ERR;
    }
    endfsent();

    lseek(fd, SBLOCK * DEV_BSIZE, L_SET);
    if (read(fd, (char *)&sblock, SBSIZE) != SBSIZE ||
	    (sblock.fs_magic != FS_MAGIC)) {

#ifdef SYSLOG
	syslog(LOG_ERR, "dfree read() failed: %m");
#endif

	return DFREE_ERR;
    }
    close(fd);

#if defined(filfree) && defined(MINFILES)
    if (filfree(fsys) < MINFILES)
	return (DFREE_INODES);
#endif

    if (((((sblock.fs_dsize) * (100 - sblock.fs_minfree) / 100)
	  - ((sblock.fs_dsize)
	     - (sblock.fs_cstotal.cs_nbfree
		* sblock.fs_frag + sblock.fs_cstotal.cs_nffree)))
	 * sblock.fs_fsize / 1024) < free_space)
	return (DFREE_BLOCKS);
    return (DFREE_OK);
}

#endif				/* READ_SUPER */

#ifdef LOAD
/*
**  GETLA -- get the current load average
**
**      This code stolen from la.c. (And subsequently stolen from sendmail,
**              conf.c for nntpd.)
**
**      Parameters:
**              none.
**
**      Returns:
**              The current load average as an integer.
**
**      Side Effects:
**              none.
*/

#if defined(USG) && !defined(WIN_TCP)
int getla()
{
    return (0);
}

#else
#include <nlist.h>
#include <sys/ioctl.h>

struct nlist Nl[]=
{

#ifdef WIN_TCP
    {"avenrun"},
#else
    {"_avenrun"},
#endif

#define X_AVENRUN       0
    {0},
};

#ifdef WIN_TCP
#define KERNEL_FILE "/unix"
#else
#define KERNEL_FILE "/vmunix"
#endif

#ifdef  WIN_TCP
#define FSCALE  64
#define FSHIFT  6
#endif

int getla()
{
    static int kmem = -1;

#ifdef FSCALE
    long avenrun[3];

#else
    double avenrun[3];

#endif
    extern off_t lseek();

    if (kmem < 0) {
	kmem = open("/dev/kmem", 0, 0);
	if (kmem < 0) {

#ifdef SYSLOG
	    syslog(LOG_ERR, "can't open /dev/kmem: %m");
#endif

	    return (-1);
	}

#ifdef WIN_TCP
	if (fcntl(kmem, F_SETFD, 1) < 0) {

#ifdef SYSLOG
	    syslog(LOG_ERR, "fcntl F_SETFD 1 on /dev/kmem failed: %m");
#endif
	}
#else				/* !WIN_TCP */
	if (ioctl(kmem, (int)FIOCLEX, (char *)0) < 0) {

#ifdef SYSLOG
	    syslog(LOG_ERR, "ioctl FIOCLEX of /dev/kmem failed: %m");
#endif
	}
#endif				/* !WIN_TCP */

	if (nlist(KERNEL_FILE, Nl) < 0 || Nl[X_AVENRUN].n_value == 0) {

#ifdef SYSLOG
	    syslog(LOG_ERR, "nlist of %s failed: %m", KERNEL_FILE);
#endif

	    return (-1);
	}
    }
    if (lseek(kmem, (off_t) Nl[X_AVENRUN].n_value, 0) == -1 ||
	    read(kmem, (char *)avenrun, sizeof(avenrun)) < sizeof(avenrun)) {
	/* thank you Ian */

#ifdef SYSLOG
	syslog(LOG_ERR, "lseek or read of /dev/kmem failed: %m");
#endif

	return (-1);
    }

#ifdef FSCALE
    return ((int)(avenrun[0] + FSCALE / 2) >> FSHIFT);
#else
    return ((int)(avenrun[0] + 0.5));
#endif
}

#endif

#endif				/* LOAD */

#else				/* !OS2PATCH */

int space(int min_free)
{
    if (min_free) ;
    return (1);
}

#endif				/* !OS2PATCH */
