/*
 * This program was hacked by Harald Kipp using a source of
 * INEWS written by Kai Uwe Rommel.
 *
 * Some parts are
 *
 *       Copyright (c) 1989-1994 by Kendra Electronic Wonderworks.
 *
 * Bug reports related to THIS modified version should be sent to
 *
 *  harald@os2point.ping.de
 *  harald@haport.sesam.com
 *  Fido: 2:2448/434
 *
 * You may freely copy or redistribute this software. However,
 * this may not apply to any part of it, if otherwise noted.
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <direct.h>
#include <time.h>
#include <ctype.h>

#include <chanlib.h>
#include <lprintf.h>
#include <getopt.h>
#include <header.h>
#include <readcfg.h>

#include "config.h"
#include "inews.h"

static CFGITEM cfgitm[] = {
    { "active"    , 0, 4, NULL               },
    { "batchfile" , 0, 4, NULL               },
    { "dupegroup" , 0, 4, NULL               },
    { "gunzip"    , 0, 4, NULL               },
    { "history"   , 0, 4, NULL               },
    { "inews"     , 0, 4, NULL               },
    { "junkgroup" , 0, 4, NULL               },
    { "mydomain"  , 0, 4, cfg.mydomain       },
    { "mynode"    , 0, 4, cfg.mynode         },
    { "newsdir"   , 0, 4, NULL               },
    { "newsserver", 0, 4, cfg.newsserver     },
    { "rnews"     , 0, 4, cfg.rnewscall      },
    { "spooldir"  , 0, 4, cfg.spooldir       },
    { "uncompress", 0, 4, NULL               }
};

static void usage(void);

/************************************************************************/
/*                                                                      */
/*  INEWS                                                               */
/*                                                                      */
/*  This program posts news articles locally and spools them for        */
/*  remote delivery.                                                    */
/*                                                                      */
/************************************************************************/
int main(int argc, char **argv)
{
    int result = 0;
    FILE *fp;
    int option;
    time_t t_proc = time(NULL);

    init_cfg();
    lopen(cfg.logfile);
    lprintf("Inews (%s)", __TIMESTAMP__);
    initexit();

    /*
     * Process command line options
     */
    while((option = getopt(argc, argv,
           "?hMUvA:c:C:d:D:e:f:F:g:N:o:r:s:S:t:")) != EOF) {
        char *cp;
        switch(option) {
        case 'A':
            strcpy(cfg.approved, optarg);
            break;
        case 'c':
            strcpy(cfg.configfile, optarg);
            break;
        case 'C':
            strcpy(cfg.grouplist, optarg);
            break;
        case 'd':
            cp = optarg;
            while(*cp) {
                switch(*cp) {
                case 'a':
                    cfg.logflg |= LOG_ACTIVE;
                    break;
                case 'd':
                    cfg.logflg |= LOG_DBM;
                    break;
                case 'f':
                    cfg.logflg |= LOG_FLUSH;
                    break;
                case 'r':
                    cfg.logflg |= LOG_RECV;
                    break;
                case 's':
                    cfg.logflg |= LOG_SEND;
                    break;
                default:
                    lprintf("Unknown option -d%c ignored", *cp);
                    break;
                }
                cp++;
            }
            break;
        case 'D':
            strcpy(cfg.distribution, optarg);
            break;
        case 'e':
            strcpy(cfg.expire, optarg);
            break;
        case 'f':
            strcpy(cfg.sender, optarg);
            break;
        case 'F':
            strcpy(cfg.followup, optarg);
            break;
        case 'g':
            cfg.grade = *optarg;
            break;
        case 'h':
            /* ignore for compatibility */
            break;
        case 'M':
            cfg.moderated = 1;
            break;
        case 'N':
            strcpy(cfg.newsgroups, optarg);
            break;
        case 'o':
            strcpy(cfg.organization, optarg);
            break;
        case 'r':
            strcpy(cfg.replyto, optarg);
            break;
        case 's':
            strcpy(cfg.spooldir, optarg);
            break;
        case 'S':
            strcpy(cfg.signfile, optarg);
            break;
        case 't':
            strcpy(cfg.subject, optarg);
            break;
        case 'U':
            cfg.uupcmode = 1;
            break;
        case 'v':
            cfg.printid = 1;
            break;
        case '?':
        default:
            usage();
            return(1);
        }
    }

    /*
     * All command line options are processed, check if one
     * argument is left to give us an article file name.
     */
    argc -= optind;
    argv += optind;
    if(argc > 1) {
        usage();
        return(1);
    }
    else if(argc == 1) {
        if((fp = freopen(argv[0], "rt", stdin)) == NULL) {
            lperror(argv[0]);
            result = 3;
        }
    }
    else
        fp = stdin;

    if(fp) {
        ReadCfg(cfg.configfile, cfgitm, sizeof(cfgitm) / sizeof(CFGITEM));
        if (validate_cfg()) {
            /*
             * Reads header from file fp into buf and returns number
             * of bytes read
             */
            size_t hdrsiz = 16384;
            char *hdrbuf = malloc(hdrsiz);
            int cc = hdrsiz;
            *hdrbuf = '\0';
            read_header(fp, hdrbuf, &cc);

            /*
             * Validate and update the header. New items are
             * appended.
             */
            if(validate_header(hdrbuf + cc, hdrsiz - cc)) {
                FILE *fpt;
                FILE *fpp;
                char *tname = tempnam(NULL, "INEWS");

                if((fpt = fopen(tname, "wb")) != NULL) {
                    write_header(fpt);
                    copy_body(fpt, fp, 0);
                    fclose(fpt);
                    if((fpt = xopen(tname, "rt")) != NULL) {
                        if(spool_news(cfg.newsserver, fpt, "rnews"))
                            result = 4;
                        fclose(fpt);
                    }
                    else {
                        lperror(tname);
                        result = 3;
                    }
                    if((fpt = xopen(tname, "rt")) != NULL) {
                        if((fpp = _popen(cfg.rnewscall, "wb")) != NULL) {
                            while((cc = fread(hdrbuf, 1, hdrsiz, fpt)) != 0)
                                fwrite(hdrbuf, 1, cc, fpp);
                            if((result = _pclose(fpp)) != 0)
                                lprintf("%s returned %d", cfg.rnewscall, result);
                        }
                        else {
                            lprintf("Failed to run %s", cfg.rnewscall);
                            result = 3;
                        }
                        fclose(fpt);
                    }
                    else {
                        lperror(tname);
                        result = 3;
                    }
                    if(unlink(tname))
                        lperror(tname);
                }
                else {
                    lperror(tname);
                    result = 3;
                }
                free(tname);
            }
            else
                result = 3;
            free(hdrbuf);
        }
        else
            result = 3;
        fclose(fp);
    }

    if(result)
        lprintf("Inews returned error %d%c", 3, '\n');
    else {
        t_proc = time(NULL) - t_proc;
        lprintf("Inews processed article in %lu second(s)%c", t_proc, '\n');
    }
    lclose();
    return(result);
}

/************************************************************************/
/*                                                                      */
/************************************************************************/
static void usage(void)
{
    puts("usage: inews [options] [article]\n"
         "options:\n"
         "  -A<approved>     approved by          -h               accept headers\n"
         "  -c<filename>     configuration file   -M               moderated\n"
         "  -C<grouplist>    create newsgroups    -N<grouplist>    to newsgroups\n"
         "  -d<log-flags>    logfile flags        -o<organization> organization\n"
         "  -D<distribution> distribution         -r<address>      reply to\n"
         "  -e<date>         expire date          -s<directory>    spool directory\n"
         "  -f<sender>       sender name (from)   -S<filename>     append signature\n"
         "  -F<followup>     followup             -t<title>        subject\n"
         "  -g<grade>        grade                -v               print returned ID\n\n"
         "logfile flags:\n"
         "  a  log active file processing\n"
         "  d  history file processing\n"
         "  f  flush logfile after each line\n"
         "  r  telegrams received\n"
         "  s  telegrams sent");
    lprintf("Inews usage displayed%c", '\n');
    lclose();
}
