/*
 * The author of this software is William Dorsey.
 * Copyright (c) 1993, 1994, 1995 by William Dorsey.  All rights reserved.
 *
 * THIS SOFTWARE IS BEING PROVIDED "AS IS", WITHOUT ANY EXPRESS OR IMPLIED
 * WARRANTY.  IN PARTICULAR, THE AUTHOR DOES NOT MAKE ANY CLAIM OR
 * WARRANTY OF ANY KIND CONCERNING THE MERCHANTABILITY OF THIS SOFTWARE OR
 * ITS FITNESS FOR ANY PARTICULAR PURPOSE.
 */

/*
 * init.c
 *
 * REVISION HISTORY
 *
 * DATE      RESPONSIBLE PARTY  DESCRIPTION
 * -------------------------------------------------------------------------
 * 93/12/08  B. Dorsey          Module created by breakup of nautilus.c
 * 93/12/31  B. Dorsey          Added config_tbl[] array
 */

#include <stdlib.h>
#include <math.h>
#include "nautilus.h"

#define	PI		3.141592653589
#define TWOPI		(2.0*PI)

/* Global variables */
struct param_t  params;                 /* operating parameters */
struct coder_t  coders[NCODERS];        /* coder table */
struct negotiate_t negotiate;           /* capability negotiation parameters */
struct config_t config_tbl[] = {
    {"SPEED", NUM, &params.port.speed},
    {"PORT", STR, &params.port.name},
    {"MODEM_INIT", STR, &params.modem.init},
    {"SND_IO", U16, &params.msdos.snd_iobase},
    {"SND_IRQ", U16, &params.msdos.snd_irq},
    {"SND_DMA", U16, &params.msdos.snd_dma},
    {"", STR, (char *) 0}
};
UINT8		beep[BEEP_MS*8];        /* assume max sampling rate of 8KHZ */

void
init(void)
{
    int             i = 0;
    int             count, mincount;
    INT16           input[256];         /* assume maximum frame size is 256 */
    UINT8           output[64];         /* assume minimum compression is 4x */

    /*
     * CODER TYPE:          ADM with switched prediction
     * CPU REQUIREMENTS:    386SX-20 or faster
     * PERFORMANCE:         SNR: >7 dB   SNRSEG: ~7 dB
     * BANDWIDTH:           8,533 bits/second
     */
    coders[i].init = apsd_init;
    coders[i].encode = apsd_encode;
    coders[i].decode = apsd_decode;
    coders[i].sample_rate = 8000;
    coders[i].frame_size = 120;
    coders[i].output_size = coders[i].frame_size / 8 + 1;
    coders[i].frames_per_pkt = 8;	/* XXX - may need tweaking */
    coders[i].bandwidth = 9600;		/* 9600 bps or faster */
    coders[i].name = "APSD";
    coders[i++].desc = "Default Nautilus Coder (9600 modem and 386SX)";

    /*
     * CODER TYPE:          ADPCM with switched prediction (2 bits/sample)
     * CPU REQUIREMENTS:    386DX-33 or faster
     * PERFORMANCE:         SNR: ~11 dB   SNRSEG: >10 dB
     * BANDWIDTH:           12,444 bits/second
     */
    coders[i].init = dmsp_init;
    coders[i].encode = dmsp_encode;
    coders[i].decode = dmsp_decode;
#ifdef sun
    coders[i].sample_rate = 8000;
    coders[i].frame_size = 144;
    coders[i].output_size = 3 * coders[i].frame_size / 16 + 1;
#else
    coders[i].sample_rate = 6000;
    coders[i].frame_size = 108;
    coders[i].output_size = coders[i].frame_size / 4 + 1;
#endif
    coders[i].frames_per_pkt = 6;	/* XXX - may need tweaking */
    coders[i].bandwidth = 14400;	/* 14400 bps or faster */
    coders[i].name = "DMSP";
    coders[i++].desc = "Enhanced Nautilus Coder (14.4 modem and 386DX33)";

    /* Setup default params */
    params.mode = -1;
    params.verbose = FALSE;
    params.port.speed = DEFAULT_DTE_SPEED;
    strcpy(params.port.name, DEFAULT_PORT);
    params.modem.speed = -1;
    strcpy(params.modem.init, DEFAULT_MODEM_INIT);
    params.coder.index = DEFAULT_CODER_INDEX;
    params.msdos.snd_iobase = -1;
    params.msdos.snd_irq = -1;
    params.msdos.snd_dma = -1;
    params.msdos.snd_type = -1;
    params.crypto.key.type = BLOWFISH;

    /* Initialize negotiation parameters */
    negotiate.major = VERSION_MAJOR;
    negotiate.minor = VERSION_MINOR;
    negotiate.coder = 255;
    negotiate.encrypt_type = params.crypto.key.type;
    negotiate.modem_speed[0] = 0;
    negotiate.modem_speed[1] = 0;

    /*
     * Determine which coders can be used by current hardware
     */

    /* Generate random input vector */
    for (i=0; i<256; i++)
        input[i] = (INT16) rand();
    
    /* Test each coder */
    for (i=0; i<NCODERS; i++) {
        /* determine minimum encoding rate for this coder */
        mincount = coders[i].sample_rate / coders[i].frame_size;
        mincount = (mincount + 10) / 9;		/* add "overhead" */

        /* measure coder execution speed */
        coders[i].init();			/* initialize coder */
        count = 0;
        Timer(1);				/* start timer */
        while (Timer(0) < 100) {
            coders[i].encode(input, output);	/* encode a block */
            count++;
	}

	/* set or clear capability field for this coder */
	if (count < mincount) {
	    if (i < 8)
	        negotiate.encode[0] &= ~(1<<i);
	    else
	        negotiate.encode[1] &= ~(1<<(i-8));
	}
	else {
	    if (i < 8)
	        negotiate.encode[0] |= (1<<i);
	    else
	        negotiate.encode[1] |= (1<<(i-8));
	}
    }
}

/*
 * Generate turnaround beep
 */
void
InitBeep(int sample_rate)
{
    int             i, samples, theta;
    INT16           beepbuf[BEEP_MS*8];		/* room for max samp rate */

    samples = (sample_rate / 1000) * BEEP_MS;
    theta = (int) ((INT32) 360 * (INT32) BEEP_FREQ / (INT32) sample_rate);
    for (i=0; i<samples; i++) {
        beepbuf[i] = (INT16) ((BEEP_AMPLITUDE>>7) *
        	     isin((int) ((INT32) i * (INT32) theta % (INT32) 360)));
    }
    Std2Raw(beepbuf, beep, samples);
}
