#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <malloc.h>
#include "mloader.h"
#include "munitrk.h"

/**************************************************************************
**************************************************************************/


typedef struct XMHEADER{
	char  id[17];			// ID text: 'Extended module: '
	char  songname[21];		// Module name, padded with zeroes and 0x1a at the end
	char  trackername[20];	// Tracker name
	UWORD version;			// (word) Version number, hi-byte major and low-byte minor
	ULONG headersize;		// Header size
	UWORD songlength;		// (word) Song length (in patten order table)
	UWORD restart;			// (word) Restart position
	UWORD numchn;			// (word) Number of channels (2,4,6,8,10,...,32)
	UWORD numpat;			// (word) Number of patterns (max 256)
	UWORD numins;			// (word) Number of instruments (max 128)
	UWORD flags;			// (word) Flags: bit 0: 0 = Amiga frequency table (see below) 1 = Linear frequency table
	UWORD tempo;			// (word) Default tempo
	UWORD bpm;				// (word) Default BPM
	UBYTE orders[256];		// (byte) Pattern order table
} XMHEADER;


typedef struct XMINSTHEADER{
	ULONG size;				// (dword) Instrument size
	char  name[22];			// (char) Instrument name
	UBYTE type;				// (byte) Instrument type (always 0)
	UWORD numsmp;			// (word) Number of samples in instrument
	ULONG ssize;			//
} XMINSTHEADER;


typedef struct XMPATCHHEADER{
	UBYTE what[96];		// (byte) Sample number for all notes
	UBYTE volenv[48];	// (byte) Points for volume envelope
	UBYTE panenv[48];	// (byte) Points for panning envelope
	UBYTE numvol;		// (byte) Number of volume points
	UBYTE numpan;		// (byte) Number of panning points
	UBYTE volsus;		// (byte) Volume sustain point
	UBYTE vollsp;		// (byte) Volume loop start point
	UBYTE vollep;		// (byte) Volume loop end point
	UBYTE pansus;		// (byte) Panning sustain point
	UBYTE panlsp;		// (byte) Panning loop start point
	UBYTE panlep;		// (byte) Panning loop end point
	UBYTE voltyp;		// (byte) Volume type: bit 0: On; 1: Sustain; 2: Loop
	UBYTE pantyp;		// (byte) Panning type: bit 0: On; 1: Sustain; 2: Loop
	UBYTE vibtyp;		// (byte) Vibrato type
	UBYTE vibsweep;		// (byte) Vibrato sweep
	UBYTE vibdepth;		// (byte) Vibrato depth
	UBYTE vibrate;		// (byte) Vibrato rate
	UWORD volfade;		// (word) Volume fadeout
	UWORD reserved[11];	// (word) Reserved
} XMPATCHHEADER;


typedef struct XMWAVHEADER{
	ULONG length;		// (dword) Sample length
	ULONG loopstart;	// (dword) Sample loop start
	ULONG looplength;	// (dword) Sample loop length
	UBYTE volume;		// (byte) Volume
	UBYTE finetune;		// (byte) Finetune (signed byte -16..+15)
	UBYTE type;			// (byte) Type: Bit 0-1: 0 = No loop, 1 = Forward loop,
//                                        2 = Ping-pong loop;
//                                        4: 16-bit sampledata
	UBYTE panning;		// (byte) Panning (0-255)
	BYTE  relnote;		// (byte) Relative note number (signed byte)
	UBYTE reserved;		// (byte) Reserved
	char  samplename[22];	// (char) Sample name
} XMWAVHEADER;


typedef struct XMPATHEADER{
	ULONG size;				// (dword) Pattern header length
	UBYTE packing;			// (byte) Packing type (always 0)
	UWORD numrows;			// (word) Number of rows in pattern (1..256)
	UWORD packsize;			// (word) Packed patterndata size
} XMPATHEADER;

typedef struct MTMNOTE{
	UBYTE a,b,c;
} MTMNOTE;


typedef struct XMNOTE{
	UBYTE note,ins,vol,eff,dat;
}XMNOTE;

XMNOTE xmpat[32][64];

/**************************************************************************
**************************************************************************/



static XMHEADER *mh;

char XM_Version[]="XM";



BOOL XM_Test(void)
{
	char id[17];
	rewind(modfp);
	if(!fread(id,17,1,modfp)) return 0;
	if(!memcmp(id,"Extended Module: ",17)) return 1;
	return 0;
}


BOOL XM_Init(void)
{
	mh=NULL;
	if(!(mh=MyCalloc(1,sizeof(XMHEADER)))) return 0;
	return 1;
}


void XM_Cleanup(void)
{
	if(mh!=NULL) free(mh);
}


void XM_ReadNote(XMNOTE *n)
{
	UBYTE cmp;
	memset(n,0,sizeof(XMNOTE));

	cmp=fgetc(modfp);

	if(cmp&0x80){
		if(cmp&1) n->note=fgetc(modfp);
		if(cmp&2) n->ins=fgetc(modfp);
		if(cmp&4) n->vol=fgetc(modfp);
		if(cmp&8) n->eff=fgetc(modfp);
		if(cmp&16) n->dat=fgetc(modfp);
	}
	else{
		n->note=cmp;
		n->ins=fgetc(modfp);
		n->vol=fgetc(modfp);
		n->eff=fgetc(modfp);
		n->dat=fgetc(modfp);
	}
}


UBYTE *XM_Convert(XMNOTE *xmtrack)
{
	int t;
	UBYTE note,ins,vol,eff,dat;

	UniReset();

	for(t=0;t<64;t++){

		note=xmtrack->note;
		ins=xmtrack->ins;
		vol=xmtrack->vol;
		eff=xmtrack->eff;
		dat=xmtrack->dat;

		if(ins!=0){
			UniInstrument(ins-1);
		}

		if(note!=0){
			UniNote(note);
		}

//		printf("Vol:%d\n",vol);
		if(vol>=0x10 && vol<=0x50){
			UniPTEffect(0xc,vol-0x10);
		}

		UniPTEffect(eff,dat);
		UniNewline();
		xmtrack++;
	}
	return UniDup();
}


UWORD twelfthroot[12]={
	8192,8679,9195,9742,10321,10935,11585,12274,13004,13777,14596,15464
};


UWORD getc2spd(int relnote)
{
	long f=8363;

	if(relnote>=0){
		f<<=(relnote/12);
		f=(f*twelfthroot[relnote%12])>>13;
	}
	else{
		relnote=-relnote;
		f>>=(relnote/12);
		f=(f<<13)/twelfthroot[relnote%12];
	}
	return f;
}



BOOL XM_Load(void)
{
	SAMPLEINFO *d;
	int t,u,v,numtrk;
	long next;

	if(!XM_Init()) return 0;

	rewind(modfp);

	// try to read module header

	if(!fread(mh,sizeof(XMHEADER),1,modfp)){
		myerr=ERROR_LOADING_HEADER;
		return 0;
	}

	/* set module variables */

	of.initspeed=mh->tempo;
	of.inittempo=mh->bpm;
	of.modtype=DupStr(mh->trackername,20);
	of.numchn=mh->numchn;
	of.numpat=mh->numpat;
	of.numtrk=(UWORD)of.numpat*of.numchn;	// get number of channels
	of.songname=DupStr(mh->songname,20); 	// make a cstr of songname
	of.numpos=mh->songlength;	           	// copy the songlength
	of.numsmp=mh->numins;

	memcpy(of.positions,mh->orders,256);

	printf("Modtype :%s\n",of.modtype);
	printf("Version :%x\n",mh->version);
	printf("Song    :%s\n",of.songname);
	printf("Speed   :%d,%d\n",of.initspeed,of.inittempo);
	printf("Channels:%d\n",of.numchn);
	printf("Numins  :%d\n",mh->numins);

	if(!AllocTracks()) return 0;
	if(!AllocPatterns()) return 0;

	numtrk=0;
	for(t=0;t<of.numpat;t++){
		XMPATHEADER ph;

		printf("Reading pattern %d\n",t);

		fread(&ph,sizeof(XMPATHEADER),1,modfp);

		printf("headln:  %ld\n",ph.size);
		printf("numrows: %d\n",ph.numrows);
		printf("packsize:%d\n",ph.packsize);

		of.pattrows[t]=ph.numrows;

//		fseek(modfp,ph.packsize,SEEK_CUR);

		/*
			Gr8.. when packsize is 0, don't try to load a pattern.. it's empty.
			This bug was discovered thanks to Khyron's module..
		*/

		if(ph.packsize>0){
			for(u=0;u<ph.numrows;u++){
				for(v=0;v<of.numchn;v++){
					XM_ReadNote(&xmpat[v][u]);
				}
			}
		}
		else{
			memset(xmpat,0,32*64*sizeof(XMNOTE));
		}

		for(v=0;v<of.numchn;v++){
			of.tracks[numtrk++]=XM_Convert(xmpat[v]);
		}
	}

	if(!AllocSampleInfo()) return 0;
	d=of.samples;

	for(t=0;t<of.numsmp;t++){
		XMINSTHEADER ih;

		printf("Reading instrument %d\n",t);

		fread(&ih,sizeof(XMINSTHEADER),1,modfp);

		printf("Size: %ld\n",ih.size);
		printf("Name: 	%22.22s\n",ih.name);
		printf("Samples:%d\n",ih.numsmp);
		printf("sampleheadersize:%ld\n",ih.ssize);

		d->samplename=DupStr(ih.name,22);
		d->seekpos=0;
		d->length=0;
		d->loopstart=0;
		d->loopend=0;
		d->flags=0;

		if(ih.numsmp>0){
			XMPATCHHEADER pth;
			XMWAVHEADER wh;

			fread(&pth,sizeof(XMPATCHHEADER),1,modfp);

//			for(u=0;u<256;u++){
//				printf("%2.2x ",fgetc(modfp));
//			}

			next=0;
			for(u=0;u<1;u++){
				ULONG f;

				fread(&wh,sizeof(XMWAVHEADER),1,modfp);
				printf("wav %d:%22.22s\n",u,wh.samplename);
				next+=wh.length;

				if(wh.type&0x10){
					wh.length>>=1;
					wh.loopstart>>=1;
					wh.looplength>>=1;
				}

				d->length=wh.length;
				d->loopstart=wh.loopstart;
				d->loopend=wh.loopstart+wh.looplength;
				d->volume=wh.volume;
				d->c2spd=8192;
				d->transpose=wh.relnote;

				printf("Type %u\n",wh.type);
				printf("Trans %d\n",wh.relnote);

				if(wh.type&0x3) d->flags|=SF_LOOP;
				if(wh.type&0x10) d->flags|=SF_16BITS;
				d->flags|=SF_DELTA;
				d->flags|=SF_SIGNED;
			}
			for(u=1;u<ih.numsmp;u++){
				fread(&wh,sizeof(XMWAVHEADER),1,modfp);
				printf("wav %d:%22.22s\n",u,wh.samplename);
				next+=wh.length;
			}

			d->seekpos=ftell(modfp);
			fseek(modfp,next,SEEK_CUR);
		}

		d++;
	}


	return 1;
}



LOADER xmload={
	NULL,
	"XM",
	"XM loader v0.1",
	XM_Test,
	XM_Load,
	XM_Cleanup
};

