/*
** Copyright (C) 1995  Jonathan Paul Griffiths.  All rights reserved.
**
** You may do anything with this code you wish EXCEPT sell it. You may sell
** any software you create using this code,  but you MUST NOT charge for
** the code itself.  Charging a distribution fee for this code is also
** FORBIDDEN.
*/

#include <malloc.h>  /* malloc */
#include <stdio.h>
#include <string.h>  /* memset */
#include "..\jlib.h"

#ifndef NULL
 #define NULL 0L;
#endif

/*+----------------------------------------------------------------------+*/
/*|sprite_init - intitialises the sprite system.                         |*/
/*+----------------------------------------------------------------------+*/

#ifdef __DEBUG_sprite_init
 #include <stdio.h>
 #define LIBDEBUG(x) x;fflush(stdout)
#else
 #define LIBDEBUG(x)
#endif

sprite_system *sprite_init(USHORT max_sprites,USHORT max_frames)
{
 sprite_system *tssp;
 int counter;

 LIBDEBUG(printf("\nEntering sprite_init()\n\n"));


 /* create space for sprite system record */
 if((tssp=(sprite_system *)malloc(sizeof(sprite_system)))==NULL){
    LIBDEBUG(printf("malloc failed at init_sprite_system line %d\n",__LINE__));
    return (sprite_system *)NULL;
 }
 LIBDEBUG(printf("malloc successful at init_sprite_system line %d\n",__LINE__));

 /* fill in max sprite info */
 #ifdef FIXED_SPRITE_NUMBERS
  tssp->no_sprites=SPR_HARD_MAX_SPRITES;
 #else
  tssp->no_sprites=max_sprites;
 #endif

 LIBDEBUG(printf("Number of sprites supported is %d\n",tssp->no_sprites));


 /* create 'is active' array */
 if((tssp->active_sprites=
 (UBYTE *)malloc(SPR_MAX_SPRITES(tssp)*sizeof(UBYTE)))==NULL){
    LIBDEBUG(printf("malloc failed at init_sprite_system line %d\n",__LINE__));
    return (sprite_system *)NULL;
 }
 LIBDEBUG(printf("malloc successful at init_sprite_system line %d\n",__LINE__));

 /* turn 'em all off */
 memset(tssp->active_sprites,(int)SPR_MAX_SPRITES(tssp),0);


 /* create 'sprite' array */
 if((tssp->sprites=
 (sprite_record **)malloc(SPR_MAX_SPRITES(tssp)*sizeof(sprite_record *)))==NULL){
    LIBDEBUG(printf("malloc failed at init_sprite_system line %d\n",__LINE__));
    return (sprite_system *)NULL;
 }
 LIBDEBUG(printf("malloc successful at init_sprite_system line %d\n",__LINE__));

 #ifdef FIXED_SPRITE_FRAMES
  tssp->no_frames=SPR_HARD_MAX_FRAMES;
 #else
  tssp->no_frames=max_frames;
 #endif

 LIBDEBUG(printf("Number of frames supported is %d\n",tssp->no_frames));


 /* create 'sprite data' array */
 if((tssp->sprite_data=
 (sprite_data_rec **)malloc(SPR_MAX_FRAMES(tssp)*sizeof(sprite_data_rec *)))==NULL){
    LIBDEBUG(printf("malloc failed at init_sprite_system line %d\n",__LINE__));
    return (sprite_system *)NULL;
 }
 LIBDEBUG(printf("malloc successful at init_sprite_system line %d\n",__LINE__));

 /* NULL them all */
 for(counter=0;counter<SPR_MAX_FRAMES(tssp);counter++){
     tssp->sprite_data[counter]=NULL;
 }

 /* initialise the sprites */
 for(counter=0;counter<SPR_MAX_SPRITES(tssp);counter++){

     if((tssp->sprites[counter]=
       (sprite_record *)malloc(sizeof(sprite_record)))==NULL){
       LIBDEBUG(printf("malloc failed at init_sprite_system line %d\n",__LINE__));
       return (sprite_system *)NULL;
     }
     LIBDEBUG(printf("malloc successful at init_sprite_system line %d\n",__LINE__));

     tssp->sprites[counter]->x=0;
     tssp->sprites[counter]->y=0;

     tssp->sprites[counter]->speed=0;
     tssp->sprites[counter]->speed_counter=0;
     tssp->sprites[counter]->xinc=0;
     tssp->sprites[counter]->yinc=0;

     tssp->sprites[counter]->animspeed=0;
     tssp->sprites[counter]->animspeed_counter=0;
     tssp->sprites[counter]->noframes=0;
     tssp->sprites[counter]->frame=0;

     tssp->sprites[counter]->animframes=NULL;

     if((tssp->sprites[counter]->buffer=
       (UBYTE *)malloc(SPR_MAX_X*SPR_MAX_Y*sizeof(UBYTE)))==NULL){
       LIBDEBUG(printf("malloc failed at init_sprite_system line %d\n",__LINE__));
       return (sprite_system *)NULL;
     }
     LIBDEBUG(printf("malloc successful at init_sprite_system line %d\n",__LINE__));

     LIBDEBUG(printf("successfully initialised sprite %d\n",counter));
 }

 /* give back the handle */
 LIBDEBUG(printf("\nLeaving sprite_init()\n\n"));
 return tssp;

}  /* init_sprite_system */


/*+----------------------------------------------------------------------+*/
/*|load_sprites - load a sprite file into a sprite system.               |*/
/*+----------------------------------------------------------------------+*/

#ifdef __DEBUG_load_sprites
 #include <stdio.h>
 #define LIBDEBUG(x) x;fflush(stdout)
#else
 #define LIBDEBUG(x)
#endif

#define _SHEIGHT  (ssys->sprite_data[count1]->height)
#define _SWIDTH   (ssys->sprite_data[count1]->width)
#define _SSIZE    (_SHEIGHT*_SWIDTH)

void generate_RLE(sprite_data_rec *srec);

UBYTE sprite_load(char *filename,sprite_system *ssys)
{
 FILE *spritefile;
 USHORT num_in_file;
 int count1,i,j;

 LIBDEBUG(printf("\nEntering load_sprites()\n"));

 if((spritefile=fopen(filename,"rb"))==NULL){
    return COULDNT_OPEN;
 }
 else{
      fread(&num_in_file,sizeof(USHORT),1,spritefile);

      LIBDEBUG(printf("%d sprites in file %s\n",num_in_file,filename));

      if(num_in_file>SPR_MAX_FRAMES(ssys)){
	 LIBDEBUG(printf("Not enough frames (%d) for sprites\n",MAX_FRAMES(ssys)));
	 return TOO_MANY_IN_FILE;
      }
      LIBDEBUG(printf("%d frames available for sprites\n",MAX_FRAMES(ssys)));


      for(count1=0;count1<num_in_file;count1++){

	  if((ssys->sprite_data[count1]=
	    (sprite_data_rec *)malloc(sizeof(sprite_data_rec)))==NULL){
	    LIBDEBUG(printf("malloc failed at load_sprites line %d\n",__LINE__));
	    return OUT_OF_MEMORY;
	  }
	  LIBDEBUG(printf("malloc successful at load_sprites line %d\n",__LINE__));

	  ssys->sprite_data[count1]->width=(UBYTE)fgetc(spritefile);
	  ssys->sprite_data[count1]->height=(UBYTE)fgetc(spritefile);

	  if((ssys->sprite_data[count1]->data=
	    (UBYTE *)malloc((_SHEIGHT*_SWIDTH)*sizeof(UBYTE)))==NULL){
	     LIBDEBUG(printf("malloc failed at load_sprites line %d\n",__LINE__));
	     return OUT_OF_MEMORY;
	  }
	  LIBDEBUG(printf("malloc succesful at load_sprites line %d\n",__LINE__));

	  fread(ssys->sprite_data[count1]->data,_SSIZE,1,spritefile);

	  ssys->sprite_data[count1]->no_rects=(UBYTE)fgetc(spritefile);

	  if((ssys->sprite_data[count1]->rect_coords=
	    (UBYTE *)malloc((ssys->sprite_data[count1]->no_rects)*sizeof(UBYTE)))==NULL){
	     LIBDEBUG(printf("malloc failed at load_sprites line %d\n",__LINE__));
	     return OUT_OF_MEMORY;
	  }
	  LIBDEBUG(printf("malloc succesful at load_sprites line %d\n",__LINE__));

	  j=0;
	  for(i=0;i<ssys->sprite_data[count1]->no_rects;i++){
	     /* read in each bounding rectangles coords */
	     ssys->sprite_data[count1]->rect_coords[j]=(UBYTE)fgetc(spritefile);
	     j++;
	     ssys->sprite_data[count1]->rect_coords[j]=(UBYTE)fgetc(spritefile);
	     j++;
	     ssys->sprite_data[count1]->rect_coords[j]=(UBYTE)fgetc(spritefile);
	     j++;
	     ssys->sprite_data[count1]->rect_coords[j]=(UBYTE)fgetc(spritefile);
	     j++;
	  }

	  generate_RLE(ssys->sprite_data[count1]);
      }

      fclose(spritefile);
 }

 LIBDEBUG(printf("\nLeaving sprites_load()\n\n"));

 return SUCCESS;
}

/* This is a private routine.  Just forget you ever saw it! */
void generate_RLE(sprite_data_rec *srec)
{
 UBYTE wcount, hcount, runlength, runtype, datum;
 UBYTE linereps, *linereparr;
 USHORT rlecount;


 LIBDEBUG(printf("\n(private) entering generate_RLE()\n"));

 /* create an array to hold the line reps */
 if((linereparr=(UBYTE *)malloc(srec->height*sizeof(UBYTE)))==NULL){
    LIBDEBUG(printf("(private) malloc failed at generate_RLE line %d\n",__LINE__));
    exit(-1);
 }
 LIBDEBUG(printf("(private) malloc successful at generate_RLE line %d\n",__LINE__));

 LIBDEBUG(printf("(private) height of sprite is %d\n",srec->height));
 LIBDEBUG(printf("(private) width of sprite is %d\n",srec->width));


 LIBDEBUG(printf("(private) RLE dump of sprite:\n"));

 rlecount = 0;

 /* this part is just to work out the size to malloc space for */
 for(hcount=0;hcount<srec->height;hcount++){
     linereps =1;

     datum=srec->data[(hcount*srec->width)];

     if(datum>0){        /* what is the ponsy bit-twiddle for this? */
	datum = 1;
     }

     LIBDEBUG(printf("(%d,",datum));

     runtype = datum; /* this is important! */
     runlength=1;

     for(wcount=1;wcount<srec->width;wcount++){
	 datum=srec->data[(hcount*srec->width)+wcount];

	 if(datum>0){
	    datum = 1;
	 }

	 if(datum == runtype){
	    ++runlength;
	 }
	 else{
	     runtype=datum;

	     LIBDEBUG(printf("%d)",runlength));
	     LIBDEBUG(printf("(%d,",runtype));

	     ++linereps;
	     runlength=1;
	 }
     }
     LIBDEBUG(printf("%d) linereps = %d \n",runlength,linereps));

     linereparr[hcount]=linereps;
     rlecount+=(linereps*2)+1;

 }

 LIBDEBUG(printf("rlecount of sprite is %d \n",rlecount));


 if((srec->pattern=(UBYTE *)malloc(rlecount*sizeof(UBYTE)))==NULL){
    LIBDEBUG(printf("(private) malloc failed at generate_RLE line %d\n",__LINE__));
    exit(-1);
 }
 LIBDEBUG(printf("(private) malloc successful at generate_RLE line %d\n",__LINE__));


 /* now we actually encode the data */
 rlecount = 0;

 for(hcount=0;hcount<srec->height;hcount++){

     srec->pattern[rlecount]= linereparr[hcount];

     ++rlecount;
     datum=srec->data[(hcount*srec->width)];

     if(datum>0){
	datum = 1;
     }

     srec->pattern[rlecount]=datum;
     ++rlecount;

     runtype = datum; /* this is important! */
     runlength=1;

     for(wcount=1;wcount<srec->width;wcount++){
	 datum=srec->data[(hcount*srec->width)+wcount];

	 if(datum>0){
	    datum = 1;
	 }

	 if(datum == runtype){
	    ++runlength;
	 }
	 else{
	     runtype=datum;

	     srec->pattern[rlecount]=runlength;
	     ++rlecount;
	     srec->pattern[rlecount]=runtype;
	     ++rlecount;

	     runlength=1;
	 }
     }
     srec->pattern[rlecount]=runlength;
     ++rlecount;
 }

 free(linereparr);

 LIBDEBUG(printf("\n(private) leaving generate_RLE()\n\n"));
}

/*+----------------------------------------------------------------------+*/
/*|sprite_turn_on - turn on a sprite                                     |*/
/*+----------------------------------------------------------------------+*/

#ifdef __DEBUG_sprite_turn_on
 #include <stdio.h>
 #define LIBDEBUG(x) x;fflush(stdout)
#else
 #define LIBDEBUG(x)
#endif

void sprite_turn_on(sprite_system *spr_sys,USHORT snum)
{
 LIBDEBUG(printf("\nEntering sprite_turn_on()\n\n"));

 LIBDEBUG(printf("Turning on sprite number %d\n",snum));

 spr_sys->active_sprites[snum]=1;

 LIBDEBUG(printf("\nLeaving sprite_turn_on()\n\n"));
}

/*+----------------------------------------------------------------------+*/
/*|sprite_turn_off - turn off a sprite                                   |*/
/*+----------------------------------------------------------------------+*/

#ifdef __DEBUG_sprite_turn_off
 #include <stdio.h>
 #define LIBDEBUG(x) x;fflush(stdout)
#else
 #define LIBDEBUG(x)
#endif

void sprite_turn_off(sprite_system *spr_sys,USHORT snum)
{
 LIBDEBUG(printf("\nEntering sprite_turn_off()\n\n"));

 LIBDEBUG(printf("Turning off sprite number %d\n",snum));

 spr_sys->active_sprites[snum]=0;

 LIBDEBUG(printf("\nLeaving sprite_turn_off()\n\n"));
}


/*+----------------------------------------------------------------------+*/
/*|sprite_set_an_frame - set a sprites animation frame                   |*/
/*+----------------------------------------------------------------------+*/

#ifdef __DEBUG_sprite_set_an_frame
 #include <stdio.h>
 #define LIBDEBUG(x) x;fflush(stdout)
#else
 #define LIBDEBUG(x)
#endif

void sprite_set_an_frame(sprite_system *ssys,USHORT snum,USHORT frame)
{
 LIBDEBUG(printf("\nEntering sprite_set_an_frame()\n\n"));

 LIBDEBUG(printf("\nSetting sprite %d's frame to %d\n",snum,frame));
 ssys->sprites[snum]->frame=frame;

 LIBDEBUG(printf("\nLeaving sprite_set_an_frame()\n\n"));

}


/*+----------------------------------------------------------------------+*/
/*|sprite_set_xy - change a sprites position                             |*/
/*+----------------------------------------------------------------------+*/

#ifdef __DEBUG_sprite_set_xy
 #include <stdio.h>
 #define LIBDEBUG(x) x;fflush(stdout)
#else
 #define LIBDEBUG(x)
#endif

void sprite_set_xy(sprite_system *spr_sys,USHORT snum,USHORT newx,USHORT newy)
{
  LIBDEBUG(printf("\nEntering sprite_set_xy()\n\n"));

  LIBDEBUG(printf("setting sprite %d's x and y to %d and %d\n",snum,newx,newy));

  spr_sys->sprites[snum]->y=newy;
  spr_sys->sprites[snum]->x=newx;

  LIBDEBUG(printf("\nLeaving sprite_set_xy()\n\n"));

}


/*+----------------------------------------------------------------------+*/
/*|find_first_free - find the number of the first free sprite.           |*/
/*+----------------------------------------------------------------------+*/

#ifdef __DEBUG_sprite_find_first_free
 #include <stdio.h>
 #define LIBDEBUG(x) x;fflush(stdout)
#else
 #define LIBDEBUG(x)
#endif

USHORT sprite_find_first_free(sprite_system *spr_sys)
{
  int count;

  LIBDEBUG(printf("\nEntering sprite_find_first_free()\n\n"));

  /* NOTE : this routine will DIE if there are no free sprites - always keep
  **        at least one free in the system.
  */
  for(count=0;spr_sys->active_sprites[count] != 0;count++);

  return count;

  LIBDEBUG(printf("\nLeaving sprite_find_first_free()\n\n"));
}

/*+----------------------------------------------------------------------+*/
/*|set_move_info - Set up the movement info for a sprite.                |*/
/*+----------------------------------------------------------------------+*/

#ifdef __DEBUG_sprite_set_move_info
 #include <stdio.h>
 #define LIBDEBUG(x) x;fflush(stdout)
#else
 #define LIBDEBUG(x)
#endif

void sprite_set_move_info(sprite_system *spr_sys,USHORT snum,
			  UBYTE speed,BYTE xinc,BYTE yinc)
{

  LIBDEBUG(printf("\nEntering sprite_set_move_info()\n\n"));


  spr_sys->sprites[snum]->speed = speed;
  spr_sys->sprites[snum]->speed_counter = speed;
  spr_sys->sprites[snum]->xinc = xinc;
  spr_sys->sprites[snum]->yinc = yinc;


  LIBDEBUG(printf("\nLeaving sprite_set_move_info()\n\n"));
}

/*+----------------------------------------------------------------------+*/
/*|set_anim_info - Set up the animation info for a sprite.               |*/
/*+----------------------------------------------------------------------+*/

#ifdef __DEBUG_sprite_set_anim_info
 #include <stdio.h>
 #define LIBDEBUG(x) x;fflush(stdout)
#else
 #define LIBDEBUG(x)
#endif

void sprite_set_anim_info(sprite_system *spr_sys,USHORT snum,
			  UBYTE anspeed,UBYTE noframes,USHORT *animpat)
{

  LIBDEBUG(printf("\nEntering sprite_set_anim_info()\n\n"));


  spr_sys->sprites[snum]->animspeed = anspeed;
  spr_sys->sprites[snum]->animspeed_counter = anspeed;
  spr_sys->sprites[snum]->noframes = noframes;
  spr_sys->sprites[snum]->frame = 0;
  spr_sys->sprites[snum]->frame_count = 0;
  spr_sys->sprites[snum]->animframes = animpat;


  LIBDEBUG(printf("\nLeaving sprite_set_anim_info()\n\n"));
}

/*+---------------------------------------------------------------------+*/
/*|update_anim_and_move - update a sprites posistion and animframe.     |*/
/*+---------------------------------------------------------------------+*/

#ifdef __DEBUG_sprite_update_anim_and_move
 #include <stdio.h>
 #define LIBDEBUG(x) x;fflush(stdout)
#else
 #define LIBDEBUG(x)
#endif

void sprite_update_anim_and_move(sprite_system *spr_sys,USHORT snum)
{
 sprite_record *current;

 current = spr_sys->sprites[snum];

 LIBDEBUG(printf("\nEntering sprite_update_anim_and_move()\n\n"));

 if(SPR_IS_ON(spr_sys,snum)){
    /* check movement */
    if(current->speed){
       if((--current->speed_counter)==0){
	  current->speed_counter=current->speed;
	  current->x+=current->xinc;
	  current->y+=current->yinc;
       }
    }

    /* check animation */
    if(current->animspeed){
       if((--current->animspeed_counter)==0){
	   current->animspeed_counter=current->animspeed;

	   if((++current->frame_count)==current->noframes){
	      current->frame_count=0;
	   }
	   current->frame = current->animframes[current->frame_count];
       }
   }
 }

 LIBDEBUG(printf("\nLeaving sprite_update_anim_and_move()\n\n"));
}


/*+----------------------------------------------------------------------+*/
/*|sprite_update_all_anim_and_move                                       |*/
/*+----------------------------------------------------------------------+*/
/*|update every active sprite.                                           |*/
/*+----------------------------------------------------------------------+*/

#ifdef __DEBUG_sprite_update_all_anim_and_move
 #include <stdio.h>
 #define LIBDEBUG(x) x;fflush(stdout)
#else
 #define LIBDEBUG(x)
#endif

void sprite_update_all_anim_and_move(sprite_system *spr_sys)
{
 int count;

 LIBDEBUG(printf("\nEntering sprite_update_all_anim_and_move()\n\n"));


 for(count=0;count<spr_sys->no_sprites;count++){
     if(SPR_IS_ON(spr_sys,count)){
	sprite_update_anim_and_move(spr_sys,count);
     }
 }

 LIBDEBUG(printf("\nLeaving sprite_update_all_anim_and_move()\n\n"));
}


