/* uclock.c - microsecond timing routines */
#include "uclock.h"

/* Constants */
#define TWO_BYTE_MODE_2  0x34 /* timer 0,2-byte read/write,mode 2, binary. */
#define TIMER_0_PORT  0x40    /* Timer 0 data port address.                */
#define TIMER_MODE    0x43    /* Timer mode port address.                  */
#define BIOS_TICK_SEGMENT   0x40    /* BIOS data segment.                  */
#define BIOS_TICK_OFFSET    0x6c    /* Address of BIOS (18.2/s) tick count */
#define SCALE    10000              /* Scale factor for timer ticks.       */

/* The following assumes 18.2 BIOS ticks per second resulting from
   the 8253 being clocked at 1.19 MHz. */
#define us_BTIK  54925    /* Micro sec per BIOS clock tick.             */
#define f_BTIK    4595    /* Fractional part of usec per BIOS tick.     */
#define us_TTIK   8381    /* Usec per timer tick * SCALE. (4/4.77 MHz)  */


void uclock_init(void)
{
 asm volatile("cli");

 outportb(TIMER_MODE, TWO_BYTE_MODE_2);
 outportb(TIMER_0_PORT, 0);        /* Initial count = 65636  */
 outportb(TIMER_0_PORT, 0);

 asm volatile("sti");
}


unsigned int uclock_read(void)
{
 unsigned char msb, lsb;
 unsigned int tim_ticks;
 unsigned int count;

 asm volatile("cli");

 outportb(TIMER_MODE, 0);                          /* Latch count.  */
 lsb = (unsigned char)inportb(TIMER_0_PORT);       /* Read count.   */
 msb = (unsigned char)inportb(TIMER_0_PORT);

 /* Get BIOS tick count */
 dosmemget(BIOS_TICK_SEGMENT*16+BIOS_TICK_OFFSET,sizeof(unsigned int),&count);

 asm volatile("sti");

 /* Merge PIT channel 0 count with BIOS tick count */
 tim_ticks = (unsigned)(-1) - ((msb << 8) | lsb);

 return((count * us_BTIK)+((int)tim_ticks*us_TTIK+(count * us_BTIK)%SCALE)/SCALE);
}

