/* fastll.c - speed oriented doubly linked list
**
** Copyright (C) 1995  Jonathan Paul Griffiths.  All rights reserved.
**
** You may do anything with this code you wish EXCEPT sell it. You may sell
** any software you create using this code,  but you MUST NOT charge for
** the code itself.  Charging a distribution fee for this code is also
** FORBIDDEN.
*/
#include "fast_ll.h"

/***************************************************************************/
/* these routines are private to the implementation:                       */
/*                                                                         */
fast_ll_node *FLLinit_node(fast_mem *fmem);
void FLLdestroy_node(fast_mem *fmem,fast_ll_node *node);
/***************************************************************************/

/* macros for ensuring the list doesn't over/under flow */
#define CHECK_ADD(list) if(list->num_items == list->max_items) exit(0)
#define CHECK_SUB(list) if(list->num_items == 0) exit(0)


/***************************************************************************/
/* fast_ll *fastll_init(int max_items)                                     */
/***************************************************************************/
fast_ll *fastll_init(int max_items)
{
 fast_ll *new;

 new = (fast_ll *)malloc(sizeof(fast_ll));

 new->list_storage = fastmem_init(max_items,sizeof(fast_ll_node));
 new->num_items = 0;
 new->max_items = max_items;
 new->head = (fast_ll_node *)NULL;
 new->tail = (fast_ll_node *)NULL;
 new->current = (fast_ll_node *)NULL;

 return(new);
}


/***************************************************************************/
/* void fastll_clear(fast_ll *list)                                        */
/***************************************************************************/
void fastll_clear(fast_ll *list)
{
 /* clear memory area */
 fastmem_clear(list->list_storage);

 /* reset head etc */
 list->num_items = list->max_items;
 list->head = (fast_ll_node *)NULL;
 list->tail = (fast_ll_node *)NULL;
 list->current = (fast_ll_node *)NULL;
}


/***************************************************************************/
/* fast_ll *fastll_destroy(fast_ll *list)                                  */
/***************************************************************************/
fast_ll *fastll_destroy(fast_ll *list)
{
 /* free memory area */
 list->list_storage = fastmem_destroy(list->list_storage);

 /* free the header record */
 free(list);

 return (fast_ll *)NULL;
}


/***************************************************************************/
/* void fastll_add_to_start(fast_ll *list,void *item)                      */
/***************************************************************************/
void fastll_add_to_start(fast_ll *list,void *item)
{
 fast_ll_node *node;

 node = FLLinit_node(list->list_storage);

 node->data = item;
 node->prev = NULL;
 list->current = node;
 CHECK_ADD(list);

 ++list->num_items;

 /* If Head == NULL, this is the first node */
 if(list->head == NULL) {
    list->head = node;
    node->next = NULL;
    list->tail = node;
    list->current = node;
  }
  else{
       list->head->prev = node;
       node->next = list->head;
       list->head = node;
 }
}


/***************************************************************************/
/* void fastll_add_to_end(fast_ll *list,void *item)                        */
/***************************************************************************/
void fastll_add_to_end(fast_ll *list,void *item)
{
 fast_ll_node *node;

 node = FLLinit_node(list->list_storage);
 node->data = item;
 node->next = NULL;
 list->current = node;
 CHECK_ADD(list);
 ++list->num_items;

 /* If Tail == NULL, this is the first node */
 if(list->tail == NULL) {
    list->tail = node;
    list->head = node;
    node->prev = NULL;
 }
 else{
    list->tail->next = node;
    node->prev = list->tail;
    list->tail = node;
 }
}


/***************************************************************************/
/* void fastll_add_after_current(fast_ll *list,void *item)                 */
/***************************************************************************/
void fastll_add_after_current(fast_ll *list,void *item)
{
  fast_ll_node *node;

  node = FLLinit_node(list->list_storage);
  node->data = item;
  CHECK_ADD(list);

  ++list->num_items;

  /* If current == NULL, this is the first node */
  if(list->current == NULL) {
     list->tail = node;
     list->head = node;
     node->prev = NULL;
     node->next = NULL;
     list->current = node;
  }
  else{
      /* If current->next == NULL, current is the last node */
      if(list->current->next == NULL) {
	 list->current->next = node;
	 node->next = NULL;
	 node->prev = list->current;
	 list->tail = node;
	 list->current = node;
      }
      else{
	  /* current is in the middle of the list somewhere */
	  node->prev = list->current;
	  node->next = list->current->next;
	  node->next->prev = node;
	  list->current->next = node;
	  list->current = node;
      }
  }
}


/***************************************************************************/
/* void fastll_add_before_current(fast_ll *list,void *item)                */
/***************************************************************************/
void fastll_add_before_current(fast_ll *list,void *item)
{
  fast_ll_node *node;

  node = FLLinit_node(list->list_storage);
  node->data = item;
  CHECK_ADD(list);

  ++list->num_items;

  /* If current == NULL, this is the first node */
  if(list->current == NULL) {
     list->tail = node;
     list->head = node;
     node->prev = NULL;
     node->next = NULL;
     list->current = node;
  }
  else{
      /* If current->prev == NULL, current is the first node */
      if(list->current->prev == NULL) {
	 list->current->prev = node;
	 node->prev = NULL;
	 node->next = list->current;
	 list->head = node;
	 list->current = node;
      }
      else{
	  /* current is in the middle of the list somewhere */
	  node->next = list->current;
	  node->prev = list->current->prev;
	  node->prev->next = node;
	  list->current->prev = node;
	  list->current = node;
      }
  }
}


/***************************************************************************/
/* void fastll_remove_current(fast_ll *list)                               */
/***************************************************************************/
void fastll_remove_current(fast_ll *list)
{
  fast_ll_node *temp;

  CHECK_SUB(list);

  if(list->current == list->head){
     fastll_remove_head(list);
  }
  else{
     if(list->current == list->tail){
	fastll_remove_tail(list);
     }
     else{
	 /* if current != head or tail then it has a node either side of it */
	 temp = list->current;
	 list->current->prev->next = list->current->next;
	 list->current->next->prev = list->current->prev;
	 list->current = temp->next;
	 FLLdestroy_node(list->list_storage,temp);
	 --list->num_items;
     }
  }
}


/***************************************************************************/
/* void fastll_remove_head(fast_ll *list)                                  */
/***************************************************************************/
void fastll_remove_head(fast_ll *list)
{
  fast_ll_node *temp;

  CHECK_SUB(list);

  --list->num_items;

  /* if head == tail,  there is only one item in the list */
  if(list->head == list->tail){
     temp = list->head;
     list->head = NULL;
     list->tail = NULL;
     list->current = NULL;
     FLLdestroy_node(list->list_storage,temp);
  }
  else{
     /* there is more than one item in the list */
     temp = list->head;
     list->head = list->head->next;
     list->head->prev = NULL;
     list->current = list->head;
     FLLdestroy_node(list->list_storage,temp);
  }
}


/***************************************************************************/
/* void fastll_remove_tail(fast_ll *list)                                  */
/***************************************************************************/
void fastll_remove_tail(fast_ll *list)
{
  fast_ll_node *temp;

  CHECK_SUB(list);

  --list->num_items;

  /* if tail == head ,  there is only one item in the list */
  if(list->tail == list->head){
     temp = list->tail;
     list->head = NULL;
     list->tail = NULL;
     list->current = NULL;
     FLLdestroy_node(list->list_storage,temp);
  }
  else{
     /* there is more than one item in the list */
     temp = list->tail;
     list->tail = list->tail->prev;
     list->tail->next = NULL;
     list->current = list->tail;
     FLLdestroy_node(list->list_storage,temp);
  }
}


/***************************************************************************/
/* these routines are private to the implementation.                       */
/***************************************************************************/
fast_ll_node *FLLinit_node(fast_mem *fmem)
{
 fast_ll_node *new;

 new = (fast_ll_node*)fastmem_allocate(fmem);

 return new;
}

void FLLdestroy_node(fast_mem *fmem,fast_ll_node *node)
{
 node = fastmem_deallocate(fmem,(void *)node);
}
