/* function.c - built-in functions */
/*
	Copyright (c) 1995, by David Michael Betz
	All rights reserved
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "function.h"
#include "execute.h"
#include "objects.h"
#include "linebuf.h"
#include "boberr.h"
#include "bobodb.h"
#include "dfile.h"

/* prototypes */
static ObjectPtr BIF_New(InterpreterContext *c,int n,ObjectPtr *p);
static ObjectPtr BIF_Clone(InterpreterContext *c,int n,ObjectPtr *p);
static ObjectPtr BIF_List(InterpreterContext *c,int n,ObjectPtr *p);
static ObjectPtr BIF_Cons(InterpreterContext *c,int n,ObjectPtr *p);
static ObjectPtr BIF_Car(InterpreterContext *c,int n,ObjectPtr *p);
static ObjectPtr BIF_SetCar(InterpreterContext *c,int n,ObjectPtr *p);
static ObjectPtr BIF_Cdr(InterpreterContext *c,int n,ObjectPtr *p);
static ObjectPtr BIF_SetCdr(InterpreterContext *c,int n,ObjectPtr *p);
static ObjectPtr BIF_Intern(InterpreterContext *c,int n,ObjectPtr *p);
static ObjectPtr BIF_Class(InterpreterContext *c,int n,ObjectPtr *p);
static ObjectPtr BIF_GetLocalProperty(InterpreterContext *c,int n,ObjectPtr *p);
static ObjectPtr BIF_Length(InterpreterContext *c,int n,ObjectPtr *p);
static ObjectPtr BIF_Print(InterpreterContext *c,int n,ObjectPtr *p);
static ObjectPtr BIF_Display(InterpreterContext *c,int n,ObjectPtr *p);
static ObjectPtr BIF_DecodeMethod(InterpreterContext *c,int n,ObjectPtr *p);
static ObjectPtr BIF_GC(InterpreterContext *c,int n,ObjectPtr *p);
static ObjectPtr BIF_FOpen(InterpreterContext *c,int n,ObjectPtr *p);
static ObjectPtr BIF_Close(InterpreterContext *c,int n,ObjectPtr *p);
static ObjectPtr BIF_GetC(InterpreterContext *c,int n,ObjectPtr *p);
static ObjectPtr BIF_PutC(InterpreterContext *c,int n,ObjectPtr *p);
static ObjectPtr BIF_CreateObjectStore(InterpreterContext *c,int n,ObjectPtr *p);
static ObjectPtr BIF_OpenObjectStore(InterpreterContext *c,int n,ObjectPtr *p);
static ObjectPtr BIF_CloseObjectStore(InterpreterContext *c,int n,ObjectPtr *p);
static ObjectPtr BIF_ObjectCount(InterpreterContext *c,int n,ObjectPtr *p);
static ObjectPtr BIF_OpenObject(InterpreterContext *c,int n,ObjectPtr *p);
static ObjectPtr BIF_CreateObject(InterpreterContext *c,int n,ObjectPtr *p);
static ObjectPtr BIF_UpdateObject(InterpreterContext *c,int n,ObjectPtr *p);
static ObjectPtr BIF_DeleteObject(InterpreterContext *c,int n,ObjectPtr *p);
static ObjectPtr BIF_ObjectNumber(InterpreterContext *c,int n,ObjectPtr *p);
static ObjectPtr BIF_ObjectVersion(InterpreterContext *c,int n,ObjectPtr *p);
static ObjectPtr BIF_CreateCursor(InterpreterContext *c,int n,ObjectPtr *p);
static ObjectPtr BIF_CloseCursor(InterpreterContext *c,int n,ObjectPtr *p);
static ObjectPtr BIF_SetCursorPosition(InterpreterContext *c,int n,ObjectPtr *p);
static ObjectPtr BIF_GetNextObject(InterpreterContext *c,int n,ObjectPtr *p);
static ObjectPtr BIF_GetPreviousObject(InterpreterContext *c,int n,ObjectPtr *p);
static ObjectPtr BIF_OpenCurrentObject(InterpreterContext *c,int n,ObjectPtr *p);
static ObjectPtr BIF_Flatten(InterpreterContext *c,int n,ObjectPtr *p);
static ObjectPtr BIF_Unflatten(InterpreterContext *c,int n,ObjectPtr *p);
static ObjectPtr BIF_NewLineBuffer(InterpreterContext *c,int n,ObjectPtr *p);
static ObjectPtr BIF_FillLineBuffer(InterpreterContext *c,int n,ObjectPtr *p);
static ObjectPtr BIF_NextToken(InterpreterContext *c,int n,ObjectPtr *p);
static ObjectPtr BIF_RestOfLine(InterpreterContext *c,int n,ObjectPtr *p);
static ObjectPtr BIF_WholeLine(InterpreterContext *c,int n,ObjectPtr *p);
static ObjectPtr BIF_IsMoreOnLine(InterpreterContext *c,int n,ObjectPtr *p);
static ObjectPtr BIF_Load(InterpreterContext *c,int n,ObjectPtr *p);
static ObjectPtr BIF_Quit(InterpreterContext *c,int n,ObjectPtr *p);
static void EnterBuiltInStream(InterpreterContext *c,char *name,Stream *s);

/* function table */
static FunctionTableEntry functionTable[] = {
{	"New",			BIF_New			},
{	"Clone",		BIF_Clone		},
{	"List",			BIF_List		},
{	"Cons",			BIF_Cons		},
{	"Car",			BIF_Car			},
{	"SetCar",		BIF_SetCar		},
{	"Cdr",			BIF_Cdr			},
{	"SetCdr",		BIF_SetCdr		},
{	"Intern",		BIF_Intern		},
{	"Class",		BIF_Class		},
{	"GetLocalProperty",	BIF_GetLocalProperty	},
{	"Length",		BIF_Length		},
{	"Print",		BIF_Print		},
{	"Display",		BIF_Display		},
{	"DecodeMethod",		BIF_DecodeMethod	},
{	"GC",			BIF_GC			},
{	"FOpen",		BIF_FOpen		},
{	"Close",		BIF_Close		},
{	"GetC",			BIF_GetC		},
{	"PutC",			BIF_PutC		},
{	"CreateObjectStore",	BIF_CreateObjectStore	},
{	"OpenObjectStore",	BIF_OpenObjectStore	},
{	"CloseObjectStore",	BIF_CloseObjectStore	},
{	"ObjectCount",		BIF_ObjectCount		},
{	"OpenObject",		BIF_OpenObject		},
{	"CreateObject",		BIF_CreateObject	},
{	"UpdateObject",		BIF_UpdateObject	},
{	"DeleteObject",		BIF_DeleteObject	},
{	"ObjectNumber",		BIF_ObjectNumber	},
{	"ObjectVersion",	BIF_ObjectVersion	},
{	"CreateCursor",		BIF_CreateCursor	},
{	"CloseCursor",		BIF_CloseCursor		},
{	"SetCursorPosition",	BIF_SetCursorPosition	},
{	"GetNextObject",	BIF_GetNextObject	},
{	"GetPreviousObject",	BIF_GetPreviousObject	},
{	"OpenCurrentObject",	BIF_OpenCurrentObject	},
{	"Flatten",		BIF_Flatten		},
{	"Unflatten",		BIF_Unflatten		},
{	"NewLineBuffer",	BIF_NewLineBuffer	},
{	"FillLineBuffer",	BIF_FillLineBuffer	},
{	"NextToken",		BIF_NextToken		},
{	"RestOfLine",		BIF_RestOfLine		},
{	"WholeLine",		BIF_WholeLine		},
{	"IsMoreOnLine",		BIF_IsMoreOnLine	},
{	"Load",			BIF_Load		},
{	"Quit",			BIF_Quit		},
{0,0}
};

/* EnterLibrarySymbols - enter the built-in functions and symbols */
void EnterLibrarySymbols(InterpreterContext *c)
{
    FunctionTableEntry *p;

    /* enter the built-in functions */
    for (p = functionTable; p->name != 0; ++p)
	EnterBuiltInFunction(c,p->name,p->handler);

    /* enter the built-in streams */
    EnterBuiltInStream(c,"stdin",c->standardInput);
    EnterBuiltInStream(c,"stdout",c->standardOutput);
    EnterBuiltInStream(c,"stderr",c->standardError);
}

/* FindLibraryFunctionHandler - find a library function handler */
CMethodHandlerPtr FindLibraryFunctionHandler(char *name)
{
    FunctionTableEntry *p;
    for (p = functionTable; p->name != 0; ++p)
	if (strcmp(p->name,name) == 0)
	    return p->handler;
    return NULL;
}

/* EnterBuiltInStream - add a built-in stream to the symbol table */
static void EnterBuiltInStream(InterpreterContext *c,char *name,Stream *s)
{
    cpush(c,InternCString(c,name));
    SetSymbolValue(top(c),NewForeignPtrObject(c,FileStreamTypeTag,(void *)s));
    drop(c,1);
}

/* BIF_New - built-in function 'New' */
static ObjectPtr BIF_New(InterpreterContext *c,int n,ObjectPtr *p)
{
    CheckArgCount(1);
    CheckType(1,ObjectP);
    return NewObject(c,GetArg(1));
}

/* BIF_Clone - built-in function 'Clone' */
static ObjectPtr BIF_Clone(InterpreterContext *c,int n,ObjectPtr *p)
{
    CheckArgCount(1);
    CheckType(1,ObjectP);
    return CloneObject(c,GetArg(1));
}

/* BIF_List - built-in function 'List' */
static ObjectPtr BIF_List(InterpreterContext *c,int n,ObjectPtr *p)
{
    ObjectPtr val = c->nilObject;
    p -= n;
    while (--n >= 0)
	val = Cons(c,*p++,val);
    return val;
}

/* BIF_Cons - built-in function 'Cons' */
static ObjectPtr BIF_Cons(InterpreterContext *c,int n,ObjectPtr *p)
{
    CheckArgCount(2);
    return Cons(c,GetArg(1),GetArg(2));
}

/* BIF_Car - built-in function 'Car' */
static ObjectPtr BIF_Car(InterpreterContext *c,int n,ObjectPtr *p)
{
    CheckArgCount(1);
    CheckType(1,ConsP);
    return Car(GetArg(1));
}

/* BIF_SetCar - built-in function 'SetCar' */
static ObjectPtr BIF_SetCar(InterpreterContext *c,int n,ObjectPtr *p)
{
    CheckArgCount(2);
    CheckType(1,ConsP);
    return SetCar(GetArg(1),GetArg(2));
}

/* BIF_Cdr - built-in function 'Cdr' */
static ObjectPtr BIF_Cdr(InterpreterContext *c,int n,ObjectPtr *p)
{
    CheckArgCount(1);
    CheckType(1,ConsP);
    return Cdr(GetArg(1));
}

/* BIF_SetCdr - built-in function 'SetCdr' */
static ObjectPtr BIF_SetCdr(InterpreterContext *c,int n,ObjectPtr *p)
{
    CheckArgCount(2);
    CheckType(1,ConsP);
    return SetCdr(GetArg(1),GetArg(2));
}

/* BIF_Intern - built-in function 'Intern' */
static ObjectPtr BIF_Intern(InterpreterContext *c,int n,ObjectPtr *p)
{
    CheckArgCount(1);
    CheckType(1,StringP);
    return InternSymbol(c,GetArg(1));
}

/* BIF_Class - built-in function 'Class' */
static ObjectPtr BIF_Class(InterpreterContext *c,int n,ObjectPtr *p)
{
    CheckArgCount(1);
    CheckType(1,ObjectP);
    return ObjectClass(GetArg(1));
}

/* BIF_GetLocalProperty - built-in function 'GetLocalProperty' */
static ObjectPtr BIF_GetLocalProperty(InterpreterContext *c,int n,ObjectPtr *p)
{
    ObjectPtr binding;
    CheckArgCount(2);
    CheckType(1,ObjectP);
    binding = GetLocalProperty(c,GetArg(1),GetArg(2));
    return binding ? binding : c->nilObject;
}

/* BIF_Length - built-in function 'Length' */
static ObjectPtr BIF_Length(InterpreterContext *c,int n,ObjectPtr *p)
{
    ObjectPtr arg;
    CheckArgCount(1);
    arg = GetArg(1);
    if (StringP(arg))
        return BoxNumber(StringSize(arg));
    else if (VectorP(arg))
        return BoxNumber(VectorSize(arg));
    else if (ConsP(arg)) {
        long len = 0;
        while (ConsP(arg)) {
            arg = Cdr(arg);
            ++len;
        }
        return BoxNumber(len);
    }
    else if (arg == c->nilObject)
        return BoxNumber(0);
    else
        TypeError(c,arg);
}

/* BIF_Print - built-in function 'Print' */
static ObjectPtr BIF_Print(InterpreterContext *c,int n,ObjectPtr *p)
{
    Stream *s;
    CheckArgRange(1,2);
    if (n == 2) {
	CheckType(2,StreamP);
	s = (Stream *)ForeignPtr(GetArg(2));
    }
    else
	s = c->standardOutput;
    PrintValue(c,GetArg(1),s);
    return c->nilObject;
}

/* BIF_Display - built-in function 'Display' */
static ObjectPtr BIF_Display(InterpreterContext *c,int n,ObjectPtr *p)
{
    Stream *s;
    CheckArgRange(1,2);
    if (n == 2) {
	CheckType(2,StreamP);
	s = (Stream *)ForeignPtr(GetArg(2));
    }
    else
	s = c->standardOutput;
    DisplayValue(c,GetArg(1),s);
    return c->nilObject;
}

/* BIF_GC - built-in function 'GC' */
static ObjectPtr BIF_GC(InterpreterContext *c,int n,ObjectPtr *p)
{
    CheckArgCount(0);
    CollectGarbage(c);
    return c->nilObject;
}

/* BIF_DecodeMethod - built-in function 'DecodeMethod' */
static ObjectPtr BIF_DecodeMethod(InterpreterContext *c,int n,ObjectPtr *p)
{
    Stream *s;
    CheckArgRange(1,2);
    CheckType(1,MethodP);
    if (n == 2) {
	CheckType(2,StreamP);
	s = (Stream *)ForeignPtr(GetArg(2));
    }
    else
	s = c->standardOutput;
    DecodeProcedure(c,GetArg(1),s);
    return c->trueObject;
}

/* BIF_FOpen - built-in function 'FOpen' */
static ObjectPtr BIF_FOpen(InterpreterContext *c,int n,ObjectPtr *p)
{
    char *fname,*mode;
    Stream *s;
    CheckArgCount(2);
    CheckType(1,StringP);
    CheckType(2,StringP);
    fname = StringDataAddress(GetArg(1));
    mode = StringDataAddress(GetArg(2));
    s = OpenFileStream(fname,mode);
    return s == NULL ? c->nilObject : NewForeignPtrObject(c,FileStreamTypeTag,(void *)s);
}

/* BIF_Close - built-in function 'Close' */
static ObjectPtr BIF_Close(InterpreterContext *c,int n,ObjectPtr *p)
{
    Stream *s;
    CheckArgCount(1);
    CheckType(1,StreamP);
    s = (Stream *)ForeignPtr(GetArg(1));
    return CloseStream(s) == 0 ? c->trueObject : c->falseObject;
}

/* BIF_GetC - built-in function 'GetC' */
static ObjectPtr BIF_GetC(InterpreterContext *c,int n,ObjectPtr *p)
{
    Stream *s;
    int ch;
    CheckArgRange(0,1);
    if (n == 1) {
	CheckType(1,StreamP);
	s = (Stream *)ForeignPtr(GetArg(1));
    }
    else
        s = c->standardInput;
    return (ch = StreamGetC(s)) < 0 ? c->nilObject : BoxNumber(ch);
}

/* BIF_PutC - built-in function 'PutC' */
static ObjectPtr BIF_PutC(InterpreterContext *c,int n,ObjectPtr *p)
{
    Stream *s;
    int ch,rch;
    CheckArgRange(1,2);
    CheckType(1,NumberP);
    if (n == 2) {
	CheckType(2,StreamP);
	s = (Stream *)ForeignPtr(GetArg(2));
    }
    else
        s = c->standardOutput;
    ch = (int)UnboxNumber(GetArg(1));
    rch = StreamPutC(ch,s);
    return rch == ch ? GetArg(1) : c->nilObject;
}

static ObjectPtr BIF_CreateObjectStore(InterpreterContext *c,int n,ObjectPtr *p)
{
    char *name;
    CheckArgCount(1);
    CheckType(1,StringP);
    name = StringDataAddress(GetArg(1));
    return CreateConference(name) == 0 ? c->trueObject : c->falseObject;
}

static ObjectPtr BIF_OpenObjectStore(InterpreterContext *c,int n,ObjectPtr *p)
{
    char *name;
    CFILE *cfp;
    int mode;
    CheckArgRange(1,2);
    CheckType(1,StringP);
    if (n == 2)
        mode = GetArg(2) == c->nilObject ? CFM_READ : CFM_UPDATE;
    else
        mode = CFM_UPDATE;
    name = StringDataAddress(GetArg(1));
    cfp = OpenConfDatabase(name,mode);
    return cfp == NULL ? c->nilObject : NewForeignPtrObject(c,ObjectStoreTypeTag,(void *)cfp);
}

static ObjectPtr BIF_CloseObjectStore(InterpreterContext *c,int n,ObjectPtr *p)
{
    int result = 1;
    CFILE *cfp;
    CheckArgCount(1);
    CheckType(1,ObjectStoreP);
    cfp = (CFILE *)ForeignPtr(GetArg(1));
    if (cfp)
	result = CloseConfDatabase(cfp);
    SetForeignPtr(GetArg(1),NULL);
    return result == 0 ? c->trueObject : c->falseObject;
}

static ObjectPtr BIF_ObjectCount(InterpreterContext *c,int n,ObjectPtr *p)
{
    CFILE *cfp;
    CheckArgCount(1);
    CheckType(1,ObjectStoreP);
    cfp = (CFILE *)ForeignPtr(GetArg(1));
    return BoxNumber(GetMessageCount(cfp));
}

static ObjectPtr BIF_OpenObject(InterpreterContext *c,int n,ObjectPtr *p)
{
    long msgno,version;
    CMESSAGE *cmsg;
    CFILE *cfp;
    Stream *s;
    CheckArgCount(3);
    CheckType(1,ObjectStoreP);
    CheckType(2,NumberP);
    CheckType(3,NumberP);
    cfp = (CFILE *)ForeignPtr(GetArg(1));
    msgno = UnboxNumber(GetArg(2));
    version = UnboxNumber(GetArg(3));
    if ((cmsg = OpenConfMessage(cfp,msgno,version)) == NULL)
	return c->nilObject;
    else if ((s = CreateObjectStream(cmsg)) == NULL) {
	CloseConfMessage(cmsg);
	return c->nilObject;
    }
    return NewForeignPtrObject(c,ObjectStreamTypeTag,(void *)s);
}

static ObjectPtr BIF_CreateObject(InterpreterContext *c,int n,ObjectPtr *p)
{
    CMESSAGE *cmsg;
    CFILE *cfp;
    Stream *s;
    CheckArgCount(1);
    CheckType(1,ObjectStoreP);
    cfp = (CFILE *)ForeignPtr(GetArg(1));
    if ((cmsg = CreateConfMessage(cfp)) == NULL)
	return c->nilObject;
    else if ((s = CreateObjectStream(cmsg)) == NULL) {
	CloseConfMessage(cmsg);
	return c->nilObject;
    }
    return NewForeignPtrObject(c,ObjectStreamTypeTag,(void *)s);
}

static ObjectPtr BIF_UpdateObject(InterpreterContext *c,int n,ObjectPtr *p)
{
    CMESSAGE *cmsg;
    CFILE *cfp;
    Stream *s;
    CheckArgCount(2);
    CheckType(1,ObjectStoreP);
    CheckType(2,NumberP);
    cfp = (CFILE *)ForeignPtr(GetArg(1));
    if ((cmsg = UpdateConfMessage(cfp,UnboxNumber(GetArg(2)))) == NULL)
	return c->nilObject;
    else if ((s = CreateObjectStream(cmsg)) == NULL) {
	CloseConfMessage(cmsg);
	return c->nilObject;
    }
    return NewForeignPtrObject(c,ObjectStreamTypeTag,(void *)s);
}

static ObjectPtr BIF_DeleteObject(InterpreterContext *c,int n,ObjectPtr *p)
{
    CFILE *cfp;
    CheckArgCount(2);
    CheckType(1,ObjectStoreP);
    CheckType(2,NumberP);
    cfp = (CFILE *)ForeignPtr(GetArg(1));
    return DeleteConfMessage(cfp,UnboxNumber(GetArg(2))) ? c->falseObject : c->trueObject;
}

static ObjectPtr BIF_ObjectNumber(InterpreterContext *c,int n,ObjectPtr *p)
{
    ObjectStream *s;
    CheckArgCount(1);
    CheckType(1,ObjectStreamP);
    s = (ObjectStream *)ForeignPtr(GetArg(1));
    return BoxNumber(s->cmsg->cm_dheader.dh_msgno);
}

static ObjectPtr BIF_ObjectVersion(InterpreterContext *c,int n,ObjectPtr *p)
{
    ObjectStream *s;
    CheckArgCount(1);
    CheckType(1,ObjectStreamP);
    s = (ObjectStream *)ForeignPtr(GetArg(1));
    return BoxNumber(s->cmsg->cm_dheader.dh_version);
}

static ObjectPtr BIF_CreateCursor(InterpreterContext *c,int n,ObjectPtr *p)
{
    CCURSOR *cursor;
    CFILE *cfp;
    long msgno;
    CheckArgCount(2);
    CheckType(1,ObjectStoreP);
    CheckType(2,NumberP);
    cfp = (CFILE *)ForeignPtr(GetArg(1));
    msgno = UnboxNumber(GetArg(2));
    if ((cursor = CreateConfCursor(cfp,msgno)) == NULL)
	return c->nilObject;
    return NewForeignPtrObject(c,CursorTypeTag,(void *)cursor);
}

static ObjectPtr BIF_CloseCursor(InterpreterContext *c,int n,ObjectPtr *p)
{
    CCURSOR *cursor;
    CheckArgCount(1);
    CheckType(1,CursorP);
    cursor = (CCURSOR *)ForeignPtr(GetArg(1));
    return CloseConfCursor(cursor) == 0 ? c->trueObject : c->falseObject;
}

static ObjectPtr BIF_SetCursorPosition(InterpreterContext *c,int n,ObjectPtr *p)
{
    CCURSOR *cursor;
    long msgno;
    CheckArgCount(2);
    CheckType(1,CursorP);
    CheckType(2,NumberP);
    cursor = (CCURSOR *)ForeignPtr(GetArg(1));
    msgno = UnboxNumber(GetArg(2));
    return SetCursorPosition(cursor,msgno) == 0 ? c->trueObject : c->falseObject;
}

static ObjectPtr BIF_GetNextObject(InterpreterContext *c,int n,ObjectPtr *p)
{
    CCURSOR *cursor;
    long msgno;
    CheckArgCount(1);
    CheckType(1,CursorP);
    cursor = (CCURSOR *)ForeignPtr(GetArg(1));
    msgno = GetNextMessage(cursor);
    return msgno == -1 ? c->nilObject : BoxNumber(msgno);
}

static ObjectPtr BIF_GetPreviousObject(InterpreterContext *c,int n,ObjectPtr *p)
{
    CCURSOR *cursor;
    long msgno;
    CheckArgCount(1);
    CheckType(1,CursorP);
    cursor = (CCURSOR *)ForeignPtr(GetArg(1));
    msgno = GetPrevMessage(cursor);
    return msgno == -1 ? c->nilObject : BoxNumber(msgno);
}

static ObjectPtr BIF_OpenCurrentObject(InterpreterContext *c,int n,ObjectPtr *p)
{
    CCURSOR *cursor;
    CMESSAGE *cmsg;
    long version;
    Stream *s;
    CheckArgCount(2);
    CheckType(1,CursorP);
    CheckType(2,NumberP);
    cursor = (CCURSOR *)ForeignPtr(GetArg(1));
    version = UnboxNumber(GetArg(2));
    if ((cmsg = OpenCurrentMessage(cursor,version)) == NULL)
	return c->nilObject;
    else if ((s = CreateObjectStream(cmsg)) == NULL) {
	CloseConfMessage(cmsg);
	return c->nilObject;
    }
    return NewForeignPtrObject(c,ObjectStreamTypeTag,(void *)s);
}

static ObjectPtr BIF_Flatten(InterpreterContext *c,int n,ObjectPtr *p)
{
    Stream *s;
    CheckArgCount(2);
    CheckType(2,StreamP);
    s = (Stream *)ForeignPtr(GetArg(2));
    return FlattenObject(c,GetArg(1),c->nilObject,s) ? c->trueObject : c->falseObject;
}

static ObjectPtr BIF_Unflatten(InterpreterContext *c,int n,ObjectPtr *p)
{
    ObjectPtr obj;
    Stream *s;
    CheckArgCount(1);
    CheckType(1,StreamP);
    s = (Stream *)ForeignPtr(GetArg(1));
    obj = UnflattenObject(c,c->nilObject,s);
    return obj ? obj : c->nilObject;
}

static ObjectPtr BIF_NewLineBuffer(InterpreterContext *c,int n,ObjectPtr *p)
{
    LineBuffer *lbuf;
    CheckArgCount(0);
    if ((lbuf = NewLineBuffer()) == NULL)
	return c->nilObject;
    return NewForeignPtrObject(c,LineBufferTypeTag,(void *)lbuf);
}

static ObjectPtr BIF_FillLineBuffer(InterpreterContext *c,int n,ObjectPtr *p)
{
    LineBuffer *lbuf;
    Stream *s;
    CheckArgCount(2);
    CheckType(1,LineBufferP);
    CheckType(2,StreamP);
    lbuf = (LineBuffer *)ForeignPtr(GetArg(1));
    s = (Stream *)ForeignPtr(GetArg(2));
    return FillLineBuffer(lbuf,s) ? c->trueObject : c->falseObject;
}

static ObjectPtr BIF_NextToken(InterpreterContext *c,int n,ObjectPtr *p)
{
    LineBuffer *lbuf;
    char buf[LBMAX];
    CheckArgCount(1);
    CheckType(1,LineBufferP);
    lbuf = (LineBuffer *)ForeignPtr(GetArg(1));
    if (!NextToken(lbuf,buf))
	return c->nilObject;
    return NewCStringObject(c,buf);
}

static ObjectPtr BIF_RestOfLine(InterpreterContext *c,int n,ObjectPtr *p)
{
    LineBuffer *lbuf;
    char buf[LBMAX];
    CheckArgCount(1);
    CheckType(1,LineBufferP);
    lbuf = (LineBuffer *)ForeignPtr(GetArg(1));
    if (!RestOfLine(lbuf,buf))
	return c->nilObject;
    return NewCStringObject(c,buf);
}

static ObjectPtr BIF_WholeLine(InterpreterContext *c,int n,ObjectPtr *p)
{
    LineBuffer *lbuf;
    char buf[LBMAX];
    CheckArgCount(1);
    CheckType(1,LineBufferP);
    lbuf = (LineBuffer *)ForeignPtr(GetArg(1));
    if (!WholeLine(lbuf,buf))
	return c->nilObject;
    return NewCStringObject(c,buf);
}

static ObjectPtr BIF_IsMoreOnLine(InterpreterContext *c,int n,ObjectPtr *p)
{
    LineBuffer *lbuf;
    CheckArgCount(1);
    CheckType(1,LineBufferP);
    lbuf = (LineBuffer *)ForeignPtr(GetArg(1));
    return IsMoreOnLine(lbuf) ? c->trueObject : c->falseObject;
}

static ObjectPtr BIF_Load(InterpreterContext *c,int n,ObjectPtr *p)
{
    char *name;
    CheckArgCount(1);
    CheckType(1,StringP);
    if (c->cc) {
        name = StringDataAddress(GetArg(1));
        return LoadFile(c->cc,name,c->standardOutput) == 0 ? c->trueObject
		                                           : c->falseObject;
    }
    else
	return c->falseObject;
}

/* BIF_Quit - built-in function 'Quit' */
static ObjectPtr BIF_Quit(InterpreterContext *c,int n,ObjectPtr *p)
{
    CheckArgCount(0);
    CallErrorHandler(c,ecExit,0);
    return 0; /* never reached */
}


