/* bobfcn.c - built-in classes and functions 
 *
 *       Copyright (c) 1991, by David Michael Betz
 *       All rights reserved
 *
 *** Revision History
 * 21-Sep-1994 Dutton      Added exec_funxtion that executed the passed
 * ....................... function name with the arguments
 *  9-Sep-1994 Dutton      Added new function strchr and changes index 
 * ....................... function to strstr
 *  6-Sep-1994 Dutton      Converted function descriptions to pre and 
 * ....................... post conditions
 *  1-Sep-1994 Dutton      Added bob load function and case manipulation 
 * ....................... functions
 * 30-Aug-1994 Dutton      Added Current and last function
 * 16-Aug-1994 Dutton      Added heaps of functionallity here. Changed 
 * ....................... typeof to return string rather than integer
 */

#include <setjmp.h>
#include <time.h>
#include "bob.h"

/* stdio dispatch table */
IODISPATCH fileio = {
    (CloseFunc)fclose,
    (GetFunc)fgetc,
    (PutFunc)fputc,
    (PutSFunc)fputs
};

/* external variables */
extern VALUE symbols;
extern jmp_buf error_trap;
extern char    last_func[TKNSIZE];     /* last sucessfully executed command */
extern char    curr_func[TKNSIZE];     /* Currently executing function */

/* forward declarations */
static void xtypeof(int argc);
static void xgc(int argc);
static void xnewvector(int argc);
static void xnewstring(int argc);
static void xsizeof(int argc);
static void xfopen(int argc);
static void xfclose(int argc);
static void xgetc(int argc);
static void xputc(int argc);
static void xprint(int argc);
static void xgetenv(int argc);
static void xsystem(int argc);
static void xstrlen ( int argc );
static void xsubstring ( int argc );
static void xstr_to_num ( int argc );
static void xnothing ( int argc );
static void xversion ( int argc );
static void xval_to_string ( int argc );
static void xkeyboard_quit ( int argc );
static void xlist_func ( int argc );
static void xstrstr ( int argc );
static void xstrchr ( int argc );
static void xdate_time ( int argc );
static void val_to_string ( char *buf, VALUE *val );
static void xchr ( int argc );
static void xascii ( int argc );
static void xcurr_func ( int argc );
static void xlast_func ( int argc );
static void xupcase ( int argc );
static void xdowncase ( int argc );
static void xeditcase ( int argc );
static void xloadbob ( int argc );
static void xexec_function ( int argc );

/* init_functions - initialize the internal functions */
void init_functions()
{
    add_function ( "typeof", xtypeof );
    add_function ( "gc", xgc );
    add_function ( "newvector", xnewvector );
    add_function ( "newstring", xnewstring );
    add_function ( "sizeof", xsizeof );
    add_function ( "fopen", xfopen );
    add_function ( "fclose", xfclose );
    add_function ( "getc", xgetc );
    add_function ( "putc", xputc );
    add_function ( "print", xprint );
    add_function ( "getenv", xgetenv );
    add_function ( "system", xsystem );
    add_function ( "strlen", xsizeof );
    add_function ( "substring", xsubstring );
    add_function ( "str_to_num", xstr_to_num );
    add_function ( "val_to_string", xval_to_string );
    add_function ( "version", xversion );
    add_function ( "nothing", xnothing );
    add_function ( "strstr", xstrstr );
    add_function ( "strchr", xstrchr );
    add_function ( "date_time", xdate_time );
    add_function ( "keyboard_quit", xkeyboard_quit );
    add_function ( "list_functions", xlist_func );
    add_function ( "chr", xchr );
    add_function ( "ascii", xascii );
    add_function ( "curr_function", xcurr_func );
    add_function ( "last_function", xlast_func );
    add_function ( "upcase", xupcase );
    add_function ( "downcase", xdowncase );
    add_function ( "editcase", xeditcase );
    add_function ( "load_bob", xloadbob );
    add_function ( "exec_function", xexec_function );
}
/******************************************************************************
 * Function:
 * BobFixStack -- Gets the arguments from the stack
 * 
 * Description:
 *
 * PRE: Bob correctly set up
 * POST: Fixes up the stack for any internal function call
 */
void BobFixStack ( int argc, int type, ... )
{
    va_list     lst;
    
    va_start ( lst, type );
    sp += argc; 
    sp->v_type = type;
    sp->v.v_integer = va_arg( lst, int );
    va_end ( lst );
}
/******************************************************************************
 * Function:
 * BobGetArgs -- Gets the arguments from the stack
 * 
 * Description:
 *
 * PRE:  Bob correctly set up
 * POST: sets up the arguments from the stack in the correct order and 
 *       checks if the number of arguments are within limits
 */
void BobGetArgs ( VALUE **arg, int argc, int mn, int mx  )
{
    int         si;             /* counter */
    
    if ( (mn) >= 0 && argc < (mn) ) error("Too few arguments" );
    if ( (mx) >= 0 && argc > (mx) ) error("Too many arguments" ); 
    if ( arg != NULL ) {
        for ( si=0; si<argc; si++ ) arg[si] = &sp[argc-1-si];
    }
}

/******************************************************************************
 * Function:
 * BobCheckType -- Checks the type of the passed arg
 * 
 * Description:
 *
 * PRE:  Bob is correctly initialized.
 * POST: returns TRUE if the arg is of the correct type. False otherwise 
 *       and sets up error
 */
int BobCheckType ( VALUE *arg, int type ) 
{
    if ( arg->v_type == type ) return ( TRUE );
    error("Bad argument type");
    return ( FALSE );
}

/******************************************************************************
 * Function:
 * del_function -- Deletes the passed function
 * 
 * Description:
 *
 * PRE:  Bob must have been correctly set up
 * POST: returns result from delentry
 */
int del_function( char *name )
{
    return ( delentry(&symbols,name) );
}

/* add_function - add a built-in function */
void add_function(name,fcn)
  char *name; void (*fcn)( int );
{
    DICT_ENTRY *sym;
    sym = addentry(&symbols,name,ST_SFUNCTION);
    set_code(&sym->de_value,fcn);
}

/* xtypeof - get the data type of a value */
static void xtypeof(argc)
  int argc;
{
    VALUE       *arg[1];
    
    BobGetArgs ( arg, argc, 1, 1 );
    BobFixStack ( argc, DT_STRING, makestring ( typename ( arg[0]->v_type ) ) );
}

/* xgc - invoke the garbage collector */
static void xgc(argc)
  int argc;
{
    BobGetArgs ( NULL, argc, 0, 0 );

    /* do the garbage collection */
    gc();
    
    /* put the value back on the stack */
    BobFixStack ( argc, DT_NIL, FALSE );
}

/* xnewvector - allocate a new vector */
static void xnewvector(argc)
  int argc;
{
    VALUE       *arg[1];
    int         size;
    
    BobGetArgs ( arg, argc, 1, 1 );
    BobCheckType ( arg[0], DT_INTEGER );
    
    /* get the size of the vector */
    BobFixStack ( argc, DT_VECTOR, newvector ( arg[0]->v.v_integer ) );
}

/* xnewstring - allocate a new string */
static void xnewstring(argc)
  int argc;
{
    VALUE       *arg[2];
    int         size;
    
    /* get the args and check */
    BobGetArgs ( arg, argc, 1, 1 );
    BobCheckType ( arg[0], DT_INTEGER );

    BobFixStack ( argc, DT_STRING, newstring ( arg[0]->v.v_integer ) );
}

/* xsizeof - get the size of a vector or string */
static void xsizeof(argc)
  int argc;
{
    VALUE       *arg[1];
    int         size;
    
    /* get the args and check */
    BobGetArgs ( arg, argc, 1, 1 );

    switch ( arg[0]->v_type) {
    case DT_VECTOR: size = arg[0]->v.v_vector->vec_size; break;
    case DT_STRING: size = arg[0]->v.v_string->str_size; break;
    default: size = 1;
    }
    BobFixStack ( argc, DT_INTEGER, size );
}

/* xfopen - open a file */
static void xfopen(argc)
  int argc;
{
    GenBuffer   name;
    char        mode[10];
    FILE        *fp;
    VALUE       *arg[2];

    /* get the args and check */
    BobGetArgs ( arg, argc, 2, 2 );
    BobCheckType ( arg[0], DT_STRING );
    BobCheckType ( arg[1], DT_STRING );

    /* do the work */
    getcstring ( name, sizeof(name), arg[0] );
    getcstring ( mode, sizeof(mode), arg[1] );
    if ( ( fp = fopen ( name, mode ) ) == NULL ) {
        BobFixStack ( argc, DT_NIL, FALSE );
    } else {
        BobFixStack ( argc, DT_IOSTREAM, newiostream ( &fileio, fp ) );
    }
}

/* xfclose - close a file */
static void xfclose(argc)
  int argc;
{
    VALUE       *arg[1];
    
    /* get the args and check */
    BobGetArgs ( arg, argc, 1, 1 );
    BobCheckType ( arg[0], DT_IOSTREAM );

    BobFixStack ( argc, DT_INTEGER, iosclose ( arg[0] ) );
}

/* xgetc - get a character from a file */
static void xgetc(argc)
  int argc;
{
    VALUE       *arg[1];
    
    /* get the args and check */
    BobGetArgs ( arg, argc, 1, 1 );
    BobCheckType ( arg[0], DT_IOSTREAM );
    BobFixStack ( argc, DT_INTEGER, iosgetc ( arg[0] ) );
}

/* xputc - output a character to a file */
static void xputc(argc)
  int argc;
{
    VALUE       *arg[2];
    
    /* get the args and check */
    BobGetArgs ( arg, argc, 2, 2 );
    BobCheckType ( arg[0], DT_INTEGER );
    BobCheckType ( arg[1], DT_IOSTREAM );

    BobFixStack ( argc, DT_INTEGER, iosputc( arg[0]->v.v_integer, arg[1] ) );
}

/* xprint - generic print function */
static void xprint(argc)
  int argc;
{
    extern VALUE stdout_iostream;
    VALUE       *arg[20];
    int n;

    BobGetArgs ( arg, argc, -1, 20 );
    for ( n=0; n<argc; n++ ) {
        print1( &stdout_iostream, FALSE, arg[n] );
    }
    BobFixStack ( argc, DT_INTEGER, n );
}

/* print1 - print one value */
print1(ios,qflag,val)
  VALUE *ios; int qflag; VALUE *val;
{
    GenBuffer   buf;
    
    if ( qflag && val->v_type == DT_STRING ) iosputc('"',ios);
    val_to_string ( buf, val );
    iosputs ( buf, ios );
    if ( qflag && val->v_type == DT_STRING ) iosputc('"',ios);
}

/* converts a value to a string */
static void val_to_string ( char *buf, VALUE *val )
{
    char name[TKNSIZE+1], *p, *s;
    VALUE *class;
    int len;
    
    switch (val->v_type) {
    case DT_NIL:
        sprintf ( buf, "nil" );
        break;
    
    case DT_CLASS:
        getcstring(name,sizeof(name),clgetname(val));
        sprintf(buf,"#<Class-%s>",name);
        break;
    
    case DT_OBJECT:
        sprintf(buf,"#<Object-%lx>",objaddr(val));
        break;
    
    case DT_VECTOR:
        sprintf(buf,"#<Vector-%lx>",vecaddr(val));
        break;

    case DT_INTEGER:
        sprintf(buf,"%ld",val->v.v_integer);
        break;
        
    case DT_RCODE:              /* remote function */
        sprintf ( buf, "#<ExtFunc-" );
        s = &buf[ strlen ( buf )];
        p = strgetdata(val);
        len = strgetsize(val);
        for ( ;--len>= 0; s++, p++ ) *s = *p;
        *s = '>'; s++;
        *s = '\0';
        break;
        
    case DT_STRING:
        p = strgetdata(val);
        len = strgetsize(val);
        s = buf;
        for ( ;--len>= 0; s++, p++ ) *s = *p;
        *s = '\0';
        break;
        
    case DT_BYTECODE:
        sprintf(buf,"#<Bytecode-%lx>",vecaddr(val));
        break;

    case DT_CODE:
        sprintf(buf,"#<Code-%lx>",val->v.v_code);
        break;

    case DT_VAR:
        class = digetclass(degetdictionary(val));
        if (!isnil(class)) {
            getcstring(name,sizeof(name),clgetname(class));
            sprintf(buf,"%s::",name);
        }
        getcstring(name,sizeof(name),degetkey(val));
        strcat ( buf, name );
        break;
        
    case DT_IOSTREAM:
        sprintf(buf,"#<Stream-%lx>",val->v.v_iostream);
        break;
        
    default:
        sprintf(buf,"BadType-%d",valtype(val));
        break;
    }
}

/* xgetenv - gets an environment variable */
static void xgetenv(argc)
  int argc;
{
    int         i;
    VALUE       *arg[1];
    GenBuffer   name;           /* name of the environment variable */
    char        *val;           /* value of the environment variable */

    BobGetArgs ( arg, argc, 1, 1 );
    BobCheckType ( arg[0], DT_STRING );
    
    getcstring ( name, sizeof ( name ), arg[0] );
    if ( ( val = getenv ( name ) ) == NULL ) val = "";
    
    BobFixStack ( argc, DT_STRING, makestring ( val ) );
}

/* xsystem - execute a system command */
static void xsystem(argc)
  int argc;
{
    GenBuffer   cmd;
    VALUE       *arg[1];
    
    BobGetArgs ( arg, argc, 1, 1 );
    BobCheckType ( arg[0], DT_STRING );

    getcstring ( cmd, sizeof ( cmd ), arg[0] );
    BobFixStack ( argc, DT_INTEGER, system ( cmd ) );
}

/* xval_to_string -- converts any argument to a string */
static void xval_to_string ( int argc )
{
    GenBuffer   buf;
    VALUE       *arg[1];
    
    BobGetArgs ( arg, argc, 1, 1 );
    val_to_string ( buf, arg[0] );
    BobFixStack ( argc, DT_STRING, makestring ( buf ) );
}

static void xsubstring ( int argc )
{
    int         slen;           /* string length */
    int         spos, i;        /* counters */
    char        *tstr=NULL;     /* tempory string */
    int         len;            /* length we want to copy */
    VALUE       *arg[3];        /* list of args */
    
    /* get the args and check */
    BobGetArgs ( arg, argc, 2, 3 );
    BobCheckType ( arg[0], DT_STRING );
    BobCheckType ( arg[1], DT_INTEGER );

    /* get the length of the passed string */
    slen = strgetsize(arg[0]);
    
    /* set up a tempory string */
    tstr = (char *)malloc ( slen + 1 );
    
    /* initialize some stuff */
    i = 0;
    spos = arg[1]->v.v_integer;
    if ( arg[2] == NULL ) len = slen;
    else len = arg[2]->v.v_integer;
    
    /* copy the string */
    while ( i < len && spos < slen ) {
        tstr[i++] = strgetdata(arg[0])[spos++];
    }
    /* terminate the string */
    tstr[i] = '\0';
    
    /* put the value back on the stack */
    if ( tstr == NULL ) {
        BobFixStack ( argc, DT_NIL, FALSE );
    } else {
        BobFixStack ( argc, DT_STRING, makestring ( tstr ) );
        free ( tstr );
    }
}

static void xstr_to_num ( int argc )
{
    char        tstr[255];
    int         num;
    VALUE       *arg[1];
    
    /* get the args and check */
    BobGetArgs ( arg, argc, 1, 1 );
    BobCheckType ( arg[0], DT_STRING );
    
    /* do the work */
    getcstring ( tstr, sizeof ( tstr ), arg[0] );
    num = atoi ( tstr );
    
    /* put the value back on the stack */
    BobFixStack ( argc, DT_INTEGER, num );
}

/* returns the version of the software */
static void xversion ( int argc )
{
    /* get the args and check */
    BobGetArgs ( NULL, argc, 0, 0 );
    
    /* put the value back on the stack */
    BobFixStack ( argc, DT_STRING, makestring ( BOBVERS ) );
}

static void xnothing ( int argc )
{
    /* get the args and check */
    BobGetArgs ( NULL, argc, 0, 0 );
    
    /* put the value back on the stack */
    BobFixStack ( argc, DT_NIL, FALSE );
}

/* does a quit */
static void xkeyboard_quit ( int argc )
{
    /* get the args and check */
    BobGetArgs ( NULL, argc, 0, 0 );
    
    longjmp ( error_trap, 1 );
    
    /* put the value back on the stack */
    BobFixStack ( argc, DT_NIL, FALSE );
}
/* lists all the available functions */
static void xlist_func ( int argc )
{
    DICT_ENTRY  *entry;
    int         size;
    char        buff[100];
    int         cnt;
    VECTOR      *vec;
    VALUE       *vdata;

    /* get the args and check */
    BobGetArgs ( NULL, argc, 0, 0 );

    for ( cnt=0, entry = digetcontents(&symbols)->v.v_var;
          entry != NULL;
          entry = entry->de_next.v.v_var) {
        
        /* make sure the correct type */
        if ( entry->de_type == ST_SFUNCTION ) cnt ++;
    }
    
    vec = newvector ( cnt );
    for ( vdata=vec->vec_data, entry=digetcontents(&symbols)->v.v_var;
          entry != NULL, cnt > 0;
          entry = entry->de_next.v.v_var ) {
        
        /* make sure the correct type */
        if ( entry->de_type != ST_SFUNCTION ) continue;
        getcstring ( buff, sizeof ( buff ), &entry->de_key );
        set_string ( vdata, makestring ( buff ) );
        vdata ++;
        cnt--;
    }
    

    /* put the value back on the stack */
    BobFixStack ( argc, DT_VECTOR, vec );
}

/* returns the position of string2 in string1 */
static void xstrstr ( int argc )
{
    VALUE       *arg[2];        /* list og arguments */
    GenBuffer   str1;           /* string we are looking in */
    GenBuffer   str2;           /* string we are looking for */
    char        *pos;           /* position in string */

    /* get the args and check */
    BobGetArgs ( arg, argc, 2, 2 );
    BobCheckType ( arg[0], DT_STRING );
    BobCheckType ( arg[1], DT_STRING );
    
    /* get the strings from the args */
    getcstring ( str1, sizeof ( str1 ), arg[0] );
    getcstring ( str2, sizeof ( str2 ), arg[1] );
    
    /* find the position and put it back on the stack */
    if ( ( pos = strstr ( str1, str2 ) ) == NULL ) {
        BobFixStack ( argc, DT_INTEGER, -1 );
    } else {
        BobFixStack ( argc, DT_INTEGER, pos - str1 );
    }
}

/* returns the position of char in string1 */
static void xstrchr ( int argc )
{
    VALUE       *arg[2];        /* list og arguments */
    GenBuffer   str1;           /* string we are looking in */
    GenBuffer   str2;           /* string we are looking for */
    char        *pos;           /* position in string */

    /* get the args and check */
    BobGetArgs ( arg, argc, 2, 2 );
    BobCheckType ( arg[0], DT_STRING );
    
    /* get the strings from the args */
    getcstring ( str1, sizeof ( str1 ), arg[0] );
    if ( valtype ( arg[1] ) == DT_STRING ) {
        getcstring ( str2, sizeof ( str2 ), arg[1] );
    } else if ( valtype ( arg[1] ) == DT_INTEGER ) {
        str2[0] = arg[1]->v.v_integer;
        str2[1] = '\0';
    } else {
        BobCheckType ( arg[1], DT_STRING );
    }
    
    /* find the position and put it back on the stack */
    if ( ( pos = strchr ( str1, str2[0] ) ) == NULL ) {
        BobFixStack ( argc, DT_INTEGER, -1 );
    } else {
        BobFixStack ( argc, DT_INTEGER, pos - str1 );
    }
}

/* returns the current date-time string */
static void xdate_time ( int argc )
{
    char        tstr[100];      /* string of the current time */
    char        *pos;           /* position in string */

    /* get the args and check */
    BobGetArgs ( NULL, argc, 0, 0 );
    
    /* get the current time */
    strcpy ( tstr, ctime ( NULL ) );
    if ( ( pos = strchr ( tstr, '\n' ) ) != NULL ) *pos = '\0';
    
    /* put the string back onto the stack */
    BobFixStack ( argc, DT_STRING, makestring ( tstr ) );
}

/* returns the string for the passed ascii value */
static void xchr ( int argc )
{
    char        tstr[2];
    int         num;
    VALUE       *arg[1];
    
    /* get the args and check */
    BobGetArgs ( arg, argc, 1, 1 );
    BobCheckType ( arg[0], DT_INTEGER );
    
    /* do the work */
    tstr[0] = arg[0]->v.v_integer;
    tstr[1] = '\0';
    
    /* put the value back on the stack */
    BobFixStack ( argc, DT_STRING, makestring ( tstr ) );
}
/* returns the ascii value of the passed string */
static void xascii ( int argc )
{
    int         num;
    VALUE       *arg[1];
    
    /* get the args and check */
    BobGetArgs ( arg, argc, 1, 1 );
    BobCheckType ( arg[0], DT_STRING );
    
    /* put the value back on the stack */
    BobFixStack ( argc, DT_INTEGER, strgetdata(arg[0])[0] );
}
/* returns the currently executing function name */
static void xcurr_func ( int argc )
{
    /* get the args and check */
    BobGetArgs ( NULL, argc, 0, 0 );
    
    /* put the value back on the stack */
    BobFixStack ( argc, DT_STRING, makestring ( curr_func ) );
}

/* returns the currently executing function name */
static void xlast_func ( int argc )
{
    /* get the args and check */
    BobGetArgs ( NULL, argc, 0, 0 );
    
    /* put the value back on the stack */
    BobFixStack ( argc, DT_STRING, makestring ( last_func ) );
}

/* converts the string to uppercase */
static void xupcase ( int argc )
{
    VALUE       *arg[1];        /* list of arguments */
    GenBuffer   str;            /* string we are looking in */
    int         i;              /* counter */
    
    /* get the args and check */
    BobGetArgs ( arg, argc, 1, 1 );
    BobCheckType ( arg[0], DT_STRING );
    
    /* get the strings from the args */
    getcstring ( str, sizeof ( str ), arg[0] );
    
    /* find the position and put it back on the stack */
    for ( i=0; str[i]; i++ ) str[i] = toupper ( str[i] );
    
    BobFixStack ( argc, DT_STRING, makestring ( str ) );
}    

/* converts the passed string to lowercase */
static void xdowncase ( int argc )
{
    VALUE       *arg[1];        /* list of arguments */
    GenBuffer   str;            /* string we are looking in */
    int         i;              /* counter */
    
    /* get the args and check */
    BobGetArgs ( arg, argc, 1, 1 );
    BobCheckType ( arg[0], DT_STRING );
    
    /* get the strings from the args */
    getcstring ( str, sizeof ( str ), arg[0] );
    
    /* find the position and put it back on the stack */
    for ( i=0; str[i]; i++ ) str[i] = tolower ( str[i] );
    
    BobFixStack ( argc, DT_STRING, makestring ( str ) );
}    

/* converts the passed string to have first character after space */
/* to be uppercase, all others to be lower case */
static void xeditcase ( int argc )
{
    VALUE       *arg[1];        /* list of arguments */
    GenBuffer   str;            /* string we are looking in */
    int         i;              /* counter */

    /* get the args and check */
    BobGetArgs ( arg, argc, 1, 1 );
    BobCheckType ( arg[0], DT_STRING );
    
    /* get the strings from the args */
    getcstring ( str, sizeof ( str ), arg[0] );
    
    /* find the position and put it back on the stack */
    for ( i=0; str[i]; i++ ) {
        
        /* if first character then uppercase */
        if ( i == 0 || str[i-1] == ' ' ) str[i] = toupper ( str[i] );
        else str[i] = tolower ( str[i] );
    }
    
    BobFixStack ( argc, DT_STRING, makestring ( str ) );
}

/* tries to load the passed bob file name */
static void xloadbob ( int argc )
{
    VALUE       *arg[1];        /* list of arguments */
    GenBuffer   str;            /* string we are looking in */
    int         i;              /* counter */
    GenBuffer   buff;           /* buffer */
    int         len;            /* length of the buffer */

    /* get the args and check */
    BobGetArgs ( arg, argc, 1, 1 );
    BobCheckType ( arg[0], DT_STRING );
    
    /* get the strings from the args */
    getcstring ( str, sizeof ( str ), arg[0] );

    /* try and load the file */
    if ( BobLoadFilename ( str ) ) {
        BobFixStack ( argc, DT_INTEGER, TRUE );
    } else {
        BobFixStack ( argc, DT_NIL, FALSE );
    }
}
/* executes the passed function name with the arguments */
static void xexec_function ( int argc )
{
    VALUE       *arg[20];       /* list of arguments */
    VALUE       parg[20];       /* list of arguments */
    VALUE       rval;           /* returned value */
    char        func[TKNSIZE];  /* function to execute */
    int         i, j;           /* counter */
    
    
    GenBuffer   str;            /* string we are looking in */
    GenBuffer   buff;           /* buffer */
    int         len;            /* length of the buffer */

    /* get the args and check */
    BobGetArgs ( arg, argc, 1, 20 );

    /* get the strings from the args */
    BobCheckType ( arg[0], DT_STRING );
    getcstring ( func, sizeof ( func ), arg[0] );
    
    for ( i=0, j=1; j<argc; j++, i++ ) parg[i] = *arg[j];
    
    if ( execute ( func, argc - 1, parg ) ) {
        rval = *sp;
        BobFixStack ( argc, rval.v_type, rval.v.v_integer );
    } else {
        BobFixStack ( argc, DT_NIL, FALSE );
    }
}
