// MAPLLIST.C -- linked list routines
// We will first pass through the ResidentNames table, creating list elements
//   Then we will pass through the NonResidentNames table, creating additional
//   elements.  Third, we will pass through the entrypoint list to associate
//   ordinals/names to an actual CS:IP (actually segno:offset)
// We also create a list from the segment table so the generated MAP
//   file will have appropriate segment information

#include "mapman.h"
#include "mapllist.h"
#include "mmnehead.h"

ENTRYPOINT* pEntryHead = NULL ; // points to first element on list
ENTRYPOINT* pEntryTail = NULL ; // points to last valid element on list
int cbEntryElements = 0 ;       // total number of elements currently on list

// allocate storage for an entry point element (contents undefined)
ENTRYPOINT *MakeEntryPointElement()
{
   ENTRYPOINT *ep ;
   ep = (ENTRYPOINT *) mymalloc( sizeof(ENTRYPOINT) ) ;

   if (NULL == ep)
   {
      myexit( RC_MALLOCERROR ) ;
   }
   else
   {
     if ( NULL == pEntryTail )
     {  // no list yet, so set pointer for valid list
        pEntryHead = ep ;
     }
     else // there is at least one element already on the list
     {
        pEntryTail->pnext = ep ;
     }

     ep->pnext = NULL ;  // no next element in the list
     pEntryTail = ep ;   // this new element is the tail (last element) on list
   }

   cbEntryElements++ ;   // the count of elements on our entry list

   return( ep ) ;
}

// return an entry point element given its ordinal or NULL if the ordinal
//   cannot be found or a list has not been created
ENTRYPOINT *GetEntryPointElement( WORD ordinal )
{
   ENTRYPOINT *ep ;

   if ( NULL == pEntryHead )
   {  // no list yet...
      return (ENTRYPOINT *) NULL ;
   }

   for ( ep = pEntryHead ; ep; ep=ep->pnext )
       if ( ordinal == ep->wAPIOrdinal )
          return(ep) ;

   return ( (ENTRYPOINT *) NULL ) ;
}

// builds an MS-specific entry table section for the generated MAP file
void DumpEntryPointList()
{
   ENTRYPOINT *ep = pEntryHead ;

   while ( ep )
   {
//      DebugString( ep->pszAPIName ) ;          // DEBUG!
//      DebugInt( ep->wAPIOrdinal ) ;            // DEBUG!
//      DebugInt( ep->wSegmentNumber ) ;         // DEBUG!
//      DebugInt( ep->wOffset ) ;                // DEBUG!

      if ( ep->wAPIOrdinal ) // ignore 0s (which are module name and description)
      {
        myprintfBuildFarPtr( ep->wSegmentNumber, ep->wOffset ) ;
        myprintfFormatStr( " %s",
             ep->pszAPIName ) ;
        myTabAsSpaces( ep->pszAPIName, 24 ) ;
        myprintfFormatStr( "%s",
             ep->pszAPIName ) ;
        myprintfStr( "\n " ) ;
      }

      ep = ep->pnext ;
   }
}

#define ISDATASEG 0x0001

// builds an MS-specific segment table section for the generated MAP file
//   It should be noted that the MAP file generated by the linker can be
//   a bit more specific.  Ours, as EXEHDR, cannot attempt to break the
//   segment into pieces ( _BSS, CDATA, and so forth ).
void DumpSegmentList()
{
   SEGMENT *sp = pSegmentHead ;
   UINT cbSeg = pNE->cbSegment ;
   int thisseg = 1 ;  // we start with the 1st segment (note that the segment
                      //   number is not given in the segment table, a segment's
                      //   position in the segment table implies its number.
                      //   The first segment table entry is the first segment)

   while ( cbSeg-- )  // this is the number of segments in the segment table
   {
      myprintfBuildFarPtr( thisseg, 0 ) ; // segments start at offset 0
      myprintfFormatInt( " 0%04XH     ", sp->len ) ; // 0lenH is the MS format
      if ( sp->flags & ISDATASEG )        // is this a DATA segment?
      {                                   // yes
        myprintfFormatInt( "Seg%X_DATA", thisseg ) ;
                                          // the following check keeps segments
                                          //   ge 10h and those lt 10h aligned
                                          //   in the generated MAP file
        if ( thisseg > 0xf )  // f (dec. 15) is the largest single-digit segno
           myTabAsSpaces( "Seg10_DATA", 23 ) ;  // two-digit segno
        else
           myTabAsSpaces( "Seg0_DATA", 23 ) ;   // one-digit segno

        myprintfFormatStr( "%s","DATA" ) ;
      }
      else
      {                                   // no, so this is a CODE segment
        myprintfFormatInt( "Seg%X_TEXT", thisseg ) ;
        if ( thisseg > 0xf )
           myTabAsSpaces( "Seg10_TEXT", 23 ) ;
        else
           myTabAsSpaces( "Seg0_TEXT", 23 ) ;

        myprintfFormatStr( "%s","CODE" ) ;
      }
      myprintfStr( "\n " ) ;

      thisseg++ ;                         // increase our segment number
      sp++ ;                              // move to next segment entry
   }
}

// the first entry of our entry table contains the module name, so return the
//   szAPIName field pointer. pEntryHead is private to this module and
//   the module name should only be accessed through this function.
char* GetModuleName()
{
   return( pEntryHead->pszAPIName ) ;
}


// Sorting the following list has been left as an exercise for the reader!
// This function is currently no different from DumpPublicsByValue().
// MapSym doesn't care so I didn't do this yet...
void DumpPublicsByName()
{
   ENTRYPOINT *ep = pEntryHead ;

   while ( ep )
   {
      if ( ep->wAPIOrdinal ) // ignore 0s (module name and description)
      {
        myprintfBuildFarPtr( ep->wSegmentNumber, ep->wOffset ) ;
        myprintfFormatStr( "       %s",
             ep->pszAPIName ) ;
        myprintfStr( "\n " ) ;
      }
      ep = ep->pnext ;
   }
}

// Sorting the following list has been left as an exercise for the reader!
// This function is currently no different from DumpPublicsByName().
// MapSym doesn't care so I didn't do this yet...
void DumpPublicsByValue()
{
   ENTRYPOINT *ep = pEntryHead ;

   while ( ep )
   {
      if ( ep->wAPIOrdinal ) // ignore 0s (module name and description)
      {
        myprintfBuildFarPtr( ep->wSegmentNumber, ep->wOffset ) ;
        myprintfFormatStr( "       %s",
             ep->pszAPIName ) ;
        myprintfStr( "\n " ) ;
      }

      ep = ep->pnext ;
   }
}

void FreeLists()
{
   ENTRYPOINT *ep = pEntryHead ;
   SEGMENT *sp = pSegmentHead ;

   // delete the individual entry list elements
   while ( ep )
   {
     void *pnext = ep->pnext ;   // remember the next value in our list

     myfree( ep ) ;

     ep = (ENTRYPOINT *) pnext  ;
   }

   // delete the segment list buffer containing segment list elements
   myfree( sp ) ;
}
