# include "stdio.h"
# include "math.h"
# include "mcodef.h"



/*-----Function declarations-----------------------------------------------*/

void   main(int argc, char *argv[]);
float  rnd_uni(long *idum);
void   rnd_gauss(float par_vec[], int dim);
extern int   accept(int dim, float par_vec[], int adapt, int *finish_ptr,
             float constraint[], int *nocont);


/*-----Global variables----------------------------------------------------*/

float    huge zufa[MAXDIM][MAXVEC]; /* stores successful random numbers */
long     rnd_uni_init;              /* serves as a seed for rnd_uni()   */



/*-----Function definitions------------------------------------------------*/

float rnd_uni(long *idum)
/**C*F****************************************************************
**                                                                  **
** SRC-FUNCTION   :rnd_uni()                                        **
** LONG_NAME      :random_uniform                                   **
**                                                                  **
** AUTHOR         :Rainer Storn                                     **
**                                                                  **
** DESCRIPTION    :rnd_uni() generates an equally distributed ran-  **
**                 dom number in the interval [0,1]. For further    **
**                 reference see Press, W.H. et alii, Numerical     **
**                 Recipes in C, Cambridge University Press, 1992.  **
**                                                                  **
** FUNCTIONS      :none                                             **
**                                                                  **
** GLOBALS        :none                                             **
**                                                                  **
** PARAMETERS     :*idum    serves as a seed value                  **
**                                                                  **
** PRECONDITIONS  :*idum must be negative on the first call.        **
**                                                                  **
** POSTCONDITIONS :*idum will be changed                            **
**                                                                  **
***C*F*E*************************************************************/
{
  int j;
  long k;
  static long idum2=123456789;
  static long iy=0;
  static long iv[NTAB];
  float temp;

  if (*idum <= 0) {
    if (-(*idum) < 1) *idum=1;
    else *idum = -(*idum);
    idum2=(*idum);
    for (j=NTAB+7;j>=0;j--) {
      k=(*idum)/IQ1;
      *idum=IA1*(*idum-k*IQ1)-k*IR1;
      if (*idum < 0) *idum += IM1;
      if (j < NTAB) iv[j] = *idum;
    }
    iy=iv[0];
  }
  k=(*idum)/IQ1;
  *idum=IA1*(*idum-k*IQ1)-k*IR1;
  if (*idum < 0) *idum += IM1;
  k=idum2/IQ2;
  idum2=IA2*(idum2-k*IQ2)-k*IR2;
  if (idum2 < 0) idum2 += IM2;
  j=iy/NDIV;
  iy=iv[j]-idum2;
  iv[j] = *idum;
  if (iy < 1) iy += IMM1;
  if ((temp=AM*iy) > RNMX) return RNMX;
  else return temp;

}/*------End of rnd_uni()--------------------------*/



void rnd_gauss(float par_vec[], int dim)
/**C*F****************************************************************
**                                                                  **
** SRC-FUNCTION   :rnd_gauss()                                      **
** LONG_NAME      :random_gauss                                     **
**                                                                  **
** AUTHOR         :Rainer Storn                                     **
**                                                                  **
** DESCRIPTION    :rnd_gauss() returns a vector with dim elements   **
**                 each of which is a normally distributed number   **
**                 with zero mean and unit variance. The algorithm  **
**                 works according to the Box-Muller method which   **
**                 needs two uniform deviates as input. For further **
**                 reference see Press, W.H. et alii, Numerical     **
**                 Recipes in C, Cambridge University Press, 1992.  **
**                                                                  **
** FUNCTIONS      :rnd_uni()                                        **
**                                                                  **
** GLOBALS        :rnd_uni_init                                     **
**                                                                  **
** PARAMETERS     :par_vec[]    contains the vector with dim        **
**                              gaussian distributed variables.     **
**                 dim          number of vector elements.          **
**                                                                  **
** PRECONDITIONS  :none                                             **
**                                                                  **
** POSTCONDITIONS :Elements of par_vec[] will be altered.           **
**                                                                  **
***C*F*E*************************************************************/
{
  float c1, c2, uniform[2];
  int   i, j;


  for (j=0; j<dim; j=j+2)
  {

/*------Calculate two random numbers ex [0,1]--------------------------*/

    for (i=0; i<2; i++)
    {
      uniform[i] = rnd_uni(&rnd_uni_init);
    }

/*------Now compute normally distributed random numbers----------------*/

    c1         = sqrt(fabs(2.0*log(uniform[0]+IOTA)));
    c2         = 2.0*pi*uniform[1];
    par_vec[j] = c1*sin(c2);
    if ((j+1) < dim) par_vec[j+1] = c1*cos(c2);
  }
}


void main(int argc, char *argv[])
/**C*F****************************************************************
**                                                                  **
** SRC-FUNCTION   :main()                                           **
** LONG_NAME      :main                                             **
**                                                                  **
** AUTHOR         :Rainer Storn                                     **
**                                                                  **
** DESCRIPTION    :Stochastic parameter optimization program due to **
**                 H. Kreutzer (Kreutzer, H., Entwurfszentrierung   **
**                 zur Erhhung der Ausbeute und zur Verbesserung   **
**                 der Eigenschaften von Schaltungen, PhD disser-   **
**                 tation, University of Stuttgart, 1984.           **
**                                                                  **
** REMARKS        :A similiar procedure has been described in       **
**                 Kjellstroem, G. and Taxen, L., Stochastic otimi- **
**                 zation in system design, IEEE Trans. CAS-25,     **
**                 pp. 702 - 715, 1978.                             **
**                 The standard deviation control mechanism is an   **
**                 idea developed by R. Storn.                      **
**                                                                  **
** FUNCTIONS      :rnd_gauss(), accept(), printf()                  **
**                                                                  **
** GLOBALS        :zufa[MAXDIM][MAXVEC], rnd_uni_init               **
**                                                                  **
***C*F*E*************************************************************/
{
 FILE     *fpin_ptr;               /* points to input file                        */
 FILE     *fpout_ptr;              /* points to output file                       */

 char     c;

 int      i, j;                    /* counters for various applications.          */
 int      dim;                     /* number of parameters (dimension of par_vec[])*/
 int      iter;                    /* counts iterations.                          */
 int      mxvecs;                  /* maximum number of vectors in one iteration. */
 int      itermax;                 /* maximum number of iterations.               */
 int      nocont;                  /* number of constraints for actual problem.   */
 int      finish;                  /* indicates finish of shrinking bounds.       */
 int      hit_flag;                /* flag to indicate "hit"(1) or "no hit"(0).   */
 int      hit_count;               /* counts hits in an iteration.                */
 int      no_of_success;           /* saves amount of successful vectors.         */
 int      no_of_vecs;              /* counts generated vectors in an iteration.   */
 int      adapt;                   /* flag to decide whether requiremens shall be */
                                   /* adapted (1) or not (0).                     */
 float    par_vec[MAXDIM];         /* holds a parameter vector.                   */
 float    temp[MAXDIM];            /* temporary buffer space.                     */
 float    xnominal[MAXDIM];        /* old nominal value for parameter vector.     */
 float    xnew[MAXDIM];            /* new nominal value for parameter vector.     */
 float    sigma[MAXDIM];           /* standard deviations for random numbers.     */
 float    reduct;                  /* reduction factor for sigma[] if no hit.     */
 float    adjust;                  /* adjustment factor in case of high yields.   */
 float    max_dev[MAXDIM];         /* maximum deviation vector.                   */
 float    constraint[NARY];        /* actual constraints for current iteration.   */
 float    yield, dsig;             /* help variables.                             */

/*-----Initializations------------------------------------------------*/

 if (argc != 3)                                 /* number of arguments */
 {
    printf("\nUsage : mco <input-file> <output-file>\n");
    exit(1);
 }

 fpin_ptr   = fopen(argv[1],"r");

 if (fpin_ptr == NULL)
 {
    printf("\nCannot open input file\n");
    exit(1);                                 /* input file is necessary */
 }

 fpout_ptr = fopen(argv[2],"r");          /* Open Output file for writing */

 if ( fpout_ptr != NULL )
 {
    printf("\nOutput file %s does already exist, \ntype y if you ",argv[2]);
    printf("want to overwrite it, \nanything else if you want to exit.\n");
    c = (char)getchar();
    if ( c != 'y')
    {
      exit(1);
    }
 }
 fclose(fpout_ptr);

 rnd_uni_init = -3; /* initialization of rnd_uni() */



/*-----Read input data------------------------------------------------*/

 fscanf(fpin_ptr,"%d",&itermax);           /*---maximum number of iterations-------*/
 fscanf(fpin_ptr,"%d",&mxvecs);            /*---number of points-------------------*/
 fscanf(fpin_ptr,"%d",&dim);               /*---number of parameters---------------*/
 fscanf(fpin_ptr,"%f",&reduct);            /*---reduction factor for sigma[].------*/

 for (i=0;i<dim;i=i+1)
 {
   fscanf(fpin_ptr,"%f",&xnominal[i]);     /*---nominal value of parameter vector--*/
 }
    
 for (i=0;i<dim;i=i+1)
 {
   fscanf(fpin_ptr,"%f",&sigma[i]);        /*---standard deviations------*/
 }
    

/*-----Close input file---------------------------------------------------*/

 fclose(fpin_ptr);


/*-----Do iterations as long as the requirements are not fulfilled----------
-------or the maximum number of iterations has not been achieved yet.-----*/

 iter          = 0;         /* Set number of current iteration to 0.             */
 finish        = 0;         /* Requirements not yet fulfilled.                   */
 no_of_success = 0;         /* no hit to memorize from last iteration.           */
 adapt         = 2;         /* flag for decision to relax the requirements.      */

 while ((iter < itermax) && (finish != 1)) /* controls iterations */
 {
   iter     = iter + 1;
   hit_flag = accept(dim, xnominal, adapt, &finish, constraint, &nocont);
   adapt    = 1;                    /* tightening of requirements enabled */

/*-----Initialize arrays---------------------------------------------*/

   for (i=0; i<dim; i++)
   {
     xnew[i]    = 0.0;  /* array for new nominal vector. */
     max_dev[i] = 0.0;  /* maximum deviations array.     */
   }


   hit_count  = 0;  /*---"no hit" in ROA (region of acceptance) for current iteration.--*/
   no_of_vecs = 0;  /*---counter for random vectors for current iteration.--------------*/


/*-----Generate new random vectors as long as there are not enough-------
-------hits or the maximum number of vectors hasn't been generated.----*/

   while((no_of_vecs < mxvecs) && (hit_count < 3*dim)) /* controls usage of vectors */
   {
     if (no_of_vecs < no_of_success)       /* use successful vectors from last iteration first. */
     {
       for (i=0; i<dim; i++)
       {
         temp[i] = zufa[i][no_of_vecs];  /* reuse successful random vector.        */
       }
     }
     else
     {
       rnd_gauss(temp,dim);            /* use new random vectors.                */
     }

     for (i=0; i<dim; i++)           /* bring in standard deviation */
     {
       par_vec[i] = temp[i]*sigma[i] + xnominal[i];
     }

/*---see if random vector hits the ROA------------------------------------*/
     hit_flag = accept(dim, par_vec, 0, &finish, constraint, &nocont);

     if (hit_flag == 1)  /* if parameter-vector produced a hit */
     {
       for (i=0; i<dim; i++) /* calculate maximum deviation vector max_dev[]*/
       {
         dsig = fabs(par_vec[i] - xnominal[i]);
         if (dsig > max_dev[i])
         {
           max_dev[i] = dsig;
         }
       }

       if (hit_count < MAXVEC)            /* if there is still space in memory. */
       {
         for (j=0; j<dim; j++)      /* store succesful random vector.             */
         {
           zufa[j][hit_count] = temp[j];
         }
       }

/*-----If vector produced a hit try also the enlarged vector i.e.----------------*/
/*-----do a line search (reusage strategy).--------------------------------------*/

       for (j=2; hit_flag == 1; j++)
       {
         for (i=0; i<dim; i++)
         {
           par_vec[i] = temp[i]*(float)j*sigma[i] + xnominal[i];
         }
         hit_flag = accept(dim, par_vec, 0, &finish, constraint, &nocont);

         if (hit_flag == 1)  /* if parameter-vector produced a hit */
         {
           /* printf("Hit No.: %d\n", j); */
           for (i=0; i<dim; i++) /* calculate maximum deviation vector max_dev[]*/
           {
             dsig = fabs(par_vec[i] - xnominal[i]);
             if (dsig > max_dev[i])
             {
               max_dev[i] = dsig;
             }
           }
         }
         else /* in case of no hit */
         {
           for (i=0; i<dim; i++)  /* restore old parameter vector */
           {
             par_vec[i] = temp[i]*((float)(j-1))*sigma[i] + xnominal[i];
           }
         }
       }

/*-----Update control variables-------------------------------------------------*/

       hit_count = hit_count + 1;    /* increase counter for successful vectors.      */


       for (j=0; j<dim; j++)     /* prepare calculation for new nominal value */
       {                         /* (first sum up all values to build the     */
         xnew[j] = xnew[j] + par_vec[j];  /* mean afterwards.)                   */
/*       xnominal[j] = xnew[j]/(float)hit_count;     */ /* Kjellstroems idea, not used here */
       }
     }/* End of tasks to fulfill after having a hit */

     no_of_vecs = no_of_vecs + 1;

   }/* End of while loop */

   no_of_success   = hit_count;          /* determine number of successful points */

   yield = ((float)hit_count)/((float)no_of_vecs);

/*-The prescribed number of random vectors has been achieved. The sums----
---for the calculation of the mean value and the standard deviation-------
---will be evaluated now for the next iteration cycle.------------------*/

   if (yield > 0.0) /* change xnominal[] and sigma[] according to hits */
   {
     if (yield > 0.4)
     {
       adjust = yield*2.5;  /* increase sigma[] if yield is too high */
     }
     else
     {
       adjust = 1.0;
     }

     for (j=0; j<dim; j++)                /* finish mean value calculation */
     {
       xnew[j] = xnew[j]/((float)hit_count);
     }
     for (i=0; i<dim; i++)
     {
       xnominal[i] = xnew[i];
       sigma[i]    = max_dev[i]*adjust;
     }
   }
   else    /* leave xnominal[] and reduce sigma[] */
   {
     for (i=0; i<dim; i++)
     {
       sigma[i] = sigma[i]*reduct;
     }
   }

/*------Print result of current iteration--------------------------------*/

   printf("\n");
   printf("Iteration: %d\n", iter);
   printf("---------------\n");

   for (i=0; i<dim; i=i+1)
   {
     printf("xnominal[%d] = %f\n",i,xnominal[i]);
   }

   for (i=0; i<dim; i=i+1)
   {
     printf("sigma[%d]    = %f\n",i,sigma[i]);
   }
   for (i=0; i<nocont; i++)
   {
     printf("constraint[%d]    = %f\n", i, constraint[i]);
   }
   printf("Number of random points = %d\n",no_of_vecs);
   printf("Number of hits          = %d\n",hit_count);
   printf("Yield in percent        = %f\n",yield*100.0);

 }/* End of iteration-while-loop */

/*-----------Print final results onto output file------------------------*/

 fpout_ptr   = fopen(argv[2],"w");

 if (fpout_ptr == NULL)
 {
    printf("\nCannot open output file\n");
    exit(1);
 }

 fprintf(fpout_ptr,"\n");
 fprintf(fpout_ptr,"Iteration: %d\n", iter);
 fprintf(fpout_ptr,"---------------\n");

 for (i=0; i<dim; i=i+1)
 {
   fprintf(fpout_ptr,"xnominal[%d] = %f\n",i,xnominal[i]);
 }

 for (i=0; i<dim; i=i+1)
 {
   fprintf(fpout_ptr,"sigma[%d]    = %f\n",i,sigma[i]);
 }
 for (i=0; i<nocont; i++)
 {
   fprintf(fpout_ptr,"constraint[%d]    = %f\n", i, constraint[i]);
 }
 fprintf(fpout_ptr,"Number of random points = %d\n",no_of_vecs);
 fprintf(fpout_ptr,"Number of hits          = %d\n",hit_count);
 fprintf(fpout_ptr,"Yield in percent        = %f\n",yield*100.0);

 fclose(fpout_ptr);

}/* End of main */
