/*--------------------------------------------------------------------------*
 *                                                                          *
 * LOGNCRIS.CMD                                                             *
 *                                                                          *
 * This script will perform the login procedures to access the Internet     *
 * via CRIS (Concentric Research Internet Services) using SLIP (Serial      *
 * Line Internet Protocol).  It can probably be easily modified for other   *
 * Internet Providers.                                                      *
 *                                                                          *
 * Once SLIP is active on OS/2, all the Internet programs may be used.      *
 *                                                                          * 
 * Portions of this script were based on ANNEX.CMD, an example REXX         *
 * procedure distributed with Warp OS/2.                                    *
 *                                                                          *
 * There are two ways to use invoke this script--via SLIP.EXE or            *
 * SLIPPM.EXE.  SLIPPM.EXE is simply a GUI program which calls SLIP.EXE.    *
 * Thus, for reasons of overhead, it may be more prudent to simply invoke   *
 * SLIP.EXE as a background process and save some computer resources for    *
 * more important things like telnet and ftp... :-)                         *
 *                                                                          *
 * Calling convention (all on one line):                                    *
 *                                                                          *
 *     LOGNCRIS.CMD <phone number> <login_id> <password> <comportname>      *
 *                  <ipaddr> <destipaddr>                                   *
 *                                                                          *
 * Using this script with SLIP.EXE with a Windowed or Full Screen Session:  *
 *                                                                          *
 *      If any of the parameters are omitted, or are specified as an        *
 *      asterick (*), the script will prompt for them.  This is most useful *
 *      with the password parameter to avoid storing the password in a text *
 *      file on the disk.                                                   *
 *                                                                          *
 *      For example, the following might be used with SLIP's "-connect"     *
 *      parameter:                                                          *
 *              -connect "logncris.cmd 999-9999 loginid * COM2"             *
 *                                                                          *
 *      which would cause logncris.cmd to initially prompt for the password.*
 *      It would then feed the "999-9999" command to the modem, and when    *
 *      CRIS answered, it would use "loginid" as a username and the password*
 *      which was specified at the prompt.                                  *
 *                                                                          *
 * Using this script with SLIPPM.EXE in a GUI session:                      *
 *                                                                          *
 *      You may not omit any parameters using the asterick (*).  Doing so   *
 *      will probably lock the keyboard/mouse interface.  Make sure that    *
 *      all parameters are specified in the "Login Script" field, such as:  *
 *                                                                          *
 *              logncris.cmd 999-9999 loginid password COM2 ip_addr ip_dest *
 *                                                                          *
 *      SLIPPM will then pass this information on to SLIP.                  *
 *                                                                          *
 *            - - - - - - - - - - - - - - - - - - - - - - - - -             *
 *                                                                          *
 * When the script runs, it is automatically passed the interface name for  *
 * the interface it is running on as the first argument, followed by the    *
 * user arguments.  This script will keep trying to dial (up to 30 times)   *
 * until a successful connection is established.                            *
 *                                                                          *
 *            - - - - - - - - - - - - - - - - - - - - - - - - -             * 
 *                                                                          * 
 *                         D I S C L A I M E R                              *
 *                                                                          *  
 * Doncha just hate to have these things?  :-]                              *
 *                                                                          * 
 * I make no warranties as to the accuracy of this script or information.   *
 * Your use of this script and information is at your own risk.  I also     *
 * do not represent the views of CRIS or am affiliated with CRIS in any     *
 * other way than a subscriber to their service.  For more information on   *
 * Concentric Research Internet Services, their voice number is             *
 * 1-800-745-2747 or 1-517-895-0500.  At the writing of this script, their  *
 * service rate was $30/month for unlimited access to the Internet using    *
 * a shell account and/or SLIP.                                             *
 *                                                                          * 
 *            - - - - - - - - - - - - - - - - - - - - - - - - -             *
 *                                                                          *
 * Author: Dave Fisher, dfisher@cris.com, 1:170/110@fidonet.org)            *
 *                      40:4372/0 (ibmNET), 81:202/201 (OS2NET)             *
 * Date: 30-Nov-1994                                                        *
 *                                                                          *
 *--------------------------------------------------------------------------*/

parse arg interface, dialcmd username password comport ip_address ip_destination_address

/*--------------------------------------------------------------------------*
 *                   Initialization and Main Script Code                    *
 *--------------------------------------------------------------------------*/

/* trace results */

/*
 * Global definitions and variables
 */

call RxFuncAdd 'SysSleep', 'RexxUtil', 'SysSleep'

cr = '0d'x
lf = '0a'x

TRUE = 1
FALSE = 0

SUCCESS = 0
FAILURE = 1
FATAL_ERROR = 2

modem_input_buffer = '' /* all characters from the COM line are put */
                        /* into this common buffer */

/*
 * Default to my personal setup for CRIS -- you can change this to
 * your own information if you do not want to pass this information
 * to the script.
 */

if ( ip_address = '' ) then
    ip_address = '199.3.12.9'

if ( ip_destination_address = '' ) then
    ip_destination_address = '192.0.2.1'

if ( comport = '' ) then
    comport = 'com2'

say ''
say 'SLIP Connection Script for CRIS (using interface' interface' on 'comport':)'
say ''

/*
 * Prompt for missing information
 */

if ( dialcmd = '' ) then
    do
    call charout, 'Dial Command: '
    parse pull dialcmd
    end

if ( username = '' | username = '*' ) then
    do
    call charout, 'User Name: '
    parse pull username
    end
else
    say 'User:' username

if ( password = '' | password = '*' ) then
    do
    call charout, 'Password: '
    password = readpass()
    end

/*
 * Try to connect to the provider until we finally DO connect!
 */

call com_flush_receive(FALSE)    /* clean out COMM/input buffer */

do cnt = 1 to 30

    say ''
    say 'Connection trial number' cnt
    say ''

    status = connect_to_provider()

    if ( status == SUCCESS ) then
        leave                       /* yeah--we got in! */

    call com_flush_receive(TRUE)    /* clean out COMM/input buffer */
end

/*
 * Configure any necessary routes, etc for SLIP connection
 */

call configure_slip_connection

exit( status )


/*-------------------------------------------------------------------------*
 * connect_to_provider                                                     *
 *                                                                         *
 *-------------------------------------------------------------------------*/

connect_to_provider:

    /******** NOTE ********** NOTE *********** NOTE ********* NOTE *********
     *
     * Make sure modem is on-hook and that buffers are flushed and
     * reset modem.
     *
     * You may need to customize this for your modem make and model and
     * COM port configuration (port baud rate, etc.)
     * 
     ******** NOTE ********** NOTE *********** NOTE ********* NOTE *********/

    say 'Resetting modem...'

    /* Toggle DTR and make sure hardware handshaking is enabled */

    mode comport':,n,8,1,RTS=ON,XON=OFF,DTR=OFF'

    call SysSleep 1

    mode comport':DTR=ON'

    /* get modem's attention */

    call com_send '+++'

    call SysSleep 2

    call com_send 'ATH' || cr

    if ( com_waitfor( 'OK', 5 ) \= SUCCESS ) then
        do
        say 'Modem not responding'
        return( FAILURE )
        end

    call com_flush_receive(TRUE)

    call com_send 'ATZ' || cr

    if ( com_waitfor( 'OK', 5 ) \= SUCCESS ) then
        do
        say 'Modem not responding'
        return( FAILURE )
        end

    call com_flush_receive(TRUE)

    /*
     * Dial the remote server and wait for connection
     */

    call charout, 'Now Dialing... '

    call com_send( 'ATDT' || dialcmd || cr )

    ring_cnt = 0

    connect = FALSE

    line = ''

    do FOREVER

        status = com_get_line( 60, line )   /* get <CR> terminated line */

        if ( status \= SUCCESS ) then
            leave

        /*
         * Keep track of responses from modem
         */

        if ( (line = 'NO DIAL TONE') | (line = 'RING') ) then
            do                  /* someone trying to call in */
            call SysSleep 2     /* wait a couple of seconds before retry */
            leave
            end

        if ( (line = 'RINGING') | (line = 'BUSY') ) then
            ring_cnt = ring_cnt + 1

        if ( ring_cnt >= 4 ) then
            do
            say ''
            say 'Remote site not answering phone'
            say '' 
            leave
            end

        if ( pos( 'CONNECT', line ) > 0 ) then
            do
            connect = TRUE
            leave
            end
    end

    if ( connect = FALSE ) then
        do
        say ''
        say 'No connection established'
        say '' 
        return( FAILURE )       /* exit script with error */
        end

    /*
     * Handle login process
     */

    if ( com_waitfor( 'if new user:', 60 ) \= SUCCESS ) then
        do
        say ''
        say 'No connection established'
        say '' 
        return( FAILURE )       /* exit script with error */
        end

    call com_flush_receive(TRUE)

    call com_send( username || cr )

    if ( com_waitfor( 'Password:', 45 ) \= SUCCESS ) then
        do
        say ''
        say 'No connection established'
        say '' 
        return( FAILURE )       /* exit script with error */
        end

    call com_flush_receive(TRUE)

    call com_send( password || cr )

    if ( com_waitfor( 'SLIP software.', 15 ) \= SUCCESS ) then
        do
        say ''
        say 'No connection established'
        say '' 
        return( FAILURE )       /* exit script with error */
        end

    call com_flush_receive(TRUE)

    say ''
    say 'Connection Established to SLIP server'
    say ''

return( SUCCESS )


/*-------------------------------------------------------------------------*
 * configure_slip_connection                                               *
 *                                                                         *
 *-------------------------------------------------------------------------*/

configure_slip_connection:

    /*
     * Now configure this host for the appropriate addresses and routing
     *
     * When using the PM Dialer (SLIPPM), these statements are mandatory.
     * If you do not do it here, all of the IAK software does not
     * recognize the link.
     *
     * If you call SLIP.EXE directly using GOCRIS.CMD, then these statements
     * are probably not necessary since SLIP calls 'ifconfig' and 'route'
     * itself and consequently takes care of "clean-up" and deletes them on
     * exit.
     */

    if ( ip_address \= '' ) then
        do
        say 'Configuring IP address =' ip_address
        say '            Destination IP address =' ip_destination_address
        'ifconfig sl0' ip_address ip_destination_address 'netmask 255.255.255.0'
        'route add default' ip_destination_address '1'
        end

return( SUCCESS )

/*--------------------------------------------------------------------------*
 * com_send                                                                 *
 *                                                                          *
 * Routine to send a character string off to the modem.                     *
 *                                                                          *
 *--------------------------------------------------------------------------*/

com_send:

    sendstring = arg(1)

    call slip_com_output interface, sendstring

return( SUCCESS )


/*-------------------------------------------------------------------------*
 * com_get_line                                                            *
 *                                                                         * 
 * Get a CR terminated line from the COM port                              *
 *                                                                         * 
 *     Arg 1 : number of seconds to wait until giving up (timeout)         *
 *     Arg 2 : returned line (without the CR)                              *
 *                                                                         *
 *-------------------------------------------------------------------------*/

com_get_line:

    timeout = arg(1)
    modem_input_line = arg(2)

    if timeout = '' then
        timeout = 5000      /* long delay if not specified */

    modem_input_line = ''

    loop_status = -1

    reset_time = time('R')  /* reset the clock timer to zero */

    do while( loop_status = -1 )
        /*
         * Stay in loop until we get a CR terminated line, or we
         * timeout
         */

        status = com_get_input()

        if ( status = FATAL_ERROR ) then
            return(FAILURE)

        if ( status = SUCCESS ) then
            do
            /* extract line from buffer if there */

            status = extract_line( 'modem_input_line' )

            if ( status = SUCCESS ) then
                do
                loop_status = SUCCESS
                leave       /* get out of while loop */
                end
            end

        if ( time('E') >= timeout ) then
            do
            say 'Timeout waiting for line from modem'
            loop_status = FAILURE
            end

        end     /* while loop_status... */

return( loop_status )

/*--------------------------------------------------------------------------*
 * com_waitfor                                                              *
 *                                                                          *
 * Waits for the supplied string to show up in the COM input.               *
 *                                                                          *
 *     Arg 1 : number of seconds to wait until giving up (timeout)          *
 *                                                                          * 
 * Return: FAILURE or SUCCESS                                               *
 *                                                                          *
 *--------------------------------------------------------------------------*/

com_waitfor:

    parse arg waitfor_string, timeout

    if timeout = '' then
        timeout = 5000      /* long delay if not specified */

    reset_time = time('R')  /* reset the clock timer to zero */

    exit_status = FAILURE

    do FOREVER
        /*
         * Stay in loop until matched string or timeout
         */

        status = com_get_input()

        if ( status = FATAL_ERROR ) then
            return(FAILURE)

        if ( status = SUCCESS ) then
            do
            /* if we have a match, extract it from input COM buffer */

            status = extract_match( waitfor_string )

            if ( status = SUCCESS ) then
                do
                exit_status = SUCCESS
                leave       /* get out of while loop */
                end
            end

        if ( time('E') >= timeout ) then
            do
            say 'Timed out waiting for string "'waitfor_string'"'
            leave   /* get out of while loop */
            end

        end     /* while loop_status... */

return( exit_status )


/*--------------------------------------------------------------------------*
 * readpass                                                                 *
 *                                                                          *
 * Routine used to read a password from the user without echoing the        *
 * password to the screen.                                                  *
 *                                                                          * 
 * Return: string entered                                                   *
 *                                                                          *
 *--------------------------------------------------------------------------*/

readpass:

    answer = ''

    do until( key = cr )
        key = slip_getch()

        if ( key \= cr ) then
            do
            answer = answer || key
            end
    end

    say ''

return( answer )


/*--------------------------------------------------------------------------*
 * com_flush_receive                                                        *
 *                                                                          *
 * Routine to flush any pending characters to be read from the COM port.    *
 * Reads everything it can until nothing new shows up for 100ms, at which   *
 * point it returns.                                                        *
 *                                                                          *
 *   Arg 1 : TRUE=echo to output, FALSE=no echo                             *
 *                                                                          * 
 *--------------------------------------------------------------------------*/

com_flush_receive:

    echo = arg(1)

    modem_input_buffer = ''     /* this has already been echo'ed on read */

    /*
     * Eat anything left in the modem or COM buffers and stop when
     * nothing new appears for 100ms
     */

    do until( line = '' )
        line = slip_com_input( interface, , 100 )
        if ( echo = TRUE) then
            call charout, line
        end

return( SUCCESS )


/*--------------------------------------------------------------------------*
 * display_mem                                                              *
 *                                                                          *
 * Bascially for debugging, display characters to output.  If characters    *
 * are unprintable, display as hexidecimal values.                          *
 *                                                                          *
 *   Arg 1 : memory segment to display                                      *
 *   Arg 2 : leading prompt before display                                  *
 *                                                                          * 
 *--------------------------------------------------------------------------*/

display_mem:

    mem = arg(1)
    prompt = arg(2)

    do curpos = 1 to length(prompt)
        call charout, substr( prompt, curpos, 1 )
        end

    call charout, '"'

    do curpos = 1 to length(mem)

        ch = substr( mem, curpos, 1 )

        if ( (c2d(ch) < 32) | (c2d(ch) > 126) ) then
            call charout, '(0x'c2x(ch)')'
        else
            call charout, ch

        end

    say '"'

return( SUCCESS )


/*--------------------------------------------------------------------------*
 * extract_line                                                             *
 *                                                                          *
 * Checks 'modem_input_bufffer' for a CR terminated line.  If found, the    *
 * line will be extracted from the buffer.  If not, no change to buffer.    *
 *                                                                          *
 *   Arg 1 : returned CR terminated line (without the CR)                   *
 *                                                                          * 
 *--------------------------------------------------------------------------*/

extract_line:

    line_var_name = arg(1)          /* passed by reference */

    buff = modem_input_buffer       /* use temp variable so we do not */
                                    /* modify the input buffer until we want to */

    x = VALUE( line_var_name, '' )  /* initialize passed variable to nothing */
                                    /* discard return to unused var 'x' */
    line = ''

    got_cr = FALSE

    do buffpos = 1 to length(buff)

        ch = substr( buff, buffpos, 1 )

        if ( ch = lf ) then
            nop             /* ignore line feeds */
        else
        if ( ch = cr ) then
            do
            /* x = display_mem( buff, 'before buff = ' ) */

            buff = substr( buff, buffpos+1, length(buff) - buffpos )

            /* x = display_mem( buff, 'after buff = ' ) */

            got_cr = TRUE
            leave
            end
        else
            do
            line = line || ch
            end
        end

    if ( got_cr = FALSE ) then
        return( FAILURE )

    modem_input_buffer = buff           /* update input buffer */

    x = VALUE( line_var_name, line )    /* return new value of extracted line */

return( SUCCESS )


/*--------------------------------------------------------------------------*
 * extract_match                                                            *
 *                                                                          *
 * Checks 'modem_input_bufffer' for a matching string.  If found, the       *
 * match will be extracted from the buffer.  If not, no change to buffer.   *
 *                                                                          *
 *   Arg 1 : pattern to match in input buffer                               *
 *                                                                          * 
 *--------------------------------------------------------------------------*/

extract_match:

    matchstr = arg(1)

    index = pos( matchstr, modem_input_buffer )

    if ( index > 0 ) then
        do
        /*
         * Extract matched string from the COM input buffer and reset
         * the input buffer to any remaining characters
         */

        index = index + length(matchstr)
        len = length(modem_input_buffer) - index + 1

        /* x = display_mem( modem_input_buff, 'before buff = ' ) */

        if ( len = 0 ) then
            modem_input_buffer = ''
        else
            modem_input_buffer = substr( modem_input_buffer, index, len )

        /* x = display_mem( modem_input_buffer, 'after buff = ' ) */

        return(SUCCESS)
        end

return(FAILURE)


/*--------------------------------------------------------------------------*
 * com_get_input                                                            *
 *                                                                          *
 * Get any input from the COM port and store in 'modem_input_buffer'.  Echo *
 * input received to the output device.                                     *
 *                                                                          *
 *--------------------------------------------------------------------------*/

com_get_input:

    line = slip_com_input( interface, , 10 )

    if ( line \= '' ) then
        do
        call charout, line

        modem_input_buffer = modem_input_buffer || line

        /*
         * Check input stream for "garbage characters".  This would
         * indicate that something is wrong with the connection (parity,
         * baud rate, remote site failure, etc.)
         */

        do curpos = 1 to length(line)
            ch = substr( line, curpos, 1 )

            if ( c2d(ch) > 126 ) then
                do
                say ''
                say 'Garbage characters detected in stream'
                say ''
                modem_input_buffer = ''
                return( FATAL_ERROR )
                end
            end

        return( SUCCESS )
        end

return(FAILURE)
