/* tty.c -- Tty mode settings
 *
 * Copyright (c) 1994 by Ezra Story.  All rights reserved.  Explicit
 * permission is given to use this source for any purpose as long as
 * the author (Ezra Story) is notified and this copyright notice is
 * included.
 *
 */
static char rcsid[] = "$Id: tty.c 1.8 1995/04/02 11:59:17 Ezra_Story Exp $";

#define INC_SYS
#define INC_IOCTL
#define INC_TTY

#include "defs.h"

export VOID TtyRaw P((int, int));
export VOID TtyRestore P((VOID));
export VOID TtyResize P((int, int, int));

#ifndef FREAD
#define FREAD 0x0001
#endif
#ifndef FWRITE
#define FWRITE 0x0002
#endif

local int   operm;
local int   inout = FREAD|FWRITE;

#ifndef CF_TERMIO

local struct sgttyb  ostty,    nstty;
local struct tchars  otchars,  ntchars;
local struct ltchars oltchars, nltchars;
local int            olmode,   nlmode;
local int            oldisc,   nldisc;

VOID
TtyRaw(min, time)
int min;
int time;
{

    struct stat st;

    if (fstat(0, &st) == 0)
        operm = st.st_mode & 06777;
    else
        operm = -1;

    /* grab the current settings */
    ioctl(0, (int)TIOCGETP, (char *)&ostty);
    ioctl(0, (int)TIOCGETC, (char *)&otchars);
    ioctl(0, (int)TIOCGLTC, (char *)&oltchars);
    ioctl(0, (int)TIOCLGET, (char *)&olmode);
    ioctl(0, (int)TIOCGETD, (char *)&oldisc);
    nstty              = ostty;
    ntchars            = otchars;
    nltchars           = oltchars;
    nlmode             = olmode;
    nldisc             = oldisc;

    /* modify them */
    nstty.sg_flags  = RAW;
    nstty.sg_erase     = nstty.sg_kill     = -1;
    ntchars.t_intrc    = ntchars.t_quitc   = -1;
    ntchars.t_eofc     = ntchars.t_brkc    = -1;
    nltchars.t_suspc   = nltchars.t_dsuspc = -1;
    nltchars.t_rprntc  = nltchars.t_flushc = -1;
    nltchars.t_werasc  = nltchars.t_lnextc = -1;

    /* set the tty to the new settings */
    if (operm != -1 && fchmod(0, S_IREAD|S_IWRITE) < 0) operm = -1;
    ioctl(0, (int)TIOCSETP, (char *)&nstty);
    ioctl(0, (int)TIOCSETC, (char *)&ntchars);
    ioctl(0, (int)TIOCSLTC, (char *)&nltchars);
    ioctl(0, (int)TIOCLSET, (char *)&nlmode);
    ioctl(0, (int)TIOCSETD, (char *)&nldisc);

}


VOID
TtyRestore(VOID)
{
    /* restore our old tty settings */
    (void)ioctl(0, (int)TIOCFLUSH, (char *)&inout);
    (void)ioctl(0, (int)TIOCSETP, (char *)&ostty);
    (void)ioctl(0, (int)TIOCSETC, (char *)&otchars);
    (void)ioctl(0, (int)TIOCSLTC, (char *)&oltchars);
    (void)ioctl(0, (int)TIOCLSET, (char *)&olmode);
    (void)ioctl(0, (int)TIOCSETD, (char *)&oldisc);
    if (operm != -1)
            (void)fchmod(0, operm);
}

#else  /* termio */

local struct termios otermio;
local struct termios ntermio;

#ifndef TCATTR
#ifndef TCSETS
#define TCSETS TCSANOW
#endif
#ifndef TCGETS
#define TCGETS TCGETP
#endif
#else
#ifndef TCSANOW
#define TCSANOW TCSETS
#endif
#endif

VOID
TtyRaw(min, time)
int min;
int time;
{
    struct stat st;

    if (fstat(0, &st) == 0)
        operm = st.st_mode & 06777;
    else
        operm = -1;

    /* get cuurent tty settings */
#ifdef CF_TCATTR
    tcgetattr(0,&otermio);
#else
    ioctl(0,TCGETS,&otermio);
#endif
    ntermio = otermio;

    /* modify */
    ntermio.c_iflag = 0;
    ntermio.c_oflag = 0;
    ntermio.c_cflag &= ~CSIZE;
    ntermio.c_cflag |= CS8;
    ntermio.c_lflag = 0;
    ntermio.c_cc[VMIN] = min;
    ntermio.c_cc[VTIME] = time;

    /* set the new settings */
    if (operm != -1 && fchmod(0, S_IREAD|S_IWRITE) < 0) operm = -1;
#ifdef CF_TCATTR
    tcsetattr(0,TCSANOW, &ntermio);
#else
    ioctl(0,TCSETS, &ntermio);
#endif

}

VOID
TtyRestore(VOID)
{
    /* restore old settings */
#ifdef CF_TCATTR
    tcsetattr(0,TCSANOW, &otermio);
#else
    ioctl(0,TCSETS,&otermio);
#endif
    if (operm != -1)
            (void)fchmod(0, operm);
}

#endif /* termio */

VOID
TtyResize(fd,x,y)
int fd,x,y;
{
#ifdef TIOCSWINSZ
    /* set window size on pty (4.3BSD) */
    struct winsize ws;
    ws.ws_row = y;
    ws.ws_col = x;
    ws.ws_xpixel = x*8;  /* doesn't mean anything in our case */
    ws.ws_ypixel = y*8;  /* so I choose a reasonable default  */
    (void)ioctl(fd, (int)TIOCSWINSZ,(char *)&ws);
#else
#ifdef TIOCSSIZE
    /* set window size on pty (Sun) */
    struct ttysize ts;
    ts.ts_lines = y;
    ts.ts_cols = x;
    (void)ioctl(fd, (int)TIOCSSIZE,(char *)&ts);
#endif
#endif
}


