/* sockcmds.c -- mlink socket commands
 *
 * Copyright (c) 1994 by Ezra Story.  All rights reserved.  Explicit
 * permission is given to use this source for any purpose as long as
 * the author (Ezra Story) is notified and this copyright notice is
 * included.
 *
 */
static char rcsid[] = "$Id: sockcmds.c 1.38 1995/04/13 02:08:20 Ezra_Story Exp $";

#define INC_SYS
#define INC_ERRNO
#define INC_STRING
#define INC_SOCKET
#define INC_IOCTL

#include "defs.h"

export  VOID    s_NewSocket P((ubyte, ubyte, ubyte));
export  VOID    s_Close P((ubyte));
export  VOID    s_Shutdown P((ubyte, ubyte));
export  VOID    s_Bind P((ubyte, uword));
export  VOID    s_Connect P((ubyte, uword, ulong));
export  VOID    s_HostByAddr P((ubyte, ulong));
export  VOID    s_HostByName P((ubyte, ubyte *));
export  VOID    s_ServByName P((ubyte, ubyte, ubyte *));
export  VOID    s_ServByPort P((ubyte, ubyte, uword));
export  VOID    s_Listen P((ubyte));
export  VOID    s_Accept P((ubyte, ubyte));
export  VOID    s_SendTo P((ubyte, uword, ulong));
export  VOID    s_Priority P((ubyte, ubyte));
export  VOID    s_OOBData P((ubyte, ubyte));
export  VOID    s_SetSockOpt P((ubyte, ubyte, ubyte));
export  VOID    s_Block P((ubyte, ubyte));
export  int     InitSockets P((VOID));

export mlsocket *sockArray[MAXSOCKS+1];
export list     sockList;
export ubyte    *tempbuf;
export int      outSock;
export int      inSock;


local char tcp_prot[] = "tcp";
local char udp_prot[] = "udp";
local struct linger sock_linger = {1, 10};
local int  ioctl_yes = 1;
local int  ioctl_no = 0;

int
InitSockets(VOID)
{
    int x;

    NewList(&sockList);
    for (x=0; x<MAXSOCKS+1; x++) sockArray[x] = 0;
    tempbuf = (ubyte *)malloc(10000);
    outSock = inSock = MAXSOCKS;
    return(1);
}
/*
 * CMD_NEWSOCKET: create a new socket.  Note, that we
 * do not check the descriptor for validity, we assume
 * the client is not overwriting something we already have!
 */
VOID
s_NewSocket(desc, domain, type)
ubyte desc, domain, type;
{
    mlsocket *ms;

    /* Translate constants */
    if (type == 1)
        type = SOCK_STREAM;
    else
        type = SOCK_DGRAM;

    /* alloc a new socket structure */
    ms = (mlsocket *)calloc(sizeof(mlsocket),1);
    if (ms == 0)
        {
        ReturnError(CMD_NEWSOCKET, desc, ENOBUFS);
        return;
        }
    ms->rdbuf = CreateFifo(2000);
    if (ms->rdbuf == 0)
        {
        free(ms);
        ReturnError(CMD_NEWSOCKET, desc, ENOBUFS);
        return;
        }
    if (type == SOCK_DGRAM)
        {
        ms->dgbuf = CreateFifo(2000);
        if (ms->dgbuf == 0)
            {
            DeleteFifo(ms->rdbuf);
            free(ms);
            ReturnError(CMD_NEWSOCKET, desc, ENOBUFS);
            return;
            }
        }
    else
        ms->dgbuf = 0;

    if (domain == 1)
        ms->state = SK_ISPTY;
    else
        ms->state = 0;
    domain = AF_INET;

    /* open a new socket */
    ms->fd = socket(domain, type, 0);
    if (ms->fd < 0)
        {
        if (ms->dgbuf) DeleteFifo(ms->dgbuf);
        DeleteFifo(ms->rdbuf);
        free(ms);
        ReturnError(CMD_NEWSOCKET, desc, errno);
        return;
        }
    NONBLOCK(ms->fd);
    setsockopt(ms->fd, SOL_SOCKET, SO_LINGER, &sock_linger, sizeof(struct linger));
    setsockopt(ms->fd, SOL_SOCKET, SO_REUSEADDR, &ioctl_yes, sizeof(int));
    ms->domain = domain;
    ms->type = type;
    ms->state |= SK_OPEN;
    ms->desc = desc;

    /* attach to global lists */
    sockArray[desc] = ms;
    ms->n.Pri = 0;
    ms->quant = cpsrate > 480 ? 132 : 64;
    Enqueue(&sockList, &ms->n);

    /* set fd bits initially ONLY for datagram sockets */
    FD_CLR(ms->fd, &sm[0].rd);
    FD_CLR(ms->fd, &sm[0].wr);
    FD_CLR(ms->fd, &sm[0].ex);
    if (type == SOCK_DGRAM)
        {
        FD_SET(ms->fd, &sm[0].rd);
        FD_SET(ms->fd, &sm[0].ex); /* does this need to be set? */
        }

    ReturnSuccess(CMD_NEWSOCKET, desc, 0, (ubyte *)0);
}

/*
 * CMD_CLOSE: terminate a socket.
 */
VOID
s_Close(desc)
ubyte desc;
{
    mlsocket *ms;

    ms = sockArray[desc];

    /* there still data to write, defer close */
    ms->state |= SK_CLOSING;
    if (AmountInFifo(ms->rdbuf, FIFO_ALL)) return;

    /* reset stream variables */
    if (inSock == desc) inSock = MAXSOCKS;
    if (outSock == desc) outSock = MAXSOCKS;

    /* remove, clr fd_set's, close and deallocate */
    sockArray[desc] = (mlsocket *)0;
    Remove(&ms->n);
    FD_CLR(ms->fd, &sm[0].rd);
    FD_CLR(ms->fd, &sm[0].wr);
    FD_CLR(ms->fd, &sm[0].ex);
    close(ms->fd);
    if (ms->dgbuf) DeleteFifo(ms->dgbuf);
    DeleteFifo(ms->rdbuf);
    free(ms);

    ReturnSuccess(CMD_CLOSE, desc, 0, (ubyte *)0);
}

/*
 * CMD_SHUTDOWN: shutdown part of an IO channel
 */
VOID
s_Shutdown(desc, how)
ubyte desc, how;
{
    shutdown(sockArray[desc]->fd, how);
    ReturnSuccess(CMD_SHUTDOWN, desc, 0, (ubyte *)0);
}

/*
 * CMD_BIND: bind ip port to socket
 */
VOID
s_Bind(desc, port)
ubyte desc;
uword port;
{
    mlsocket        *ms;
    int             nmlen;
    uword           op;
    ubyte           out[2];
    struct sockaddr_in nmin;

    ms = sockArray[desc];

    /* inet, we call bind, and return the address
     * it gave us.
     */

    bzero((char *)&nmin, sizeof(nmin));
    nmin.sin_family = AF_INET;
    nmin.sin_port   = htons(port);
    nmin.sin_addr.s_addr = INADDR_ANY;
    nmlen = sizeof(struct sockaddr_in);
#ifdef BSD44
    nmin.sin_len = nmlen;
#endif

    if (bind(ms->fd, (struct sockaddr *)&nmin, nmlen) < 0)
        ReturnError(CMD_BIND, desc, errno);
    else
        {
        getsockname(ms->fd, (struct sockaddr *)&nmin, &nmlen);
        op = ntohs(nmin.sin_port);
        out[0] = (op >> 8) & 0xff;
        out[1] = op & 0xff;
        ms->state |= SK_BOUND;
        ReturnSuccess(CMD_BIND, desc, 2, out);
        }
}

/*
 * CMD_CONNECT: connect to a server socket
 */
VOID
s_Connect(desc, port, addr)
ubyte desc;
uword port;
ulong addr;
{
    mlsocket           *ms;
    int                nmlen;
    ubyte              out[2];
    uword              op;
    struct sockaddr_in nmin;
    int                connectin;
    struct timeval     tv;

    ms = sockArray[desc];
    ms->port = port;
    ms->addr = addr;
    bzero((char *)&nmin, sizeof(nmin));
    Gettimeofday(&tv);

    /* test timeout */
    if (ms->state & SK_CONNECTING)
        {
        if (ms->lsttry != tv.tv_sec)
            {
            ms->lsttry = tv.tv_sec;
            ms->pings++;
            if (ms->pings > 30)
                {
                ms->state &= ~SK_CONNECTING;
                ReturnError(CMD_CONNECT, desc, ETIMEDOUT);
                return;
                }
            }
        else
            return;
        }
    else
        {
        ms->lsttry = tv.tv_sec;
        ms->pings = 0;
        }

    /* redirect rlogin anyhost to shell.  We fork a shell in
     * a pty and point the socket descriptor at it.
     */
    if ((ms->domain == AF_INET && (port == 513 || port == 514)) ||
       (ms->state & SK_ISPTY))
        {
        connectin = m_forkpty();
        if (connectin < 0)
            {
            errno = EINVAL;
            connectin = -1;
            }
        else
            {
            dup2(connectin, ms->fd);
            close(connectin);
            NONBLOCK(ms->fd);
            ms->domain = AF_UNIX;
            ms->state |= SK_ISPTY;
            port = 1;
            }
        }
    else if (ms->domain == AF_INET)
        {
        nmin.sin_family = AF_INET;
        nmin.sin_port = htons(port);
        nmin.sin_addr.s_addr = htonl(addr);
        nmlen = sizeof(struct sockaddr_in);
#ifdef BSD44
        nmin.sin_len = nmlen;
#endif
        connectin = connect(ms->fd, (struct sockaddr *)&nmin, nmlen);
        }

    /* do connect */
    if (connectin < 0)
        {
        if ((errno == EALREADY) || (errno == EINPROGRESS) || (errno == EWOULDBLOCK))
            {
            ms->state |= SK_CONNECTING;
            return;
            }
        if (errno != EISCONN)
            {
            ms->state &= ~SK_CONNECTING;
            ReturnError(CMD_CONNECT, desc, errno);
            return;
            }
        }

    /* success, reunblock socket, return def binding */
    ms->state &= ~SK_CONNECTING;
    getsockname(ms->fd, (struct sockaddr *)&nmin, &nmlen);
    op = ntohs(nmin.sin_port);
    out[0] = (op >> 8) & 0xff;
    out[1] = op & 0xff;
    ms->state |= SK_CONNECTED;
    ReturnSuccess(CMD_CONNECT, desc, 2, out);

    /* set masks for active socket */
    FD_SET(ms->fd, &sm[0].rd);
    FD_SET(ms->fd, &sm[0].ex);

    /* rlogin hack */
    if (ms->state & SK_ISPTY)
        {
        out[0] = 0x80;
        SendCommand(CMD_OOBDATA, ms->desc, 1, out);
        }

}

/*
 * CMD_HBYADDR: return hostname associated with inet addr
 */
VOID
s_HostByAddr(desc, addr)
ubyte desc;
ulong addr;
{
    ulong iaddr;
    struct hostent *he;
    char *hostn;

    iaddr = htonl(addr);

    if ((he = gethostbyaddr((char *)&iaddr, 4, AF_INET))==0)
        ReturnError(CMD_HBYADDR, desc, ENOENT);
    else
        {
        hostn = he->h_name;
        if (hostn == 0)
            ReturnError(CMD_HBYADDR, desc, ENOBUFS);
        else
            ReturnSuccess(CMD_HBYADDR, desc, strlen(hostn)+1, hostn);
        }

}

/*
 * CMD_HBYNAME: return inet addr associated with hostname
 */
VOID
s_HostByName(desc, name)
ubyte desc;
ubyte *name;
{
    struct hostent *he;
    ulong iaddr;
    ubyte out[4];

    if ((he = gethostbyname(name))==0)
        ReturnError(CMD_HBYNAME, desc, ENOENT);
    else
        {
        iaddr = ntohl( *((ulong *)he->h_addr) );
        out[0] = (iaddr >> 24) & 0xff;
        out[1] = (iaddr >> 16) & 0xff;
        out[2] = (iaddr >>  8) & 0xff;
        out[3] = iaddr & 0xff;
        ReturnSuccess(CMD_HBYNAME, desc, 4, out);
        }


}

/*
 * CMD_SBYNAME: return port associated with service name/protocol
 */
VOID
s_ServByName(desc, prot, name)
ubyte desc, prot;
ubyte *name;
{
    struct servent *sve;
    char *protocol;
    ubyte out[2];
    uword port;

    /* IPPROTO_TCP or UDP */
    if (prot == 6)
        protocol = tcp_prot;
    else
        protocol = udp_prot;

    if ((sve = getservbyname(name, protocol))==0)
        ReturnError(CMD_SBYNAME, desc, ENOENT);
    else
        {
        port = ntohs( sve->s_port );
        out[0] = (port >> 8) & 0xff;
        out[1] = port & 0xff;
        ReturnSuccess(CMD_SBYNAME, desc, 2, out);
        }
}

/*
 * CMD_SBYPORT: return name associated with service port
 */
VOID
s_ServByPort(desc, prot, port)
ubyte desc,prot;
uword port;
{
    struct servent *sve;
    char *protocol;

    /* translate from remote constant */
    if (prot == 6)
        protocol = tcp_prot;
    else
        protocol = udp_prot;

    if ((sve = getservbyport(port, protocol))==0)
        ReturnError(CMD_SBYNAME, desc, ENOENT);
    else
        ReturnSuccess(CMD_SBYNAME, desc, strlen(sve->s_name)+1, sve->s_name);
}

/*
 * CMD_LISTEN: set socket to listen for connections
 */
VOID
s_Listen(desc)
ubyte desc;
{
    mlsocket *ms;

    ms = sockArray[desc];

    if (listen(ms->fd, 5))
        ReturnError(CMD_LISTEN, desc, errno);
    else
        {
        ms->state |= SK_LISTENING;
        ReturnSuccess(CMD_LISTEN, desc, 0, (ubyte *)0);
        FD_SET(ms->fd, &sm[0].rd);
        }
}

/*
 * CMD_ACCEPT: accept an incoming connection on a socket
 */
VOID
s_Accept(desc, newdesc)
ubyte desc, newdesc;
{
    mlsocket *ms;
    struct sockaddr_in nmin;
    int             nmlen;
    int             nfd;
    int             domain;
    uword           op;
    ulong           iaddr;
    ubyte           out[10];


    /* attempt to accept */
    ms = sockArray[desc];
    nmlen = sizeof(struct sockaddr_in);

    if (ms->state & SK_LISTENING)
        {
        ms->state &= ~SK_KNOCKED;
        FD_SET(ms->fd, &sm[0].rd);
        FD_CLR(ms->fd, &sm[1].rd);   /* clear out set bit */
        }

    if ((nfd = accept(ms->fd, (struct sockaddr *)&nmin, &nmlen)) < 0)
        ReturnError(CMD_ACCEPT, desc, errno);
    else
        {
        /* reset server socket */
        domain = ms->domain;

        /* create new socket */
        ms = (mlsocket *)calloc(sizeof(mlsocket),1);
        if (ms == 0)
            {
            ReturnError(CMD_ACCEPT, desc, ENOBUFS);
            return;
            }
        ms->rdbuf = CreateFifo(2000);
        if (ms->rdbuf == 0)
            {
            free(ms);
            ReturnError(CMD_ACCEPT, desc, ENOBUFS);
            return;
            }
        ms->dgbuf = 0;

        ms->fd = nfd;
        NONBLOCK(ms->fd);
        setsockopt(ms->fd, SOL_SOCKET, SO_LINGER, &sock_linger, sizeof(struct linger));
        setsockopt(ms->fd, SOL_SOCKET, SO_REUSEADDR, &ioctl_yes, sizeof(int));
        ms->domain = domain;
        ms->type = SOCK_STREAM;
        ms->state = SK_OPEN|SK_CONNECTED|SK_BOUND;
        ms->desc = newdesc;

        sockArray[newdesc] = ms;
        ms->n.Pri = 0;
        ms->quant = cpsrate > 480 ? 132: 64;
        Enqueue(&sockList, &ms->n);
        FD_SET(ms->fd, &sm[0].rd);
        FD_SET(ms->fd, &sm[0].ex);

        op = ntohs(nmin.sin_port);
        out[0] = op / 0x0100;
        out[1] = op % 0x0100;
        iaddr = ms->addr = ntohl(nmin.sin_addr.s_addr);
        out[2] = iaddr / 0x01000000;
        out[3] = (iaddr % 0x01000000) / 0x010000;
        out[4] = (iaddr % 0x010000)   / 0x0100;
        out[5] = (iaddr % 0x0100);
        getsockname(nfd, (struct sockaddr *)&nmin, &nmlen);
        op = ms->port = ntohs(nmin.sin_port);
        out[6] = op / 0x0100;
        out[7] = op % 0x0100;
        ReturnSuccess(CMD_ACCEPT, desc, 8, out);
        }

}


/*
 * CMD_PRIORITY: set socket priority
 */
VOID
s_Priority(desc, pri)
ubyte desc, pri;
{
    mlsocket *ms;

    ms = sockArray[desc];

    Remove(&ms->n);
    ms->n.Pri = ((int)pri)-128;
    ms->quant = cpsrate > 480 ? 132 : 64;
    Enqueue(&sockList, &ms->n);

    ReturnSuccess(CMD_PRIORITY, desc, 0, (ubyte *)0);

}

/*
 * CMD_OOBDATA: send out of band data byte.  Yeah, there's
 * absolutely NO error checking, but 99.99% of the time, this'll
 * make it through.
 */
VOID
s_OOBData(desc, bt)
ubyte desc, bt;
{
    send(sockArray[desc]->fd, &bt, 1, MSG_OOB);
}

/*
 * CMD_SETSOCKOPT: set socket options.
 */
VOID
s_SetSockOpt(desc, option, value)
ubyte desc, option, value;
{
    int opt;

    /*
     * We don't do any link --> local constant mapping, cause
     * it would be a real pain in the ass. :-)
     */

    opt = value;
    if (setsockopt(sockArray[desc]->fd, SOL_SOCKET, option, &opt, sizeof(opt)))
        {
        ReturnError(CMD_SETSOCKOPT, desc, errno);
        }
    else
        {
        ReturnSuccess(CMD_SETSOCKOPT, desc, 0, (ubyte *)0);
        }
}


/*
 * CMD_BLOCK: client blocks writes
 */
VOID
s_Block(desc, yesno)
ubyte desc, yesno;
{
    int x;

    x = sockArray[desc]->state;
    x = yesno ? x | SK_BLOCKED : x & ~SK_BLOCKED;
    sockArray[desc]->state = x;
    ReturnSuccess(CMD_BLOCK, desc, 0, (ubyte *)0);

}

