/* sockchk.c -- interface from mlink to/from sockets
 *
 * Copyright (c) 1994 by Ezra Story.  All rights reserved.  Explicit
 * permission is given to use this source for any purpose as long as
 * the author (Ezra Story) is notified and this copyright notice is
 * included.
 *
 */
static char rcsid[] = "$Id: sockchk.c 1.26 1995/04/22 19:39:33 Ezra_Story Exp $";

#define INC_SYS
#define INC_ERRNO
#define INC_SOCKET

#include "defs.h"

export  VOID     CheckSocketRd P((VOID));
export  VOID     CheckSocketWr P((VOID));
export  VOID     CheckSocketEx P((VOID));
export  VOID     WriteSock P((mlsocket *, ubyte *, int));

local   VOID     PtyWrite P((int, ubyte *, int));

/*
 * Check for socket data out.  This is the priority loop.
 */
VOID
CheckSocketRd(VOID)
{
    mlsocket *ms,*ms2;
    struct sockaddr_in nmin;
    int nmlen, num;

    ms = (mlsocket *)&sockList;

    for (ms = (mlsocket *)sockList.Head; ms2 = (mlsocket *)ms->n.Succ; ms = ms2)
        {
        /* if we're blocked by the other side, we have to give up
         * our slice.
         */
        if (ms->state & SK_BLOCKED) continue;

        if (ms->state & SK_CONNECTING)
            {
            s_Connect(ms->desc, ms->port, ms->addr);
            continue;
            }

        if (FD_ISSET(ms->fd, &sm[1].rd))
            {
            if (ms->type == SOCK_STREAM)
                {
                /* if we've got a pending accept(), and we haven't told the
                 * client yet, do so now.
                 */
                if ((ms->state & SK_LISTENING) && !(ms->state & SK_KNOCKED))
                    {
                    SendCommand(CMD_KNOCK, ms->desc, 0, (ubyte *)0);
                    ms->state |= SK_KNOCKED;
                    FD_CLR(ms->fd, &sm[0].rd);
                    continue;
                    }

                if (ms->state & SK_CONNECTED)
                    {
                    /* otherwise, we're getting data.  We grab our quantum
                    * and send on its way.  If we get an error,
                    * we tell the host that we've disconnected.
                    */
                    if (ms->state & SK_ISPTY)
                        {
                        /* pty's aren't sockets
                         */
                        num = read(ms->fd, tempbuf, ms->quant);
                        if (num < 0) num = 0;
                        }
                    else
                        num = recv(ms->fd, tempbuf, ms->quant, 0);

                    if (num > 0)
                        WriteData(ms->desc, tempbuf, num);
                    else if (num == 0)
                        {
                        SendCommand(CMD_DISCONNECT,ms->desc, 0, (ubyte *)0);
                        ms->state &= ~SK_CONNECTED;
                        FD_CLR(ms->fd, &sm[0].rd);
                        }
                    }
                }
            else
                {
                /* datagram socket.  If we've emptyed our previous
                 * datagram, grab this new one.  Otherwise, let it
                 * sit on the port a while.
                 */
                if (!AmountInFifo(ms->dgbuf, FIFO_ALL))
                    {
                    num = recvfrom(ms->fd, tempbuf, 4000, 0, (struct sockaddr *)&nmin, &nmlen);
                    if (num > 0)
                        {
                        ulong hport, haddr;
                        hport = ntohs(nmin.sin_port);
                        haddr = ntohl(nmin.sin_addr.s_addr);
                        WriteFifo(ms->dgbuf, tempbuf, num, FIFO_STRM);
                        tempbuf[0] = (num >> 8) & 0xff;
                        tempbuf[1] = num & 0xff;
                        tempbuf[2] = (hport >> 8) & 0xff;
                        tempbuf[3] = hport & 0xff;
                        tempbuf[4] = (haddr >> 24) & 0xff;
                        tempbuf[5] = (haddr >> 16) & 0xff;
                        tempbuf[6] = (haddr >> 8) & 0xff;
                        tempbuf[7] = haddr & 0xff;
                        WriteData(ms->desc, tempbuf, 8);
                        }

                    }

                }

            }

        /* datagram sockets have a local buffer, so
         * we have to check it.
         */
        if ((ms->type == SOCK_DGRAM) && AmountInFifo(ms->dgbuf, FIFO_ALL))
            {
            num = ReadFifo(ms->dgbuf, tempbuf, ms->quant, FIFO_ALL);
            WriteData(ms->desc, tempbuf, num);
            }
        }

}

/*
 * check for exceptional conditions (oob data)
 */
VOID
CheckSocketEx(VOID)
{
    mlsocket *ms,*ms2;
    ubyte out[2];

    /* socket disconnect is also an exception,
     * but we rely on reads to catch it
     */
    for (ms = (mlsocket *)sockList.Head; ms2 = (mlsocket *)ms->n.Succ; ms = ms2)
        if (FD_ISSET(ms->fd, &sm[1].ex))
            if (recv(ms->fd, out, 1, MSG_OOB) > 0)
                SendCommand(CMD_OOBDATA, ms->desc, 1, out);
}

/*
 * Check stream sockets for data to send.
 */
VOID
CheckSocketWr(VOID)
{
   mlsocket *ms,*ms2;
   int x,num;
   ubyte out[2];
   ubyte *tmpb;

    for (ms = (mlsocket *)sockList.Head; ms2 = (mlsocket *)ms->n.Succ; ms = ms2)
        if (FD_ISSET(ms->fd, &sm[1].wr))
            {
            /* If there is data to send, we send out the
             * buffer in reasonable sized chunks until
             * the buffer is empty or an error occurs.
             */
            x = 1;
            while (num = FifoNextBlock(ms->rdbuf, &tmpb, FIFO_ALL))
                {
                if ((x = send(ms->fd, tmpb, num, 0)) <= 0)
                    break;
                ReadFifo(ms->rdbuf, 0, x, FIRD_FLSH|FIFO_ALL); /* flush */
                }

            /* If we've emptyed the buffer, or if we
             * disconnected while writing, we don't
             * want to select to write anymore, and
             * close if no longer defered.  If we
             * were blocked, unblock us.
             */
            if (!num || ((x <= 0) && errno == EPIPE))
                {
                ReadFifo(ms->rdbuf, 0, AmountInFifo(ms->rdbuf, 0), FIRD_FLSH|FIFO_ALL);
                if (ms->state & SK_WEBLOCK)
                    {
                    out[0] = 0;
                    SendCommand(CMD_BLOCK, ms->desc, 1, out);
                    ms->state &= ~SK_WEBLOCK;
                    }
                FD_CLR(ms->fd, &sm[0].wr);
                if (ms->state & SK_CLOSING)
                    {
                    s_Close(ms->desc);
                    continue;
                    }
                }
            /* If we still have data in the buffer, then
             * the socket has blocked on us.  Tell the
             * client.  We also add a tiny delay to keep
             * us from spinning on this data too much.
             */
            else if (num)
                {
                struct timeval tv;
                if (!(ms->state & SK_WEBLOCK))
                    {
                    out[0] = 1;
                    SendCommand(CMD_BLOCK, ms->desc, 1, out);
                    ms->state |= SK_WEBLOCK;
                    }
                tv.tv_sec = 0;
                tv.tv_usec = 250000; /* 1/4 second */
                select(0,0,0,0,&tv);
                }
            }
}


/*
 * Called by DeMux to write data to socket.
 */
VOID
WriteSock(ms, buf, len)
mlsocket *ms;
ubyte *buf;
int len;
{
    int num;
    struct sockaddr_in sa;
    ubyte out[4];
    ubyte *ob;

    if (ms->type == SOCK_STREAM)
        {
        /* Attempt to write all out data first before
         * defering to the main write loop.  If we fail
         * to write out all the data, we set the write
         * bit for the socket.  Pty's never get defered
         * because they always "work".
         */
         if (ms->state & SK_ISPTY)
            {
            PtyWrite(ms->fd, buf, len);
            num = 0;
            }
         else
            {
            num = send(ms->fd, buf, len, 0);
            if (num < len)
                {
                num = (num < 0) ? 0 : num;
                WriteFifo(ms->rdbuf, buf+num, len-num, FIFO_STRM);
                FD_SET(ms->fd, &sm[0].wr);
                }
            }
        }
    else
        {
        /* Datagram sockets dont really block, so we
         * simply buffer the data until we've got a whole
         * packet and send it out.
         */
        WriteFifo(ms->rdbuf, buf, len, FIFO_STRM);
        num = AmountInFifo(ms->rdbuf, FIFO_ALL);
        if (num > 8)
            {
            num -= 8;
            ReadFifo(ms->rdbuf, out, 2, FIRD_PEEK|FIFO_ALL);
            len = DATAWORD(out);
            if (num >= len)
                {
                ReadFifo(ms->rdbuf, tempbuf, len + 8, FIFO_ALL);
                bzero((char *)&sa, sizeof(sa));
#ifdef BSD44
                sa.sin_len = sizeof(struct sockaddr_in);
#endif
                sa.sin_family = AF_INET;
                sa.sin_port = htons(DATAWORD(tempbuf+2));
                sa.sin_addr.s_addr = htonl(DATALONG(tempbuf+4));
                (void)sendto(ms->fd, tempbuf+8, len, 0, (struct sockaddr *)&sa, sizeof(struct sockaddr_in));
                }
            }
        }
}

/* write() replacement for rlogin pty sockets, designed to
 * implement the rlogin resize protocol.
 */
VOID
PtyWrite(pty, buf, len)
int pty;
ubyte *buf;
int len;
{
    static int rsind = 0;
    static int rows, cols;
    static ubyte savebuf[] = {0xff, 0xff, 's', 's'};
    ubyte *b;


    for (b=buf; len>0; len--,b++)
    {
        switch(rsind)
        {
        case 0: /* 0xff == start of cookie, dump buffer so far */
            if (*b == savebuf[rsind])
                {
                if (b-buf)
                    write(pty, buf, b-buf);
                buf = b;
                rsind++;
                }
            break;
        case 1: /* 0xff */
        case 2: /* 's' */
        case 3: /* 's' */
            if (*b == savebuf[rsind])
                rsind++;
            else
                rsind = 0;
            break;
        case 4: /* rowsh */
            rows = *b << 8;
            rsind++;
            break;
        case 5: /* rowsl */
            rows += *b;
            rsind++;
            break;
        case 6: /* colsh */
            cols = *b << 8;
            rsind++;
            break;
        case 7: /* colsl */
            cols += *b;
            rsind++;
            break;
        case 8: /* yh */
        case 9: /* yl */
        case 10: /* xh */
            rsind++;
            break;
        case 11: /* xl */
            TtyResize(pty, cols, rows);
            rsind++;
            break;
        case 12: /* end, skip whole cookie */
            buf = b;
            if (*b == savebuf[0])
                {
                rsind = 1;
                break;
                }

        default: /* never happens (famous last words) */
            rsind = 0;
            break;
        }
    }

    /* dump rest of buffer if not in cookie */
    if ((b-buf)&&(rsind==0))
        write(pty, buf, b-buf);
}

