/* fifo.c -- all purpose fifo buffering scheme
 *
 * "Ported" from my custom amiga library (ezra.lib) :-)
 */
char rcsid[] = "$Id: fifo.c 1.14 1995/04/22 19:38:55 Ezra_Story Exp $";

#define INC_SYS
#define INC_STRING

#include "defs.h"



/*
 *  Creates a fifo.  Allocs fifo's memory pool, sets initial values.
 */
Fifo
CreateFifo(minsize)
int minsize;
{
    Fifo    f;

    if (!(f = (Fifo)malloc(sizeof(fifo)))) return(0);

    NewList((list *)&f->blst);
    f->size = 0;
    f->oobsize  = 0;
    f->minsize = minsize;

    return(f);
}

/*
 *  Destroys a fifo.  Traverses list, freeing all the nodes and frees
 *  the root structure.
 */
VOID
DeleteFifo(f)
Fifo f;
{
    node *n,*n2;

    for (n = f->blst.Head; n2 = n->Succ; n = n2)
        {
        Remove((node *)n);
        free(n);
        }

    free(f);
}


/*
 *  Write bytes into fifo.  If oobflags is set, it writes out of band data.
 */
int
WriteFifo(f, buf, len, oobflag)
Fifo f;
ubyte *buf;
int len;
char oobflag;
{
    fnode *fn;
    int x,y;

    if (!(y = len)) return(0);

    /* if its not oob data, first try to find available space on
     * the last node in the list.
     */
    if (!oobflag)
        {
        fn = (fnode *)f->blst.TailPred;
        if (fn != (fnode *)&f->blst && fn->left && !fn->oob)
            {
            x = MIN(fn->left, len);
            bcopy(buf, fn->buf+fn->size, x);
            fn->size += x;
            fn->left -= x;
            len -= x;
            buf += x;
            f->size += x;
            }
        }

    /* Create a new node, and add the (remaining) data to it.  If
     * it is an OOB node, just allocate enough for the data, and
     * mark it as such.
     */
    if (len)
        {
        x = oobflag ? len : MAX(len, f->minsize);
        if (fn = (fnode *)malloc(sizeof(fnode) + x))
            {
            fn->size = len;
            fn->left = x-len;
            fn->buf  = (ubyte *)(fn+1);
            bcopy(buf, fn->buf, len);
            AddTail((list *)&f->blst, (node *)fn);
            if (fn->oob = oobflag)
                f->oobsize += len;
            else
                f->size += len;
            len = 0;
            }
        }

    return(y-len);
}

/*
 *  Get bytes from fifo.  If oobflag is 0, it gets inband data only.  If
 *  oobflag is 1 is gets out of band data only.  If oobflag is 2, then it
 *  gets both.  If the flag is 10.. then only peek at the data, don't
 *  remove it from the queue.  0x10,0x11 and 0x12 work as expected. 0x20 works
 *  like 0x10 excpet no data is actually read.. it just depletes the fifo
 *  by the given amount.
 */

int
ReadFifo(f, buf, len, oobflag)
Fifo f;
ubyte *buf;
int len;
char oobflag;
{
    char peek;
    char deplete;
    int x;
    fnode  *fn,*fn2;

    if (!(x = len)) return(0);

    peek = oobflag & 0x10;
    deplete = oobflag & 0x20;
    oobflag &= 0x0f;

    /* we traverse the buffer list from the top until we've
     * gotten all the data we need, only considering node types
     * that coincide with the oobflag given to us.
     */
    for (fn = (fnode *)f->blst.Head; (fn2=(fnode *)fn->n.Succ) && len; fn = fn2)
        {
        if (oobflag == 2 || (!oobflag && !fn->oob) || (oobflag && fn->oob))
            {

            if (len >= fn->size)
                {
                /* We need all the data from this node.  If its not
                 * the only node on the list, or its an OOB data node,
                 * we toast it after we're done.  Otherwise, its the
                 * last non-oob node on the list, so we just reset
                 * it's pointers to avoid an alloc/dealloc loop nightmare
                 * for buffering small amounts of data.
                 */
                if (!deplete) bcopy(fn->buf, buf, fn->size);
                buf += fn->size;
                len -= fn->size;

                if (!peek)
                    {
                    if (fn->oob)
                        f->oobsize -= fn->size;
                    else
                        f->size -= fn->size;
                    if (fn2->n.Succ || fn->oob)
                        {
                        Remove((node *)fn);
                        free(fn);
                        }
                    else
                        {
                        fn->left += fn->buf - ((ubyte *)(fn+1));
                        fn->buf = (ubyte *)(fn+1);
                        fn->size = 0;
                        }
                    }
                }
            else
                {
                /* We only need a part of a node.  Just copy that
                 * part out and update the pointers.
                 */
                if (!deplete) bcopy(fn->buf, buf, len);
                if (!peek)
                    {
                    fn->buf += len;
                    if (fn->oob)
                        f->oobsize -= len;
                    else
                        f->size -= len;
                    fn->size -= len;
                    }
                len = 0;
                }
            }
        }
    x -= len;

    return(x);
}


/*
 *  Returns amount of data in the Fifo.  Depending on the oobflag, it will
 *  return the amount of normal data (0), amount of oobdata (1) or the
 *  total amount which includes both oob and normal data (2).
 */
int
AmountInFifo(f, oobflag)
Fifo f;
char oobflag;
{
    int x;

    switch (oobflag)
        {
        case 0:
            x = f->size;
            break;
        case 1:
            x = f->oobsize;
            break;
        case 2:
            x = f->size + f->oobsize;
            break;
        default: /* sanity */
            x = 0;
            break;
        }
    return(x);
}

/*
 * Returns TRUE if read pointer is at an out of band data mark.
 */
int
FifoAtMark(f)
Fifo f;
{
    int x;

    x = IsListEmpty((list *)&f->blst) ? 0 : ((fnode *)f->blst.Head)->oob;
    return(x);
}

/* quick write of one char to a fifo (stream only!) */
int
WriteFifoChar(f, c)
Fifo f;
ubyte c;
{
    fnode *fn;

    fn = (fnode *)f->blst.TailPred;
    if (fn != (fnode *)&f->blst && fn->left && !fn->oob)
        {
        fn->buf[fn->size++] = c;
        fn->left--;
        f->size++;
        c = 1;
        }
    else
        {
        if (fn = (fnode *)malloc(sizeof(fnode) + f->minsize))
            {
            fn->size = 1;
            fn->left = f->minsize-1;
            fn->buf  = (ubyte *)(fn+1);
            *fn->buf = c;
            AddTail((list *)&f->blst, (node *)fn);
            f->size++;
            c = 1;
            }
        else
            c = 0;
        }

    return(c);
}

/* quick read of one char from a fifo (all, no flags) */
int
ReadFifoChar(f, buf)
Fifo f;
ubyte *buf;
{
    int x;
    fnode  *fn;

    x = 0;
    if (f->size + f->oobsize)
        {
        x = 1;
        fn = (fnode *)f->blst.Head;
        *buf = *fn->buf;
        fn->size--;

        if (fn->oob) f->oobsize--;
        else f->size--;

        if (!fn->size)
            {
            if (fn->n.Succ->Succ || fn->oob)
                {
                Remove((node *)fn);
                free(fn);
                }
            else
                {
                fn->left += fn->buf - ((ubyte *)(fn+1));
                fn->buf = (ubyte *)(fn+1);
                fn->size = 0;
                }
            }
        else
            {
            fn->buf++;
            }
        }

    return(x);
}

/* returns a pointer to, and the length of the next
 * filled block in the fifo of the specified type.
 */
int
FifoNextBlock(f, buf, oobflag)
Fifo f;
ubyte **buf;
char oobflag;
{
    fnode *fn, *fn2;

    for (fn = (fnode *)f->blst.Head; fn2=(fnode *)fn->n.Succ; fn = fn2)
        if (oobflag == 2 || (!oobflag && !fn->oob) || (oobflag && fn->oob))
            {
            *buf = fn->buf;
            return(fn->size);
            }
    return(0);
}
