/* tsterr.cpp - test error manager */

#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include "errmgr.hpp"


static int ask(const char *prompt);
static int write_complaint_file(const char *filename,
                                const char **msgs);

const int NAME_LEN = 32;


/* we write more than one error dictionary to test the LIFO code: messages */
/* in later dictionaries override the ones in earlier dictionaries. */

static const char *msgs1[] = {
    "override: this message should have been overridden!\n",
    "first_dict: this message came from the first dictionary defined - OK\n",
    "one_of_each: %d int, %ld long, %lx long, %u unsigned - OK\n",
    0
};

static const char *msgs2[] = {
    "override: this message overrides one in the first file - OK\n",
    "malformed: this text doesn't match the format specification\n",
    "second_of_each: %15.5e float, %6.4f float, %8g float - OK\n",
    "third_of_each: %c char, %i int, %o octal, percent %% - OK\n",
    0
};

/* msgs3[] has some errors in it and so won't be installed by err_mgr. */
/* we test this by defining a key here that's also in msgs2[] or msgs1[] */
/* and writing an error that uses that key. */

static const char *msgs3[] = {
    "override: this message should never be printed.\n",
    "abcdef missing ':'\n",
    0
};


int main(int argc,char *argv[])
{
    int errs = 0;
    char tempname1[NAME_LEN],tempname2[NAME_LEN],tempname3[NAME_LEN];
    error_mgr *bad_err_mgr;

    /* allocating an error_mgr will cause the program to halt. thus, we */
    /* don't want to do it every time. */

    if (ask("Allocate a second error_mgr")) {
        bad_err_mgr = new error_mgr;
        fprintf(stderr,"Was able to allocate a second error_mgr!\n");
        return EXIT_FAILURE;
    }

    /* create three dictionary files, one of which has some errors in it. */
    /* we assume the complaint dictionary code (complain.cpp) has been */
    /* tested already (tstcompl.cpp) and so the dictionaries have rather */
    /* ordinary messages in them. */

    tmpnam(tempname1);
    if (write_complaint_file(tempname1,msgs1))
        return EXIT_FAILURE;
    tmpnam(tempname2);
    if (write_complaint_file(tempname2,msgs2))
        return EXIT_FAILURE;
    tmpnam(tempname3);
    if (write_complaint_file(tempname3,msgs3))
        return EXIT_FAILURE;

    if (!err_mgr.define_dictionary(tempname1) ||
        !err_mgr.define_dictionary(tempname2) ||
        err_mgr.define_dictionary(tempname3)) {
        fprintf(stderr,"define_dictionary() failed\n");
        ++errs;
    }

    /* now print messages to test the keyed error message facility. */
    /* most of these will be overridden. every warning printed should */
    /* have "OK" at the end of it. */

    err_mgr.warn("$override: failed to replace key!\n");
    err_mgr.warn("$first_dict: failed to replace key!\n");
    err_mgr.warn("$blot: this message should not be replaced - OK\n");
    err_mgr.warn("%s - OK\n","An unkeyed message");

    /* this message, not the one in the dictionary, should be printed. */
    /* the dictionary form is missing the %s. */

    err_mgr.warn("$malformed: has a string in it - %s\n","OK");

    /* bad message text - can't determine the key */

    err_mgr.warn("$x+y: bad key name - OK\n");
    err_mgr.warn("$: missing key name - OK\n");

    /* print something of everything - %d, %c, %ld, %u, %e, %f, %g */

    err_mgr.warn("$one_of_each: int %3d, long %ld, long %lx, unsigned %u\n",
                 34,12345678L,0x123456,0x8000);
    err_mgr.warn("$second_of_each: float %10.5e, float %5.3f, float %10g\n",
                 7.35e-10,3.75,27.1234);
    err_mgr.warn("$third_of_each: char %c, int %i, octal %o\n",
                 63,756,03577);

    /* test the new post() routine. */

    err_mgr.post("a posted message: %d - OK\n",5);

    remove(tempname1);
    remove(tempname2);
    remove(tempname3);

    if (!errs)
        printf("all tests succeeded\n");
    return errs ? EXIT_FAILURE : EXIT_SUCCESS;

}  /* end of main() */

static int ask(const char *prompt)
{
    /* ask for permission to do the requested task. the default (<CR>) */
    /* is yes. */

    char buf[80];

    fprintf(stderr,"%s [yes]? ",prompt);
    fgets(buf,sizeof(buf),stdin);       /* can't always read from stderr */
    return toupper(buf[0]) != 'N';

}  /* end of ask() */

static int write_complaint_file(const char *filename,
                                const char **msgs)
{
    /* write the text to the named file. return 1 on failure, 0 on success. */

    int i;
    FILE *tempfile;

    if ((tempfile = fopen(filename,"w")) == NULL) {
        fprintf(stderr,"Unable to create temporary file\n");
        return 1;
    }
    for (i = 0; msgs[i]; ++i)
        fputs(msgs[i],tempfile);
    fclose(tempfile);
    return 0;

}  /* end of write_complaint_file() */
