#include "FileIO.Ch"
#include "FiveWin.ch"

//
// A Class for Reading/Using Norton Guide.
//
//----------------------------------------------------------------------------//

CLASS NgFile
   //
EXPORT:
   VAR      cFile             // The FileName
   VAR      nHandle           // Handle for the File
   VAR      cTitle            // Title of the Ng
   VAR      aCredit           // Credit of the Ng
   VAR      cHeader           // Header of the Ng
   VAR      aMnuStruct        // Structures of different Menus
   //
   VAR      lWindows          // Are we Under Windows ?
   VAR      aMenus            // The Ng Menus

   METHOD   New               // Constructor
   METHOD   Open              // Open the File and Initialize the Structure
   METHOD   Close             // Close the File
   METHOD   GetEntry          // Read the Entry of a specified menu
   METHOD   Read              // Read a Entry

PROTECTED:
   METHOD   Credits
   METHOD   InitMenus
   METHOD   GetMenus
ENDCLASS

//----------------------------------------------------------------------------//
//
METHOD New( cFile, lWindows ) Class NgFile
   DEFAULT lWindows := .F.    // DOS Mode
   ::nHandle   := -1
   ::cFile     := cFile
   ::lWindows  := lWindows
   IF FILE( ::cFile )
      ::Open( .F. )
      ::Close()
   ENDIF
RETURN Self

//----------------------------------------------------------------------------//
// Open a NG File
METHOD Open( lNormalMode ) Class NgFile
   LOCAL cBuffer := SPACE( 48 )
   // DEFAULT lNormalMode := .T.
   lNormalMode := IIF( lNormalMode==Nil, .T., lNormalMode )
   ::nHandle:= FOpen( ::cFile )
   IF ( ::nHandle > 0 )
      FRead( ::nHandle, @cBuffer, 48 )
      If ( SubStr( cBuffer, 1, 2 ) == "NG" ) .OR. ;
         ( SubStr( cBuffer, 1, 2 ) == "EH" )
         // Header
         ::cHeader   := cBuffer
         ::cTitle    := SubStr( cBuffer, 9, 40 )
         ::cTitle    := STRTRAN( ::cTitle, Chr(0), Chr(32) )
         ::cTitle    := ALLTRIM( ::cTitle )
         IF ::lWindows
            ::cTitle := OemToAnsi( ::cTitle )
         ENDIF
         ::Credits()
         //
         IF lNormalMode
            ::InitMenus()
            ::GetMenus()
         ENDIF
       Else
          FClose( ::nHandle )
          ::nHandle := -1
       EndIf
   EndIf
RETURN Self

//----------------------------------------------------------------------------//
// Closing the current NG
METHOD Close() Class NgFile
   IF ( ::nHandle > 0 )
      FClose( ::nHandle )
      ::nHandle := -1
   ENDIF
RETURN Self

//----------------------------------------------------------------------------//
// Reading the elements for a Menu
METHOD GetEntry( nTop, nSub ) Class NgFile
LOCAL nTotal, nOffset
LOCAL oNgEntry
// Number of Top Menus
nTotal      := Asc( Substr( ::cHeader, 7, 1 ) )
//
IF ( nTotal <= 0 )
   // The first Pos is Always 378
   nOffset := 378
ELSE
   nOffset := Ng2DWord( SubStr( ::aMnuStruct[nTop], 3 + ( nSub - 1) * 4, 4 ) )
ENDIF
//
oNgEntry := ::Read( nOffset )
//
RETURN oNgEntry

//----------------------------------------------------------------------------//
// Reading the element(s) for a entry
METHOD Read( nOffset ) Class NgFile
LOCAL oNgEntry
//
oNgEntry := NgEntry():New( nOffset, ::nHandle, ::lWindows )
//
RETURN oNgEntry

//----------------------------------------------------------------------------//
// Credit for the current NG
METHOD Credits() Class NgFile
LOCAL cBuffer, nCpt
cBuffer := Space( 66 )
::aCredit := { }
// 5 lines ( non coded ) of 66 characters starting at position 48
For nCpt := 0 To 4
   // Point nCpt line
   FSeek( ::nHandle, 48 + 66 * nCpt, FS_SET )
   // Reading the line
   cBuffer := FReadStr( ::nHandle, 66 )
   cBuffer := STRTRAN( cBuffer, Chr(0), Chr(32) )
   If ( !Empty( cBuffer ) )
      IF ::lWindows
         cBuffer := OemToAnsi( cBuffer )
      ENDIF
      // Adding the line
      AADD( ::aCredit, Alltrim( cBuffer ) )
   EndIf
Next
RETURN Self


//----------------------------------------------------------------------------//
//
METHOD InitMenus() Class NgFile
LOCAL nTotal, nPos, nCpt, cBuffer, cTemp
// Number of Top Menus
nTotal      := Asc( Substr( ::cHeader, 7, 1 ) )
//
// The first Pos is Always 378
nPos  := 378
::aMnuStruct := { }
FOR nCpt := 1 TO nTotal
   //
   cBuffer := Space(26)
   FSeek( ::nHandle, nPos )
   FRead( ::nHandle, @cBuffer, 26)
   cTemp := cBuffer
   //
   cBuffer := SPACE( Ng2Word( SubStr( cTemp, 3, 2 ) ) )
   //
   FRead( ::nHandle, @cBuffer, Ng2Word( SubStr( cTemp, 3, 2 ) ) )
   AAdd( ::aMnuStruct, SubStr( cTemp, 5, 2) + cBuffer )
   FSeek( ::nHandle, nPos + 2 )
   cBuffer := Space( 2 )
   FRead( ::nHandle, @cBuffer, 2 )
   nPos := nPos + 26 + Ng2Word( cBuffer )
NEXT
RETURN Self

//----------------------------------------------------------------------------//
//
METHOD GetMenus() Class NgFile
LOCAL nTop, cTemp, nPos, nSub
LOCAL nCpt, nCpt2, nCpt3
//
nTop := Asc( Substr( ::cHeader, 7, 1 ) )
//
::aMenus    := Array( nTop )
//
FOR nCpt := 1 TO nTop
   nSub  := Ng2Word( SubStr( ::aMnuStruct[ nCpt ], 1, 2 ) ) - 1
   nPos  := 11 + 12 * nSub
   cTemp := SubStr( ::aMnuStruct[ nCpt ], nPos )
   IF ::lWindows
      ::aMenus[ nCpt ] := { OemToAnsi( Ng2String( cTemp ) ), Array( nSub ) }
   ELSE
      ::aMenus[ nCpt ] := { Ng2String( cTemp ), Array( nSub ) }
   ENDIF
   //
   FOR nCpt2 := 1 TO nSub
      nPos  := nPos + NgStrLen( cTemp )
      cTemp := SubStr( ::aMnuStruct[ nCpt ], nPos )
      IF ::lWindows
         ::aMenus[ nCpt, 2, nCpt2 ] := OemToAnsi( Ng2String( cTemp ) )
      ELSE
         ::aMenus[ nCpt, 2, nCpt2 ] := Ng2String( cTemp )
      ENDIF
   NEXT
NEXT
RETURN Self




/*
//----------------------------------------------------------------------------//
// Convert a 2 Bytes String in NG to Word
Function Ng2Word( cString )
   LOCAL cLow, cHigh, nRetVal
   LOCAL nRet2
   // We take the 2 bytes
   // We apply the XOR Mask of NG
   cLow     := Chr( nXor( ASC( LEFT( cString, 1 ) ),  26 ) )
   cHigh    := Chr( nXor( ASC( RIGHT( cString, 1 ) ), 26 ) )
   // And we make it a Signed Word
   nRetVal  := Bin2I( cLow + cHigh )
   nRet2    := cNg2Word( cString )
   IF ( nRetVal <> nRet2 )
      ? "Ng2Word",ASC( LEFT( cString, 1 ) ),ASC( RIGHT( cString, 1 ) ),nRetVal,nRet2
   ENDIF
RETURN ( nRetVal )

//----------------------------------------------------------------------------//
// Convert a 4 Bytes String in NG to Word
Function Ng2DWord( cString )
   LOCAL cByte1, cByte2, cByte3, cByte4
   LOCAL nRetVal, nRet2
   // We take each byte
   cByte1   := Chr( nXor( ASC( SubStr( cString, 1, 1 ) ),  26 ) )
   cByte2   := Chr( nXor( ASC( SubStr( cString, 2, 1 ) ),  26 ) )
   cByte3   := Chr( nXor( ASC( SubStr( cString, 3, 1 ) ),  26 ) )
   cByte4   := Chr( nXor( ASC( SubStr( cString, 4, 1 ) ),  26 ) )
   // And Make it a Long / Double Word Signed
   nRetVal  := Bin2L( cByte1 + cByte2 + cByte3 + cByte4 )
   nRet2    := cNg2DWord( cString )
   IF ( nRetVal <> nRet2 )
      ? "Ng2DWord",ASC( cByte1 ),ASC( cByte2 ),ASC( cByte3 ),ASC( cByte4 ),nRetVal,nRet2
   ENDIF
RETURN ( nRetVal )

//----------------------------------------------------------------------------//
//
Function NgStrLen( cString )
LOCAL cRetVal
LOCAL nCpt, nChar, nRet2
   cRetVal := ""
   nCpt    := 1
   WHILE ( nCpt < LEN( cString ) )
      nChar := nXOR( Asc( SUBSTR( cString, nCpt, 1 ) ), 26 )
      IF ( nChar == 0 )
         Exit
      ENDIF
      nCpt ++
   ENDDO
   nRet2    := cNgStrLen( cString )
   IF ( nCpt <> nRet2 )
      ? "NgStrLen",cString,nCpt,nRet2
   ENDIF
Return nCpt

//----------------------------------------------------------------------------//
// Convert a NG String to a Normal String
Function Ng2String( cString )
   LOCAL nChar, cRetVal, nCpt
   LOCAL cRet2
   cRetVal := ""
   nCpt    := 1
   WHILE ( nCpt < LEN( cString ) )
      nChar := nXOR( Asc( SUBSTR( cString, nCpt, 1 ) ), 26 )
      DO CASE
         CASE ( nChar == 000 )      // End of String
            Exit
         CASE ( nChar == 255 )      // Repeat Space
            nCpt ++
            nChar := nXOR( Asc( SUBSTR( cString, nCpt, 1 ) ), 26 )
            cRetVal := cRetVal + Replicate( " ", nChar )
            nCpt ++
            Loop
         CASE ( nChar == 094 )      // "^" Control Character
            nCpt++
            nChar := nXOR( Asc( SUBSTR( cString, nCpt, 1 ) ), 26 )
            IF ( UPPER( Chr( nChar ) ) == "A" ) .OR. ;
               ( UPPER( Chr( nChar ) ) == "C" )
               // The next two char are used by this control
               // so we skip them
               nCpt := nCpt + 2
            ENDIF
            nCpt ++
            Loop
      ENDCASE
      IF ( nChar == 0 )
         Exit
      ENDIF
      cRetVal := cRetVal + Chr( nChar )
      nCpt ++
   ENDDO
   cRet2 := cNg2String( cString )
   IF ( cRetVal <> cRet2 )
      ? "Ng2String",cString,cRetval,cRet2
   ENDIF
RETURN ( cRetVal )
*/

