// Error handler system adapted to FiveWin

/*************
*   Errsysw.prg
*   From Standard Clipper 5.0 error handler
*	Compile:  /m/n/w
*/

#include "error.ch"
#include "FiveWin.ch"

external _fwGenError   // Link FiveWin generic Error Objects Generator

#define NTRIM(n)		( LTrim(Str(n)) )

/*************
*	ErrorSys()
*
*	Note:  automatically executes at startup
*/
proc ErrorSys()
    ErrorBlock( { | e | ErrorDialog( e ) } )
return


/*************
*   ErrorDialog()
*/
static func ErrorDialog( e ) // -> logical  or quits App.

   local oDlg, oLbx
   local lRet    // if lRet == nil -> default action: QUIT
   local i, j, cMessage, aStack := {}
   local oSay, hLogo := FWBitMap()
   local nButtons := 1
   local cErrorLog := "Application Error occurred at: " + ;
                      DToC( Date() ) + ", " + Time() + CRLF
   local aVersions := GetVersion()
   local aTasks    := GetTasks()

   cErrorLog += "Application path and name: " + GetModuleFileName( GetInstance() ) + CRLF + CRLF

   cErrorLog += "Free System resources: % " + AllTrim( Str( GetFreeSystemResources( 0 ) ) ) + CRLF + ;
                "     GDI    resources: % " + AllTrim( Str( GetFreeSystemResources( 1 ) ) ) + CRLF + ;
                "     User   resources: % " + AllTrim( Str( GetFreeSystemResources( 2 ) ) ) + CRLF + CRLF

   cErrorLog += "Windows and MsDos versions: " + ;
                Str( aVersions[ 1 ], 2 ) + "." + ;
                Str( aVersions[ 2 ], 2 ) + ", " + ;
                Str( aVersions[ 3 ], 2 ) + "." + ;
                Str( aVersions[ 4 ], 2 ) + CRLF

   cErrorLog += "Application max file handles permited: ( SetHandleCount() ) " + Str( SetHandleCount(), 3 ) + CRLF + CRLF
   // cErrorLog += "Application file handles not used: " + Str( GetFreeFileHandles(), 3 ) + CRLF + CRLF

   cErrorLog += "Windows total applications running: " + Str( GetNumTasks(), 3 ) + CRLF
   for i = 1 to Len( aTasks )
      cErrorLog += " " + Str( i, 3 ) + " " + aTasks[ i ] + CRLF
   next
   cErrorLog += CRLF


   // by default, division by zero yields zero
   if ( e:genCode == EG_ZERODIV )
       return (0)
   end


   // for network open error, set NETERR() and subsystem default
   if ( e:genCode == EG_OPEN .and. e:osCode == 32 .and. e:canDefault )

       NetErr(.t.)
       return .f.       // Warning: Exiting!

   end


   // for lock error during APPEND BLANK, set NETERR() and subsystem default
   if ( e:genCode == EG_APPENDLOCK .and. e:canDefault )

       NetErr(.t.)
       return .f.       // OJO SALIDA

   endif

   if e:canRetry
      nButtons++
   endif

   if e:canDefault
      nButtons++
   endif


   // build error message
   cMessage := ErrorMessage(e)
   cErrorLog += "Error description: " + cMessage + CRLF

   i := 2    // we don't disscard any info again !
   while ( i < 74 )

       if ! Empty(ProcName(i))
          AAdd( aStack, "Called from " + Trim(ProcName(i)) + ;
                        "(" + NTRIM(ProcLine(i)) + ")" )
          cErrorLog += ATail( aStack ) + CRLF
       endif

       i++
   end

   cErrorLog += CRLF + "Variables in use" + CRLF + "================" + CRLF
   cErrorLog += "   Procedure     Type   Value" + CRLF
   cErrorLog += "   ==========================" + CRLF

   i := 2    // we don't disscard any info again !
   while ( i < 74 )

       if ! Empty( ProcName( i ) )
          cErrorLog += "   " + Trim( ProcName( i ) ) + CRLF
          for j = 1 to ParamCount( i )
             cErrorLog += "     Param " + Str( j, 3 ) + ":    " + ;
                          ValType( GetParam( i, j ) ) + ;
                          "    " + cGetInfo( GetParam( i, j ) ) + CRLF
          next
          for j = 1 to LocalCount( i )
             cErrorLog += "     Local " + Str( j, 3 ) + ":    " + ;
                          ValType( GetLocal( i, j ) ) + ;
                          "    " + cGetInfo( GetLocal( i, j ) ) + CRLF
          next
       endif

       i++
   end


   cErrorLog += CRLF + "DataBases in use" + CRLF + "================" + CRLF
   for i = 1 to 255
      if ! Empty( Alias( i ) )
         cErrorLog += CRLF + Str( i, 3 ) + ": " + If( Select() == i, "=> ", "   " ) + ;
                      Alias( i ) + CRLF
         cErrorLog += "     Indexes in use" + CRLF
         for j = 1 to 15
            if ! Empty( ( Alias( i ) )->( IndexKey( j ) ) )
               cErrorLog += Space( 8 ) + ;
                            If( ( Alias( i ) )->( IndexOrd() ) == j, "=> ", "   " ) + ;
                            ( Alias( i ) )->( IndexKey( j ) ) + CRLF
            endif
         next
      endif
   next

   i = 1
   cErrorLog += CRLF + "Classes in use:" + CRLF
   cErrorLog += "===============" + CRLF
   while ! Empty( __ClassNam( i ) )
      cErrorLog += "   " + Str( i, 3 ) + " " + __ClassNam( i++ ) + CRLF
   end

   // Generates a file with an Error Log
   MemoWrit( "Error.log", cErrorLog )

   DEFINE DIALOG oDlg FROM 0, 0 TO 20, 58 ;
      TITLE OemToAnsi( FWVERSION + " ERROR Window" )

   @ 0, 0 SAY oSay PROMPT OemToAnsi( cMessage ) CENTERED BORDER OF oDlg

   oSay:nTop     = 10
   oSay:nLeft    = 21
   oSay:nBottom  = 36
   oSay:nRight   = 206

   @ 3,  3 SAY "&Stack List" OF oDlg
   @ 3, 17 SAY "See Error.log file" OF oDlg

   i = aStack[ 1 ]

   @ 4, 2 LISTBOX oLbx VAR i ITEMS aStack OF oDlg SIZE 193, 72

   oLbx:nTop  -= 5
   oLbx:nLeft += 3

   @ 11, if( nButtons > 1, 3, 13 ) BUTTON "&Quit"     OF oDlg ACTION oDlg:End() ;
                                           SIZE 40, 12
   if e:canRetry
      @ 11, if( nButtons == 2, 24, 13 ) BUTTON "&Retry"    OF oDlg ACTION ( lRet  := .t., oDlg:End() ) ;
                                        SIZE 40, 12
   endif

   if e:canDefault
      @ 11, 24 BUTTON "&Default"  OF oDlg ACTION ( lRet  := .f., oDlg:End() ) ;
                                        SIZE 40, 12
   endif

   ACTIVATE DIALOG oDlg CENTERED ;
      ON PAINT ( DrawBitmap( oDlg:hDC, hLogo, 34, 4 ),;
                 DrawBitmap( oDlg:hDC, hLogo, 34, 420 ) )

   DeleteObject( hLogo )

   if lRet == nil .or. ( !LWRunning() .and. lRet )
      SET RESOURCES TO
      ErrorLevel(1)
      QUIT              // OJO QUIT
   endif

return lRet


/*************
*	ErrorMessage()
*/
static func ErrorMessage(e)

	// start error message
    local cMessage := if( empty( e:osCode ), ;
                          if( e:severity > ES_WARNING, "Error ", "Warning " ),;
                          "(DOS Error " + NTRIM(e:osCode) + ") " )

	// add subsystem name if available
    cMessage += if( ValType( e:subsystem ) == "C",;
                    e:subsystem()                ,;
                    "???" )

	// add subsystem's error code if available
    cMessage += if( ValType( e:subCode ) == "N",;
                    "/" + NTRIM( e:subCode )   ,;
                    "/???" )
	// add error description if available
	if ( ValType(e:description) == "C" )
        cMessage += "  " + e:description
	end

	// add either filename or operation
    cMessage += if( !Empty( e:filename ),;
                    ": " + e:filename   ,;
                    if( !Empty( e:operation ),;
                        ": " + e:operation   ,;
                        "" ) )
return cMessage

//----------------------------------------------------------------------------//
// returns extended info for a certain variable type

static function cGetInfo( uVal )  

   local cType := ValType( uVal )

   do case
      case cType == "C"
           return '"' + cValToChar( uVal ) + '"'

      case cType == "O"
           return "Class: " + uVal:ClassName()

      case cType == "A"
           return "Len: " + Str( Len( uVal ), 4 )

      otherwise
           return cValToChar( uVal )
   endcase

return nil

//----------------------------------------------------------------------------//
