
//***************************************************************************
//
// Program:				FS_DEMO.EXE
//
// Purpose:				To demostrate the use of the CFileSystem class.
//
// Copyright:			Copyright 1993, 1994 Scott P. Leslie, All Rights Reserved.
//
// Version:				Version 1.3 for MS Windows 3.1 and MSVC 1.0.
//
// Important:			Read the README.WRI file included with this distribution
//								before using this code.
//
// Shareware:			This code is ShareWare.  If you use it, you should register
//								it with the author.  If you do not register it after 14 days
//								of use, you must discontinue using it.
//
//								Contact the author at ScotLeslie@aol.com for more info.
//
// Distribution:	You may distribute this code unmodified as long as all
//								accompanying documentation and notes are also distributed.
//
// Disclaimer:		This code is provided as is with no implied or express
//								warranties.  You should test this code for your particular
//								use on non-critical data before using it in a production
//								system.
//
//***************************************************************************


#include "stdio.h"

#include "filesys.h"


void
main()
{
	CFileSystem	fs;

	VERIFY(fs.MakeDirectory("tests\\foo"));
	VERIFY(fs.MakeDirectory("tests\\bar"));
	VERIFY(fs.RenameDirectory("tests\\foo", "tests\\bar") == FALSE);
	VERIFY(fs.RenameDirectory("tests\\foot", "tests\\bar") == FALSE);
	VERIFY(fs.RenameDirectory("tests\\foo", "tests\\foot"));
	VERIFY(fs.RenameDirectory("tests\\foot", "tests\\bar\\foo") == FALSE);
	VERIFY(fs.RenameDirectory("tests\\bar", "tests\\this1234567890") == TRUE);		// NewName is truncated.
	VERIFY(fs.DeleteDirectory("tests\\foot"));
	VERIFY(fs.DeleteDirectory("tests\\this1234"));

	printf("GetFreeSpace a:\\ = %ld\n", fs.GetFreeSpace("a:\\"));
	printf("GetFreeSpace c:\\ = %ld\n", fs.GetFreeSpace("c:\\"));
	printf("GetFreeSpace d:\\ = %ld\n", fs.GetFreeSpace("d:\\"));
	printf("GetFreeSpace e:\\ = %ld\n", fs.GetFreeSpace("e:\\"));
	printf("GetFreeSpace f:\\ = %ld\n", fs.GetFreeSpace("f:\\"));

	printf("GetTotalSpace a:\\ = %ld\n", fs.GetTotalSpace("a:\\"));
	printf("GetTotalSpace c:\\ = %ld\n", fs.GetTotalSpace("c:\\"));
	printf("GetTotalSpace d:\\ = %ld\n", fs.GetTotalSpace("d:\\"));
	printf("GetTotalSpace e:\\ = %ld\n", fs.GetTotalSpace("e:\\"));
	printf("GetTotalSpace f:\\ = %ld\n", fs.GetTotalSpace("f:\\"));

	printf("IsReadOnlyFS c:\\ = %d\n", fs.IsReadOnlyFileSystem("c:\\"));
	printf("IsReadOnlyFS a:\\ = %d\n", fs.IsReadOnlyFileSystem("a:\\"));


	// Test Full Path Name Retrieval.
	CString fpn;
	fpn = fs.GetFullPathName("c:\\foo\\bar");
	printf("Full Path Name for c:\\foo\\bar = %s\n", (const char *) fpn);
	fpn = fs.GetFullPathName("c:foo\\bar");
	printf("Full Path Name for c:foo\\bar = %s\n", (const char *) fpn);
	fpn = fs.GetFullPathName("\\foo\\bar");
	printf("Full Path Name for \\foo\\bar = %s\n", (const char *) fpn);
	fpn = fs.GetFullPathName("\\foo\\bar\\what.txt");
	printf("Full Path Name for \\foo\\bar\\what.txt = %s\n", (const char *) fpn);
	fpn = fs.GetFullPathName("foo\\bar\\what.txt");
	printf("Full Path Name for foo\\bar\\what.txt = %s\n", (const char *) fpn);

	fpn = fs.GetFullPathName("d:\\foo\\bar\\.\\what.txt");
	printf("Full Path Name for d:\\foo\\bar\\.\\what.txt = %s\n", (const char *) fpn);
	fpn = fs.GetFullPathName("d:\\foo\\bar\\.\\.\\what.txt");
	printf("Full Path Name for d:\\foo\\bar\\.\\.\\what.txt = %s\n", (const char *) fpn);
	fpn = fs.GetFullPathName("d:\\foo\\bar\\.\\text\\.\\what.txt");
	printf("Full Path Name for d:\\foo\\bar\\.\\text\\.\\what.txt = %s\n", (const char *) fpn);
	fpn = fs.GetFullPathName("d:\\foo\\.\\");
	printf("Full Path Name for d:\\foo\\.\\ = %s\n", (const char *) fpn);
	fpn = fs.GetFullPathName(".\\foo");
	printf("Full Path Name for .\\foo = %s\n", (const char *) fpn);
	fpn = fs.GetFullPathName("\\.\\");
	printf("Full Path Name for \\.\\ = %s\n", (const char *) fpn);

	fpn = fs.GetFullPathName("d:\\foo\\..\\");
	printf("Full Path Name for d:\\foo\\..\\ = %s\n", (const char *) fpn);
	fpn = fs.GetFullPathName("d:\\foo\\..\\bar\\what.txt");
	printf("Full Path Name for d:\\foo\\..\\bar\\what.txt = %s\n", (const char *) fpn);
	fpn = fs.GetFullPathName("d:\\foo\\bar\\..\\..\\what.txt");
	printf("Full Path Name for d:\\foo\\bar\\..\\..\\what.txt = %s\n", (const char *) fpn);
	fpn = fs.GetFullPathName("d:\\foo\\..\\bar\\..\\what.txt");
	printf("Full Path Name for d:\\foo\\..\\bar\\..\\what.txt = %s\n", (const char *) fpn);
	fpn = fs.GetFullPathName("\\..\\");
	printf("Full Path Name for \\..\\ = %s\n", (const char *) fpn);
	fpn = fs.GetFullPathName("..");
	printf("Full Path Name for .. = %s\n", (const char *) fpn);
	fpn = fs.GetFullPathName("d:\\..\\");
	printf("Full Path Name for d:\\..\\ = %s\n", (const char *) fpn);

	fpn = fs.GetFullPathName("");
	printf("Full Path Name for '' = %s\n", (const char *) fpn);
	fpn = fs.GetFullPathName("\\");
	printf("Full Path Name for \\ = %s\n", (const char *) fpn);

	VERIFY(fs.CompareDirectories("tests\\test0\\*.*", "tests\\test1\\*.*", FALSE, TRUE) == FALSE);
	VERIFY(fs.CompareDirectories("tests\\test1\\*.*", "tests\\test1\\*.*", TRUE, TRUE) == TRUE);
	VERIFY(fs.CompareDirectories("tests\\test1\\*.*", "tests\\test2\\*.*") == TRUE);
	VERIFY(fs.CompareDirectories("tests\\test2\\*.*", "tests\\test3\\*.*") == FALSE);
	VERIFY(fs.CompareDirectories("tests\\test2\\*.*", "tests\\test4\\*.*") == FALSE);

	// Get the file system name

	CString FileSystemName = fs.GetCurrentFileSystem();
	CString VolLabel = fs.GetVolumeLabel(FileSystemName);
	printf("Volume Label for %s = %s\n", (const char *) FileSystemName, (const char *) VolLabel);

	// Get the current directory information

	CString CurrentDir = fs.GetCurrentDirectory();
	printf("Current Directory = %s\n", (const char *) CurrentDir);

	// Get the number of bytes under the current directory

	LONG lDirSize = fs.GetDirectorySize(CurrentDir, "*.*", TRUE);
	printf("Current Directory Size = %ld\n", lDirSize);

	// Print out a directory listing.
	CStringList *pDirList = fs.GetDirectory("*.*", CFileSystem::normal, TRUE);
	printf("\nDirectory Listing:\n");
	CFileSystem::Sort(pDirList);
	for (int i = 0; i < pDirList->GetCount(); i++)
	{
		CFileStatus Status;
		CString FileName = pDirList->GetAt(pDirList->FindIndex(i));

		if (CFile::GetStatus((const char *) FileName, Status))
		{
			printf("\t%s\t%ld\n", (const char *) FileName, Status.m_size);
		} // if
	} // for
	delete pDirList;
	pDirList = NULL;


	//**********************************************************************
	// Test Cases
	//**********************************************************************


	// Verify the status of the test directory.
	CStringList *pDir = fs.GetDirectory("tests\\*.*", CFileSystem::allfiles, TRUE);
	VERIFY(pDir->Find("tests\\test1.dat"));
	VERIFY(pDir->Find("tests\\test2.dat"));
	delete pDir;
	pDir = NULL;

	// Test making directories.
	VERIFY(fs.MakeDirectory("tests\\subdir1"));
	VERIFY(fs.DirectoryExists("tests\\subdir1"));
	VERIFY(fs.MakeDirectory("tests\\subdir2"));
	VERIFY(fs.DirectoryExists("tests\\subdir2"));
	VERIFY(fs.DirectoryExists("tests\\subdir3") == FALSE);

	// Test making a path (ie. Multiple directories with one call).
	VERIFY(fs.MakePath("tests\\subdir3\\lower1"));
	VERIFY(fs.DirectoryExists("tests\\subdir3\\lower1"));
	VERIFY(fs.DirectoryExists("tests\\subdir3\\lower2") == FALSE);
	VERIFY(fs.MakePath("tests\\subdir3\\lower2"));
	VERIFY(fs.MakePath("tests\\subdir3\\lower2") == FALSE);

	// Test file copying.
	VERIFY(fs.FileExists("tests\\test3.dat") == FALSE);
	VERIFY(fs.CopyFile("tests\\test1.dat", "tests\\test3.dat"));
	VERIFY(fs.FileExists("tests\\test3.dat"));
	VERIFY(fs.CopyFile("tests\\test1.dat", "tests\\test2.dat") == FALSE);
	VERIFY(fs.CopyFile("tests\\test3.dat", "tests\\test3.dat") == FALSE);
	VERIFY(fs.CopyFile("tests\\test1.dat", "tests\\subdir1\\test2.dat"));
	VERIFY(fs.CopyFile("tests\\test1.dat", "tests\\subdir2\\test2.dat"));

	// Test file comparisons.
	VERIFY(fs.CompareFiles("tests\\test1.dat", "tests\\test1.dat"));
	VERIFY(fs.CompareFiles("tests\\test1.dat", "tests\\test3.dat", 20000));
	VERIFY(fs.CompareFiles("tests\\test1.dat", "tests\\test3.dat"));
	VERIFY(fs.DeleteFile("tests\\test3.dat"));
	VERIFY(fs.CompareFiles("tests\\test1.dat", "tests\\test3.dat") == FALSE);
	VERIFY(fs.CompareFiles("tests\\test2.dat", "tests\\test3.dat") == FALSE);

	// Test file renaming.
	VERIFY(fs.RenameFile("tests\\test1.dat", "tests\\test1.dat"));
	VERIFY(fs.CopyFile("tests\\test1.dat", "tests\\test3.dat"));
	VERIFY(fs.RenameFile("tests\\test3.dat", "tests\\test4.dat") == TRUE);
	VERIFY(fs.RenameFile("tests\\test2.dat", "tests\\test4.dat") == FALSE);
	VERIFY(fs.CopyFile("tests\\test2.dat", "tests\\test5.dat"));
	VERIFY(fs.RenameFile("tests\\test4.dat", "tests\\test5.dat") == FALSE);
	VERIFY(fs.DeleteFile("tests\\test4.dat"));
	VERIFY(fs.DeleteFile("tests\\test5.dat"));

	// Test directory deletion.
	VERIFY(fs.DeleteDirectory("tests\\subdir1", FALSE) == FALSE);
	VERIFY(fs.DeleteDirectory("tests\\subdir1", TRUE));
	VERIFY(fs.DeleteDirectory("tests\\subdir2", TRUE));
	VERIFY(fs.DeleteDirectory("tests\\subdir3", TRUE));
	VERIFY(fs.DeleteDirectory("tests\\subdir4", TRUE) == FALSE);
	VERIFY(fs.DeleteDirectory("tests\\subdir4", FALSE) == FALSE);

	// Test filesystem changing.
	CString FileSystem = fs.GetCurrentFileSystem();
	if (fs.FileSystemExists("x:\\") == FALSE)
	{
		VERIFY(fs.ChangeFileSystem("x:\\") == FALSE);
	} // if
	if (fs.FileSystemExists("y:\\") == FALSE)
	{
		VERIFY(fs.ChangeFileSystem("y:\\") == FALSE);
	} // if
	if (fs.FileSystemExists("z:\\") == FALSE)
	{
		VERIFY(fs.ChangeFileSystem("z:\\") == FALSE);
	} // if
	VERIFY(fs.ChangeFileSystem(FileSystem));

	// Test Parsing Functions.
	VERIFY(fs.GetPath("c:\\foo\\bar\\what.txt") == "c:\\foo\\bar\\");
	VERIFY(fs.GetPath("c:\\foo\\bar\\") == "c:\\foo\\bar\\");
	VERIFY(fs.GetPath("c:\\foo\\bar") == "c:\\foo\\");

	VERIFY(fs.GetFileName("c:\\foo\\bar\\what.txt") == "what.txt");
	VERIFY(fs.GetFileName("what.txt") == "what.txt");
	VERIFY(fs.GetFileName("c:\\foo\\bar\\") == "");

	VERIFY(fs.GetExtension("c:\\foo\\bar\\what.txt") == "txt");
	VERIFY(fs.GetExtension("c:\\foo\\bar\\what.txt.john") == "john");

	VERIFY(fs.GetFileSystem("c:\\foo\\bar\\what.txt") == "c:\\");
	VERIFY(fs.GetFileSystem("c:") == "c:");
	VERIFY(fs.GetFileSystem("\\foo\\bar\\what.txt") == "");

	VERIFY(fs.GetBaseFileName("c:\\foo\\bar\\what.txt") == "what");
	VERIFY(fs.GetBaseFileName("what.txt") == "what");

	// Test file information functions.
	CFileStatus FileStatus;
	CTime time;
	unsigned long lSize;
	BYTE attribute;

	VERIFY(fs.GetFileStatus("tests\\test1.dat", FileStatus));
	VERIFY(fs.GetFileCreateTime("tests\\test1.dat", time));
	VERIFY(fs.GetFileModifyTime("tests\\test1.dat", time));
	VERIFY(fs.GetFileAccessTime("tests\\test1.dat", time));
	VERIFY(fs.GetFileSize("tests\\test1.dat", lSize));
	VERIFY(fs.GetFileAttribute("tests\\test1.dat", attribute));


	// Test FileSystem Volume information.
	CStringList *pFSList = fs.GetFileSystemList();
	for (POSITION pos = pFSList->GetHeadPosition(); pos != NULL; )
	{
		CString FSName = pFSList->GetNext(pos);

		CString VolLabel = fs.GetVolumeLabel(FSName);
		if (VolLabel != "")
		{
			printf("Volume Label for %s = %s\n", (const char *) FSName, (const char *) VolLabel);
		} // if
		else
		{
			printf("Volume Label for %s = << Not Available >>\n", (const char *) FSName);
		} // else

		printf("FileSystem Type = %ld\n", fs.GetFileSystemType(FSName));
		printf("FileSystem CWD = %s\n", fs.GetCurrentDirectory(FSName));

	} // for
	delete pFSList;
	pFSList = NULL;
} // main 

