/****************************************************************************
			    TSPHEUR.C (TSPHEUR.EXE)

		 System main functions are contained in this unit.
 ****************************************************************************/

#include <stdio.h>
#include <alloc.h>
#include <dos.h>
#include <stdlib.h>
#include <ctype.h>
#include <stdarg.h>
#include "defs.h"
#include "conorder.h"
#include "tspheur.h"

/* variables */
INT   tsp_stflag = 0;         /* source/term flag */
INT   tsp_hamilton = 0;       /* create hamilton cycle */
INT   tsp_disttype = 1;       /* 1 = manhatten length, 2 = distance formula */
INT   tsp_3d = 0;             /* three dimensional flag */
INT   tsp_errorocr = 0;       /* has a non-fatal error occured? */
FILE *tsp_out = NULL;         /* main output stream */
FILE *tsp_cityfile = NULL;    /* points to city locations file */
FILE *tsp_dumpfile = NULL;    /* info line dump file */
FILE *tsp_outfile = NULL;     /* text file to dump pins into */
CHAR  tsp_outname[100];       /* file name to dump pins to */
CHAR  tsp_cityfilename[100];  /* city file name */
CHAR  tsp_outfilename[100];   /* out file for connection order name */
INT   tsp_openfile = 0;       /* is a city or matrix file open? */
INT   tsp_lnum = 0;           /* current line number */
INT   tsp_rand = 0;           /* call random city function? */
INT   tsp_randx = 30;         /* x random range */
INT   tsp_randy = 30;         /* y random range */
INT   tsp_randz = 30;         /* z random range */
INT   tsp_doit = 0;           /* can heuristic be executed? */
LONG  tsp_lens[3];            /* lengths of created nets */
FLOAT tsp_times[5];           /* timing variables */

/* imports */
EXTERN CON_CITYTYPE *con_cities[CON_CITYMAX + 1];
EXTERN INT con_citycount, con_numexclude;
EXTERN INT con_sterm[3];

/* dos_time() returns seconds from midnight
 */
FLOAT dos_time()
    {
    UNION REGS r;
    FLOAT f;

    r.h.ah = 0x2c;
    CALL int86( 0x21, &r, &r );
    f = (FLOAT) r.h.ch * 3600 + (FLOAT) r.h.cl * 60 + (FLOAT) r.h.dh;
    return( f + (FLOAT) r.h.dl / 100 );
    }

/* dos_timelapse() calculates the ammount of time lapsed in a given period
 */
FLOAT dos_timelapse()
    {
    STATIC FLOAT t = 0;
    FLOAT r;

    r = dos_time() - t;
    t = dos_time();
    if( r < 0.00 )
	r = 0.00;
    return( r );
    }

/* fil_readline() reads a line from a filestream
 */
VOID fil_readline( CHAR *s, FILE *stream )
    {
    INT ok = 1, a;
    CHAR tmp[255];

    if( !s || !stream )
	return;
    strcpy( s, "" );
    strcpy( tmp, "" );
    while( ok )
	{
	tsp_lnum++;
	CALL fgets( tmp, 255, stream );
	tmp[strlen( tmp ) - 1] = 0;
	for( a = strlen( tmp ) - 1; (tmp[a] == ' ' && a > 0); a-- )
	    ;
	tmp[a + 1] = 0;
	if( tmp[a] == '+' || tmp[a] == '\\' || tmp[a] == '/' )
	    tmp[a] = 0;
	else
	    ok = 0;
	strcat( s, tmp );
	if( strlen( s ) == 0 )
	    ok = 1;
	if( feof( stream ) )
	    ok = 0;
	}
    }

/* mem_alloc() allocates memory and uses the mem_ok array to return if the
   allocation was successful.
 */
CHAR *mem_alloc( INT num )
    {
    CHAR *tmpptr;

    tmpptr = (CHAR *) malloc( num );
    if( !tmpptr )
	tsp_terminate( "Out of memory." );
    return( tmpptr );
    }

/* mem_free() frees a mem_alloc allocated pointer
 */
VOID mem_free( CHAR *ptr )
    {
    free( ptr );
    }

/* str_digits() returns true is isdigit is true for all charaters in s
 */
INT str_digits( CHAR *s )
    {
    INT a;

    if( !strlen( s ) )
	return( 0 );
    for( a = 0; a < strlen( s ); a++ )
	if( !isdigit( s[a] ) && s[a] != ' ' )
	    return( 0 );
    return( 1 );
    }

/* str_nextstr() places the next str terminated by ',' or CHAR c into tmp
 */
INT str_nextstr( CHAR *s, CHAR *tmp, CHAR c )
    {
    INT r, a;

    if( !strlen( s ) || !tmp )
	return( 0 );
    for( a = 0; (s[a] != ',' && s[a] != c && a < strlen( s )); a++ )
	;
    if( a != strlen( s ) )
	{
	strcpy( tmp, s );
	r = 2;
	if( s[a] == ',' )
	    r = 1;
	tmp[a] = 0;
	return( r );
	}
    return( 0 );
    }

/* str_same() returns true if two strings are identical except for case
 */
INT str_same( CHAR *s1, CHAR *s2 )
   {
   INT a;

   if( !strlen( s1 ) || !strlen( s2 ) )
       return( 0 );
   if( strlen( s1 ) != strlen( s2 ) )
       return( 0 );
   for( a = 0; a < strlen( s1 ); a++ )
       if( toupper( s1[a] ) != toupper( s2[a] ) )
	   return( 0 );
   return( 1 );
   }

/* str_skipspace() eliminates space proceeding a string
 */
VOID str_skipspace( CHAR *s )
    {
    INT a;

    if( !s )
	return;
    for( a = 0; (s[a] == ' ' && a < strlen( s )); a++ )
	;
    if( a != strlen( s ) )
	strcpy( s, s + a );
    }

/* tsp_argtostr() makes a string out of the command line arguments
 */
CHAR *tsp_argtostr( INT argc, CHAR *argv[] )
    {
    CHAR *ret;
    INT i;

    ret = mem_alloc( 255 );
    strcpy( ret, "" );
    for( i = 0; i < argc; i++ )
	 {
	 strcat( ret, argv[i] );
	 strcat( ret, " " );
	 }
    return( ret );
    }

/* tsp_dumpinfo() writes an information line to the end of the dump file
 */
VOID tsp_dumpinfo()
    {
    fprintf( tsp_dumpfile, "%3d %2d %6.2f %6.2f %6.2f %6.2f ",
	     con_citycount, con_numexclude,
	     tsp_times[1], tsp_times[2], tsp_times[3], tsp_times[4] );
    fprintf( tsp_dumpfile, "%5ld %5ld %5ld ",
	     tsp_lens[0], tsp_lens[1], tsp_lens[2] );
    fprintf( tsp_dumpfile, " f:%1d%1d%1d ",
	     tsp_stflag, tsp_disttype == 2, tsp_3d );
    if( tsp_rand )
	{
	if( tsp_outfile )
	    fprintf( tsp_dumpfile, "( %s )\n", tsp_outname );
	else
	    fprintf( tsp_dumpfile, " (%3d,%3d,%3d)\n",
		     tsp_randx, tsp_randx, tsp_3d ? tsp_randz : 0 );
	}
    if( tsp_openfile == 1 )
	fprintf( tsp_dumpfile, " ( city file )\n" );
    fclose( tsp_dumpfile );
    }

/* tsp_error() prints out a non fatal error
 */
VOID tsp_error( CHAR *s1, CHAR *s2 )
    {
    if( strlen( s1 ) )
	{
	fprintf( stdout, "Error: %s\n", s1 );
	if( strlen( s2 ) )
	    fprintf( stdout, "       %s\n", s2 );
	}
    tsp_errorocr = 1;
    }

/* tsp_parsenum() will return the ith argument as a null terminated string
 */
INT tsp_parsenum( CHAR *str, CHAR *ret, INT i )
    {
    INT a, b = 0, c = 0;
    CHAR tmp[255];

    if( !str || !ret )
	return( 0 );
    strcpy( tmp, str );
    for( a = 0; (a <= i && tmp[c]); a++ )
	 {
	 for( b = c; (tmp[b] == ' ' && tmp[b]); b++ )
	      ;
	 for( c = b; (tmp[c] != ' ' && tmp[c]); c++ )
	      ;
	 }
     if( b == c || a <= i )
	 return( 0 );
     tmp[c] = 0;
     strcpy( ret, tmp + b );
     return( 1 );
     }

/* tsp_parse() will parse a string of options contained in a string
 */
VOID tsp_parse( CHAR *str )
    {
    CHAR tmp[255], baseword[100], parenstr[100];
    INT ok = 1, a = 0, b, i, paren, pend, valid;
    FILE *tfile;

    strcpy( parenstr, NULL );
    while( ok )
	{
	valid = 0;
	a++;
	if( tsp_parsenum( str, tmp, a ) )
	    {
	    strcpy( baseword, tmp + 1 );
	    paren = 0;
	    for( i = 0; (i < strlen( baseword ) && !paren); i++ )
		 if( baseword[i] == '(' )
		     paren = i;
	    if( paren )
		{
		pend = 0;
		for( i = 0; (i < strlen( baseword ) && !pend); i++ )
		    if( baseword[i] == ')' )
			pend = i;
		strcpy( parenstr, baseword + paren + 1);
		baseword[paren] = 0;
		parenstr[pend - paren - 1] = 0;
		if( strlen( parenstr ) == 0 )
		    paren = 0;
		}
	    if( str_same( baseword, "dform" ) )
		{
		valid = 1;
		tsp_disttype = 2;
		}
	    if( str_same( baseword, "ston" ) )
		{
		valid = 1;
		tsp_stflag = 1;
		con_sterm[1] = 1;
		con_sterm[2] = 2;
		}
	    if( str_same( baseword, "hamil" ) )
		{
		valid = 1;
		tsp_hamilton = 1;
		}
	    if( str_same( baseword, "dim" ) )
		{
		valid = 1;
		tsp_3d = 0;
		i = 0;
		if( paren )
		    i = atoi( parenstr );
		if( i == 3 )
		    tsp_3d = 1;
		if( !paren || (i != 2 && i != 3) )
		    tsp_error( "Option -dim requires an argument",
			       "of either 2 or 3." );
		}
	    if( str_same( baseword, "ord" ) )
		{
		/* THIS OPTION IS NOT FULLY IMPLEMENTED */
		valid = 1;
		tfile = NULL;
		if( paren )
		    {
		    b = 0;
		    for( i = 0; i < strlen( parenstr ); i++ )
			if( parenstr[i] == '.' )
			    b = 1;
		    if( !b )
			strcat( parenstr, ".ord" );
		    tfile = fopen( parenstr, "w" );
		    if( tfile )
			{
			tsp_out = tfile;
			strcpy( tsp_outfilename, parenstr );
			}
		    else
			tsp_error( "Data stream of option -out",
				   "is illegal." );
		    }
		else
		    tsp_error( "Option -out requires an argument", "" );
		}
	    if( str_same( baseword, "dump" ) )
		{
		valid = 1;
		tfile = NULL;
		if( paren )
		    {
		    tfile = fopen( parenstr, "a" );
		    if( tfile )
			tsp_dumpfile = tfile;
		    else
			tsp_error( "Data stream of option -dump",
				   "can not be opened for append." );
		    }
		else
		    tsp_error( "Option -dump requires an argument.", "" );
		}
	    if( str_same( baseword, "file" ) )
		{
		valid = 1;
		tfile = NULL;
		if( paren )
		    {
		    tfile = fopen( parenstr, "w" );
		    strcpy( tsp_outname, parenstr );
		    if( tfile )
			tsp_outfile = tfile;
		    else
			tsp_error( "Access denied to file defined",
				   "in -file command." );
		    }
		else
		    tsp_error( "Option -file requires an argument.", "" );
		}
	    if( str_same( baseword, "cityfile" ) )
		{
		valid = 1;
		if( !tsp_openfile )
		    {
		    tfile = NULL;
		    if( paren )
			{
			b = 0;
			for( i = 0; i < strlen( parenstr ); i++ )
			    if( parenstr[i] == '.' )
				b = 1;
			if( !b )
			    strcat( parenstr, ".cty" );
			tfile = fopen( parenstr, "r" );
			if( tfile )
			    {
			    tsp_openfile = 1;
			    tsp_cityfile = tfile;
			    tsp_lnum = 0;
			    strcpy( tsp_cityfilename, parenstr );
			    b = 0;
			    for( i = 0; (i < strlen( parenstr ) && !b); i++ )
				if( parenstr[i] == '.' )
				    b = i;
			    parenstr[b] = 0;
			    sprintf( tmp, "tspheur -ord(%s)", parenstr );
			    tsp_parse( tmp );
			    }
			else
			    tsp_error( "Data stream of option -cityfile",
				       "can not be opened." );
			}
		    else
			tsp_error( "Option -cityfile requires an",
				   "argument." );
		    }
		else
		    tsp_error( "Option -cityfile found within",
			       "a city data file." );
		}
	    if( str_same( baseword, "rx" ) )
		{
		valid = 1;
		tsp_randx = 30;
		i = 0;
		if( paren )
		    i = atoi( parenstr );
		if( i >= 10 && i <= 100 )
		    tsp_randx = i;
		if( !paren || i < 10 || i > 100 )
		    tsp_error( "Option -rx requires an argument",
			       "from 10 to 100. Default 30 set." );
		}
	    if( str_same( baseword, "ry" ) )
		{
		valid = 1;
		tsp_randy = 30;
		i = 0;
		if( paren )
		    i = atoi( parenstr );
		if( i >= 10 && i <= 100 )
		    tsp_randy = i;
		if( !paren || i < 10 || i > 100 )
		    tsp_error( "Option -ry requires an argument",
			       "from 10 to 100. Default 30 set." );
		}
	    if( str_same( baseword, "rz" ) )
		{
		valid = 1;
		tsp_randz = 30;
		i = 0;
		if( paren )
		    i = atoi( parenstr );
		if( i >=10 && i <= 100 )
		    tsp_randz = i;
		if( !paren || i < 10 || i > 100 )
		    tsp_error( "Option -rz requires an argument",
			       "from 10 to 100. Default 30 set." );
		}
	    if( str_same( baseword, "random" ) )
		{
		valid = 1;
		if( !tsp_openfile )
		    {
		    con_citycount = 50;
		    tsp_doit = 1;
		    tsp_rand = 1;
		    i = 0;
		    if( paren )
			i = atoi( parenstr );
		    if( i >= 10 && i <= CON_CITYMAX )
			con_citycount = i;
		    if( !paren || i < 10 || i > CON_CITYMAX )
			{
			sprintf( tmp, "from 10 to %d. Default 50 set.",
				 CON_CITYMAX );
			tsp_error( "Option -random requires an argument", tmp );
			}
		    }
		else
		    tsp_error( "Option -random found within a",
			       "city data file." );
		}
	    if( !valid )
		{
		sprintf( parenstr, "Invalid option. '%s'", tmp );
		tsp_error( parenstr, "" );
		}
	    }
	else
	    {
	    ok = 0;
	    if( a == 1 )
		{
		tsp_printf( "Optional commands.\n" );
		tsp_printf( "------------------\n" );
		tsp_printf( "-dform   = use distance formula\n" );
		tsp_printf( "-ston    = source/term on\n" );
		tsp_printf( "-hamil   = create hamilton cycle\n" );
		tsp_printf( "-dim(n)  = set dimensions to n\n" );
		tsp_printf( "-r?(n)   = set x, y, z random range\n" );
		tsp_printf( "-dump(f) = dump info line to file f\n" );
		tsp_printf( "-ord(f)  = output connect order to f\n" );
		tsp_printf( "-file(f) = output random cities\n\n" );
		tsp_printf( "One of the following is required.\n" );
		tsp_printf( "---------------------------------\n" );
		tsp_printf( "-cityfile(f)   = read cities from f\n" );
		tsp_printf( "-random(n)     = use n random cities\n" );
		tsp_printf( "\n" );
		exit( 1 );
		}
	    }
	}
    }

/* tsp_prepare() prepares for a run of the heuristic. It properly sets
   source/term and hamilton cycle variables
 */
VOID tsp_prepare()
    {
    if( (tsp_stflag || tsp_hamilton) )
	{
	if( tsp_hamilton )
	    {
	    *con_cities[con_citycount + 1] = *con_cities[con_citycount];
	    con_citycount++;
	    tsp_stflag = 1;
	    con_sterm[1] = con_citycount - 1;
	    con_sterm[2] = con_citycount;
	    }
	con_cities[con_sterm[1]]->srctrm = 1;
	con_cities[con_sterm[2]]->srctrm = 1;
	}
    }

/* tsp_printf() is the formated output for this program
 */
VOID tsp_printf( CHAR *format, ... )
    {
    CHAR s[255];
    va_list ap;

    va_start( ap, format );
    CALL vsprintf( s, format, ap );
    va_end( ap );
    fprintf( stdout, s );
    if( tsp_out != stdout && tsp_out )
	fprintf( tsp_out, s );
    }

/* tsp_readcityfile() reads in and processes the contents of tsp_cityfile
 */
VOID tsp_readcityfile()
    {
    CHAR s[255], t[255];
    INT ax, ay, az, tx = 0, ty = 0, tz = 0, done = 0, err;

    fil_readline( s, tsp_cityfile );
    if( strlen( s ) )
	{
	tsp_parse( s );
	while( !done )
	    {
	    err = 0;
	    fil_readline( s, tsp_cityfile );
	    str_skipspace( s );
	    if( s[0] == '(' )
		{
		str_skipspace( s );
		strcpy( s, s + 1 );
		ax = str_nextstr( s, t, ')' );
		if( ax == 1 )
		    {
		    if( str_digits( t ) )
			tx = atoi( t );
		    else
			err = 1;
		    strcpy( s, s + strlen( t ) + 1 );
		    str_skipspace( s );
		    ay = str_nextstr( s, t, ')' );
		    if( ((ay == 1 && tsp_3d) || (ay == 2)) && !err )
			{
			if( str_digits( t ) )
			    ty = atoi( t );
			else
			    err = 1;
			strcpy( s, s + strlen( t ) + 1 );
			if( tsp_3d && !err )
			    {
			    str_skipspace( s );
			    az = str_nextstr( s, t, ')' );
			    if( az == 2 )
				tz = atoi( t );
			    else
				err = 1;
			    }
			}
		    else
			if( ay == 1 )
			    err = 3;
			else
			    err = 1;
		    }
		else
		    if( ax == 2 )
			err = 2;
		    else
			err = 1;
		}
	    else
		if( s[0] == '[' )
		    done = 2;
		else
		    if( !strlen( s ) )
			done = 1;
		    else
			err = 1;
	    switch( err )
		{
		case 1: sprintf( t, "Invalid character on line %d", tsp_lnum );
			sprintf( s, "of -cityfile stream." );
			break;
		case 2: sprintf( t, "Premature ')' on line %d", tsp_lnum );
			sprintf( s, "of -cityfile stream." );
			break;
		case 3: sprintf( t, "Option -dim(3) required for" );
			sprintf( s, "line %d of -cityfile stream.", tsp_lnum );
			break;
		default: break;
		}
	    if( err )
		tsp_error( t, s );
	    else
		if( !done )
		    {
		    con_citycount++;
		    con_cities[con_citycount]->x = tx;
		    con_cities[con_citycount]->y = ty;
		    tz = tsp_3d ? tz : 0;
		    con_cities[con_citycount]->z = tz;
		    }
	    }
	}
    else
	tsp_error( "Option -cityfile stream does not",
		   "have any declarations." );
    if( con_citycount > 10  )
	tsp_doit = 1;
    else
	tsp_error( "Option -cityfile stream declares",
		   "less than 10 cities." );
    }

/* tsp_run() is the calls the heuristic once all variables are set
 */
VOID tsp_run()
    {
    tsp_prepare();


    CALL dos_timelapse();
    con_makewires();
    tsp_times[0] = dos_timelapse();
    tsp_printf( " (%.2f sec)\n", tsp_times[0] );
    con_sortwires();
    tsp_times[1] = dos_timelapse();
    tsp_printf( " (%.2f sec)\n", tsp_times[1] );
    tsp_lens[0] = con_mintree();
    tsp_times[2] = dos_timelapse();
    tsp_printf( " (%.2f sec)\n", tsp_times[2] );
    tsp_lens[1] = con_heuristic( 2 );
    tsp_times[3] = dos_timelapse();
    tsp_printf( " (%.2f sec)\n", tsp_times[3] );
    tsp_lens[2] = con_neighbor();
    tsp_times[4] = dos_timelapse();
    tsp_printf( " (%.2f sec)\n\n", tsp_times[4] );
    tsp_printf( " Min tree length: %5ld (%5.2f sec)\n",
	     tsp_lens[0], tsp_times[1] + tsp_times[2] );
    tsp_printf( "Heuristic length: %5ld (%5.2f sec)\n",
	     tsp_lens[1], tsp_times[1] + tsp_times[2] + tsp_times[3] );
    tsp_printf( " Neighbor length: %5ld (%5.2f sec)\n",
	     tsp_lens[2], tsp_times[1] + tsp_times[4] );
    if( tsp_dumpfile )
	tsp_dumpinfo();
    }

/* tsp_terminate() issues a terminal error message
 */
VOID tsp_terminate( CHAR *s )
    {
    fprintf( stdout, "\n\nTSP Fatal Error: %s\n\n", s );
    if( tsp_out )
	fclose( tsp_out );
    exit( 1 );
    }

/* tsp_writecities() writes out the cities used in a connection order.
 */
VOID tsp_writecities()
    {
    INT a, b;

    if( con_citycount % 2 != 0 )
	return; /* does not handle odd numbers */
    if( tsp_outfile )
	{
	fprintf( tsp_outfile, "   Pin      X      Y         Pin      X      Y\n");
	fprintf( tsp_outfile, "  Number  Coord  Coord      Number  Coord  Coord\n");
	for( a = 1; a < (con_citycount / 2 + 1); a++ )
	    {
	    b = a + (con_citycount + 1) / 2;
	    fprintf( tsp_outfile, "%3s%3d%5s%3d%4s%3d%8s%3d%5s%3d%4s%3d\n",
		     "", a, "", con_cities[a]->x, "", con_cities[a]->y,
		     "", b, "", con_cities[b]->x, "", con_cities[b]->y );
	    }
	fclose( tsp_outfile );
	}
    }

/* main() is where control flows into
 */
VOID main( INT argc, CHAR *argv[] )
    {
    tsp_printf( "\nDetrimental Wire Exclusion TSP Heuristic" );
    tsp_printf( "\n       (C) 1994 Paul J. Martino\n\n" );
    tsp_parse( tsp_argtostr( argc, argv ) );
    if( tsp_errorocr )
	fprintf( stdout, "\n" );
    tsp_errorocr = 0;
    if( tsp_out )
	{
	fprintf( tsp_out, "\nDetrimental Wire Exclusion TSP Heuristic" );
	fprintf( tsp_out, "\n       (C) 1994 Paul J. Martino\n\n" );
	}
    srand( (INT) dos_time() % 10000 );
    con_alloc();
    con_reset();
    if( tsp_rand )
	con_randcities();
    if( tsp_openfile == 1 )
	tsp_readcityfile();
    if( tsp_errorocr )
	fprintf( stdout, "\n" );
    if( tsp_doit )
	{
	tsp_run();
	if( tsp_outfile )
	    tsp_writecities();
	}
    else
	if( !tsp_openfile )
	    tsp_error( "Option -random or -cityfile",
		       "expected." );
    if( tsp_out )
	fclose( tsp_out );
    }


