program SolarSystemBetaVersion;

uses INPUT,
     VGA256,
     Crt;

const ScreenX = 160;
      ScreenY = 90;

      DaysPerMove: Real = 3;

type OrbitalPtr = ^OrbitalType;

     OrbitalType = Record
       Name: String[20];
       Year,
       X,
       Y,
       LastX,
       LastY,
       Radius,
       Speed,
       Mass,
       Distance: Real;
       Orbitals: Array[1..20] of OrbitalPtr;
     end;

var Sun:        OrbitalPtr;
    ZoomSun,
    ZoomSpace,
    ZoomPlanet: Real;
    ShowText:   Boolean;


function FindDistance (x1,y1,x2,y2: real): real;
begin
  FindDistance := Sqrt (Sqr(y2-y1) + Sqr(x2-x1));
end;



procedure SetUpData;
const Position : ARRAY[1..10,1..5] of real =

{       X      Y     LastX     LastY     Raidus
       (AU)   (AU)   (AU)      (AU)      (10,000*AU)  }
  ( (  0.000,  0,    0.000,    0.000,    46.52471),
    (  2.457,  0,    2.457,  -10.534,     0.16303),
    (  4.591,  0,    4.591,   -7.706,     0.40455),
    (  6.350,  0,    6.350,   -6.553,     0.42634),
    (  9.677,  0,    9.677,   -5.309,     0.22684),
    ( 33.039,  0,   33.039,   -2.874,     4.77279),
    ( 59.430,  0,   59.430,   -2.123,     4.01075),
    (121.806,  0,   121.806,  -1.496,     1.69788),
    (190.868,  0,   190.868,  -1.195,     1.62435),
    (250.444,  0,   250.444,  -1.043,     0.07520) );

    Name : ARRAY[1..10] of String[20] =
    ( 'Sun','Mercury','Venus','Earth','Mars',
      'Jupiter','Saturn','Uranus','Neptune','Pluto');

var i,j: byte;
begin
  New(Sun);
  For i := 1 to 20 do
    Sun^.Orbitals[i] := nil;
  Sun^.Name   := Name[1];
  Sun^.X      := Position[1,1];
  Sun^.Y      := Position[1,2];
  Sun^.Radius := Position[1,5];

  for j := 1 to 9 do
  begin
    New (Sun^.Orbitals[j]);
    For i := 1 to 20 do
      Sun^.Orbitals[j]^.Orbitals[i] := nil;
    Sun^.Orbitals[j]^.Name   := Name[j+1];
    Sun^.Orbitals[j]^.X      := Position[j+1,1];
    Sun^.Orbitals[j]^.Y      := Position[j+1,2];
    Sun^.Orbitals[j]^.LastX  := Position[j+1,3];
    Sun^.Orbitals[j]^.LastY  := Position[j+1,4] * DaysPerMove/60;
    Sun^.Orbitals[j]^.Radius := Position[j+1,5];
  end;
end;

procedure DisposeSolarSystem (Sun: OrbitalPtr);
var i: byte;
begin
  for i := 1 to 9 do
    Dispose (Sun^.Orbitals[i]);
  Dispose (Sun);
end;


procedure ShowSolarSystem (Xoffs,Yoffs: Real);

procedure DrawOrbital (Obj: OrbitalType; XOffs,YOffs: Real);
var i: byte;
    x,y: real;
begin
  x := Round ( ( Obj.X ) * ZoomSpace + ScreenX + XOffs);
  y := Round ( ( Obj.Y ) * ZoomSpace + ScreenY + YOffs);

  if Trunc (Obj.Radius) = 46
  then Circle ( Round(x),
                Round(y),
                Round ( Obj.Radius * ZoomSun / 10000),255,VADDR)
  else if (abs(round(x)-160) < 160) and (abs(round(y)-90) < 90)
  then Circle ( Round(x),
                Round(y),
                Round( Obj.Radius * ZoomPlanet / 10000), 255,VADDR);

   if ShowText then
   if Abs (FindDistance (Obj.X,Obj.Y,0,0)*ZoomSpace - 5) > 4.99999 then
   WriteGraphString ( Obj.Name,
                   Round ( ( Obj.X ) * ZoomSpace + ScreenX + XOffs ),
                   Round ( ( Obj.Y ) * ZoomSpace + ScreenY + YOffs ) - 4,
              255,VADDR);

  For i := 1 to 20 do if (Obj.Orbitals[i] <> nil)
    then DrawOrbital (Obj.Orbitals[i]^,Obj.X+XOffs,Obj.Y+YOffs);
end;

begin
  LineClip (Round(XOffs+ScreenX),0,Round(XOffs+ScreenX),200,1,VADDR);
  LineClip (0,Round(YOffs+ScreenY),320,Round(YOffs+ScreenY),1,VADDR);
  DrawOrbital (Sun^,XOffs,YOffs);
end;

Function PowerReal ( Base : real; Exponent : real ) : real;
begin {PowerReal}
  if (Base > 0) then
    PowerReal := Exp(Exponent*ln(Base))
  else
    begin {else}
    if (100*round(Exponent) = round(100*Exponent)) then
      begin {Integer check}
        if (round(Exponent) mod 2 = 0)
        then PowerReal := Exp(Exponent*ln(Abs(Base)))
        else PowerReal :=-Exp(Exponent*ln(Abs(Base)))
    end;
  end;
end;

function FindDeltaV (R: real): real;
begin
  FindDeltaV := PowerReal (6.35,3) / Sqr(R);
end;


procedure MoveSystem (Sun: OrbitalPtr; Days: real);

procedure MoveOrbital (var Obj: OrbitalType; Days: real);
var DeltaV,DeltaX,DeltaY,ChangeX,ChangeY,Angle: real;
begin
  with Obj do
  begin
    DeltaV := FindDeltaV(FindDistance(LastX,LastY,0,0))*sqr(days/60);       { Find Delta }
    if (LastX > 0) then Angle := ArcTan(LastY/LastX);        { Find Angle from comit }
    if (LastX < 0) then Angle := ArcTan(LastY/LastX) - pi;
    DeltaX := DeltaV*Cos(angle);            { Get X component for DeltaV }
    DeltaY := DeltaV*Sin(angle);            { Get Y component for DeltaV }
    ChangeX := ((LastX - X) - DeltaX);
    ChangeY := ((LastY - Y) - DeltaY);
    X := LastX;
    Y := LastY;
    LastX := ChangeX + LastX;
    LastY := ChangeY + LastY;
  end;
end;

var i: byte;
begin
  for i := 1 to 20 do
    if (Sun^.Orbitals[i] <> nil)
    then MoveOrbital (Sun^.Orbitals[i]^,DaysPerMove);
end;

procedure ShowData;
var S: String;
begin
  Str (ZoomSpace:4:2,S);
  WriteGraphString ('Space:  '+S,1,1,15,VADDR);
  Str (ZoomPlanet:4:2,S);
  WriteGraphString ('Planet: '+S,1,10,15,VADDR);
  Str (ZoomSun:4:2,S);
  WriteGraphString ('Sun:    '+S,1,20,15,VADDR);
end;

procedure SetColors;
begin
  setcolor (0,0,0,0); { black }
  setcolor (255,63,63,63); { white }
  setcolor (1,63,0,63); { purple }
end;


var ch,ch2: char;
    XOffs,YOffs: real;
begin
  SetupData;
  StartGraphics;
  SetUpVirtual;
  Cls (0,Vaddr);
  ShowText := True;
  ZoomSpace := 8;
  ZoomPlanet := 70000;
  ZoomSun := 2250;
  XOffs := 0;
  YOffs := 0;
  SetColors;
  repeat
  Cls(0,VADDR);
  MoveSystem (Sun,DaysPerMove);
  ShowSolarSystem(XOffs,YOffs);
  if ShowText then ShowData;
  GetKey(ch,ch2);
  case upcase(ch) of
    'I':  begin
            ZoomSpace := ZoomSpace * 1.5;
            ZoomPlanet := ZoomPlanet * 1.5;
            ZoomSun := ZoomSun * 1.5;
          end;
    'D':  begin
            ZoomSpace := ZoomSpace / 1.5;
            ZoomPlanet := ZoomPlanet / 1.5;
            ZoomSun := ZoomSun / 1.5;
          end;
    'T':  ShowText  := Not(ShowText);
  end;
  case ch2 of
    #80: YOffs := YOffs + 5;
    #72: YOffs := YOffs - 5;
    #77: XOffs := XOffs + 5;
    #75: XOffs := XOffs - 5;
    #59: ZoomPlanet := ZoomPlanet * 1.5;
    #60: ZoomPlanet := ZoomPlanet / 1.5;
    #61: ZoomSun := ZoomSun * 1.5;
    #62: ZoomSun := ZoomSun / 1.5;
  end;
  Flip (Vaddr,Vga);
  until ch = #27;
  ShutDown;                 { Destroy Virtual }
  StartText;                { Go to text mode }
  DisposeSolarSystem(Sun);  { Return memory to comuter }
end.

