

#ifndef _visualobj_h_
#define _visualobj_h_





/*
 *
 *          Copyright (C) 1994, M. A. Sridhar
 *  
 *
 *     This software is Copyright M. A. Sridhar, 1994. You are free
 *     to copy, modify or distribute this software  as you see fit,
 *     and to use  it  for  any  purpose, provided   this copyright
 *     notice and the following   disclaimer are included  with all
 *     copies.
 *
 *                        DISCLAIMER
 *
 *     The author makes no warranties, either expressed or implied,
 *     with respect  to  this  software, its  quality, performance,
 *     merchantability, or fitness for any particular purpose. This
 *     software is distributed  AS IS.  The  user of this  software
 *     assumes all risks  as to its quality  and performance. In no
 *     event shall the author be liable for any direct, indirect or
 *     consequential damages, even if the  author has been  advised
 *     as to the possibility of such damages.
 *
 */




#include "base/map.h"
#include "base/binding.h"

#include "ui/uidefs.h"
#include "ui/rectangl.h"
#include "ui/event.h"
#include "ui/cursor.h"
#include "ui/font.h"
#include "ui/color.h"

class UI_Dialog;
class UI_CompositeVObject;
class UI_Controller;
class UI_Application;
class UI_DisplaySurface;


// This  class is the root of the inheritance hierarchy for the GUI
// objects of YACL. It encapsulates the common features of a View (or a
// window). A View is any visible area on the desktop available to the 
// user for input, output or both. Every  View is divided into two parts-
// the client area and the non client area. As the name suggests the 
// client area is the one available to the user for interaction while
// the nonclient area is used by the system for its use.
// The coordinates of the rectangle are always interpreted relative to 
// the parent's top left corner.
//
// YACL does not allow views as C++ static objects. Nor can a VisualObject 
// be explicitly destroyed. That is, {\it never\/} do a {\tt delete} of a
// VisualObject or any of its derivatives! In order to destroy a
// VisualObject, it must
// be sent an {\small\tt Event_Destroy} via the controller. Before it is
// destroyed, 
// the method WantToQuit() is invoked; if the latter returns TRUE, then the
// controller destroys the view and all its children. This
// method may be overriden in derived classes. A VisualObject may also be
// unequivocally destroyed by sending it an Event_Quit.
//
// Every VisualObject inherits two static protected instance variables {\tt
// _Application} and {\tt _Controller}, which are respectively the
// application and controller pointers for this application.
// 
// Of special interest is the method HandleEvent (UI_Event* );
// All events originating in a VisualObject are passed to the HandleEvent
// method oh that VisualObject which provides default interpretation
// to these events by invoking the corresponding method outlined in
// the protected section. Being virtual, all these methods may be overwrriten
// in derived classes so that individual VisualObjects may interpret the
// same event differently. The HandleEvent () method may also be overriden-but
// must only be done after a clear understanding of how YACL works.
//
//     Every visual object exports a method \\
//          {\small\tt    UI_Font& UI_VisualObject::Font() }\\
// which returns a reference to the font used by the object. By default,
// each visual object uses the same font as its parent; the root of the
// window hierarchy creates a font when it is created, and all its
// descendants inherit this font. When a client program modifies the
// return value of the Font() method of a particular visual object v, this
// font change is propagated to all the descendants of v in
// the view tree; but the fonts used by v's ancestors are not affected.






enum UI_MouseButton { UIM_None = 0, UIM_Left, UIM_Middle, UIM_Right };

enum UIVObj_AttributeEnum {
    UIV_Title, UIV_Model, UIV_Shape, UIV_Enabled, UIV_Visible
};


class UI_VisualObject: public CL_Object {

public:

    // ----------------- Querying and setting attributes ----------------

    virtual CL_Object* Model ();
    // Return the model corresponding to the visual object.

    virtual CL_String& Title ();
    // Return the title. Note that the return value can be assigned to,
    // thereby modifying the title.

    virtual UI_Rectangle Area () const; 
    // Return the entire canvas comprising this view, including title bar etc. 

    virtual UI_Rectangle& ClientArea ();
    // Return the client area of the canvas. Note that assigning to
    // the return value causes the object to resize and/or reposition itself.


    virtual UI_CompositeVObject* Parent () const;
    // Return the visual object enclosing me. Return NULL if this is the
    // root of the visual object hierarchy.

    virtual UI_ViewID ViewID () const {return _id;} ;
    // Returns   the integer  ID  that  is  passed  to   the view during
    // construction.  Unlike the handle, the  ID may be used to identify
    // objects uniquely between  runs of the  same application. If no ID
    // is passed during to the  constructor, then the  handle and id are
    // the same. In this case, the id is thus assigned at run time.

    virtual bool WantToQuit ();
    // WantToQuit: return TRUE if this object can be closed down. The
    // default implementation returns TRUE. Can be overridden by derived
    // classes.

    // ------------------- View-related methods ---------------
    
    virtual bool Enable ();
    // Allow view to capture events
    // By default, every view is enabled at creation.

    virtual bool Disable ();
    // Prevent view from capturing events.

    virtual void MakeInvisible ();
    
    virtual void MakeVisible ();

    virtual bool IsVisible ();

    // ------------------ Display resource methods -------------------
    
    virtual UI_DisplaySurface* DisplaySurface ();
    // Return the display surface for this visual object

    UI_Cursor& Cursor ();
    // Return the cursor displayed when the mouse is on this
    // object. Whenever the mouse enters the client area of a particular
    // visual object, the controller sets the current cursor to be the one
    // returned by this method. (The only exception to this is when the
    // controller is currently in "wait" state, displaying a wait cursor.)
    
    virtual UI_Font&   Font ();
    // Font used by this object. The return value can be modified.

    UI_Color& Foreground(); 
    // The foreground color of the visual object.
    
    void Foreground (const UI_Color &c);
    // Set the foreground color of the visual object to the specified
    // color.
    
    UI_Color& Background();
    // The background color used by the visual object.
    
    void Background (const UI_Color& c);
    // Set the background color of the visual object to the specified
    // color.
    
    
    virtual char* MSWindowsName () const;

#if defined (__X_MOTIF__)
    virtual _WidgetClassRec *XName () const;

#endif


    // ------------------ Basic methods ----------------------------
    
    const char* ClassName () const { return "UI_VisualObject";};




    // ---------------- Initialization and termination -------------

    virtual void Initialize ();
    // Initialize: called by the Controller, just after the visual element
    // of this object has been created. Derived classes that override this
    // method must call Initialize on their parent class as the first action
    // in the overriding function.

    virtual void Finalize  () {};
    // Finalize: Called by the Controller to inform this object that is is
    // being  unconditionally shut down. It is
    // generally safer to override ConditionalFinalize() than to override
    // Finalize(). Finalize is called if (and immediately after) the
    // object's WantToQuit returns TRUE, and just before the object itself
    // is destroyed. Derived classes that override this
    // method must call Finalize on their parent class as the last action
    // in the overriding function.

protected:
    // 
    // Display Context methods:
    // 

    // A display context does not exist untill a call to CreateDisplayContex()
    // One thus created stays alive untill a call to DestroyDisplayObject()
    // or till the VisualObject terminates.

    virtual UI_DisplaySurface& CreateDisplaySurface ();

    virtual void DestroyDisplaySurface ();

    friend UI_Controller;

    // 
    // ------------------Construction---------------------
    // 

    UI_VisualObject (UI_CompositeVObject* parent, UI_ViewID id,
                    const UI_Rectangle& shape, long style = -1);
    // Create a view with the given rectangle as the canvas.
    // All Visualobjects are enabled at creation. The id, if assigned may be
    // used to uniquely identify the object between runs.

    UI_VisualObject (UI_CompositeVObject* parent, UI_ViewID id,
                     UI_ViewHandle handle = 0);
    // For resource-based construction (only under MS-Windows)

    ~UI_VisualObject ();



    // ------------------Event Handling------------------

    // The following event-handling methods are intended only for being
    // overridden by derived classes; they are never (and should never be)
    // called by any object except the Controller.
    
    virtual bool HandleEvent (UI_Event* anEvent);
    // HandleEvent: the "gateway" through which this object receives
    // events from the controller. This method is called by the controller on
    // receiving an event on this object (the event destination). This
    // virtual method is a dummy and simply calls the non-virtual method
    // ProcessEvent (UI_Event* e) that actually process all events.
    // Therefore, the HandleEvent method may be overridden in derived 
    // classes to capture special events and deal with them.However,
    // the overwritten method {\it must\/} invoke ProcessEvent(e)
    // to deal with events not handled by it. A sample implementation
    // of this method in a derived class may be:
    // \par{\small\begin{verbatim}
    //      bool UI_DerivedClass::HandleEvent (UI_Event *e)
    //      {
    //           if(e satisfies some condition) {
    //               do the special processing;
    //           
    //           } else
    //               return ProcessEvent(e);
    //      } 
    // \end{verbatim}
    // }\par

     
    bool ProcessEvent (UI_Event *e);
    // ProcessEvent: A non-virtual method that provides default
    // implementation for all events that occur on every object. Special
    // events, specific to a class, may be taken care of by the
    // HandleEvent() method.
 
    // The following comprise the entire set of events that are
    // recognised by any visual object.
    // Each event returns a bool response -TRUE if the event is
    // recognized and processed by it , FALSE otherwise. Each of the
    // following methods gets invoked by the ProcessEvent() method when
    // the corresponding event occurs on the object. If the object
    // wishes to recognize the event, it must provide an
    // appropriate implementation and return TRUE.
    // The default implementation provided in this top class, for each
    // of the methods, do nothing and return FALSE.
    // Point 'origin' refers to the upper leftmost corner of the 
    // enclosing visual object unless otherwise mentioned.
         

    // 
    // --------------------Hard Events------------------
    // 
 
    // 
    // Mouse Events:
    // 
   
   
    virtual bool ButtonDown (const UI_Point& curPos, UI_MouseButton btn,
                              bool shiftKey, bool ctrlKey); 
    // Inform the view that the mouse button 'btn' was pressed on it. The
    // parameters are the position of the mouse, the mouse button, and
    // whether the shift or control key was depressed at the time of click.

    virtual bool ButtonUp (const UI_Point& curPos, UI_MouseButton btn);
    // Inform the view that the mouse button 'btn' was released on it. The
    // parameters are the position of the mouse and the mouse button.
  
    virtual bool DoubleClick (const UI_Point& curPos, UI_MouseButton btn);
    // Inform the view that the left mouse button was double-clicked on it.
    // The parameter is where the left mouse button double clicked.

    virtual bool ViewEnter (const UI_Point& p);
    // Informs the view that the mouse has just moved into its client area.
    // (This has nothing to do with focus.)
    // The parameter is the point of entry.
 
    virtual bool ViewLeave (const UI_Point& p);
    // Inform the view that the mouse has just moved out of its client area. 
    // The parameter is the point of exit.

    virtual bool MouseMove (const UI_Point& cursorPos);
    // Inform view that the mouse has moved on its client area
 
    // 
    // KeyBoard Events: 
    // 

    virtual bool GetFocus ();
    // Inform the view that it has focus.

    virtual bool LoseFocus ();
    // Inform the view that it has lost focus.

    virtual bool KeyHit (char aKey);
    // Inform the view with current keyboard control that a key  has
    // been pressed. The paremeter is the ASCII value of the key.


    // 
    // Motion and Sizing Events:
    // 

    virtual bool Resize (long new_width, long new_height);
    // Inform view that it has been resized

    virtual bool ViewMove (const UI_Point& new_position);
    // Inform the view that it has been moved. The parameter is the new
    // position of the top left corner of the view.

    // Scrolling events:

    virtual bool Scroll (long position) {return FALSE;};

    virtual bool FinishScroll () {return FALSE;};
    
    virtual bool ScrollToBegin () {return FALSE;};
    
    virtual bool ScrollToEnd () {return FALSE;};
    
    virtual bool ScrollForwardLine () {return FALSE;};
    
    virtual bool ScrollBackwardLine () {return FALSE;};
    
    virtual bool ScrollForwardPage () {return FALSE;};
    
    virtual bool ScrollBackwardPage () {return FALSE;};
    
    virtual bool ScrollToPosition (long position) {return FALSE;};
    

    
    // 
    // ----------------Soft Events------------------------
    // 


    // 
    // Display:
    // 

    virtual bool Paint () {return FALSE;};
    // Inform view to display a representation of itself on the display
    // surface.

    virtual bool Select () { return FALSE; };
    // We received an Event_Select. This means that this VisualObject was
    // somehow "selected" -- exact semantics if this is left to derived
    // classes.

    // 
    // ------------- Creating and destroying the visual element -----
    //

    virtual bool MakeVisualElement ()
        {NotImplemented ("MakeVisualElement"); return FALSE;};
    // MakeVisualElement: create the visual aspect of this view. 

    virtual bool DestroyVisualElement ();

    virtual CL_String InstanceName ();
    // Used only under X windows, to specify the instance name for the
    // widget when creating it. The default implementation returns the value
    // returned by the Application's InstanceName method.


#if defined(__X_MOTIF__)
    virtual void _SetupStyle (void* arg, short& argn);
    // [YACL internal use only] X-windows-specific: set the resources for
    // this widget. Called by MakeVisualElement. The first parameter will be
    // a pointer to an Arg array.

#endif
    
    // ------------------- Miscellaneous methods -----------------------

    void SetShapeRectangle (const UI_Rectangle& r);
    // Set our shape rectangle. This method is to ensure that when we assign
    // to our shape rectangle, the shape doesn't notify us that we changed
    // it.
    
    void _MakeNewFont ();
    // _MakeNewFont: called to create a new font for ourselves

    UI_Font* OurFont ();
    // OurFont: return the font used by this object

    virtual bool _ShapeRectChanged (CL_Object&, long);
    // The application has invoked a method on our shape rectangle that
    // modified it; so our shape rectangle is telling us that it's modified.

    virtual bool _TitleChanged (CL_Object&, long);
    // TitleChanged: called by our title string to notify us that it has
    // changed. 

    virtual bool _FontChanged (CL_Object&, long);
    // The method that our font object uses to notify that someone
    // modified our font.

public:


    // -------------------- For YACL internal use only -------------------

    virtual UI_ViewHandle ViewHandle () const;
    // [YACL internal use only.]
    // Returns the integer handle that identifies this view. A handle
    // is assigned by the application at run time and may thus change
    // values between runs.

    virtual bool SetFont (UI_Font* font);
    // [Internal use only]
    // set the font used for this object in the underlying
    // window system. Return TRUE on success, FALSE on failure. Child
    // classes override this method according to their implementation.

#if defined(__MS_WINDOWS__)
    virtual void  SetStyle (ulong style);
    // [MS/Windows-specific; internal use only.]

    ulong Style () const {return _style;};
    // [MS/Windows-specific; internal use only.]

#elif defined (__X_MOTIF__)

    // -------------------- For YACL internal use only ------------------

    operator struct _WidgetRec* () { return _xwidget; };
    // [Internal use only]
    // Return the Widget corresponding to the visual object for the X
    // implementation.

#endif
    
    
protected:
    // 
    // ------------------Instance variables------------------
    // 
  
    UI_CompositeVObject* _parent;
    UI_Rectangle         _shape;
    CL_String            _title;
    
#if defined(__MS_WINDOWS__)
    UI_ViewHandle        _handle;
    long                 _style;

#elif defined(__X_MOTIF__)
    UI_ViewHandle        _xwidget;
    
#endif
    UI_ViewID            _id;              // An assigned ID used to uniquely
                                           // identify a view
    bool                 _created;         // Has the interface element for
                                           // this  object been created?
                                           // (Set by Initialize)
    bool                 _visible;         // Are we currently visible?
    UI_DisplaySurface*   _displaySurface;  // Not initialized until call to
                                           // CreateDisplaySurface
    UI_Cursor            _cursor;
    UI_Font*             _font;
    bool                 _ownFont;         // Do we own our font object?
    UI_Color             _bgColor;
    UI_Color             _fgColor;
    
    CL_Object*           _model  ; 


    //  Static instance variables:
    static UI_Controller*  _Controller;
    static UI_Application* _Application;
    
#if defined(__X_MOTIF__)    
    static _WidgetRec*     _shell;
    
#endif


private:
    void _Init (UI_CompositeVObject* parent, UI_ViewID id);
    

};


inline CL_Object* UI_VisualObject::Model()
{
    return _model;
}


inline UI_ViewHandle UI_VisualObject::ViewHandle() const
{
#if defined (__MS_WINDOWS__)
    return _handle;
    
#elif defined (__X_MOTIF__)
    return (UI_ViewHandle) (_xwidget);
    
#endif
}



inline UI_DisplaySurface* UI_VisualObject::DisplaySurface ()
{
    return _displaySurface;
}

inline UI_Cursor& UI_VisualObject::Cursor ()
{
    return _cursor;
}


inline UI_Color& UI_VisualObject::Foreground ()
{
    return _fgColor;
}

inline UI_Color& UI_VisualObject::Background ()
{
    return _bgColor;
}


inline bool UI_VisualObject::IsVisible ()
{
    return _visible;
}

inline UI_CompositeVObject* UI_VisualObject::Parent() const
{
    return _parent;
}

inline UI_Rectangle& UI_VisualObject::ClientArea()
{
    return _shape;
}

inline CL_String& UI_VisualObject::Title() 
{
    return _title;
}



#endif
