

#ifndef _controller_h_
#define _controller_h_





/*
 *
 *          Copyright (C) 1994, M. A. Sridhar
 *  
 *
 *     This software is Copyright M. A. Sridhar, 1994. You are free
 *     to copy, modify or distribute this software  as you see fit,
 *     and to use  it  for  any  purpose, provided   this copyright
 *     notice and the following   disclaimer are included  with all
 *     copies.
 *
 *                        DISCLAIMER
 *
 *     The author makes no warranties, either expressed or implied,
 *     with respect  to  this  software, its  quality, performance,
 *     merchantability, or fitness for any particular purpose. This
 *     software is distributed  AS IS.  The  user of this  software
 *     assumes all risks  as to its quality  and performance. In no
 *     event shall the author be liable for any direct, indirect or
 *     consequential damages, even if the  author has been  advised
 *     as to the possibility of such damages.
 *
 */



// Authors:   M. A. Sridhar
//            N. Bhowmik


#include "base/tree.h"

#include "ui/visualob.h"
#include "ui/rectangl.h"
#include "ui/event.h"
#include "ui/applic.h"
#include "ui/cursor.h"

// The Controller   class encapsulates the object  creation/disposal, event
// capturing/dispatching   details of a  YACL    application.  In any  YACL
// application, there    can   be  only   one    instance  of   the   class
// UI_Controller. This class owns and manages the central event queue.  All
// hard events are captured and added to the queue by this class while soft
// events generated  by other YACL objects are also added by it.  All event
// dispatching and disposal is taken care of by this class. In other words,
// any YACL object is free to create an  event and add it  to the queue via
// the controller. The disposal of this event,  after it has been processed
// is the controller's responsibility.  Similarly, all  object  disposal is
// also taken care of by the controller. As documented elsewhere, YACL does
// not allow  static  objects. Nor does  it  allow explicit  deletion.  Any
// object      that  needs  to      be     destroyed must    be    sent  an
// Event_Destroy or Event_Quit. The controller ensures that the object and ALL
// its  children are  then destroyed. It does  so by maintaining  a tree of
// all visual objects in the application.
// 
// In addition to  event and object management,  the controller also  provides
// methods for handling the mouse and keyboard.




class UI_Controller: public CL_Object {

public:

    UI_Controller (UI_Application* appl);

    ~UI_Controller ();


    //
    //----------Controller Services/events provided to a  VisualObject  
    //          

    //
    // -------------Mouse Control----------------------------
    //
    
    void GiveMouseTo (const UI_VisualObject& aView);
    // Direct all mouse events to aView irrespective of cursor position.

    void ReleaseMouse ();
    // Stop directing all mouse events to a particular view.


   

    //
    //--------------KeyBoard Control-------------------
    //
    
    void GiveFocusTo (const UI_VisualObject& v);
    // Direct all keyboard events to v.

    // ----------------------- Cursor display -----------------------

    void BeginWait ();
    // Begin a ``wait state'' by displaying the platform-specific wait
    // cursor. Until {\tt EndWait} is called, the mouse cursor will not change.

    void EndWait ();
    // End the wait state begun by {\tt BeginWait}.

    //
    //----------------Modal Input--------------------------
    //

     
    void Refresh ();
    // Paints the top window. (Unused?)
    
    void Beep ();
    // Provide a short beep.
    
    // ----------------- Event methods -------------------

    bool EventLoop (CL_AbstractBinding* termination = NULL,
                    CL_AbstractBinding* eventFilter = NULL);
    // A  generalized event loop that does the following: retrieve an event,
    // invoke eventFilter on that event, and dispatch the event if
    // eventFilter returns TRUE. Then consult the termination binding with
    // the event as parameter, and stop the loop if it returns TRUE.
    //
    //    The function returns immediately if there are no events available.
    // The return value of the function is FALSE if there are no events
    // available, and TRUE if the return occurred because either the root of
    // the view tree was destroyed, or because the termination binding
    // returned TRUE.
    //
    //    The default value of both parameters is NULL; a NULL termination
    // binding is assumed to always return FALSE, while a NULL eventFilter
    // binding is assumed to always return TRUE.
    
    void Run ();
    // The {\tt Run} method is simply a convenient way of iterating the
    // {\tt EventLoop} until the app gets a Quit event. Whenever the
    // {\tt EventLoop() }
    // method returns FALSE (indicating absence of events), this method
    // calls {\tt IdleAction ()} on the Application object.

    void DispatchPendingEvents ();
    // Dispatch all pending hard events. Must be called periodically when in
    // long processing loops when a progress dialog needs to be maintained.

    void AddEvent (UI_Event* );
    // Add an event to the event queue.

    // ----------------------- View methods ----------------------
    
    UI_CompositeVObject* Root () {return _root;};
    // Return the root of the view tree.

    CL_IntegerTree* ViewTree ();
    // Return a pointer to the view tree.
    
    UI_VisualObject* operator[] (UI_ViewHandle); 
    // Return the visual object with the given handle.

    void Register (UI_VisualObject* view, UI_CompositeVObject* parent);
    // Add a view to the visual object tree  as child of specified view. If
    // parent is NULL, this method assumes that the root is being created,
    // and that the first parameter is a composite.

    void MakeTopWindow (UI_CompositeVObject* root);

    bool Destroy (UI_VisualObject* aView);
    // Invoke the destructor of the view and those of the subtree rooted
    // at it, if any.

    long GetNextWidgetCount ();
    //
    

     // ###Can't think of any other
     // way to let handle be assigned in the constr. of VisualObj.
     // rather than by the controller. Will transfer this to Root obj.


    //
    //Initialize
    //

    
#if defined(__MS_WINDOWS__)
    short Initialize (HANDLE hInstance, HANDLE hPrevInstance, LPSTR lpCmdln,
                      short nCmdShow);
    // [MS/Windows-specific method]
    
#elif defined(__X_MOTIF__)
    short Initialize (int& argc,char *argv[]);
    // [X-windows-specific method]

    struct _WidgetRec* ShellWidget () { return _shell; };
    // [X-windows-specific method] For YACL internal use only.
    
#endif

    const char* ClassName () const { return "UI_Controller";};


protected:

    // ------- General event methods --------------


    bool ProcessSoftEvent (UI_Event* e);

    bool DispatchEvent (UI_Event* e);


    bool ProcessNativeEvent (CL_AbstractBinding* eventFilter);
    // ProcessNativeEvent: retrieve an event from the underlying window
    // system and process it if the eventFilter allows the processing.
    // Return TRUE if an event was available to 
    // process, and FALSE if not.

    UI_Event* RemoveEvent ();
    // Remove an event from the queue and return it.


    //
    // Instance variables:
    //
 
    UI_CompositeVObject* _root; 
    CL_IntegerTree*      _viewTree; // Maintains all views in their proper
                                    // hierarchical order

    CL_ObjectSequence*   _eventQueue;  //Hold events as they are generated
    UI_VisualObject*     _current;  // Object that is currently under the
                                    // mouse (not necessarily has focus)
    
    CL_IntPtrMap         _visualObjMap;

#if defined(__MS_WINDOWS__)
    HANDLE hInst;    
    HANDLE hPrevInst;

#elif defined(__X_MOTIF__)
    struct _WidgetRec *_shell;

    UI_VisualObject*   _focus; // The object currently in focus

    static void XEventHandler (struct _WidgetRec* w, void *, XEvent* xevent,
                               char *);
    // [X-windows-specific]

    
#endif
    short widgetCount;

private:
    void _InitController ();
    
    bool _SetCurrentCursor (UI_Cursor&);
    // Set the current cursor being shown. Return TRUE if a cursor was set,
    // FALSE otherwise.
    
    bool DisposeNode (const CL_Object&, long);
    // Destroy a view (used for traversing the view tree).

    bool RootDestroyed (CL_Object&, long);
    // Are we finished? (The parameters are dummy, needed in order to use
    // this method in a binding.)

    bool TranslateNativeEvent (NativeEventStruct& msg, UI_Event& event);
    // Translate the native even into a YACL event. Return TRUE if a knwon
    // event, FALSE otherwise.

    bool DispatchNativeEvent (UI_Event& e);
    // Dispatch the event. Return FALSE if default processing must be done
    // after this, TRUE otherwise. Most events require default processing
    // to be done; the one exception is when a window refuses to be closed.

#if defined(__MS_WINDOWS__)
    friend long FAR PASCAL YACLWindowProc (HWND ,unsigned, WORD, LONG);
    // [MS-Windows-specific]
    
    friend long FAR PASCAL YACLDialogProc (HWND ,unsigned, WORD, LONG);
    // [MS-Windows-specific]
    
    long WindowProc (HWND hwnd, unsigned msg, WORD wParam, LONG lp);
    // [MS-Windows-specific]
    
    long DialogProc (HWND hwnd, unsigned msg, WORD wParam, LONG lp);
    // [MS-Windows-specific]
    
#endif

    UI_Cursor _defaultCursor;
    bool      _inWaitState;


#if defined(__X_MOTIF__)
public:
    typedef char* String;
    static void LButtonDouble (struct _WidgetRec*,
                               XEvent*, String*, unsigned int*);
    // [X-windows-specific; internal use only]
    
    static void MButtonDouble (struct _WidgetRec*,
                               XEvent*, String*, unsigned int*);
    // [X-windows-specific; internal use only]
    
    static void RButtonDouble (struct _WidgetRec*,
                               XEvent*, String*, unsigned int*);
    // [X-windows-specific; internal use only]

    ulong RegisterTimeOut (long msec, void (*function) (void*, ulong*),
                           void* client_data);
    // [X-windows-specific; internal use only]

    void UnregisterTimeOut (ulong);
    // [X-windows-specific; internal use only]


    
#endif

};


inline CL_IntegerTree* UI_Controller::ViewTree ()
{
    return _viewTree;
}


#if defined(__MS_WINDOWS__)
long FAR PASCAL _export YACLDialogProc (HWND hwnd, unsigned msg, WORD
                                        wParam, LONG lp);
#endif


#endif
