// Simplified Windows Checkbox Control

// compilation parameter

const int BORDER = 4; // pixels between focus rectangle and box

#include <windows.h>

extern "C" {
  long far pascal _export SCheckBoxProc(HWND, UINT, WPARAM, LPARAM);
}

/*-----------------------------------------------------------------------------
This function registers the SCHECKBOX class. It is called automatically when
the program starts up.

The external variables _hPrev and _hInstance duplicate the arguments
hPrevInstance and hInstance, which are passed to WinMain(). If the startup
code does not supply these external variables, you must pass the arguments to
this function and call it explicitly before invoking any SCHECKBOX control.
-----------------------------------------------------------------------------*/

extern HINSTANCE _hPrev, _hInstance;

static void register_check_box(void)
{
  if (!_hPrev)
  {
    WNDCLASS w;
    w.cbClsExtra = 0;
    w.cbWndExtra = 2*sizeof(WORD);
    w.hbrBackground = (HBRUSH) COLOR_WINDOW + 1;
    w.hCursor = LoadCursor(NULL, IDC_ARROW);
    w.hIcon = NULL;
    w.hInstance = _hInstance;
    w.lpfnWndProc = SCheckBoxProc;
    w.lpszClassName = "SCHECKBOX";
    w.lpszMenuName = NULL;
    w.style = 0;
    RegisterClass(&w);
  }
}

#pragma startup register_check_box

// status bits                   bit is set if this control --

const int CHECKED     = 1<<0; // is checked
const int FOCUS       = 1<<1; // has the input focus
const int DISABLED    = 1<<2; // is disabled

/*-----------------------------------------------------------------------------
This function receives all messages directed to the control.
-----------------------------------------------------------------------------*/

long far pascal _export SCheckBoxProc(HWND handle, UINT message,
  WPARAM wParam, LPARAM lParam)
{
  WORD status = GetWindowWord(handle, 0);
  WORD size = GetWindowWord(handle, 2);
  switch (message)
  {
    case WM_CREATE:
    {
      size = ((CREATESTRUCT far *) lParam)->cy;
      if (((CREATESTRUCT far *) lParam)->cx < size)
        size = ((CREATESTRUCT far *) lParam)->cx;
      SetWindowWord(handle, 2, size);
      SetWindowWord(handle, 0,
        ((CREATESTRUCT far *) lParam)->style & WS_DISABLED ? DISABLED : 0);
      return 0;
    }
    case WM_PAINT:
    {
      PAINTSTRUCT paint;
      HDC dc = BeginPaint(handle, &paint);
      HPEN pen = CreatePen(PS_SOLID, 1,
        GetSysColor(status & DISABLED ? COLOR_GRAYTEXT : COLOR_WINDOWTEXT));
      HPEN old_pen = SelectObject(dc, pen);
      HBRUSH brush = CreateSolidBrush(GetSysColor(COLOR_WINDOW));
      HBRUSH old_brush = SelectObject(dc, brush);
      // draw box
      Rectangle(dc, BORDER, BORDER, size - BORDER, size - BORDER);
      // check by drawing an X inside the box
      if (status & CHECKED)
      {
        MoveTo(dc, BORDER, BORDER);
        LineTo(dc, size - (BORDER + 1), size - (BORDER + 1));
        MoveTo(dc, BORDER, size - (BORDER + 1));
        LineTo(dc, size - (BORDER + 1), BORDER);
      }
      // draw a focus rectangle around the box
      if (status & FOCUS)
      {
        RECT r;
        r.left = r.top = 0;
        r.right = r.bottom = size;
        DrawFocusRect(dc, &r);
      }
      SelectObject(dc, old_pen);
      DeleteObject(pen);
      SelectObject(dc, old_brush);
      DeleteObject(brush);
      EndPaint(handle, &paint);
      return 0;
    }
    case WM_ENABLE:
      SetWindowWord(handle, 0,
        wParam ? status & ~DISABLED : status | DISABLED);
      InvalidateRect(handle, NULL, TRUE);
      return 0;
    case WM_LBUTTONDOWN:
      SetWindowWord(handle, 0, status ^ CHECKED);
      InvalidateRect(handle, NULL, TRUE);
      SetFocus(handle);
      SendMessage(GetParent(handle), WM_COMMAND, GetMenu(handle),
        MAKELONG(handle, BN_CLICKED));
      return 0;
    case WM_SETFOCUS:
      SetWindowWord(handle, 0, status | FOCUS);
      InvalidateRect(handle, NULL, TRUE);
      return 0;
    case WM_KILLFOCUS:
      SetWindowWord(handle, 0, status & ~FOCUS);
      InvalidateRect(handle, NULL, TRUE);
      return 0;
    case WM_GETDLGCODE:
      return DLGC_BUTTON;
    case WM_KEYDOWN:
      if (wParam == ' ')
      {
        SetWindowWord(handle, 0, status ^ CHECKED);
        InvalidateRect(handle, NULL, TRUE);
        SetFocus(handle);
        SendMessage(GetParent(handle), WM_COMMAND, GetMenu(handle),
          MAKELONG(handle, BN_CLICKED));
        return 0;
      }
      break;
    case BM_SETCHECK:
      SetWindowWord(handle, 0, wParam ? status | CHECKED : status & ~CHECKED);
      InvalidateRect(handle, NULL, TRUE);
      return 0;
    case BM_GETCHECK:
      return status & CHECKED;
  }
  return DefWindowProc(handle, message, wParam, lParam);
}

