// Marquee Control

#include <windows.h>

// compilation parameters

const unsigned TIMER_INTERVAL = 100; // milliseconds
const unsigned BORDER_WIDTH = 6;     // pixels
const int BRUSH_SIZE = 8;            // pixels
const int SCALE_FACTOR = 10;

#include "marquee.h"

extern "C" {
  long far pascal _export MarqueeProc(HWND, UINT, WPARAM, LPARAM);
}

/*-----------------------------------------------------------------------------
This function registers the MARQUEE class. It is called automatically when
the program starts up.

The external variables _hPrev and _hInstance duplicate the arguments
hPrevInstance and hInstance, which are passed to WinMain(). If the startup
code does not supply these external variables, you must pass the arguments to
this function and call it explicitly before invoking any MARQUEE control.
-----------------------------------------------------------------------------*/

extern HINSTANCE _hPrev, _hInstance;

static void register_marquee(void)
{
  if (!_hPrev)
  {
    WNDCLASS w;
    w.cbClsExtra = 0;
    w.cbWndExtra = sizeof(HGLOBAL);
    w.hbrBackground = (HBRUSH) COLOR_WINDOW + 1;
    w.hCursor = LoadCursor(NULL, IDC_ARROW);
    w.hIcon = NULL;
    w.hInstance = _hInstance;
    w.lpfnWndProc = MarqueeProc;
    w.lpszClassName = "MARQUEE";
    w.lpszMenuName = NULL;
    w.style = CS_DBLCLKS;
    RegisterClass(&w);
  }
}

#pragma startup register_marquee

struct MARQUEE
{
  int position;
  int speed;
  unsigned timer;
  RECT top;
  RECT bottom;
  RECT left;
  RECT right;
  HFONT font;
  char text[1];
};

/*-----------------------------------------------------------------------------
This function receives all messages directed to the control.
-----------------------------------------------------------------------------*/

long far pascal _export MarqueeProc(HWND handle, UINT message,
  WPARAM wParam, LPARAM lParam)
{
  MARQUEE far *p;
  HGLOBAL hp = GetWindowWord(handle, 0);
  if (hp != NULL)
    p = (MARQUEE far *) GlobalLock(hp);
  switch (message)
  {
    case WM_CREATE:
    {
      hp = GlobalAlloc(GMEM_MOVEABLE | GMEM_ZEROINIT, sizeof(MARQUEE) +
        lstrlen(((CREATESTRUCT far *) lParam)->lpszName));
      if (hp == NULL)
        return -1L;
      p = (MARQUEE far *) GlobalLock(hp);
      p->speed = 20;
      {
        unsigned height = ((CREATESTRUCT far *) lParam)->cy;
        unsigned width = ((CREATESTRUCT far *) lParam)->cx;
        p->top.top = 0;
        p->top.bottom = BORDER_WIDTH-1;
        p->top.left = 0;
        p->top.right = width;
        p->bottom.top = height - (BORDER_WIDTH-1);
        p->bottom.bottom = height;
        p->bottom.left = 0;
        p->bottom.right = width;
        p->left.top = BORDER_WIDTH-2;
        p->left.bottom = height - (BORDER_WIDTH-2);
        p->left.left = 0;
        p->left.right = BORDER_WIDTH-1;
        p->right.top = BORDER_WIDTH-2;
        p->right.bottom = height - (BORDER_WIDTH-2);
        p->right.left = width - (BORDER_WIDTH-1);
        p->right.right = width;
      }
      p->timer = SetTimer(handle, 1, TIMER_INTERVAL, NULL);
      lstrcpy(p->text, ((CREATESTRUCT far *) lParam)->lpszName);
      SetWindowWord(handle, 0, (WORD) hp);
      GlobalUnlock(hp);
      return 0;
    }
    case WM_SETTEXT:
    {
      HGLOBAL new_hp = GlobalAlloc(GMEM_MOVEABLE | GMEM_ZEROINIT,
        sizeof(MARQUEE) + lstrlen((char far *) lParam));
      if (new_hp != NULL)
      {
        MARQUEE far *new_p = (MARQUEE far *) GlobalLock(new_hp);
        *new_p = *p;
        lstrcpy(new_p->text, (char far *) lParam);
        InvalidateRect(handle, NULL, TRUE);
        GlobalUnlock(hp);
        GlobalFree(hp);
        hp = new_hp;
      }
      GlobalUnlock(hp);
      return 0;
    }
    case WM_SETFONT:
    {
      p->font = (HFONT) wParam;
      InvalidateRect(handle, NULL, TRUE);
      GlobalUnlock(hp);
      return 0;
    }
    case WM_GETFONT:
      GlobalUnlock(hp);
      return (long) (WORD) p->font;
    case WM_PAINT:
    {
      PAINTSTRUCT paint;
      HDC dc = BeginPaint(handle, &paint);
      // display text
      {
        HFONT old_font;
        if (p->font != NULL)
          old_font = SelectObject(dc, p->font);
        SetTextColor(dc, GetSysColor(COLOR_WINDOWTEXT));
        SetBkColor(dc, GetSysColor(COLOR_WINDOW));
        TEXTMETRIC tm;
        GetTextMetrics(dc, &tm);
        char far *text = p->text;
        int y = BORDER_WIDTH+1;
        while (*text != 0)
        {
          unsigned length = 0;
          while (text[length] != 0 && text[length] != '\n')
            length++;
          TextOut(dc, BORDER_WIDTH+1, y, text, length);
          text += length;
          if (*text == '\n')
            text++;
          y += tm.tmHeight;
        }
        if (p->font != NULL)
          SelectObject(dc, old_font);
      }
      // draw border
      {
        HPEN old_pen = SelectObject(dc, GetStockObject(NULL_PEN));
        int HORIZONTAL_PATTERN[] =
          {0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F, 0x0F};
        HBITMAP horizontal_bitmap =
          CreateBitmap(BRUSH_SIZE, BRUSH_SIZE, 1, 1, HORIZONTAL_PATTERN);
        HBRUSH horizontal_brush = CreatePatternBrush(horizontal_bitmap);
        int VERTICAL_PATTERN[] =
          {0x00, 0x00, 0x00, 0x00, 0xFF, 0xFF, 0xFF, 0xFF};
        HBITMAP vertical_bitmap = CreateBitmap(BRUSH_SIZE, BRUSH_SIZE, 1, 1, VERTICAL_PATTERN);
        HBRUSH vertical_brush = CreatePatternBrush(vertical_bitmap);
        HBRUSH old_brush;
        unsigned offset = p->position/SCALE_FACTOR;
        UnrealizeObject(horizontal_brush);
        SetBrushOrg(dc, offset, 0);
        old_brush = SelectObject(dc, horizontal_brush);
        Rectangle(dc, p->top.left, p->top.top, p->top.right, p->top.bottom);
        SelectObject(dc, old_brush);
        UnrealizeObject(vertical_brush);
        SetBrushOrg(dc, 0, offset);
        SelectObject(dc, vertical_brush);
        Rectangle(dc, p->right.left, p->right.top, p->right.right,
          p->right.bottom);
        SelectObject(dc, old_brush);
        if (offset != 0) offset = BRUSH_SIZE - offset;
        UnrealizeObject(horizontal_brush);
        SetBrushOrg(dc, offset, 0);
        SelectObject(dc, horizontal_brush);
        Rectangle(dc, p->bottom.left, p->bottom.top, p->bottom.right,
          p->bottom.bottom);
        SelectObject(dc, old_brush);
        UnrealizeObject(vertical_brush);
        SetBrushOrg(dc, 0, offset);
        SelectObject(dc, vertical_brush);
        Rectangle(dc, p->left.left, p->left.top, p->left.right,
          p->left.bottom);
        SelectObject(dc, old_brush);
        SelectObject(dc, old_pen);
        DeleteObject(vertical_brush);
        DeleteObject(vertical_bitmap);
        DeleteObject(horizontal_brush);
        DeleteObject(horizontal_bitmap);
      }
      EndPaint(handle, &paint);
      GlobalUnlock(hp);
      return 0;
    }
    case MA_GETSPEED:
    {
      long n = p->speed;
      GlobalUnlock(hp);
      return n;
    }
    case MA_SETSPEED:
      if ((short) wParam > SCALE_FACTOR*BRUSH_SIZE - 1)
        wParam = SCALE_FACTOR*BRUSH_SIZE - 1;
      else if ((short) wParam < 1 - SCALE_FACTOR*BRUSH_SIZE)
        wParam = 1 - SCALE_FACTOR*BRUSH_SIZE;
      p->speed = (short) wParam;
      if (p->timer != NULL)
      {
        KillTimer(handle, p->timer);
        p->timer = NULL;
      }
      if (p->speed != 0)
        p->timer = SetTimer(handle, 1, TIMER_INTERVAL, NULL);
      // fall through to case WM_TIMER
    case WM_TIMER:
    {
      p->position += p->speed;
      while (p->position < 0)
        p->position += SCALE_FACTOR*BRUSH_SIZE;
      while (p->position >= SCALE_FACTOR*BRUSH_SIZE)
        p->position -= SCALE_FACTOR*BRUSH_SIZE;
      InvalidateRect(handle, &p->top, TRUE);
      InvalidateRect(handle, &p->bottom, TRUE);
      InvalidateRect(handle, &p->left, TRUE);
      InvalidateRect(handle, &p->right, TRUE);
      GlobalUnlock(hp);
      return 0;
    }
    case WM_GETTEXT:
    {
      unsigned i;
      for (i = 0; p->text[i] != 0 && i + 1 < wParam; i++)
        ((char far *) lParam)[i] = p->text[i];
      ((char far *) lParam)[i++] = 0;
      GlobalUnlock(hp);
      return i;
    }
    case WM_GETTEXTLENGTH:
    {
      unsigned length = lstrlen(p->text);
      GlobalUnlock(hp);
      return length;
    }
    case WM_DESTROY:
      if (p->timer != NULL) KillTimer(handle, p->timer);
      GlobalUnlock(hp);
      GlobalFree(hp);
      return 0;
    //case WM_ERASEBKGND:
    //  p->invalidate = TRUE;
    //  GlobalUnlock(hp);
    //  return 0;
  }
  GlobalUnlock(hp);
  return DefWindowProc(handle, message, wParam, lParam);
}


