;;; CLASS BROWSER FOR C++
;;; $Id: br-save.el,v 3.1 1995/02/17 18:19:36 mmann Exp $
;;;
;;; **********************************************************************
;;; Copyright (C) 1993, 1994 Gerd Moellmann. All rights reserved.
;;; Altenbergstr. 6, D-40235 Duesseldorf, Germany
;;; 100025.3303@COMPUSERVE.COM
;;; Suggestions, comments and requests for improvements are welcome.
;;; **********************************************************************
;;;
;;; This version works with both Emacs version 18 and 19, and I want
;;; to keep it that way. It requires the CL-19 Common Lisp compatibility
;;; package for Emacs 18 and 19.
;;;
;;; This file contains the code related to writing trees to disk.
;;; 

;; This file may be made part of the Emacs distribution at the option
;; of the FSF.

;; This code is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY.  No author or distributor
;; accepts responsibility to anyone for the consequences of using it
;; or for whether it serves any particular purpose or works at all,
;; unless he says so in writing.  Refer to the GNU Emacs General Public
;; License for full details.

;; Everyone is granted permission to copy, modify and redistribute
;; this code, but only under the conditions described in the
;; GNU Emacs General Public License.   A copy of this license is
;; supposed to have been given to you along with GNU Emacs so you
;; can know your rights and responsibilities.  It should be in a
;; file named COPYING.  Among other things, the copyright notice
;; and this notice must be preserved on all copies.

(require 'cl-19 "cl")
(require 'backquote)
(require 'br-macro)
(require 'br-struc)


;;;
;;; Write regular expressions to FILENAME. If FILENAME is NIL
;;; read a filename from the minibuffer.
;;;

;;###autoload
(defun tree-write-regexps (&optional filename)
  (interactive "FFile to save regexps in: ")
  (let ((temp-buffer (get-buffer-create "*Tree Output"))
	(old-standard-output standard-output)
	(tree @tree))
    (unwind-protect
	(save-excursion
	  (set-buffer (setq standard-output temp-buffer))
	  (erase-buffer)
	  (mapcar 'tree-write-class-regexps tree)
	  (write-file filename)
	  (message "Regexps written to file %s." filename))
      (kill-buffer temp-buffer)
      (setq standard-output old-standard-output))))

;;;
;;; Write regular expressions of a single class. Replace former
;;; regular expressions in the tree with buffer positions in the
;;; file written.
;;; 

(defun tree-write-class-regexps (class)
  (let ((point (point)))
    (message "%s..." (class-name (tree-class class)))
    (prin1 (class-file (tree-class class)))
    (prin1 (class-pattern (tree-class class)))
    (setf (class-pattern (tree-class class)) point
	  (class-file (tree-class class)) nil)
    (mapcar 'tree-write-class-regexps (tree-subclasses class))
    (dolist (func member-lists)
      (dolist (m (funcall func class))
	(setq point (point))
	(prin1 (member-file m))
	(prin1 (member-pattern m))
	(setf (member-pattern m) point
	      (member-file m) nil
	      point (point))
	(prin1 (member-definition-file m))
	(prin1 (member-definition-pattern m))
	(setf (member-definition-pattern m) point
	      (member-definition-file m) nil)))))

;;;
;;; Save tree to file is was loaded from.
;;; ###autoload


(defun tree-save ()
  "Save tree in same file it was loaded from."
  (interactive)
  (tree-write (or buffer-file-name @tags-filename)))

;;;
;;; Write tree to disk.
;;; 

;;###autoload
(defun tree-write (&optional filename)
  "Write the tree data structure to a file. Note that we have to make
up a local reference to the tree to be written because switching to
another buffer will make the local variable @TREE unavailable."
  (interactive "FFile to save tree in: ")
  (let ((temp-buffer (get-buffer-create "*Tree Output"))
	(old-standard-output standard-output)
	(header @header)
	(tree @tree))
    (unwind-protect
	(save-excursion
	  (set-buffer (setq standard-output temp-buffer))
	  (erase-buffer)
	  (insert (if (memq 'quiroz-cl browse-options)
		      "[tree-header " "[cl-struct-tree-header ")
		  (tree-header-version header)
		  (tree-header-regexp-file header)
		  "()]")
	  (insert " ")
	  (mapcar 'tree-write-class tree)
	  (write-file filename)
	  (message "Tree written to file %s." filename))
      (kill-buffer temp-buffer)
      (set-buffer-modified-p nil)
      (tree-update-mode-line)
      (setq standard-output old-standard-output))))

;;;
;;; Write single class to a buffer.
;;; 

(defun tree-write-class (class)
  (message "%s..." (class-name (tree-class class)))
  (insert "[tree ")
  (prin1 (tree-class class))		;class name
  (insert "(")				;list of subclasses
  (mapcar 'tree-write-class (tree-subclasses class))
  (insert ")")
  (dolist (func member-lists)
    (prin1 (funcall func class))
    (insert "\n"))
  (insert "()")				;superclasses slot
  (prin1 (tree-mark class))
  (insert "]\n"))

;;; end of `save.el'.