/*
** Program fdb - family tree database generation and manipulation
**
** Copyright (C) 1994 Andy Burrows 
**
**            email: cadellin@corum.me.man.ac.uk (130.88.29.14)
**
** This program is free software; you can redistribute it and/or modify it
** under the terms of the GNU General Public Licence as published by the Free
** Software Foundation; either version 1, or any later version.
** 
** This program is distributed in the hope that it will be useful, but
** WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
** or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
** more details. It should be included in this distribution in a file 
** called COPYING
**
*/

/*
   this is the main file for the family tree database
*/

/* include the standard headers */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

/* include the X11 and XView headers */

#include <X11/X.h>
#include <xview/xview.h>
#include <xview/frame.h>
#include <xview/panel.h>
#include <xview/canvas.h>
#include <xview/font.h>
#include <xview/scrollbar.h>
#include <xview/screen.h>
#include <xview/win_input.h>
#include <xview/svrimage.h>
#include <xview/icon.h>

/* include the database definitions */

#include "definitions.h"

/* include the function prototypes */

#include "prototypes.h"

/* include the main fdb icon information */

unsigned short main_icon_bits[] = {
#include "fdb.icon"
};

/* include the descent line icon information */

unsigned short descent_icon_bits[] = {
#include "descent.icon"
};

/* include the descendants tree icon information */

unsigned short tree_icon_bits[] = {
#include "tree.icon"
};

main(argc,argv)

int      argc;
char   **argv;
{
/* define local variables */

int             i;
char            temp_string[255];

/*
   initialise a few things
*/

gen_canvas_height = 1000;   /* height of the descent line paint window in pixels */
gen_canvas_width  = 1000;   /* width of the descent line paint window in pixels */
des_canvas_height = 1000;   /* height of the descendants tree paint window in pixels */
des_canvas_width  = 1000;   /* width of the descendants tree paint window in pixels */

/* set the initial minimum text width to 0 */

min_text_width = 0;

/* 
   initialise the max. number of generations in the plots
   MAX_INDICES => all available
*/

max_gen_generations = MAX_INDICES;
max_des_generations = MAX_INDICES;

current_index = -1;     /* index of the current database entry */

strcpy(file_name,"");   /* file name to load/save database */

first_time = 1; /* set the first time flag */ 

/* initialise output type to be PostScript */

output_type_flag = POSTSCRIPT;

/*
   initialise XView, allowing it to deal with any command line
   arguments which relate to it
*/

xv_init(XV_INIT_ARGC_PTR_ARGV, &argc, argv, NULL);

/* 
   if there is one argument left after XView has stripped off
   those relating to it, try to interpret it as a database file
*/

if(argc == 2)
    strcpy(new_file_name,argv[1]);
else
    strcpy(new_file_name,"");

/* 
   first time in, nullify all database entries and display structure pointers
*/

for(i=0;i<MAX_INDICES;i++)
    {
    entry[i]      = (struct db_record *)     NULL;
    generation[i] = (struct db_generation *) NULL;
    }

/************************************************************
                            FRAMES
************************************************************/

/*
   create the frame which will hold the entire fdb display
*/

sprintf(temp_string,"Family Tree Database V%s - cadellinsoft '94",FDB_VERSION);
fdb_frame = (Frame) xv_create(0, FRAME, 
          FRAME_LABEL,               temp_string,
          FRAME_NO_CONFIRM,          FALSE, /* requests confirmation before exit */
          FRAME_SHOW_FOOTER,         TRUE,
          FRAME_SHOW_RESIZE_CORNER,  FALSE, /* I don't want the user resizing the frame */
          XV_WIDTH,                  750,
          XV_HEIGHT,                 550,
/* old values
          XV_WIDTH,                  550,
          XV_HEIGHT,                 715,
*/
          NULL);

/* find the display and screen number associated with the base frame */

display       = (Display *) xv_get(fdb_frame, XV_DISPLAY);
screen        = (Xv_Screen) xv_get(fdb_frame, XV_SCREEN);
screen_number = (int)       xv_get(screen, SCREEN_NUMBER);

/* find the size of the screen  - the subtracted amounts allow for edging etc. */

max_window_width  = DisplayWidth(display, screen_number) - 25;
max_window_height = DisplayHeight(display, screen_number) - 50;

/*
   create the frame which will hold the descent line
*/

sprintf(temp_string,"Descent Line V%s - cadellinsoft '94",FDB_VERSION);
gen_frame = (Frame) xv_create(fdb_frame, FRAME, 
          FRAME_LABEL,               temp_string,
          FRAME_NO_CONFIRM,          TRUE, /* doesn't request confirmation before exit */
          FRAME_SHOW_FOOTER,         FALSE,
          FRAME_SHOW_RESIZE_CORNER,  FALSE, /* I don't want the user resizing the frame */
          XV_WIDTH,                  1024,
          XV_HEIGHT,                 630,
          XV_SHOW,                   FALSE,   /* initially invisible */
          NULL);

/* find the display associated with the descent line frame */

gen_display = (Display *) xv_get(gen_frame, XV_DISPLAY);

/*
   create the frame which will hold the descendants tree
*/

sprintf(temp_string,"Descendants Tree V%s - cadellinsoft '94",FDB_VERSION);
des_frame = (Frame) xv_create(fdb_frame, FRAME, 
          FRAME_LABEL,               temp_string,
          FRAME_NO_CONFIRM,          TRUE, /* doesn't request confirmation before exit */
          FRAME_SHOW_FOOTER,         FALSE,
          FRAME_SHOW_RESIZE_CORNER,  FALSE, /* I don't want the user resizing the frame */
          XV_WIDTH,                  1024,
          XV_HEIGHT,                 630,
          XV_SHOW,                   FALSE,   /* initially invisible */
          NULL);

/* find the display associated with the base frame */

des_display = (Display *) xv_get(des_frame, XV_DISPLAY);

/*
   create the frame which will hold the "Save as" file selection list
*/

save_frame = (Frame) xv_create(fdb_frame, FRAME, 
          FRAME_LABEL,       "Save as File",
          FRAME_SHOW_RESIZE_CORNER,  FALSE, /* I don't want the user resizing the frame */
          XV_WIDTH,           500,
          XV_HEIGHT,          330,
          XV_SHOW,            FALSE,   /* initially invisible */
          NULL);

/*
   create the frame which will hold the "Load From" file selection list
*/

load_frame = (Frame) xv_create(fdb_frame, FRAME, 
          FRAME_LABEL,       "Load From File",
          FRAME_SHOW_RESIZE_CORNER,  FALSE, /* I don't want the user resizing the frame */
          XV_WIDTH,           500,
          XV_HEIGHT,          330,
          XV_SHOW,            FALSE,   /* initially invisible */
          NULL);

/*
   create the frame which will hold the "Choose Entry" selection list
*/

entries_frame = (Frame) xv_create(fdb_frame, FRAME, 
          FRAME_SHOW_RESIZE_CORNER,  FALSE, /* I don't want the user resizing the frame */
          XV_WIDTH,           520,
          XV_HEIGHT,          300,
          XV_SHOW,            FALSE,   /* initially invisible */
          NULL);

/*
   create the frame which will hold the Error Popup
*/

error_frame = (Frame) xv_create(fdb_frame, FRAME, 
          FRAME_LABEL,        "Advisory!",
          FRAME_SHOW_RESIZE_CORNER,  FALSE, /* I don't want the user resizing the frame */
          XV_WIDTH,           450,
          XV_HEIGHT,          80,
          XV_SHOW,            FALSE,   /* initially invisible */
          NULL);

/*
   create the frame which will hold the Settings menu
*/

settings_frame = (Frame) xv_create(fdb_frame, FRAME, 
          FRAME_LABEL,        "Properties",
          FRAME_SHOW_RESIZE_CORNER,  FALSE, /* I don't want the user resizing the frame */
          XV_WIDTH,           650,
          XV_HEIGHT,          250,
          XV_SHOW,            FALSE,   /* initially invisible */
          NULL);

/************************************************************
                             ICONS
************************************************************/

/*
   create and set the main fdb icon
*/

main_image = (Server_image) xv_create(0, SERVER_IMAGE,
                XV_WIDTH,           64,
                XV_HEIGHT,          64,
                SERVER_IMAGE_BITS,  main_icon_bits,
                NULL);

main_icon = (Icon) xv_create(fdb_frame, ICON,
                XV_HEIGHT,          84,
                ICON_IMAGE,         main_image,
                NULL);

xv_set(fdb_frame, FRAME_ICON, main_icon, NULL);

/*
   create and set the descent line window icon
*/

descent_image = (Server_image) xv_create(0, SERVER_IMAGE,
                XV_WIDTH,           64,
                XV_HEIGHT,          64,
                SERVER_IMAGE_BITS,  descent_icon_bits,
                NULL);

descent_icon = (Icon) xv_create(gen_frame, ICON,
                ICON_IMAGE,         descent_image,
                NULL);

xv_set(gen_frame, FRAME_ICON, descent_icon, NULL);

/*
   create and set the descendants tree window icon
*/

tree_image = (Server_image) xv_create(0, SERVER_IMAGE,
                XV_WIDTH,           64,
                XV_HEIGHT,          64,
                SERVER_IMAGE_BITS,  tree_icon_bits,
                NULL);

tree_icon = (Icon) xv_create(des_frame, ICON,
                ICON_IMAGE,         tree_image,
                NULL);

xv_set(des_frame, FRAME_ICON, tree_icon, NULL);

/************************************************************
                            CANVASES
************************************************************/

/*
   create the canvas for displaying the family tree
*/

gen_canvas = (Canvas) xv_create(gen_frame, CANVAS,
              XV_X,                  0,
              XV_Y,                  0,
              XV_WIDTH,              1024,
              XV_HEIGHT,             600,
              CANVAS_X_PAINT_WINDOW, TRUE,          /* use Xlib to draw on canvas */
              CANVAS_REPAINT_PROC,   draw_line,
              CANVAS_AUTO_SHRINK,    FALSE,
              CANVAS_AUTO_EXPAND,    FALSE,
              CANVAS_WIDTH,          gen_canvas_width,  /* varies as family tree changes */
              CANVAS_HEIGHT,         gen_canvas_height, /* varies as family tree changes */
              WIN_RETAINED,          TRUE, /* let XView handle redraws due to scrolling */
              CANVAS_RETAINED,       TRUE, /* let XView handle redraws due to scrolling */
              WIN_COLLAPSE_EXPOSURES,  TRUE, /* collapse expose events together before
                                                calling the repaint procedure */
              OPENWIN_SPLIT,                 /* action when a view is split */
                  OPENWIN_SPLIT_INIT_PROC,  new_view_gen, 
              NULL,
              NULL);

/* add scrollbars */

(void) xv_create(gen_canvas, SCROLLBAR,
                SCROLLBAR_DIRECTION,   SCROLLBAR_VERTICAL,
                SCROLLBAR_SPLITTABLE,                  TRUE,
                NULL);

(void) xv_create(gen_canvas, SCROLLBAR,
                SCROLLBAR_DIRECTION,   SCROLLBAR_HORIZONTAL,
                SCROLLBAR_SPLITTABLE,                  TRUE,
                NULL);

/* establish the event callback and mask for the paint window */

xv_set(canvas_paint_window(gen_canvas),
                WIN_EVENT_PROC,        user_gen_event,  /* handles input to canvas window */
                WIN_X_EVENT_MASK,    
                    ButtonPressMask, NULL, 
                NULL);

/*
   create the canvas for displaying the descendants tree
*/

des_canvas = (Canvas) xv_create(des_frame, CANVAS,
              XV_X,                  0,
              XV_Y,                  0,
              XV_WIDTH,              1024,
              XV_HEIGHT,             600,
              CANVAS_X_PAINT_WINDOW, TRUE,          /* use Xlib to draw on canvas */
              CANVAS_REPAINT_PROC,   draw_descendants,
              CANVAS_AUTO_SHRINK,    FALSE,
              CANVAS_AUTO_EXPAND,    FALSE,
              CANVAS_WIDTH,          des_canvas_width,  /* varies as family tree changes */
              CANVAS_HEIGHT,         des_canvas_height, /* varies as family tree changes */
              WIN_RETAINED,          TRUE, /* let XView handle redraws due to scrolling */
              CANVAS_RETAINED,       TRUE, /* let XView handle redraws due to scrolling */
              WIN_COLLAPSE_EXPOSURES,  TRUE, /* collapse expose events together before
                                                calling the repaint procedure */
              OPENWIN_SPLIT,                 /* action when a view is split */
                  OPENWIN_SPLIT_INIT_PROC,  new_view_des, 
              NULL,
              NULL);

/* add scrollbars */

(void) xv_create(des_canvas, SCROLLBAR,
                SCROLLBAR_DIRECTION,   SCROLLBAR_VERTICAL,
                SCROLLBAR_SPLITTABLE,                  TRUE,
                NULL);

(void) xv_create(des_canvas, SCROLLBAR,
                SCROLLBAR_DIRECTION,   SCROLLBAR_HORIZONTAL,
                SCROLLBAR_SPLITTABLE,                  TRUE,
                NULL);

/* establish the event callback and mask for the paint window */

xv_set(canvas_paint_window(des_canvas),
                WIN_EVENT_PROC,        user_des_event,  /* handles input to canvas window */
                WIN_X_EVENT_MASK,    
                    ButtonPressMask, NULL, 
                NULL);

/* establish the font in the canvas paint windows */

/*
font = (Xv_Font) xv_find(fdb_frame, FONT, FONT_NAME, VIEW_FONT, NULL);
*/
font = (Xv_Font) xv_find(fdb_frame, FONT, 
                FONT_FAMILY,         FONT_FAMILY_ROMAN, 
                FONT_STYLE,          FONT_STYLE_BOLD,
                FONT_SIZE,           LARGER_FONT,
                NULL);

if(!font)
    {
/*    printf("cannot find font %s\n",VIEW_FONT); */
    printf("cannot find required standard font\n");
    /* use the base frames font if we can't find the desired font */
    font = (Xv_Font) xv_get(fdb_frame, XV_FONT);
    }

/* create a smaller version of the main font */

small_font = (Xv_Font) xv_find(fdb_frame, FONT, 
                FONT_FAMILY,         FONT_FAMILY_ROMAN, 
                FONT_STYLE,          FONT_STYLE_BOLD,
                FONT_SIZE,           SMALLER_FONT,
                NULL);

/*
small_font = (Xv_Font) xv_find(fdb_frame, FONT, 
                FONT_RESCALE_OF,     font,    WIN_SCALE_LARGE,
                NULL);
*/

if(!small_font)
    {
/*    printf("cannot find font %s\n",VIEW_FONT); */
    printf("cannot find required smaller font\n");
    /* use the base frames font if we can't find the desired font */
    font = (Xv_Font) xv_get(fdb_frame, XV_FONT);
    }

/* create a graphics context and set the normal font in it */

gc_values.font = (Font) xv_get(font, XV_XID);
gc_values.foreground = BlackPixel(display, DefaultScreen(display));
gc_values.background = WhitePixel(display, DefaultScreen(display));
gc = XCreateGC(display, RootWindow(display, DefaultScreen(display)), 
         GCForeground | GCBackground | GCFont, &gc_values);

/* create a graphics context and set the small font in it */

gc_values.font = (Font) xv_get(small_font, XV_XID);
gc_values.foreground = BlackPixel(display, DefaultScreen(display));
gc_values.background = WhitePixel(display, DefaultScreen(display));
gc_small = XCreateGC(display, RootWindow(display, DefaultScreen(display)), 
         GCForeground | GCBackground | GCFont, &gc_values);

/************************************************************
                           PANELS
************************************************************/

/*
   create the main database entry control panel
*/

fdb_panel = (Panel) xv_create(fdb_frame, PANEL,
                XV_X,             0,
                XV_Y,             35,
                XV_WIDTH,         750,
                XV_HEIGHT,        550,
                XV_HELP_DATA,     "fdb:fdb",
                WIN_BORDER,     TRUE,
                NULL);

/*
   create the control panel to hold the row of menu buttons
*/

menu_panel = (Panel) xv_create(fdb_frame, PANEL,
                XV_X,             0,
                XV_Y,             0,
                XV_WIDTH,         750,
                XV_HEIGHT,        35,
                XV_HELP_DATA,     "fdb:menus",
                WIN_BORDER,     TRUE,
                NULL);
 
/*
   create the control panel for the "Load From File" selector
*/

load_panel = (Panel) xv_create(load_frame, PANEL,
                XV_WIDTH,                  500,
                XV_HEIGHT,                 330,
                XV_HELP_DATA,              "fdb:load_file",
                NULL);

/*
   create the control panel for the "Save as File" selector
*/

save_panel = (Panel) xv_create(save_frame, PANEL,
                XV_WIDTH,                  500,
                XV_HEIGHT,                 330,
                XV_HELP_DATA,              "fdb:save_file",
                NULL);

/*
   create the control panel for the "Select Entry" selector
*/

entries_panel = (Panel) xv_create(entries_frame, PANEL,
                XV_WIDTH,                  520,
                XV_HEIGHT,                 300,
                XV_HELP_DATA,              "fdb:select_entry",
                NULL);

/*
   create the control panel for the Error Popup
*/

error_panel = (Panel) xv_create(error_frame, PANEL,
                XV_WIDTH,           450,
                XV_HEIGHT,          80,
                XV_HELP_DATA,       "fdb:error_popup",
                NULL);

/*
   create the control panel for the Settings menu
*/

settings_panel = (Panel) xv_create(settings_frame, PANEL,
                XV_WIDTH,           650,
                XV_HEIGHT,          250,
                XV_HELP_DATA,       "fdb:settings_menu",
                NULL);

/*
   create the control panel for the descent display
*/

gen_panel = (Panel) xv_create(gen_frame, PANEL,
                XV_X,               0,
                XV_Y,               600,
                XV_WIDTH,           1024,
                XV_HEIGHT,          30,
                XV_HELP_DATA,       "fdb:descent_tree",
                NULL);

/*
   create the control panel for the descendants display
*/

des_panel = (Panel) xv_create(des_frame, PANEL,
                XV_X,               0,
                XV_Y,               600,
                XV_WIDTH,           1024,
                XV_HEIGHT,          30,
                XV_HELP_DATA,       "fdb:descendants_tree",
                NULL);

/************************************************************
                      SCROLLING LISTS 
************************************************************/

/*
   list of files to select for loading database from
*/

load_list = (Panel_item) xv_create(load_panel, PANEL_LIST,
                XV_X,                      50,
                XV_Y,                      33,
                PANEL_CHOOSE_NONE,         TRUE,
                PANEL_LIST_WIDTH,          380,
                PANEL_LIST_DISPLAY_ROWS,   10,
                PANEL_NOTIFY_PROC,         load_file_select,
                XV_HELP_DATA,              "fdb:load_list",
                NULL);

/*
   list of files to select for saving database to
*/

save_list = (Panel_item) xv_create(save_panel, PANEL_LIST,
                XV_X,                      50,
                XV_Y,                      33,
                PANEL_CHOOSE_NONE,         TRUE,
                PANEL_LIST_WIDTH,          380,
                PANEL_LIST_DISPLAY_ROWS,   10,
                PANEL_NOTIFY_PROC,         save_file_select,
                XV_HELP_DATA,              "fdb:save_list",
                NULL);

/*
   list of spouses for display selection
*/

spouse_list = (Panel_item) xv_create(fdb_panel, PANEL_LIST,
                PANEL_LABEL_STRING,        "Spouse(s):",
                XV_X,                      X_OFFSET,
                XV_Y,                      Y_OFFSET + 8 * Y_SEPARATION,
                PANEL_CHOOSE_NONE,         TRUE,
                PANEL_LIST_WIDTH,          260,
                PANEL_LIST_DISPLAY_ROWS,   3,
                PANEL_NOTIFY_PROC,         display_spouse,
                XV_HELP_DATA,              "fdb:spouse_list",
                NULL);

/*
   list of children for display selection
*/

child_list = (Panel_item) xv_create(fdb_panel, PANEL_LIST,
                PANEL_LABEL_STRING,        "Child(ren):",
                XV_X,                      X_OFFSET + 360,
                XV_Y,                      Y_OFFSET + 8 * Y_SEPARATION,
                PANEL_CHOOSE_NONE,         TRUE,
                PANEL_LIST_WIDTH,          260,
                PANEL_LIST_DISPLAY_ROWS,   3,
                PANEL_NOTIFY_PROC,         display_child,
                XV_HELP_DATA,              "fdb:child_list",
                NULL);

/*
   list of entries for selection
*/

entries_list = (Panel_item) xv_create(entries_panel, PANEL_LIST,
                XV_X,                      0,
                XV_Y,                      0,
                PANEL_CHOOSE_NONE,         TRUE,
                PANEL_LIST_WIDTH,          500,
                PANEL_LIST_DISPLAY_ROWS,   13,
                PANEL_NOTIFY_PROC,         entry_select,
                XV_HELP_DATA,              "fdb:entry_list",
                NULL);

/************************************************************
                          BUTTONS 
************************************************************/

/*
   create the "cancel load from file" selection button
*/

load_cancel_button = (Panel_item) xv_create(load_panel, PANEL_BUTTON,
                XV_X,                     200,
                XV_Y,                     290,
                PANEL_LABEL_STRING,       "Cancel Load",
                PANEL_NOTIFY_PROC,        cancel_load_from,
                XV_HELP_DATA,             "fdb:load_cancel_button",
                NULL);

/*
   create the "cancel save as file" selection button
*/

save_cancel_button = (Panel_item) xv_create(save_panel, PANEL_BUTTON,
                XV_X,                     200,
                XV_Y,                     290,
                PANEL_LABEL_STRING,       "Cancel Save",
                PANEL_NOTIFY_PROC,        cancel_save_as,
                XV_HELP_DATA,             "fdb:save_cancel_button",
                NULL);

/*
   create the "display mother" button
*/

display_mother_button = (Panel_item) xv_create(fdb_panel, PANEL_BUTTON,
                XV_X,                     X_OFFSET,
                XV_Y,                     Y_OFFSET + 7 * Y_SEPARATION,
                PANEL_NOTIFY_PROC,        display_mother,
                XV_HELP_DATA,             "fdb:display_mother_button",
                NULL);
                
/*
   create the "display father" button
*/

display_father_button = (Panel_item) xv_create(fdb_panel, PANEL_BUTTON,
                XV_X,                     X_OFFSET + 350,
                XV_Y,                     Y_OFFSET + 7 * Y_SEPARATION,
                PANEL_NOTIFY_PROC,        display_father,
                XV_HELP_DATA,             "fdb:display_father_button",
                NULL);
                
/*
   create the "cancel select entry" selection button
*/

cancel_select_button = (Panel_item) xv_create(entries_panel, PANEL_BUTTON,
                XV_X,                     200,
                XV_Y,                     270,
                PANEL_LABEL_STRING,       "Cancel Select",
                PANEL_NOTIFY_PROC,        cancel_selection,
                XV_HELP_DATA,             "fdb:cancel_selection_button",
                NULL);

/*
   create the "dismiss error message display" button
*/

error_ok_button = (Panel_item) xv_create(error_panel, PANEL_BUTTON,
                XV_X,                     170,
                XV_Y,                     40,
                PANEL_LABEL_STRING,       "Acknowledged",
                PANEL_NOTIFY_PROC,        dismiss_error_frame,
                XV_HELP_DATA,             "fdb:error_ok_button",
                NULL);

/*
   create the "dismiss settings menu display" button
*/

settings_ok_button = (Panel_item) xv_create(settings_panel, PANEL_BUTTON,
                XV_X,                     285,
                XV_Y,                     210,
                PANEL_LABEL_STRING,       "Dismiss",
                PANEL_NOTIFY_PROC,        dismiss_settings_frame,
                XV_HELP_DATA,             "fdb:settings_ok_button",
                NULL);

/*
   create the "dismiss descent display" button
*/

dismiss_gen_button = (Panel_item) xv_create(gen_panel, PANEL_BUTTON,
                PANEL_LABEL_STRING,       "Dismiss Display",
                PANEL_NOTIFY_PROC,        dismiss_gen_frame,
                XV_HELP_DATA,             "fdb:dismiss_gen_button",
                NULL);

/*
   create the "dismiss descendants display" button
*/

dismiss_des_button = (Panel_item) xv_create(des_panel, PANEL_BUTTON,
                PANEL_LABEL_STRING,       "Dismiss Display",
                PANEL_NOTIFY_PROC,        dismiss_des_frame,
                XV_HELP_DATA,             "fdb:dismiss_des_button",
                NULL);

/************************************************************
                 MENUS AND ASSOCIATED BUTTONS 
************************************************************/

/* 
   create the file menu and attach it to a button
*/

file_menu = (Menu) xv_create(0, MENU,
                MENU_NOTIFY_PROC,      file_menu_selection,
                MENU_STRINGS,          "Load", "", "Save", "Save as...", "", "Properties...", 
                                       "", "Exit",NULL,
                NULL);

(void) xv_create(menu_panel, PANEL_BUTTON,
                PANEL_LABEL_STRING,    "File",
                PANEL_ITEM_MENU,       file_menu,
                XV_HELP_DATA,          "fdb:file_menu",
                NULL);

/* find the menu item associated with the "Load" option */

load_file_item = (Menu_item) xv_find(file_menu, MENUITEM,
                MENU_STRING,          "Load",
                NULL);

/* find the menu item associated with the "Save as..." option */

save_file_item = (Menu_item) xv_find(file_menu, MENUITEM,
                MENU_STRING,          "Save as...",
                NULL);

/* 
   create the edit menu and attach it to a button
*/

edit_menu = (Menu) xv_create(0, MENU,
                MENU_NOTIFY_PROC,      edit_menu_selection,
                MENU_STRINGS,          "Add Mother Entry", "Add Father Entry", 
                                       "Add Spouse Entry", "Add Child Entry", "", 
                                       "Add New Entry", "",
                                       "Delete Current Entry", NULL,
                NULL);

(void) xv_create(menu_panel, PANEL_BUTTON,
                PANEL_LABEL_STRING,    "Add/Delete",
                PANEL_ITEM_MENU,       edit_menu,
                XV_HELP_DATA,          "fdb:edit_menu",
                NULL);

/* 
   create the select menu and attach it to a button
*/

select_menu = (Menu) xv_create(0, MENU,
                MENU_NOTIFY_PROC,      select_menu_selection,
                MENU_STRINGS,          "Select New Current Entry", "",
                                       "Select Mother Entry", "Select Father Entry",
                                       "Select Spouse Entry", "Select Child Entry", 
                                       NULL,
                NULL);

(void) xv_create(menu_panel, PANEL_BUTTON,
                PANEL_LABEL_STRING,    "Select",
                PANEL_ITEM_MENU,       select_menu,
                XV_HELP_DATA,          "fdb:select_menu",
                NULL);

/* 
   create the deselect menu and attach it to a button
*/

deselect_menu = (Menu) xv_create(0, MENU,
                MENU_NOTIFY_PROC,      deselect_menu_selection,
                MENU_STRINGS,          "Deselect Mother Entry", "Deselect Father Entry",
                                       "Deselect Spouse Entry", "Deselect Child Entry", 
                                       NULL,
                NULL);

(void) xv_create(menu_panel, PANEL_BUTTON,
                PANEL_LABEL_STRING,    "Deselect",
                PANEL_ITEM_MENU,       deselect_menu,
                XV_HELP_DATA,          "fdb:deselect_menu",
                NULL);

/* 
   create the output menu and attach it to a button
*/

output_menu = (Menu) xv_create(0, MENU,
                MENU_NOTIFY_PROC,      output_menu_selection,
                MENU_STRINGS,          "Output Current Person", 
                                       "Output Entire Database", 
                                       "Output Current Descent Line",
                                       "Output Current Descendants Tree",
                                       NULL,
                NULL);

/* 
   obtain handles on the descent line and descendants tree options so that they
   can be deactivated when output type is set to LaTeX
*/

plot_des_line_item = (Menu_item) xv_find(output_menu, MENUITEM,
                     MENU_STRING,          "Output Current Descent Line",
                     NULL);

plot_gen_tree_item = (Menu_item) xv_find(output_menu, MENUITEM,
                     MENU_STRING,          "Output Current Descendants Tree",
                     NULL);
   
(void) xv_create(menu_panel, PANEL_BUTTON,
                PANEL_LABEL_STRING,    "Output",
                PANEL_ITEM_MENU,       output_menu,
                XV_HELP_DATA,          "fdb:output_menu",
                NULL);

/* 
   create the family tree view menu and attach it to a button
*/

view_menu = (Menu) xv_create(0, MENU,
                MENU_NOTIFY_PROC,      view_menu_selection,
                MENU_STRINGS,          "View Male Descent Line", "View Female Descent Line",
                                       "", "View Descendants Tree",
                                       NULL,
                NULL);

(void) xv_create(menu_panel, PANEL_BUTTON,
                PANEL_LABEL_STRING,    "View",
                PANEL_ITEM_MENU,       view_menu,
                XV_HELP_DATA,          "fdb:view_menu",
                NULL);

/************************************************************
             DATABASE ENTRY/DISPLAY ELEMENTS 
************************************************************/

/*
   create the title selection abbreviated choice options
*/

title_choice = (Panel_item) xv_create(fdb_panel, PANEL_CHOICE_STACK,
                PANEL_LAYOUT,            PANEL_HORIZONTAL,
                PANEL_LABEL_STRING,      "Title:",
                PANEL_CHOICE_STRINGS,    titles[0], titles[1], titles[2],
                                         titles[3], titles[4], titles[5],
                                         titles[6], titles[7], titles[8],
                                         titles[9], titles[10], titles[11],
                                         titles[12], titles[13], titles[14],
                                         titles[15], titles[16], titles[17],
                                         titles[18], titles[19], titles[20],
                                         titles[21], titles[22], titles[23],
                                         titles[24], titles[25], titles[26],
                                         titles[27], titles[28], titles[29],
                                         NULL,
                PANEL_CHOICE_NCOLS,      6,  /* display options in 4 columns */
                PANEL_NOTIFY_PROC,       title_chosen,
                XV_X,                    X_OFFSET,
                XV_Y,                    Y_OFFSET - 5,
                XV_HELP_DATA,            "fdb:title_choice",
                NULL);

/*
   create the surname entry/display text item
*/

surname_text = (Panel_item) xv_create(fdb_panel, PANEL_TEXT,
                PANEL_LABEL_STRING,           "Surname:",
                PANEL_VALUE_DISPLAY_LENGTH,   20,
                PANEL_VALUE_STORED_LENGTH,    MAX_SURNAME - 2,
                PANEL_NOTIFY_STRING,          "\n\r\t",
                PANEL_NOTIFY_PROC,            enter_surname,
                XV_X,                         X_OFFSET + 150,
                XV_Y,                         Y_OFFSET,
                XV_HELP_DATA,                 "fdb:surname_text",
                NULL);

/*
   create the forenames entry/display text item
*/

forenames_text = (Panel_item) xv_create(fdb_panel, PANEL_TEXT,
                PANEL_LABEL_STRING,           "Forenames:",
                PANEL_VALUE_DISPLAY_LENGTH,   30,
                PANEL_VALUE_STORED_LENGTH,    MAX_FORENAMES - 2,
                PANEL_NOTIFY_STRING,          "\n\r\t",
                PANEL_NOTIFY_PROC,            enter_forenames,
                XV_X,                         X_OFFSET + 400,
                XV_Y,                         Y_OFFSET,
                XV_HELP_DATA,                 "fdb:forenames_text",
                NULL);

/*
   create the gender selection option
*/

gender_choice = (Panel_item) xv_create(fdb_panel, PANEL_CHOICE,
                PANEL_LABEL_STRING,           "Sex:",
                PANEL_CHOICE_STRINGS,         "Male", "Female", NULL,
                PANEL_NOTIFY_PROC,            gender_chosen,
                XV_X,                         X_OFFSET,
                XV_Y,                         Y_OFFSET - 5 + Y_SEPARATION,
                XV_HELP_DATA,                 "fdb:gender_choice",
                NULL);

/*
   create the status selection option
*/

status_choice = (Panel_item) xv_create(fdb_panel, PANEL_CHOICE,
                PANEL_LABEL_STRING,           "Status:",
                PANEL_CHOICE_STRINGS,         "Living", "Deceased", NULL,
                PANEL_NOTIFY_PROC,            status_chosen,
                XV_X,                         X_OFFSET + 450,
                XV_Y,                         Y_OFFSET - 5 + Y_SEPARATION,
                XV_HELP_DATA,                 "fdb:status_choice",
                NULL);

/*
   create the maiden name entry/display text item
*/

maiden_name_text = (Panel_item) xv_create(fdb_panel, PANEL_TEXT,
                PANEL_LABEL_STRING,           "Maiden Name:",
                PANEL_VALUE_DISPLAY_LENGTH,   20,
                PANEL_VALUE_STORED_LENGTH,    MAX_SURNAME - 2,
                PANEL_NOTIFY_STRING,          "\n\r\t",
                PANEL_NOTIFY_PROC,            enter_maiden_name,
                XV_X,                         X_OFFSET + 150,
                XV_Y,                         Y_OFFSET + Y_SEPARATION,
                XV_HELP_DATA,                 "fdb:maiden_name_text",
                NULL);

/*
   create the birth date entry/display text item
*/

birth_date_text = (Panel_item) xv_create(fdb_panel, PANEL_TEXT,
                PANEL_LABEL_STRING,           "Born:",
                PANEL_VALUE_DISPLAY_LENGTH,   18,
                PANEL_VALUE_STORED_LENGTH,    MAX_DATE - 2,
                PANEL_NOTIFY_STRING,          "\n\r\t",
                PANEL_NOTIFY_PROC,            enter_birth_date,
                XV_X,                         X_OFFSET,
                XV_Y,                         Y_OFFSET + 2 * Y_SEPARATION,
                XV_HELP_DATA,                 "fdb:birth_date_text",
                NULL);

/*
   create the birth place entry/display text item
*/

birth_place_text = (Panel_item) xv_create(fdb_panel, PANEL_TEXT,
                PANEL_LABEL_STRING,           "At:",
                PANEL_VALUE_DISPLAY_LENGTH,   35,
                PANEL_VALUE_STORED_LENGTH,    MAX_STRING_LENGTH - 2,
                PANEL_NOTIFY_STRING,          "\n\r\t",
                PANEL_NOTIFY_PROC,            enter_birth_place,
                XV_X,                         X_OFFSET + 230,
                XV_Y,                         Y_OFFSET + 2 * Y_SEPARATION,
                XV_HELP_DATA,                 "fdb:birth_place_text",
                NULL);

/*
   create the baptism date entry/display text item
*/

baptism_date_text = (Panel_item) xv_create(fdb_panel, PANEL_TEXT,
                PANEL_LABEL_STRING,           "Baptised:",
                PANEL_VALUE_DISPLAY_LENGTH,   18,
                PANEL_VALUE_STORED_LENGTH,    MAX_DATE - 2,
                PANEL_NOTIFY_STRING,          "\n\r\t",
                PANEL_NOTIFY_PROC,            enter_baptism_date,
                XV_X,                         X_OFFSET,
                XV_Y,                         Y_OFFSET + 3 * Y_SEPARATION,
                XV_HELP_DATA,                 "fdb:baptism_date_text",
                NULL);

/*
   create the baptism place entry/display text item
*/

baptism_place_text = (Panel_item) xv_create(fdb_panel, PANEL_TEXT,
                PANEL_LABEL_STRING,           "At:",
                PANEL_VALUE_DISPLAY_LENGTH,   35,
                PANEL_VALUE_STORED_LENGTH,    MAX_STRING_LENGTH - 2,
                PANEL_NOTIFY_STRING,          "\n\r\t",
                PANEL_NOTIFY_PROC,            enter_baptism_place,
                XV_X,                         X_OFFSET + 230,
                XV_Y,                         Y_OFFSET + 3 * Y_SEPARATION,
                XV_HELP_DATA,                 "fdb:baptism_place_text",
                NULL);

/*
   create the marriage date entry/display text item
*/

marriage_date_text = (Panel_item) xv_create(fdb_panel, PANEL_TEXT,
                PANEL_LABEL_STRING,           "Married:",
                PANEL_VALUE_DISPLAY_LENGTH,   18,
                PANEL_VALUE_STORED_LENGTH,    MAX_DATE - 2,
                PANEL_NOTIFY_STRING,          "\n\r\t",
                PANEL_NOTIFY_PROC,            enter_marriage_date,
                XV_X,                         X_OFFSET,
                XV_Y,                         Y_OFFSET + 4 * Y_SEPARATION,
                XV_HELP_DATA,                 "fdb:marriage_date_text",
                NULL);

/*
   create the marriage place entry/display text item
*/

marriage_place_text = (Panel_item) xv_create(fdb_panel, PANEL_TEXT,
                PANEL_LABEL_STRING,           "At:",
                PANEL_VALUE_DISPLAY_LENGTH,   35,
                PANEL_VALUE_STORED_LENGTH,    MAX_STRING_LENGTH - 2,
                PANEL_NOTIFY_STRING,          "\n\r\t",
                PANEL_NOTIFY_PROC,            enter_marriage_place,
                XV_X,                         X_OFFSET + 230,
                XV_Y,                         Y_OFFSET + 4 * Y_SEPARATION,
                XV_HELP_DATA,                 "fdb:marriage_place_text",
                NULL);

/*
   create the death date entry/display text item
*/

death_date_text = (Panel_item) xv_create(fdb_panel, PANEL_TEXT,
                PANEL_LABEL_STRING,           "Died:",
                PANEL_VALUE_DISPLAY_LENGTH,   18,
                PANEL_VALUE_STORED_LENGTH,    MAX_DATE - 2,
                PANEL_NOTIFY_STRING,          "\n\r\t",
                PANEL_NOTIFY_PROC,            enter_death_date,
                XV_X,                         X_OFFSET,
                XV_Y,                         Y_OFFSET + 5 * Y_SEPARATION,
                XV_HELP_DATA,                 "fdb:death_date_text",
                NULL);

/*
   create the death place entry/display text item
*/

death_place_text = (Panel_item) xv_create(fdb_panel, PANEL_TEXT,
                PANEL_LABEL_STRING,           "At:",
                PANEL_VALUE_DISPLAY_LENGTH,   35,
                PANEL_VALUE_STORED_LENGTH,    MAX_STRING_LENGTH - 2,
                PANEL_NOTIFY_STRING,          "\n\r\t",
                PANEL_NOTIFY_PROC,            enter_death_place,
                XV_X,                         X_OFFSET + 230,
                XV_Y,                         Y_OFFSET + 5 * Y_SEPARATION,
                XV_HELP_DATA,                 "fdb:death_place_text",
                NULL);

/*
   create the resting place entry/display text item
*/

resting_place_text = (Panel_item) xv_create(fdb_panel, PANEL_TEXT,
                PANEL_LABEL_STRING,           "Resting Place:",
                PANEL_VALUE_DISPLAY_LENGTH,   50,
                PANEL_VALUE_STORED_LENGTH,    MAX_STRING_LENGTH - 2,
                PANEL_NOTIFY_STRING,          "\n\r\t",
                PANEL_NOTIFY_PROC,            enter_resting_place,
                XV_X,                         X_OFFSET,
                XV_Y,                         Y_OFFSET + 6 * Y_SEPARATION,
                XV_HELP_DATA,                 "fdb:resting_place_text",
                NULL);

/*
   create the occupation entry/display text item
*/

occupation_text = (Panel_item) xv_create(fdb_panel, PANEL_TEXT,
                PANEL_LABEL_STRING,           "Occupation(s):",
                PANEL_VALUE_DISPLAY_LENGTH,   73,
                PANEL_VALUE_STORED_LENGTH,    MAX_OCCUPATION - 2,
                PANEL_NOTIFY_STRING,          "\n\r\t",
                PANEL_NOTIFY_PROC,            enter_occupation,
                XV_X,                         X_OFFSET,
                XV_Y,                         Y_OFFSET + 11 * Y_SEPARATION,
                XV_HELP_DATA,                 "fdb:occupation_text",
                NULL);

/*
   create the notes entry/display text item
*/

notes_text = (Panel_item) xv_create(fdb_panel, PANEL_TEXT,
                PANEL_LABEL_STRING,           "Notes:",
                PANEL_VALUE_DISPLAY_LENGTH,   80,
                PANEL_VALUE_STORED_LENGTH,    MAX_STRING_LENGTH - 2,
                PANEL_NOTIFY_STRING,          "\n\r\t",
                PANEL_NOTIFY_PROC,            enter_notes,
                XV_X,                         X_OFFSET,
                XV_Y,                         Y_OFFSET + 12 * Y_SEPARATION,
                XV_HELP_DATA,                 "fdb:notes_text",
                NULL);

/*
   create the birth date source text item
*/

birth_source_text = (Panel_item) xv_create(fdb_panel, PANEL_TEXT,
                PANEL_LABEL_STRING,           "Birth Source:",
                PANEL_VALUE_DISPLAY_LENGTH,   75,
                PANEL_VALUE_STORED_LENGTH,    MAX_STRING_LENGTH - 2,
                PANEL_NOTIFY_STRING,          "\n\r\t",
                PANEL_NOTIFY_PROC,            enter_birth_source,
                XV_X,                         X_OFFSET,
                XV_Y,                         Y_OFFSET + 13 * Y_SEPARATION,
                XV_HELP_DATA,                 "fdb:birth_source_text",
                NULL);

/*
   create the baptism date source text item
*/

baptism_source_text = (Panel_item) xv_create(fdb_panel, PANEL_TEXT,
                PANEL_LABEL_STRING,           "Baptism Source:",
                PANEL_VALUE_DISPLAY_LENGTH,   72,
                PANEL_VALUE_STORED_LENGTH,    MAX_STRING_LENGTH - 2,
                PANEL_NOTIFY_STRING,          "\n\r\t",
                PANEL_NOTIFY_PROC,            enter_baptism_source,
                XV_X,                         X_OFFSET,
                XV_Y,                         Y_OFFSET + 14 * Y_SEPARATION,
                XV_HELP_DATA,                 "fdb:baptism_source_text",
                NULL);

/*
   create the marriage date source text item
*/

marriage_source_text = (Panel_item) xv_create(fdb_panel, PANEL_TEXT,
                PANEL_LABEL_STRING,           "Marriage Source:",
                PANEL_VALUE_DISPLAY_LENGTH,   72,
                PANEL_VALUE_STORED_LENGTH,    MAX_STRING_LENGTH - 2,
                PANEL_NOTIFY_STRING,          "\n\r\t",
                PANEL_NOTIFY_PROC,            enter_marriage_source,
                XV_X,                         X_OFFSET,
                XV_Y,                         Y_OFFSET + 15 * Y_SEPARATION,
                XV_HELP_DATA,                 "fdb:marriage_source_text",
                NULL);

/*
   create the date of death source text item
*/

death_source_text = (Panel_item) xv_create(fdb_panel, PANEL_TEXT,
                PANEL_LABEL_STRING,           "Death Source:",
                PANEL_VALUE_DISPLAY_LENGTH,   74,
                PANEL_VALUE_STORED_LENGTH,    MAX_STRING_LENGTH - 2,
                PANEL_NOTIFY_STRING,          "\n\r\t",
                PANEL_NOTIFY_PROC,            enter_death_source,
                XV_X,                         X_OFFSET,
                XV_Y,                         Y_OFFSET + 16 * Y_SEPARATION,
                XV_HELP_DATA,                 "fdb:death_source_text",
                NULL);

/*
   create the "Load From File" text entry
*/

load_file_text = (Panel_item) xv_create(load_panel, PANEL_TEXT,
                PANEL_LABEL_STRING,           "File name:",
                PANEL_VALUE_DISPLAY_LENGTH,   20,
                PANEL_VALUE_STORED_LENGTH,    MAX_STRING_LENGTH - 2,
                PANEL_NOTIFY_STRING,          "\n\r",
                PANEL_NOTIFY_PROC,            enter_load_file_text,
                XV_X,                         120,
                XV_Y,                         255, 
                NULL);

/*
   create the "Save as File" text entry
*/

save_file_text = (Panel_item) xv_create(save_panel, PANEL_TEXT,
                PANEL_LABEL_STRING,           "File name:",
                PANEL_VALUE_DISPLAY_LENGTH,   20,
                PANEL_VALUE_STORED_LENGTH,    MAX_STRING_LENGTH - 2,
                PANEL_NOTIFY_STRING,          "\n\r",
                PANEL_NOTIFY_PROC,            enter_save_file_text,
                XV_X,                         120,
                XV_Y,                         255, 
                NULL);

/**************
   MESSAGES
**************/

/*
   create the message which will contain the current working directory
   for the load file panel
*/

load_message = (Panel_item) xv_create(load_panel, PANEL_MESSAGE,
                XV_X,                         20,
                XV_Y,                         11,
                NULL);

/*
   create the message which will contain the current working directory
   for the save file panel
*/

save_message = (Panel_item) xv_create(save_panel, PANEL_MESSAGE,
                XV_X,                         20,
                XV_Y,                         11,
                NULL);

/*
   create the error message display item
*/

error_message_item = (Panel_item) xv_create(error_panel, PANEL_MESSAGE,
                XV_X,                         20,
                XV_Y,                         10,
                PANEL_LABEL_BOLD,             TRUE,
                PANEL_LABEL_STRING,           "",
                NULL);

/*********************************************
   CHOICES (not related to database input)
*********************************************/

/*
   create the output type choice
*/

output_choice = (Panel_item) xv_create(settings_panel, PANEL_CHOICE,
                PANEL_LABEL_STRING,           "Output Type:",
                PANEL_CHOICE_STRINGS,         "PostScript", "LaTeX", NULL,
                PANEL_NOTIFY_PROC,            output_type_chosen,
                XV_X,                         185,
                XV_Y,                         160,
                XV_HELP_DATA,                 "fdb:output_choice",
                NULL);

/*
   create the add spaces in text output choice entry
*/

add_spaces_choice = (Panel_item) xv_create(settings_panel, PANEL_CHECK_BOX,
                   XV_X,                    150,
                   XV_Y,                    125,
                   PANEL_LABEL_STRING,      "Add Spaces for Blank Entries in Text Output:",
                   PANEL_CHOICE_STRINGS,    "", NULL,
                   PANEL_NOTIFY_PROC,       add_spaces_chosen,
                   XV_HELP_DATA,            "fdb:add_spaces_choice",
                   NULL);

/*
   create the "use all available generations in descent line" tick box
*/

use_all_gen_choice = (Panel_item) xv_create(settings_panel, PANEL_CHECK_BOX,
                   XV_X,                    490,
                   XV_Y,                    15,
                   PANEL_LABEL_STRING,      "Use All Available",
                   PANEL_CHOICE_STRINGS,    "", NULL,
                   PANEL_NOTIFY_PROC,       use_all_gen,
                   XV_HELP_DATA,            "fdb:use_all_gen_choice",
                   NULL);

/*
   create the "use all available generations in descendants tree" tick box
*/

use_all_des_choice = (Panel_item) xv_create(settings_panel, PANEL_CHECK_BOX,
                   XV_X,                    490,
                   XV_Y,                    45,
                   PANEL_LABEL_STRING,      "Use All Available",
                   PANEL_CHOICE_STRINGS,    "", NULL,
                   PANEL_NOTIFY_PROC,       use_all_des,
                   XV_HELP_DATA,            "fdb:use_all_des_choice",
                   NULL);

/*****************
   SLIDERS
*****************/

/*
   slider to allow variation of the minimum horizontal separation between
   leaves in the descendants tree
*/

min_text_width_slider = (Panel_item) xv_create(settings_panel, PANEL_SLIDER,
                   XV_X,                    100,
                   XV_Y,                    90,
                   PANEL_LABEL_STRING,     "Descendants Tree Horizontal Spacing:",
                   PANEL_MIN_VALUE,        0,
                   PANEL_MAX_VALUE,        200,
                   PANEL_SLIDER_WIDTH,     100,
                   PANEL_TICKS,            10,
                   PANEL_NOTIFY_PROC,      set_min_text_width,
                   PANEL_VALUE,            min_text_width,
                   PANEL_SHOW_VALUE,       FALSE,
                   PANEL_SLIDER_END_BOXES, FALSE,
                   XV_HELP_DATA,           "fdb:width_slider",
                   NULL);

/*
   slider to allow selection of the maximum depth of displayed/plotted descent line
*/

descent_depth_slider = (Panel_item) xv_create(settings_panel, PANEL_SLIDER,
                   XV_X,                    10,
                   XV_Y,                    20,
                   PANEL_LABEL_STRING,     "Max. Generations in Descent Line:        ",
                   PANEL_MIN_VALUE,        1,
                   PANEL_MAX_VALUE,        15,
                   PANEL_SLIDER_WIDTH,     100,
                   PANEL_TICKS,            1,
                   PANEL_VALUE,            max_gen_generations,
                   PANEL_SHOW_VALUE,       TRUE,
                   PANEL_SLIDER_END_BOXES, FALSE,
                   PANEL_NOTIFY_PROC,      depth_gen_chosen,
                   XV_HELP_DATA,           "fdb:depth_gen_choice",
                   NULL);

/*
   slider to allow selection of the maximum depth of displayed/plotted descendants tree
*/

descendants_depth_slider = (Panel_item) xv_create(settings_panel, PANEL_SLIDER,
                   XV_X,                    10,
                   XV_Y,                    50,
                   PANEL_LABEL_STRING,     "Max. Generations in Descendants Tree:",
                   PANEL_MIN_VALUE,        1,
                   PANEL_MAX_VALUE,        15,
                   PANEL_SLIDER_WIDTH,     100,
                   PANEL_TICKS,            1,
                   PANEL_VALUE,            max_des_generations,
                   PANEL_SHOW_VALUE,       TRUE,
                   PANEL_SLIDER_END_BOXES, FALSE,
                   PANEL_NOTIFY_PROC,      depth_des_chosen,
                   XV_HELP_DATA,           "fdb:depth_des_choice",
                   NULL);

/* 
   initialise the database 
*/

initialise_database();

/* create and display the left footer */

sprintf(footer_string,"Current Database File: %s",file_name);
xv_set(fdb_frame, FRAME_LEFT_FOOTER, footer_string, NULL);

/* set the main icon label to the current database file name */

xv_set(main_icon, ICON_LABEL, file_name, NULL);

/*
   set the display items up with the information for the current person
*/

update_panel_items();

/*
   hand control over to XView
*/

xv_main_loop(fdb_frame);
exit(0);
}

