EXTPROC CEnvi
//*******************************************************
//*** DelOld.cmd - Act like DEL, but delete file that ***
//*** ver.1        are more than a certain number of  ***
//***              days old.                          ***
//*******************************************************

#include <OptParms.lib>

main(argc,argv)
{
   // get the command-line options
   Prompt = OptionalParameter(argc,argv,"P");
   Recurse = OptionalParameter(argc,argv,"R");
   System = OptionalParameter(argc,argv,"S");
   Hidden = OptionalParameter(argc,argv,"H");

   // get file name and age
   FileSpec = argv[1];
   Age = atoi(argv[2]);

   // if parameters weren't valid then print instructions and leave
   if ( argc != 3  ||  Age < 1 ) {
      Instructions();
      return EXIT_FAILURE;
   }

   // Whether or not recursion is selected, perform in the current
   // directory
   DelOld(FileSpec,Age,Prompt,System,Hidden);

   if ( Recurse ) {
      // There can be A LOT of files if recursion is selected, and so
      // do one directory at a time
      FileParts = SplitFileName(FileSpec);
      DirPart = FileParts.dir;
      strcat(DirPart,"*");
      DirList = Directory(DirPart,True,
                          FATTR_RDONLY|FATTR_HIDDEN|FATTR_SYSTEM|FATTR_SUBDIR|FATTR_ARCHIVE,
                          FATTR_SUBDIR);
      if ( DirList ) {
         // run DelOld in each directory
         DirCount = 1 + GetArraySpan(DirList);
         for ( i = 0; i < DirCount; i++ ) {
            sprintf(Spec,"%s\\%s%s",DirList[i].name,FileParts.name,FileParts.ext);
            DelOld(Spec,Age,Prompt,System,Hidden);
         }
      }
   }
}

DelOld(pFileSpec,pAge,pPrompt,pSystem,pHidden)
   // delete each file matching FileSpec that is at least pAge old
{
   // Build list of matching files
   Attrib = FATTR_RDONLY | FATTR_ARCHIVE;
   if ( pSystem ) Attrib |= FATTR_SYSTEM;
   if ( pHidden ) Attrib |= FATTR_HIDDEN;
   FileList = Directory(pFileSpec,False,Attrib);

   if ( !FileList )
      // no matching files were found, so return
      return;

   // Get the current time, and translate old time to seconds because
   // that is the simplest way to compare
   AgeInSeconds = pAge * 24 * 60 * 60;

   FileCount = 1 + GetArraySpan(FileList);
   for ( i = 0; i < FileCount; i++ ) {
      if ( AgeInSeconds <= difftime(time(),FileList[i].Write) ) {
         DeleteFile(FileList[i],pPrompt);
      }
   }
}

DeleteFile(pFileData,pPrompt)
{
   strftime(TimeBuf,"%m-%d-%y",localtime(pFileData.Write));
   // print name of file, date, and age in days
   printf("%-58s %s Age: %.1f\n",pFileData.name,TimeBuf,
          float(difftime(time(),pFileData.Write)) / (24 * 60 * 60) ); 

   // if rd_only then explain that cannot delete
   if ( FATTR_RDONLY & pFileData.attrib ) {
      printf("\n\a\tCANNOT DELETE READ-ONLY FILE!!!\n");
   } else {
      if ( pPrompt ) {
         printf("Delete? (Y/N) ");
         Answer = toupper(getch());
         while ( !strchr("YN",Answer) ) {
            printf("\a");
            Answer = toupper(getch());
         }
         printf("%c\n",Answer);
      }
      if ( !pPrompt  ||  Answer == 'Y' )
         remove(pFileData.name);
   }
}



Instructions()
{
   puts("\a")
   puts(`DelOld - Delete old file; files more than X days old`)
   puts(``)
   puts(`USAGE: DelOld <FileSpec> <Age> [/P] [/R] [/S] [/H]`)
   puts(``)
   puts(`WHERE: FileSpec - Any file specification, including wildcards`)
   puts(`       Age - Age, in days, of files to delete`)
   puts(`       /P - Prompt before deleting each file`)
   puts(`       /R - Recursive; work in current and subirectories`)
   puts(`       /S - Include files with the SYSTEM attribute`)
   puts(`       /H - Include files with the HIDDEN attribute`)
   puts(``)
   puts(`EXAMPLE: This will delete all *.bak file on C: at least 10 days old`)
   puts(`            DelOld C:\*.bak 10 /R /S /H`)
   puts(``)
}

