/* FTP client (interactive user) code */
#include <stdio.h>
#include <ctype.h>
#include <sys/stat.h>
#include "global.h"
#include "mbuf.h"
#include "session.h"
#include "cmdparse.h"
#include "proc.h"
#include "tty.h"
#include "socket.h"
#include "ftp.h"
#include "ftpcli.h"
#include "commands.h"
#include "netuser.h"
#include "dirutil.h"
#include "files.h"

#define	DIRBUF	256


static int dotype __ARGS((int argc,char *argv[],void *p));
static int near getresp __ARGS((struct ftpcli *ftp,int mincode));
static void near sendport __ARGS((int s,struct sockaddr_in *socket));


/* Common code to LIST/NLST/RETR and mget
 * Returns number of bytes received if successful
 * Returns -1 on error
 */
static long near
getsub(ftp,command,remotename,localname,offset)
struct ftpcli *ftp;
char *command,*remotename,*localname;
long offset;
{
	unsigned long total;
	FILE *fp;
	int cnt = 0, resp, i, control, savmode, vsave, prevstate, typewait = 0;
	char *mode;
	struct sockaddr_in lsocket, lcsocket;
	int32 startclk, rate;

	if(ftp == NULLFTP)
		return -1;

	control = ftp->control;
	savmode = ftp->type;

	switch(ftp->type){
	case IMAGE_TYPE:
	case LOGICAL_TYPE:
		mode = WRITE_BINARY;
		break;
	case ASCII_TYPE:
		mode = WRITE_TEXT;
		break;
	}
	if(offset)
		mode = "rb+";

	/* Open the file */
	if(localname == NULLCHAR){
		fp = NULLFILE;
	} else if((fp = open_file(localname,mode,0,1)) == NULLFILE)
		return -1;

	if (fp) {
		if(fseek(fp,offset,SEEK_SET))   {
			tprintf("Can't position %s: %s\n",localname,sys_errlist[errno]);
			fclose(fp);
			return -1;
		}
	}
	/* Open the data connection */
	ftp->data = socket(AF_INET,SOCK_STREAM,0);
	listen(ftp->data,0);	/* Accept only one connection */
	prevstate = ftp->state;
	ftp->state = RECEIVING_STATE;

	/* Send TYPE message, if necessary */
	if(strcmp(command,"LIST") == 0 || strcmp(command,"NLST") == 0){
		/* Directory listings are always in ASCII */
		ftp->type = ASCII_TYPE;
	}
	if(ftp->typesent != ftp->type){
		switch(ftp->type){
		case LOGICAL_TYPE:
			usprintf(control,"TYPE L %d\n",ftp->logbsize);
			break;
		case ASCII_TYPE:
		case IMAGE_TYPE:
			usprintf(control,"TYPE %s\n",(ftp->type == ASCII_TYPE) ? "A" : "I");
			break;
		}
		ftp->typesent = ftp->type;
		if(!ftp->batch){
			resp = getresp(ftp,200);
			if(resp == -1 || resp > 299)
				goto failure;
		} else
			typewait = 1;
	}
	/* Send the PORT message. Use the IP address
	 * on the local end of our control connection.
	 */
	i = SOCKSIZE;
	getsockname(ftp->data,(char *)&lsocket,&i); /* Get port number */
	i = SOCKSIZE;
	getsockname(ftp->control,(char *)&lcsocket,&i);
	lsocket.sin_addr.s_addr = lcsocket.sin_addr.s_addr;
	sendport(control,&lsocket);
	if(!ftp->batch){
		/* Get response to PORT command */
		resp = getresp(ftp,200);
		if(resp == -1 || resp > 299)
			goto failure;
	}
	/* Generate the command to start the transfer */
	usputs(control,command);
	if(remotename != NULLCHAR)
		usprintf(control," %s",remotename);
	if(offset)
		usprintf(control," %ld",offset);
	usputs(control,"\n");

	if(ftp->batch){
		/* Get response to TYPE command, if sent */
		if(typewait){
			resp = getresp(ftp,200);
			if(resp == -1 || resp > 299)
				goto failure;
		}
		/* Get response to PORT command */
		resp = getresp(ftp,200);
		if(resp == -1 || resp > 299)
			goto failure;
	}
	/* Get the intermediate "150" response */
	resp = getresp(ftp,100);
	if(resp == -1 || resp >= 400)
		goto failure;

	/* Wait for the server to open the data connection */
	ftp->data = accept(ftp->data,NULLCHAR,&cnt);
	startclk = msclock();

	/* If output is to the screen, temporarily disable hash marking */
	vsave = ftp->verbose;
	if(vsave >= V_HASH && fp == NULLFILE)
		ftp->verbose = V_NORMAL;
	total = recvfile(fp,ftp->data,ftp->type,ftp->verbose >= V_HASH ? ftp->verbose : 0);
	/* Immediately close the data connection; some servers (e.g., TOPS-10)
	 * wait for the data connection to close completely before returning
	 * the completion message on the control channel
	 */
	close_s(ftp->data);
	ftp->data = -1;

#ifdef	CPM
	if(fp != NULLFILE && ftp->type == ASCII_TYPE)
		putc(CTLZ,fp);
#endif

	if(fp != NULLFILE && fp != stdout)
		fclose(fp);
	if(remotename == NULLCHAR)
		remotename = "";
	if(total == -1)
		tprintf("%s %s: Error/abort during data transfer\n",command,remotename);

	/* Get the "Sent" message */
	getresp(ftp,200);

	if(total != -1 && ftp->verbose >= V_SHORT) {
		startclk = msclock() - startclk;
		rate = (startclk != 0) ? (total*1000)/startclk : 0;
		tprintf("%s %s: %lu bytes in %lu sec (%lu/sec)\n",
			command,remotename,total,startclk/1000,rate);
	}
	ftp->state = prevstate;
	ftp->verbose = vsave;
	ftp->type = savmode;
	return total;

failure:
	/* Error, quit */
	if(fp != NULLFILE && fp != stdout)
		fclose(fp);
	close_s(ftp->data);
	ftp->data = -1;
	ftp->state = prevstate;
	ftp->type = savmode;
	return -1;
}

/* Common code to put, mput */
static long near
putsub(ftp,remotename,localname)
register struct ftpcli *ftp;
char *remotename,*localname;
{
	int i = SOCKSIZE, resp, typewait = 0, prevstate;
	unsigned long total;
	FILE *fp;
	struct sockaddr_in lsocket, lcsocket;
	int32 startclk, rate;

	int control = ftp->control;
	char *mode = (ftp->type == IMAGE_TYPE) ? READ_BINARY : READ_TEXT;

	/* Open the file */
	if((fp = fopen(localname,mode)) == NULLFILE)
		return -1;

	if(ftp->type == ASCII_TYPE && isbinary(fp)){
		tprintf("Warning: type is ASCII and %s appears to be binary\n",localname);
	}
	/* Open the data connection */
	ftp->data = socket(AF_INET,SOCK_STREAM,0);
	listen(ftp->data,0);
	prevstate = ftp->state;
	ftp->state = SENDING_STATE;

	/* Send TYPE message, if necessary */
	if(ftp->typesent != ftp->type){
		switch(ftp->type){
		case LOGICAL_TYPE:
			usprintf(control,"TYPE L %d\n",ftp->logbsize);
			break;
		case ASCII_TYPE:
		case IMAGE_TYPE:
			usprintf(control,"TYPE %s\n",(ftp->type == ASCII_TYPE) ? "A" : "I");
		}
		ftp->typesent = ftp->type;
		/* Get response to TYPE command */
		if(!ftp->batch){
			resp = getresp(ftp,200);
			if(resp == -1 || resp > 299){
				goto failure;
			}
		} else {
			typewait = 1;
		}
	}
	/* Send the PORT message. Use the IP address
	 * on the local end of our control connection.
	 */
	getsockname(ftp->data,(char *)&lsocket,&i);
	i = SOCKSIZE;
	getsockname(ftp->control,(char *)&lcsocket,&i);
	lsocket.sin_addr.s_addr = lcsocket.sin_addr.s_addr;
	sendport(control,&lsocket);
	if(!ftp->batch){
		/* Get response to PORT command */
		resp = getresp(ftp,200);
		if(resp == -1 || resp > 299){
			goto failure;
		}
	}
	/* Generate the command to start the transfer */
	usprintf(control,"STOR %s\n",remotename);

	if(ftp->batch){
		/* Get response to TYPE command, if sent */
		if(typewait){
			resp = getresp(ftp,200);
			if(resp == -1 || resp > 299){
				goto failure;
			}
		}
		/* Get response to PORT command */
		resp = getresp(ftp,200);
		if(resp == -1 || resp > 299){
			goto failure;
		}
	}
	/* Get the intermediate "150" response */
	resp = getresp(ftp,100);
	if(resp == -1 || resp >= 400){
		goto failure;
	}

	/* Wait for the data connection to open. Otherwise the first
	 * block of data would go out with the SYN, and this may confuse
	 * some other TCPs
	 */
	accept(ftp->data,NULLCHAR,(int *)NULL);

	startclk = msclock();

	total = sendfile(fp,ftp->data,ftp->type,ftp->verbose >= V_HASH ? ftp->verbose : 0);
	close_s(ftp->data);
	ftp->data = -1;
	fclose(fp);

	if(total == -1)
		tprintf("STOR %s: Error/abort during data transfer\n",remotename);

	getresp(ftp,200);

	if(total != -1 && ftp->verbose >= V_SHORT) {
		startclk = msclock() - startclk;
		rate = (startclk != 0) ? (total*1000)/startclk : 0;
		tprintf("STOR %s: %lu bytes in %lu sec (%lu/sec)\n",
			remotename,total,startclk/1000,rate);
	}
	ftp->state = prevstate;
	return total;

failure:
	/* Error, quit */
	fclose(fp);
	close_s(ftp->data);
	ftp->data = -1;
	ftp->state = prevstate;
	return -1;
}

/* send PORT message */
static void near
sendport(s,socket)
int s;
struct sockaddr_in *socket;
{
	/* Send PORT a,a,a,a,p,p message */
	usprintf(s,"PORT %u,%u,%u,%u,%u,%u\n",
		hibyte(hiword(socket->sin_addr.s_addr)),
		lobyte(hiword(socket->sin_addr.s_addr)),
		hibyte(loword(socket->sin_addr.s_addr)),
		lobyte(loword(socket->sin_addr.s_addr)),
		hibyte(socket->sin_port),
		lobyte(socket->sin_port));
}

/* Wait for, read and display response from FTP server. Return the result code.
 */
static int near
getresp(ftp,mincode)
struct ftpcli *ftp;
int mincode;	/* Keep reading until at least this code comes back */
{
	char line[LINELEN];
	int rval = -1;

	usflush(ftp->control);
	for(;;){
		/* Get line */
		if(recvline(ftp->control,line,LINELEN) == -1){
			break;
		}
		rip(line);		/* Remove cr/lf */
		if((rval = atoi(line)) >= 400 || ftp->verbose >= V_NORMAL)
			tprintf("%s\n",line);	/* Display to user */

		/* Messages with dashes are continued */
		if(line[3] != '-' && rval >= mincode)
			break;
	}
	return rval;
}

/* Issue a prompt and read a line from the user */
static int near
getline(sp,prompt,buf,n)
struct session *sp;
char *prompt;
char *buf;
int n;
{
	/* If there's something already there, don't issue prompt */
	if(socklen(sp->input,0) == 0)
		tputs(prompt);

	usflush(sp->output);
	usflush(sp->s);
	return recvline(sp->input,buf,n);
}

/* Attempt to log in the user whose name is in ftp->username and password
 * in pass
 */
static char * near
ftpcli_login(ftp,host)
struct ftpcli *ftp;
char *host;
{
	char buf[LINELEN], *cp, *cp1;
	FILE *fp;

	if((fp = fopen(Hostfile,READ_TEXT)) == NULLFILE){
		return NULLCHAR;
	}

	while(fgets(buf,LINELEN,fp),!feof(fp)){
		if(buf[0] == '#')
			continue;	/* Comment */
		rip(buf);
		if((cp = strchr(buf,' ')) == NULLCHAR)
			/* Bogus entry */
			continue;
		*cp++ = '\0';		/* Now points to user name */
		if(strcmp(host,buf) == 0)
			break;		/* Found host name */
	}
	if(feof(fp)){
		/* User name not found in file */
		fclose(fp);
		return NULLCHAR;
	}
	fclose(fp);

	/* Look for space after user field in file */
	if((cp1 = strchr(cp,' ')) == NULLCHAR)
		/* if not there then we'll prompt */
		ftp->password = NULLCHAR;
	else
		*cp1++ = '\0';		/* Now points to password */
	if(strcmp(cp,"*") == 0)
		cp1 = "anonymous";
	ftp->password = strxdup(cp1);
	return strxdup(cp);
}


/* ------------------------- FTP-Client Subcmds ------------------------- */

/* Abort a GET or PUT operation in progress.
 * Note: this will leave the partial file on the local or remote system
 * This function is called from config.h
 */
int
doabort(argc,argv,p)
int argc;
char *argv[];
void *p;
{
	struct session *sp = (struct session *)p;
	struct ftpcli *ftp = sp->cb.ftp;

	/* Default is the current session, but it can be overridden with
	 * an argument.
	 */
	if(argc > 1)
		sp = sessptr(argv[1]);

	if(sp == NULLSESSION || sp->type != FTP) {
		tputs("Not an active FTP session\n");
	} else {
		switch(ftp->state){
		case COMMAND_STATE:
			tputs("No active transfer\n");
			return 1;
		case SENDING_STATE:		/* defined as 1 */
		case RECEIVING_STATE:	/* defined as 2 */
			/* Send a premature EOF.
			 * Unfortunately we can't just reset the connection
			 * since the remote side might end up waiting forever
			 * for us to send something.
			 * If receiving state just blow away the socket
			 */
			shutdown(ftp->data,ftp->state);
			ftp->abort = 1;
			return 0;
		}
	}
	return -1;
}

static int
doascii(argc,argv,p)
int argc;
char *argv[];
void *p;
{
	char *args[2];

	args[1] = "A";
	return dotype(2,args,p);
}

/* Enable/disable command batching */
static int
dobatch(argc,argv,p)
int argc;
char *argv[];
void *p;
{
	struct ftpcli *ftp;

	if((ftp = (struct ftpcli *)p) == NULLFTP)
		return -1;
	setbool(&ftp->batch,"FTP batching",argc,argv);
	return 0;
}

static int
dobinary(argc,argv,p)
int argc;
char *argv[];
void *p;
{
	char *args[2];

	args[1] = "I";
	return dotype(2,args,p);
}

/* Translate 'cd' to 'cwd' for convenience */
static int
doftpcd(argc,argv,p)
int argc;
char *argv[];
void *p;
{
	struct ftpcli *ftp;

	if((ftp = (struct ftpcli *)p) == NULLFTP)
		return -1;
	usprintf(ftp->control,"CWD %s\n",argv[1]);
	return getresp(ftp,200);
}

/* Start receive transfer. Syntax: get <remote name> [<local name>] */
static int
doget(argc,argv,p)
int argc;
char *argv[];
void *p;
{
	struct ftpcli *ftp;
	char *remotename, *localname;

	if((ftp = (struct ftpcli *)p) == NULLFTP)
		return -1;
	remotename = argv[1];
	localname = (argc < 3) ? remotename : argv[2];
	getsub(ftp,"RETR",remotename,localname,0L);
	return 0;
}

/* Set verbosity to high (convenience command) */
static int
dohash(argc,argv,p)
int argc;
char *argv[];
void *p;
{
	struct ftpcli *ftp;

	if((ftp = (struct ftpcli *)p) == NULLFTP)
		return -1;
	ftp->verbose = V_HASH;
	return 0;
}

/* List remote directory. Syntax: dir <remote files> [<local name>] */
static int
dolist(argc,argv,p)
int argc;
char *argv[];
void *p;
{
	struct ftpcli *ftp;
	char *remotename,*localname;

	if((ftp = (struct ftpcli *)p) == NULLFTP)
		return -1;
	remotename = argv[1];
	localname = (argc < 3) ? NULLCHAR : argv[2];
	getsub(ftp,"LIST",remotename,localname,0L);
	return 0;
}

/* Remote directory list, short form. Syntax: ls <remote files> [<local name>] */
static int
dols(argc,argv,p)
int argc;
char *argv[];
void *p;
{
	struct ftpcli *ftp;
	char *remotename, *localname;

	if((ftp = (struct ftpcli *)p) == NULLFTP)
		return -1;
	remotename = argv[1];
	localname = (argc < 3) ? NULLCHAR : argv[2];
	getsub(ftp,"NLST",remotename,localname,0L);
	return 0;
}

/* Get a collection of files */
static int
domget(argc,argv,p)
int argc;
char *argv[];
void *p;
{
	struct ftpcli *ftp;
	FILE *files, *filel;
	char tmpname[80], *buf, *local;
#ifdef	MSDOS
	char *c;
#endif
	int i, inlist;
	long r;

	if((ftp = (struct ftpcli *)p) == NULLFTP)
		return -1;

	tmpnam(tmpname);
	buf = mxallocw(DIRBUF);

	ftp->state = RECEIVING_STATE;

	for(i = 1; i < argc; i++) {
	  if(ftp->abort)
		break;
      if(argv[i][0] == '@') {
         inlist = 1;
		 if((filel = open_file(&argv[i][1],"r",ftp->control,0)) == NULLFILE)
            continue;
		 if((files = open_file(tmpname,"w",ftp->control,0)) == NULLFILE) {
			fclose(filel);
            continue;
         }
		 while(fgets(buf,DIRBUF,filel) != NULLCHAR)
            fputs(buf,files);
		 fclose(files);
		 fclose(filel);
		 if((files = open_file(tmpname, "r",ftp->control,0)) == NULLFILE){
			fclose(filel);
            continue;
         }
      } else {
		inlist = 0;
		r = getsub(ftp,"NLST",argv[i],tmpname,0L);
		if(ftp->abort)
			break;
		if(r == -1 || (files = open_file(tmpname,"r",0,1)) == NULLFILE){
			unlink(tmpname);
			continue;
		}
	  }
		/* The tmp file now contains a list of the remote files.
		 * If any cannot be read, it must be because we were aborted
		 * or reset locally, so break out if a transfer fails.
		 */
		while(fgets(buf,DIRBUF,files) != NULLCHAR){
			rip(buf);
			local = strxdup(buf);
#ifdef	MSDOS
			if(inlist){
				strrev(local);
				strtok(local, "\\/[]<>,?#~()&%");
				strrev(local);
			}
			if((c = strstr(local, ".")) != NULLCHAR) {
				c++;
				c = strtok(c, ".");			/* remove 2nd period if any */
			}
#endif
			getsub(ftp,"RETR",buf,local,0L);
			xfree(local);
			if(ftp->abort) {
				ftp->abort = 0;
				fclose(files);
				unlink(tmpname);
				ftp->state = COMMAND_STATE;
				xfree(buf);
				return 1;
			}
		}
		fclose(files);
		unlink(tmpname);
	}
	xfree(buf);
	ftp->abort = 0;
	ftp->state = COMMAND_STATE;
	return 0;
}

/* Translate 'mkdir' to 'xmkd' for convenience */
static int
domkdir(argc,argv,p)
int argc;
char *argv[];
void *p;
{
	struct ftpcli *ftp;

	if((ftp = (struct ftpcli *)p) == NULLFTP)
		return -1;
	usprintf(ftp->control,"XMKD %s\n",argv[1]);
	return getresp(ftp,200);
}

/* Put a collection of files */
static int
domput(argc,argv,p)
int argc;
char *argv[];
void *p;
{
	struct ftpcli *ftp;
	FILE *files;
	int i;
	char *buf, tmpname[80];

	if((ftp = (struct ftpcli *)p) == NULLFTP)
		return -1;

	tmpnam(tmpname);
	if((files = open_file(tmpname,"w+",0,1)) == NULLFILE){
		unlink(tmpname);
		return -1;
	}

	for(i = 1; i < argc; i++)
		getdir(argv[i],0,files);

	rewind(files);
	buf = mxallocw(DIRBUF);
	ftp->state = SENDING_STATE;
	while(fgets(buf,DIRBUF,files) != NULLCHAR){
		rip(buf);
		putsub(ftp,buf,buf);
		if(ftp->abort)
			break;
	}
	fclose(files);
	unlink(tmpname);
	xfree(buf);
	ftp->state = COMMAND_STATE;
	ftp->abort = 0;
	return 0;
}

/* NO-OP function */
static int
donothing(argc,argv,p)
int argc;
char *argv[];
void *p;
{
	return 0;
}

/* Send a file. Syntax: put <local name> [<remote name>] */
static int
doput(argc,argv,p)
int argc;
char *argv[];
void *p;
{
	struct ftpcli *ftp;
	char *remotename, *localname;

	if((ftp = (struct ftpcli *)p) == NULLFTP)
		return -1;
	localname = argv[1];
	remotename = (argc < 3) ? localname : argv[2];
	putsub(ftp,remotename,localname);
	return 0;
}

/* Close session */
static int
doquit(argc,argv,p)
int argc;
char *argv[];
void *p;
{
	struct ftpcli *ftp;

	if((ftp = (struct ftpcli *)p) == NULLFTP)
		return -1;
	usputs(ftp->control,"QUIT\n");
	getresp(ftp,200);			/* Get the closing message */
	getresp(ftp,200);			/* Wait for the server to close */
	return -1;
}

/* Translate 'rmdir' to 'xrmd' for convenience */
static int
dormdir(argc,argv,p)
int argc;
char *argv[];
void *p;
{
	struct ftpcli *ftp;

	if((ftp = (struct ftpcli *)p) == NULLFTP)
		return -1;
	usprintf(ftp->control,"XRMD %s\n",argv[1]);
	return getresp(ftp,200);
}

/* Start receive transfer restart. Syntax: get <remote name> [<local name>] */
static int
dorest(argc,argv,p)
int argc;
char *argv[];
void *p;
{
	struct ftpcli *ftp;
	char *remotename, *localname;
	long offset;
	struct stat st;

	if((ftp = (struct ftpcli *)p) == NULLFTP)
		return -1;

	/*-------------------------------------------------------------------*
	*  local and remotename required
	*--------------------------------------------------------------------*/
	remotename = argv[1];
	localname = (argc < 3) ? remotename : argv[2];

	if (stat(localname,&st))   {
		tprintf("Cannot find %s\n",localname);
		return -1;
	}
	/*-------------------------------------------------------------------*
	* adjust offset to latest 1K Boundary
	*--------------------------------------------------------------------*/
	offset = st.st_size & 0xfffffc00L;
	/*-------------------------------------------------------------------*
	* let 'em swing
	*--------------------------------------------------------------------*/
	getsub(ftp,"REST",remotename,localname,offset);
	return 0;
}

/* Handle "type" command from user */
static int
dotype(argc,argv,p)
int argc;
char *argv[];
void *p;
{
	struct ftpcli *ftp;

	if((ftp = (struct ftpcli *)p) == NULLFTP)
		return -1;
	if(argc < 2){
		switch(ftp->type){
		case IMAGE_TYPE:
		case ASCII_TYPE:
			tprintf("%s\n",(ftp->type == ASCII_TYPE) ? "Ascii" : "Image");
			break;
		case LOGICAL_TYPE:
			tprintf("Logical bytesize %u\n",ftp->logbsize);
			break;
		}
		return 0;
	}
	switch(tolower(*argv[1])) {
	case 'i':
	case 'b':
		ftp->typesent = ftp->type = IMAGE_TYPE;
		usputs(ftp->control,"TYPE I\n");
		break;
	case 'a':
		ftp->typesent = ftp->type = ASCII_TYPE;
		usputs(ftp->control,"TYPE A\n");
		break;
	case 'l':
		ftp->typesent = ftp->type = LOGICAL_TYPE;
		ftp->logbsize = atoi(argv[2]);
		usprintf(ftp->control,"TYPE L %s\n",argv[2]);
		break;
	default:
		tprintf("Invalid type %s\n",argv[1]);
		return 0;
	}
	return getresp(ftp,200);
}

/* Control verbosity level */
static int
doverbose(argc,argv,p)
int argc;
char *argv[];
void *p;
{
	struct ftpcli *ftp;

	if((ftp = (struct ftpcli *)p) == NULLFTP)
		return -1;
	setshort(&ftp->verbose,"Verbose",argc,argv);
	return 0;
}

/* Handle top-level FTP command */
int
doftp(argc,argv,p)
int argc;
char *argv[];
void *p;
{
	struct session *sp;
	struct ftpcli ftp;
	struct sockaddr_in fsocket;
	int resp, vsave, control;
	char *buf, *bufsav, *cp;

    struct cmds Ftpcmds[] = {
        "",         donothing,  0, 0, NULLCHAR,
        "ascii",    doascii,    0, 0, NULLCHAR,
        "batch",    dobatch,    0, 0, NULLCHAR,
        "binary",   dobinary,   0, 0, NULLCHAR,
        "cd",       doftpcd,    0, 2, "cd <dir>",
        "dir",      dolist,     0, 0, NULLCHAR,
        "list",     dolist,     0, 0, NULLCHAR,
        "get",      doget,      0, 2, "get <remotefile> <localfile>",
        "hash",     dohash,     0, 0, NULLCHAR,
        "ls",       dols,       0, 0, NULLCHAR,
        "mget",     domget,     0, 2, "mget <file> [<file> ...]",
        "mkdir",    domkdir,    0, 2, "mkdir <dir>",
        "mput",     domput,     0, 2, "mput <file> [<file> ...]",
        "nlst",     dols,       0, 0, NULLCHAR,
        "quit",     doquit,     0, 0, NULLCHAR,
        "rmdir",    dormdir,    0, 2, "rmdir <dir>",
        "put",      doput,      0, 2, "put <localfile> <remotefile>",
        "restart",  dorest,    0, 2, "restart <remotefile> <localfile>",
        "type",     dotype,     0, 0, NULLCHAR,
        "verbose",  doverbose,  0, 0, NULLCHAR,
        NULLCHAR,   NULLFP,     0, 0, NULLCHAR,
    };

	/* Allocate a session control block */
	if((sp = newsession(argv[1],FTP,0,1)) == NULLSESSION){
		tputs(Nosess);
		return -1;
	}
	memset((char *)&ftp,0,sizeof(ftp));
	ftp.verbose = V_NORMAL;
	ftp.control = ftp.data = -1;

	sp->cb.ftp = &ftp;	/* Downward link */
	ftp.session = sp;	/* Upward link */

	fsocket.sin_family = AF_INET;
	fsocket.sin_port = (argc < 3) ? IPPORT_FTP : atoi(argv[2]);
	tprintf("Resolving %s... ",sp->name);

	if((fsocket.sin_addr.s_addr = resolve(sp->name)) == 0){
		tprintf(Badhost,sp->name);
		goto quit2;
	}
	/* Open the control connection */
	if((control = sp->s = ftp.control = socket(AF_INET,SOCK_STREAM,0)) == -1){
		tputs(Nosocket);
		goto quit2;
	}
	sockmode(sp->s,SOCK_ASCII);
	setflush(sp->s,-1);
	tprintf("Trying %s...\n",psocket((struct sockaddr *)&fsocket));
	if(connect(control,(char *)&fsocket,sizeof(fsocket)) == -1)
		goto quit;
	tprintf("FTP session connected to %s\n",sp->name);
	log(sp->s,"FTP  connect");

	/* Wait for greeting from server */
	resp = getresp(&ftp,200);
	if(resp >= 400)
		goto quit;
	/* Now process responses and commands */
	buf = mxallocw(LINELEN);
	while(resp != -1){
		if(resp == 220){
			/* Sign-on banner; prompt for and send USER command */
			if((cp = ftpcli_login(&ftp, sp->name)) == NULLCHAR){
				getline(sp,"Enter user name: ",buf,LINELEN);
				if(buf[0] != '\n'){
					usprintf(control,"USER %s",buf);
					resp = getresp(&ftp,200);
				} else
					resp = 200;	/* dummy */
			} else {
				usprintf(control,"USER %s\n",cp);
				xfree(cp);
				resp = getresp(&ftp,200);
			}
		} else if(resp == 331) { /* Password prompt; get password */
			if(ftp.password == NULLCHAR) {
				/* turn off echo */
				sp->ttystate.echo = 0;
				getline(sp,"Password: ",buf,LINELEN);
				sp->ttystate.echo = 1;
				tputs("\n");
				if(buf[0] != '\n'){
					usprintf(control,"PASS %s",buf);
					resp = getresp(&ftp,200);
				} else {
					resp = 200;	/* dummy */
				}
			} else {
				usprintf(control,"PASS %s\n",ftp.password);
				resp = getresp(&ftp,200);
			}
		} else {
			/* Test the control channel first */
			if(sockstate(control) == NULLCHAR) {
				xfree(buf);
				goto quit;
			}
			getline(sp,"ftp> ",buf,LINELEN);

			/* Copy because cmdparse modifies the original */
			bufsav = strxdup(buf);
			if((resp = cmdparse(Ftpcmds,buf,&ftp)) != -1){
				/* Valid command, free buffer and get another */
				xfree(bufsav);
			} else {
				/* Not a local cmd, send to remote server */
				usputs(control,bufsav);
				xfree(bufsav);
				/* Enable display of server response */
				vsave = ftp.verbose;
				ftp.verbose = V_NORMAL;
				resp = getresp(&ftp,200);
				ftp.verbose = vsave;
			}
		}
	}
	xfree(buf);
quit:
	cp = sockerr(control);
	tprintf("FTP session closed: %s\n",cp != NULLCHAR ? cp : "EOF");

	if(ftp.fp != NULLFILE && ftp.fp != stdout)
		fclose(ftp.fp);
	if(ftp.data != -1)
		close_s(ftp.data);
	if(ftp.control != -1)
		close_s(ftp.control);
	keywait(NULLCHAR,1);
	if(ftp.session != NULLSESSION)
		freesession(ftp.session);
	if(ftp.password != NULLCHAR)
		xfree(ftp.password);
	return 0;
quit2:
	keywait(NULLCHAR,1);
	freesession(sp);
	return -1;
}

