(* This unit provides a code 3 of 9 barcode output object *)

Unit TsBar3o9;

Interface
Uses
  WinTypes,
  Win31,
  Strings,
  WinProcs;

Const
  (*    0.13 suggested  3.0 Ratio    *)

  (* New format used with english units - 1/100 inch *)
  BarWidth : Real = 0.15;  (* 1/10 inch per bar   WAS 0.18  WAS 0.16*)
  ExtraAir : Real = 0; { 0.02; } {0.08; } (* 1/100 inch extra space on spaces *)
  BarHeight : Real = 4.0;  (* 4 - 10ths of an inch *)
  Ratio : Real = 3.0;  (* Ratio of narrow to wide bars *)

{Procedure PrintBarCode2(PDC: HDC;XPos,YPos: Integer;BarCode: String;Heading: Boolean);}

Procedure PrintBarCode(PDC: HDC;XPos,YPos: Real;BarCode: String;Heading: Boolean);

Type
  tBarStyle = (btSpace,btBar,btWideSpace,btWideBar,btNone);

  tBarCode = Object
     CodeStr : String;  (* This holds the barcode for disecting *)
    CurrentBar : Word;

    cFlag : tBarStyle; (* Cary flag for the GetNextBarSize *)
    MoreToGet : Boolean;  (* Used in GetNextBarSize as a read ahead flag *)

    StrPos : Integer; (* This holds the position in the current Barcode *)
                      (* 0 = Working on Start Bar           *)
		      (*   < Length(CodeStr) = Working on Data *)
                      (*   > Length(CodeStr) = Working on Stop Bar *)
    CharPos : Word;   (* This holds the position in the current char of the barcode *)

    Procedure SetBarCode(tCodeStr: String);
      (* Set the barcode to be processed *)

    Function GetNextBar(Var BarType: tBarStyle): Boolean;
      (* Returns true if there is another bar *)
      (* Returns in BarType one of:
               btSpace,btBar,btNone *)
    Function GetNextBarSize(Var BarType: tBarStyle): Boolean;
      (* As above but will return the wide & narrow size bar *)
      (* Note that this will skip through the bar code faster *)
      (* because it is returning compressed bar/space information *)
      (* Returns in BarType one of:
               btSpace,btBar,btWideSpace,btWideBar,btNone *)
  End;

Implementation

Const
  StartStop = $8BBA;  (* Start/Stop char for all 3 of 9 barcodes *)
  FirstChar = ' ';
  LastChar  = 'Z';
Const
  Bars : Array[FirstChar..LastChar] of Word =   (* Bar code information *)

(*    0000  1111  2222  3333  4444  5555  6666  7777  8888  9999 *)
{32= } (            $8EBA,$0000,$0000,$0000,$888A,$A222,$0000,$0000,
{40=(}  $0000,$0000,$0000,$8A22,$0000,$8AEE,$E2BA,$88A2,$A3BA,$E8AE,
{50=2}  $B8AE,$EE2A,$A3AE,$E8EA,$B8EA,$A2EE,$E8BA,$B8BA,$0000,$0000,
{60=<}  $0000,$0000,$0000,$0000,$0000,$EA2E,$BA2E,$EE8A,$AE2E,$EB8A,
{70=F}  $BB8A,$A8EE,$EA3A,$BA3A,$AE3A,$EA8E,$BA8E,$EEA2,$AE8E,$EBA2,
{80=P}  $BBA2,$AB8E,$EAE2,$BAE2,$AEE2,$E2AE,$8EAE,$E3AA,$8BAE,$E2EA,
{90=Z}  $8EEA);

(* How it works ---
(* Each word stores a barcode char in binary form *)
(* when the word is converted to base 2  you can see *)
(* the bars and spaces come to life- where there are *)
(* multiple bars side by side they indicate a wide bar *)
(* Note: Bars and spaces are either 1 or 3 wide.       *)
 
Procedure tBarCode.SetBarCode(tCodeStr: String);
Begin
  CodeStr := tCodeStr;
  StrPos := 0;
  CharPos := $8000;
  CurrentBar := StartStop;
  cFlag := btNone;
  MoreToGet := True;
End;

Function tBarCode.GetNextBar(Var BarType: tBarStyle): Boolean;
Begin
  If StrPos = -1 Then
    Begin
      GetNextBar := False;
      Exit;
    End;
  GetNextBar := True;
  If CurrentBar and CharPos = CharPos Then
    BarType := btBar
  Else
    BarType := btSpace;

  CharPos := CharPos Shr 1;

  If CharPos = 0 Then  (* End of the current char *)
    Begin
      CharPos := $8000;

      Inc(StrPos);
      If StrPos <= Length(CodeStr) Then  (* Get more data *)
        Begin
          If (CodeStr[StrPos] < FirstChar) or
             (CodeStr[StrPos] > LastChar) Then
            Begin
              RunError(201);  (* You can't pass a char that is out of range *)
            End
          Else
            Begin
              CurrentBar := Bars[CodeStr[StrPos]];
            End;
	End;

      If StrPos > Length(CodeStr) Then  (* Time of the stop data *)
        Begin
          CurrentBar := StartStop;	
	End ;

      If StrPos > (Length(CodeStr)+1) Then  (* End of the road *)
        Begin
          StrPos := -1;   (* Flag that we are done ! - This will get the last char out *)
	End
    End;
End;

Function tBarCode.GetNextBarSize(Var BarType: tBarStyle): Boolean;
(* This function will scan ahead to see if there is a wide bar    *)
(* or space and returns the extended btWideSpace & btWideBar info *)

Var
  Temp : tBarStyle;
Begin
  GetNextBarSize := True;

  If (cFlag = btNone) and (Not MoreToGet) Then
    Begin
      GetNextBarSize := False;
      Exit;
    End;

  If cFlag = btNone Then
    Begin
      MoreToGet := GetNextBar(cFlag);
    End;

  If MoreToGet Then
    Begin
      MoreToGet := GetNextBar(Temp);
      If MoreToGet Then
        Begin
          If cFlag = Temp Then
            Begin
              If cFlag = btSpace Then
                cFlag := btWideSpace;
              If cFlag = btBar Then
                cFlag := btWideBar;
              MoreToGet := GetNextBar(Temp);  (* Clear out the extra bar Must 1 or 3 in a row *)
              BarType := cFlag;
              cFlag := btNone;
            End
          Else
            Begin
              BarType := cFlag;
              cFlag := Temp;
            End;
        End
      Else
        Begin
          BarType := cFlag;
          cFlag := btNone;  (* There are no more to get and we are finished *)
          MoreToGet := False;
        End;
    End;
End;

(*************************************************************)

Procedure PrintBarCode(PDC: HDC;XPos,YPos: Real;BarCode: String;Heading: Boolean);
Var
  MyBarCode : tBarCode;
  BarType : tBarStyle;
  Point: tPoint;
  pS : Array[0..40] of Char;
  PrevObject : THandle;
  Rect : tRect;
Begin
  PrevObject := SelectObject(PDC,GetStockObject(Black_Pen));
  MyBarCode.SetBarCode(BarCode);
  XPos := XPos * 100;
  YPos := -(YPos * 100);
  With Rect Do
    Begin
      Top    := Trunc(YPos);
      Bottom := Trunc(YPos) + Trunc((BarHeight*100));
    End;

  While MyBarCode.GetNextBarSize(BarType) Do
    Begin
      Case BarType of
        btSpace:
          Begin
            XPos := XPos + Trunc(BarWidth*100 + ExtraAir*100);
          End;
        btWideSpace:
          Begin
            XPos := XPos + Trunc((BarWidth*Ratio)*100 + ExtraAir*100);
          End;

        btBar:
          Begin
            With Rect Do
              Begin
                Left   := Trunc(XPos);
                Right  := Left + Trunc(BarWidth*100);
                FillRect(PDC,Rect,GetStockObject(Black_Brush));
                XPos := XPos + BarWidth*100;
              End;
          End;
        btWideBar:
          Begin
            With Rect Do
              Begin
                Left   := Trunc(XPos);
                Right  := Left + Trunc((BarWidth*Ratio)*100);
                FillRect(PDC,Rect,GetStockObject(Black_Brush));
                XPos := XPos + (BarWidth*Ratio)*100;
              End;
          End;
       End; (* case *)
    End;

  If Heading Then
    Begin
      GetCurrentPositionEX(PDC,@Point);
      SetTextAlign(PDC,ta_Center+ta_Bottom);
      StrPCopy(pS,BarCode);
      TextOut(pdc,Trunc(XPos) + ((Point.X-Trunc(XPos)) Div 2),
          Trunc(YPos+(BarHeight*100)),pS,Length(BarCode));
    End;
  SelectObject(PDC,PrevObject);
End;


End.

