/*
    The source code contained within this file is protected under the
    laws of the United States of America and by International Treaty.
    Unless otherwise noted, the source contained herein is:

    Copyright (c)1990-1994 BecknerVision Inc - No Rights Reserved

    Written by John Wm Beckner
    BecknerVision Inc

        ** THIS FILE IS PUBLIC DOMAIN **

        ** Use of this code is at your own risk, absolutely no guarantees
        ** are made to its usefulness  - you must determine for yourself if
        ** this code is of any use to you.  If you do not agree with this, do
        ** not use this code. **
*/

STATIC aReportData, nLinesLeft, aReportTotals
STATIC aGroupTitles, lFirstPass, lFormFeeds, nMaxLinesAvail
STATIC lHasTotals, nDataWidth, nLeftMargin, nColumns, nGroups, lToFile
STATIC cExprBuff
STATIC cOffsetsBuff
STATIC cLengthsBuff


#include "beckner.inc"
#include "frmdef.ch"
#include "error.ch"

////////////////
////////////////
//
// Purpose:
//    Reads an FRM file into an array
//
// Syntax:
//    fCreateFrm(<cFile>) -> aReport
//
// Formal Arguments: (1)
//    Name        Description
//    ___________ ____________
//    cFile       File name
//
// Returns:
//    aReport     Report parameters
//
// Examples:
//    FUNCTION TestIt()
//       LOCAL cFile    := "Test", aCheck, nCtr
//
//       LOCAL aColumn1 := {32,              /* column width               */;
//                .n.,                       /* total?                     */;
//                0,                         /* decimal positions          */;
//                "Name",                    /* column expression          */;
//                "Name",                    /* column header 1            */;
//                "",                        /* column header 2            */;
//                "",                        /* column header 3            */;
//                ""                         /* column header 4            */}
//
//       LOCAL aColumn2 := {12,              /* column width               */;
//                .y.,                       /* total?                     */;
//                2,                         /* decimal positions          */;
//                "Balance",                 /* column expression          */;
//                "Balance",                 /* column header 1            */;
//                "",                        /* column header 2            */;
//                "",                        /* column header 3            */;
//                ""                         /* column header 4            */}
//
//       LOCAL aColumn3 := {1,               /* column width               */;
//                .n.,                       /* total?                     */;
//                0,                         /* decimal positions          */;
//                "Transform(Over21, [Y])",  /* column expression          */;
//                ">",                       /* column header 1            */;
//                "2",                       /* column header 2            */;
//                "1",                       /* column header 3            */;
//                "?"                        /* column header 4            */}
//
//       LOCAL aReport  := {"Name List",     /* page heading 1             */;
//                "",                        /* page heading 2             */;
//                "",                        /* page heading 3             */;
//                "",                        /* page heading 4             */;
//                80,                        /* page width                 */;
//                10,                        /* left margin                */;
//                10,                        /* right margin               */;
//                58,                        /* printable lines per page   */;
//                .n.,                       /* double space?              */;
//                .n.,                       /* plain page?                */;
//                .y.,                       /* eject after report?        */;
//                .n.,                       /* eject before report?       */;
//                "",                        /* group expression           */;
//                "",                        /* group header               */;
//                .n.,                       /* summary only?              */;
//                .n.,                       /* eject after group          */;
//                "",                        /* subgroup expression        */;
//                "",                        /* subgroup header            */;
//                3,                         /* number of columns          */;
//                aColumn1,                  /* column 1 definition        */;
//                aColumn2,                  /* column 2 definition        */;
//                aColumn3                   /* column 3 definition        */}
//
//       CreateTest()
//       fCreateFrm(cFile, aReport)
//       REPORT FORM Test TO PRINTER NOCONSOLE
//       aCheck := aReadFrm(cFile)
//       FOR nCtr := 1 to 19
//          ? aCheck[nCtr]=aReport[nCtr]     /* .t. */
//       NEXT
//       CLOSE Test
//    ENDFUNCTION
//
//    #include "alias.ch"
//    STATIC FUNCTION CreateTest()
//       fCreateDbf("Test/Name/C/32/Balance/N/12/2/Date/D/Over21/L/Notes/M")
//       USE Test NEW EXCLUSIVE
//       ADDRECORD ALIAS Test
//       Test->Name    := "Abbott, Jake"
//       Test->Balance := 0.00
//       Test->Date    := CtoD("9/26/94")
//       Test->Over21  := .y.
//       Test->Notes   := ""
//       ADDRECORD ALIAS Test
//       Test->Name    := "Beckner, John Wm"
//       Test->Balance := 15000.00
//       Test->Date    := CtoD("9/26/94")
//       Test->Over21  := .y.
//       Test->Notes   := "These are notes!"
//       ADDRECORD ALIAS Test
//       Test->Name    := "Beckner, Elizabeth Anne"
//       Test->Balance := 0.58
//       Test->Date    := CtoD("12/18/96")
//       Test->Over21  := .n.
//       Test->Notes   := ""
//       ADDRECORD ALIAS Test
//       Test->Name    := "Beckner, Joseph Alan"
//       Test->Balance := 1.68
//       Test->Date    := CtoD("6/10/98")
//       Test->Over21  := .n.
//       Test->Notes   := "These are too!"
//       ADDRECORD ALIAS Test
//       Test->Name    := "Zumundi, Beavis"
//       Test->Balance := 123.45
//       Test->Date    := CtoD("")
//       Test->Over21  := .y.
//       Test->Notes   := ""
//       GO TOP
//    ENDFUNCTION
//
// Files:
//    (<cFile>.frm)
//
// Description:
//    Reads an FRM file into an array.  The elements of the array
//    are defined in the example above.
//
// See Also:
//    fCreateFrm()
//    fCreateLbl()
//    fReadLbl()
//    pRepGen()
//
// Category:
//    File Function
//
// Revisions:
//    01/26/94 Added comment blocks
//    02/18/94 12:50:00 Forced FRM extension
//                      Commented out IF USED...ENDIF code
//
////////////////
////////////////

FUNCTION fReadFRM(cFilename)
   __FrmLoad(fExtNew(cFilename, "FRM"))
   RETURN iFrm()
ENDFUNCTION

////////////////
////////////////
//
// Purpose:
//    Modified REPORT FORM function - internal
//
// Syntax:
//    see REPORT FORM command
//
// Description:
//    This file was modified by Jo W French, then re-modified by John Wm
//    Beckner.  It is noted that this file is Copyright (c)1990 Nantucket
//    Corp and is used with permission.  The primary correction was in two
//    areas, the reading of the plus_byte, specifically the eject before
//    printing flag, and the adding of aBeckner for use with fReadFrm().
//
// See Also:
//    fCreateFrm()
//    fReadFrm()
//
// Category:
//    Internal Function
//
// Revisions:
//    01/26/94 Added comment blocks
//
////////////////
////////////////


/***
*   Modified Frmrun.prg ( Clipper 5.0 REPORT FORM runtime system ).
*   Original Copyright (c) 1990 Nantucket Corp.  All rights reserved.
*   Modified by Jo W. French dba Practical Computing 08-20-1991.
*
*   08/20/91 Check for invalid Right Margin. If bad, ignore.
*   08/20/91 Cleanup aPageHeader for next report run.
*   08/20/91 Print Group Headers when SUMMARY report.
*
*   08/20/91 SEE NOTES at EOF for 2 Patches to FRMBACK.PRG.
*
*                    REPORT LAYOUT
*       |<                  Physical Page Width                 >|
*       |      |<             Report Page Width (RL)            >|
*       | Set  |Report|                                   |Report|
*       |Margin| Left |<        Data  Area Width         >| Right|
*       |      |Margin|           nDataWidth              |Margin|
*       | nLeftMargin |    (See #define NO_PAGE_ONE_LINE) |      |
*       |      |      | Page #|<   HEADING 1 ;   >|       |      |
*       |      |      |   15  |<   HEADING 2     >| 15    |      |
*       |      |      |       |    nHEADINGWidth          |      |
*       |      |      |       |  or, if won't fit:        |      |
*       |      |      | Page #|<      HEADING 1 ;        >|      |
*       |      |      |   15  |<      HEADING 2          >|      |
*       |      |      | Date                              |      |
*       |      |      |<         Report Header 1         >|      |
*       |      |      |<         Report Header 2         >|      |
*       |      |      | Blank                             |      |
*       |      |      | Column || Column ||   Column      |      |
*       |      |      | Heading|| Heading||   Heading     |      |
*       |      |      | Blank                             |      |
*       |      |      | Blank  (See #define STD_HEADER    |      |
*       |      |      | Record Header (Group, SubGroup)   |      |
*       |      |      | Record Data (Wrapped, Parsed)     |      |
*       |      |      | Record Footer (SubTotal, etc.)    |      |
*       |      |      |        etc.                       |      |
*
*  Notes: If TO FILE is specified, the TO FILE name is compared to
*         ALTFILE and EXTRAFILE.
*         If FILE TO name is unique, an EXTRAFILE will be opened and closed.
*         If FILE TO name == ALTFILE and ALTFILE is open, it will be appended
*            to and left open.
*         If FILE TO name == ALTFILE and ALTFILE is not open, ALTFILE will
*            be overwritten and closed.
*         If FILE TO name == EXTRAFILE and EXTRAFILE is _OPEN_OR_CLOSED_,
*            it will be opened, appended to, and CLOSED.
*
*   Compile: /n/w/m
*/


#define PAGE_ONE_LINE   // Comment out to remove 1st Page top blank line.
#define STOPPIT         // Comment out to prevent Escaping from report.
#define SHORT_RL        // Comment out to force Column Headings + 3 lines.
#define PLAIN_FF        // Comment out to have NO form feeds in PLAIN report.
#define FILE_FF         // Comment out to have NO form feeds in FILE report.

FUNCTION __ReportForm( cFRMName, lPrinter, cAltFile, lNoConsole, bFor, ;
                      bWhile, nNext, nRecord, lRest, lPlain, cHEADING, ;
                      lBEject, lSummary )

  LOCAL lPrintOn, lConsoleOn           // Status of PRINTER and CONSOLE
  LOCAL cExtraFile, lExtraState := .F. // Status of EXTRA
  LOCAL nCol, nGroup
  LOCAL xBreakVal, lBroke := .F.
  LOCAL err, nSetMargin, nLine, nRepWide := 0
  lHasTotals := lToFile := .F.

  // Resolve parameters
  IF cFRMName == NIL
    err           := ErrorNew()
    err:severity  := 2
    err:genCode   := EG_ARG
    err:subSystem := "FRMLBL"
    Eval(ErrorBlock(), err)
  ELSE
    IF AT( ".", cFRMName ) == 0
      cFRMName := TRIM( cFRMName ) + ".FRM"
    ENDIF
  ENDIF

  cHEADING := IF( cHEADING == NIL, "", cHEADING )

  // Set output devices
  lPrintOn   := IF(lPrinter, SET( _SET_PRINTER, lPrinter ), SET( _SET_PRINTER) )
  lConsoleOn := IF(lNoConsole, SET( _SET_CONSOLE, .F.), SET( _SET_CONSOLE) )

  lFormFeeds := IF(lPrinter, .T., .F.)

  IF !Empty(cAltFile)       // To file
    cAltFile := UPPER(cAltFile)
    lToFile     := .T.
    IF cAltFile == UPPER(SET(_SET_ALTFILE)) .and. SET(_SET_ALTERNATE)
      // If an AltFile was open, append to it.
      SET(_SET_ALTFILE, cAltFile, .T. )
    ELSE
      // Note: Anomaly SET(_SET_EXTRA) always returns NIL.
      cExtraFile := UPPER(SET(_SET_EXTRAFILE))
      IF cAltFile == cExtraFile
        // If an ExtraFile existed, append to it.
        SET(_SET_EXTRAFILE, cAltFile, .T.)
      ELSE
        SET(_SET_EXTRAFILE, cAltFile, .F.)
      ENDIF
      SET(_SET_EXTRA, .T.)
      lExtraState := .F.
    ENDIF
  ENDIF

  BEGIN SEQUENCE

    aReportData    := __FrmLoad( cFRMName )   // Load the frm into an array

    // Modify aReportData based on the report parameters
    aReportData[ RP_SUMMARY ] := IF( lSummary, .T., aReportData[ RP_SUMMARY ] )
    aReportData[ RP_BEJECT  ] := IF( lBEject , .F., aReportData[ RP_BEJECT  ] )
    // Parse cHEADING and store in aReportData[ RP_HEADING ].
    aReportData[ RP_HEADING ] := STRTRAN(cHEADING,";",CHR(13)+CHR(10))

    IF lPlain                // Set plain report flag
      aReportData[ RP_PLAIN ] := .T.
      cHEADING                := ""

#ifndef PLAIN_FF
      lFormFeeds              := .F.
#endif

    ENDIF

    // Subtract 1 from RL spacing (for convenience).
    aReportData[ RP_SPACING ] := IF(aReportData[ RP_SPACING ] > 0, ;
                                    aReportData[ RP_SPACING ] - 1, 0)
    // Set variables.
    nSetMargin     := SET( _SET_MARGIN, 0 )

    nLeftMargin    := aReportData[ RP_LMARGIN ] + nSetMargin

    nDataWidth     := aReportData[ RP_WIDTH] - aReportData[ RP_LMARGIN ] - ;
                                               aReportData[ RP_RMARGIN ]

    lFirstPass     := .T.           // Set the first pass flag
    nLinesLeft     := aReportData[ RP_LINES ]
    nColumns       := LEN(aReportData[RP_COLUMNS])
    nGroups        := LEN(aReportData[RP_GROUPS])


    // Initialize aReportTotals to track both group and report totals, then
    // set the column total elements to 0 if they are to be totaled, otherwise
    // leave them NIL

    aReportTotals := ARRAY( nGroups + 1, nColumns )

    // Set aReportTotals to 0, determine if there are totals in report.
    FOR nCol := 1 TO nColumns

      nRepWide += ( aReportData[RP_COLUMNS,nCol,RC_WIDTH] + 1 )

#ifdef SHORT_RL
      // Minimize RL Column Header.
      FOR nline := LEN(aReportData[RP_COLUMNS,nCol,RC_HEADER]) TO 1 STEP -1
        IF !EMPTY(aReportData[RP_COLUMNS,nCol,RC_HEADER,nline])
          EXIT
        ELSE
           ASIZE(aReportData[RP_COLUMNS,nCol,RC_HEADER], ;
                 LEN(aReportData[RP_COLUMNS,nCol,RC_HEADER]) - 1)
        ENDIF
      NEXT
#endif

      IF aReportData[RP_COLUMNS,nCol,RC_TOTAL]
        lHasTotals := .T.
        FOR nGroup := 1 TO LEN(aReportTotals)
          aReportTotals[nGroup,nCol] := 0
        NEXT
      ENDIF
    NEXT

    nRepWide --

    /* Check for bad right margin setting.*/
    IF nDataWidth < nRepWide
      nDataWidth += aReportData[RP_RMARGIN]     // Add Back Right Margin
    ENDIF

    // Initialize aGroupTitles as an array
    aGroupTitles := ARRAY( nGroups )

    // Check for "before report" eject.
    IF aReportData[ RP_BEJECT ]
      EjectPage()
    ENDIF

    // Generate and print the initial report header.
    NewPage(.F.)

    // Execute the actual report based on matching records
    DBEval( { || ExecuteReport() }, bFor, bWhile, nNext, nRecord, lRest )

    // End of Report Processing; however, if lHasTotals.

    IF lHasTotals
      TotalsHeader(.T.)
    ENDIF


    // Check for "after report" eject.
    IF aReportData[ RP_AEJECT ]
      EjectPage()
    ENDIF


  RECOVER USING xBreakVal

    lBroke := .T.

  END  // Sequence

  // clean up
  SET( _SET_PRINTER, lPrintOn   )     // Set the printer back to prior state
  SET( _SET_CONSOLE, lConsoleOn )     // Set the console back to prior state
  SET( _SET_MARGIN,  nSetMargin )     // Set the margin  back to prior state

  IF lToFile                // To file
    IF (cAltFile == UPPER(SET(_SET_ALTFILE)) .and. SET(_SET_ALTERNATE)) .or. ;
                                                          lExtraState
      * Do Nothing
    ELSE
      SET( _SET_EXTRA, .F. )
      SET( _SET_EXTRAFILE, cExtraFile, .T.)
    ENDIF
  ENDIF

  // Recover the space
  aReportData    := NIL
  aReportTotals  := NIL
  aGroupTitles   := NIL
  lFirstPass     := NIL
  nLinesLeft     := NIL
  lFormFeeds     := NIL
  nMaxLinesAvail := NIL
  lHasTotals     := NIL
  nDataWidth     := NIL
  nLeftMargin    := NIL
  nColumns       := NIL
  nGroups        := NIL
  lToFile        := NIL
  NewPage(.T.)               // Sets aPageHeader and nPageNumber to NIL

  IF lBroke
    // keep the break value going
    BREAK xBreakVal
  END

RETURN NIL

/***
*   ExecuteReport() --> NIL
*   Executed by DBEVAL() for each record that matches record scope
*/

////////////////
////////////////
//
// Purpose:
//    Executed once for each processed record in the report
//
// Syntax:
//    ExecuteReport() -> NIL
//
// Formal Arguments: (0)
//
// Returns:
//    NIL
//
// Description:
//    Executed once for each processed record in the report.
//
// Category:
//    Internal Function
//
// Revisions:
//    01/26/94 Added comment blocks
//
////////////////
////////////////


STATIC FUNCTION ExecuteReport
  LOCAL aRecordToPrint := {}           // Current record to print
  LOCAL nCol                           // Counter for the column work
  LOCAL nGroup                         // Counter for the group work
  LOCAL lGroupChanged  := .F.          // Has any group changed?
  LOCAL nMaxLines                      // Number of lines needed by record
  LOCAL nLine                          // Counter for each record line
  LOCAL cLine                          // Current line of text for parsing
  LOCAL cAdjLine                       // Used for parsing semi-colons
  LOCAL lGroupOneChange                // Used for Group 1 eject
  LOCAL nHdrLength

#ifdef STOPPIT
  IF Inkey() == 27
    BREAK
  ENDIF
#endif

  // Create / Print totals to date.
  IF (!lFirstPass .and. lHasTotals)
    TotalsHeader(.F.)
  ENDIF

  // Add to aRecordToPrint in the event that the group has changed and
  // new group headers need to be generated

  // Cycle through the groups

  IF nGroups > 0
    lGroupOneChange := (MakeAStr(EVAL(aReportData[RP_GROUPS, 1, RG_EXP]),;
           aReportData[RP_GROUPS, 1, RG_TYPE]) != aGroupTitles[1])

    IF aReportData[RP_GROUPS,1 ,RG_AEJECT] .and. lGroupOneChange .and. ;
                               !lHasTotals .and. !lFirstPass
      NewPage(.F.)

    ENDIF

  ENDIF

  FOR nGroup := 1 TO nGroups
    // If group 1 or group 2 have changed
    IF lGroupOneChange .or. ;
         MakeAStr(EVAL(aReportData[RP_GROUPS,nGroup,RG_EXP]),;
         aReportData[RP_GROUPS,nGroup,RG_TYPE]) != aGroupTitles[nGroup]

      AADD( aRecordToPrint, "" )   // The blank line

      AADD( aRecordToPrint, IF(nGroup==1,"** ","* ") +;
            aReportData[RP_GROUPS,nGroup,RG_HEADER] + " " +;
            MakeAStr(EVAL(aReportData[RP_GROUPS,nGroup,RG_EXP]), ;
            aReportData[RP_GROUPS,nGroup,RG_TYPE]) )
    ENDIF
  NEXT

  nHdrLength := LEN(aRecordToPrint)

  // Only run through the record detail if this is NOT a summary report
  IF !aReportData[ RP_SUMMARY ]
    // Determine the max number of lines needed by each expression
    nMaxLines := 1
    FOR nCol := 1 TO nColumns
      IF aReportData[RP_COLUMNS,nCol,RC_TYPE] $ "CM"
        IF aReportData[RP_COLUMNS,nCol,RC_TYPE] == "C"
          cAdjLine := STRTRAN(EVAL(aReportData[RP_COLUMNS,nCol,RC_EXP]), ;
                              ";", CHR(13)+CHR(10))
        ELSE
          cAdjLine := EVAL(aReportData[RP_COLUMNS,nCol,RC_EXP])
        ENDIF
        nMaxLines := MAX(XMLCOUNT(cAdjLine, ;
                     aReportData[RP_COLUMNS,nCol,RC_WIDTH]), nMaxLines)
      ENDIF
    NEXT

    // Size aRecordToPrint to the maximum number of lines it will need, then
    // fill it with nulls

    ASIZE( aRecordToPrint, nHdrLength + nMaxLines)
    AFILL( aRecordToPrint, "" , nHdrLength + 1)

    // Load the current record into aRecordToPrint
    FOR nCol := 1 TO nColumns
      FOR nLine := 1 TO nMaxLines
        // Check to see if it's a memo or character
        IF aReportData[RP_COLUMNS,nCol,RC_TYPE] $ "CM"
          IF aReportData[RP_COLUMNS,nCol,RC_TYPE] == "C"
            cAdjLine := STRTRAN(EVAL(aReportData[RP_COLUMNS,nCol,RC_EXP]), ;
                             ";", CHR(13)+CHR(10))
          ELSE
            cAdjLine := EVAL(aReportData[RP_COLUMNS,nCol,RC_EXP])
          ENDIF
          // Load the current line of the current column into cLine
          cLine := XMEMOLINE(TRIM(cAdjLine),;
                   aReportData[RP_COLUMNS,nCol,RC_WIDTH], nLine )
          cLine := PADR( cLine, aReportData[RP_COLUMNS,nCol,RC_WIDTH] )
        ELSE
          IF nLine == 1
            cLine := TRANSFORM(EVAL(aReportData[RP_COLUMNS,nCol,RC_EXP]),;
                     aReportData[RP_COLUMNS,nCol,RC_PICT])
            cLine := PADR( cLine, aReportData[RP_COLUMNS,nCol,RC_WIDTH] )
          ELSE
            cLine := SPACE( aReportData[RP_COLUMNS,nCol,RC_WIDTH])
          ENDIF
        ENDIF
        // Add it to the existing report line
        aRecordToPrint[ nHdrLength + nLine ] += cLine
        IF nCol < nColumns
          aRecordToPrint[ nHdrLength + nLine ] += " "
        ENDIF
      NEXT
    NEXT

  ENDIF    // Was this a summary report?

  // Remove blank line if first line after report header.
  IF nHdrLength > 0 .and. (lFirstPass .or. nLinesLeft < LEN(aRecordToPrint))
    ADEL(aRecordToPrint, 1)
    ASIZE(aRecordToPrint, LEN(aRecordToPrint) - 1)
  ENDIF

  // Print aRecordToPrint
  PrintArray(aRecordToPrint, .F.)

  // Tack on the spacing for double/triple/etc.
  nLine := aReportData[ RP_SPACING ]
  WHILE (nLine > 0 .and. nLinesLeft > 0)
    PrintIt()
    nLine --
    nLinesLeft --
  END


  lFirstPass := .F.

  // Reset the group expressions in aGroupTitles
  FOR nGroup := 1 TO nGroups
    aGroupTitles[nGroup] := MakeAStr(EVAL(aReportData[RP_GROUPS,nGroup,RG_EXP]),;
                            aReportData[RP_GROUPS,nGroup,RG_TYPE])
  NEXT

  // Add to the group totals
  FOR nCol := 1 TO nColumns
    // If this column should be totaled, do it
    IF aReportData[RP_COLUMNS,nCol,RC_TOTAL]
      // Cycle through the groups
      FOR nGroup := 1 TO LEN( aReportTotals )
        aReportTotals[nGroup ,nCol] += ;
          EVAL( aReportData[RP_COLUMNS,nCol,RC_EXP] )
      NEXT
    ENDIF
  NEXT

RETURN NIL


/***
*   NewPage(lDestruct) --> NIL
*    Create Page Header. Eject and Send to print. Reset nLinesLeft.
*/

////////////////
////////////////
//
// Purpose:
//    Internal function
//
// Description:
//    Internal function
//
// Category:
//    Internal Function
//
// Revisions:
//    01/26/94 Added comment blocks
//
////////////////
////////////////

STATIC FUNCTION NewPage(lDestruct)
  STATIC aPageHeader, nPageNumber
  LOCAL nLinesInHeader := 0
  LOCAL nCol, nLine, nMaxColLength, nGroup, nHEADINGWidth

  IF lDestruct
    aPageHeader := NIL
    nPageNumber := NIL
    RETURN NIL                  // NOTE
  ENDIF

  IF !aReportData[ RP_PLAIN ]   // If not a plain paper report
    nLinesLeft := nMaxLinesAvail := aReportData[RP_LINES]
  ELSE

#ifdef PLAIN_FF
    nLinesLeft := nMaxLinesAvail := aReportData[RP_LINES]
#else
    nLinesLeft := nMaxLinesAvail := 1000
#endif

  ENDIF

  IF aPageHeader == NIL
    aPageHeader := {}
    nPageNumber := 1

    // Create the header and drop it into aPageHeader
    // Start with the command line HEADING.

    IF !aReportData[ RP_PLAIN ]         // If not a plain paper report

      // ****   Command line HEADING ****
      nHEADINGWidth := IF(nDataWidth > 30, nDataWidth - 30, ;
                       IF(nDataWidth > 15, nDataWidth - 15, 0 ))

      IF aReportData[RP_HEADING] == ""    // the heading
        AADD( aPageHeader, SPACE(15) )    // Reserved for Page No.
      ELSE
        nLinesInHeader := XMLCOUNT( aReportData[RP_HEADING], nHEADINGWidth )
        FOR nLine := 1 TO nLinesInHeader
          AADD( aPageHeader, SPACE(15) + ;
          PADC(TRIM(XMEMOLINE(aReportData[RP_HEADING],nHEADINGWidth,nLine)),;
               nHEADINGWidth))
        NEXT
      ENDIF
      AADD( aPageHeader, DTOC(DATE()) )

    ENDIF

    // ****   Report Form Header from the FRM ****
    FOR nLine := 1 TO LEN( aReportData[RP_HEADER] )
      IF !EMPTY(aReportData[RP_HEADER, nLine])
        AADD( aPageHeader, PADC(aReportData[ RP_HEADER, nLine ], nDataWidth) )
      ENDIF
    NEXT

    // S87 compat.
    AADD( aPageHeader, "" )

    // ****   Report Form Column Headings from the FRM ****

    nLinesInHeader := LEN( aPageHeader )
    // Determine the longest column header
    nMaxColLength := 0
    FOR nCol := 1 TO nColumns
        nMaxColLength := MAX( LEN(aReportData[RP_COLUMNS,nCol,RC_HEADER]), ;
                              nMaxColLength )
    NEXT
    FOR nCol := 1 TO nColumns
      ASIZE( aReportData[RP_COLUMNS,nCol,RC_HEADER], nMaxColLength )
    NEXT

    FOR nLine := 1 TO nMaxColLength
      AADD( aPageHeader, "" )
    NEXT

    FOR nCol := 1 TO nColumns       // Cycle through the columns

      FOR nLine := 1 TO nMaxColLength

        IF aReportData[RP_COLUMNS,nCol,RC_HEADER,nLine] == NIL
           aPageHeader[ nLinesInHeader + nLine ] += ;
                   SPACE( aReportData[RP_COLUMNS,nCol,RC_WIDTH] )
        ELSE
          IF aReportData[RP_COLUMNS,nCol,RC_TYPE] == "N"
            aPageHeader[ nLinesInHeader + nLine ] += ;
              PADL(aReportData[RP_COLUMNS,nCol,RC_HEADER,nLine],;
                          aReportData[RP_COLUMNS,nCol,RC_WIDTH])
          ELSE
            aPageHeader[ nLinesInHeader + nLine ] += ;
              PADR(aReportData[RP_COLUMNS,nCol,RC_HEADER,nLine],;
                          aReportData[RP_COLUMNS,nCol,RC_WIDTH])
          ENDIF
        ENDIF
        IF nCol < nColumns
          aPageHeader[ nLinesInHeader + nLine ] += " "
        ENDIF
      NEXT
    NEXT

    // Insert a blank line after heading, if column headings.

    IF nMaxColLength > 0
      AADD( aPageHeader, "" )
    ENDIF

#ifdef PAGE_ONE_LINE
    QOUT()
    nLinesLeft --
    nMaxLinesAvail --
#endif

  ENDIF
  // aPageHeader has been initialized. Stuff page number if not PLAIN

  IF !aReportData[ RP_PLAIN ]
    aPageHeader[ 1 ] := STUFF( aPageHeader[ 1 ], 1, 14, ;
                                   "Page No." + STR(nPageNumber,6) )
  ENDIF

  IF !lFirstPass
    EjectPage()
  ENDIF

  // Send it for printing ( nLinesLeft decremented ).
  PrintArray(aPageHeader, .F.)

  // Set nMaxLinesAvail
  nMaxLinesAvail := nLinesLeft

  // Increment the page number
  nPageNumber++

RETURN NIL


/***
*   TotalsHeader( <lEndOfReport> ) --> NIL
*    Create and Print Record Header when Totals are included.
*/

////////////////
////////////////
//
// Purpose:
//    Internal function
//
// Description:
//    Internal function
//
// Category:
//    Internal Function
//
// Revisions:
//    01/26/94 Added comment blocks
//
////////////////
////////////////

STATIC FUNCTION TotalsHeader(lEndOfReport)
  Local nCol, nGroup, lAnySubTotals, lGroupOneChange
  Local aTotalsHeader := {}
  // Determine if any of the groups have changed.  If so, add the appropriate
  // line to aTotalsHeader for totaling out the previous records
  IF !lFirstPass         // Don't bother
    IF nGroups > 0
      lGroupOneChange := (MakeAStr(EVAL(aReportData[RP_GROUPS, 1, RG_EXP]),;
         aReportData[RP_GROUPS, 1, RG_TYPE]) != aGroupTitles[1])
    ENDIF

    // Make a pass through all the groups
    FOR nGroup :=  nGroups TO 1 STEP -1
      // make sure group has subtotals
      lAnySubTotals := .F.
      FOR nCol := 1 TO nColumns
        IF aReportData[RP_COLUMNS,nCol,RC_TOTAL]
          lAnySubTotals := .T.
          EXIT           // NOTE
        ENDIF
      NEXT

      IF !lAnySubTotals
        LOOP            // NOTE
      ENDIF
      // **** SubTotals to Date ****
      // If group 1 or group 2 has changed since the last record
      IF (lGroupOneChange .or. ;
          MakeAStr(EVAL(aReportData[RP_GROUPS, nGroup, RG_EXP]), ;
          aReportData[RP_GROUPS,nGroup,RG_TYPE]) != aGroupTitles[nGroup])

        AADD( aTotalsHeader, IF(nGroup==1,"** Subtotal **","* Subsubtotal *") )
        AADD( aTotalsHeader, "" )

        // Cycle through the columns, adding either the group
        // amount from aReportTotals or spaces wide enough for
        // the non-totaled columns
        FOR nCol := 1 TO nColumns
          IF aReportData[RP_COLUMNS,nCol,RC_TOTAL]

            aTotalsHeader[ LEN(aTotalsHeader) ] += ;
              TRANSFORM(aReportTotals[nGroup+1,nCol], ;
              aReportData[RP_COLUMNS,nCol,RC_PICT])

            // Zero out the group totals column from aReportTotals
            aReportTotals[nGroup+1,nCol] := 0
          ELSE
            aTotalsHeader[ LEN(aTotalsHeader) ] += ;
            SPACE(aReportData[RP_COLUMNS,nCol,RC_WIDTH])
          ENDIF
          IF nCol < nColumns
            aTotalsHeader[ LEN(aTotalsHeader) ] += " "
          ENDIF
        NEXT
      ENDIF
    NEXT

  ENDIF     // !lFirstPass

  IF lEndOfReport   // Add Grand Total to aTotalsHeader.
    AADD( aTotalsHeader, "*** Total ***" )
    AADD( aTotalsHeader, "" )

    FOR nCol := 1 TO nColumns
      IF aReportData[RP_COLUMNS,nCol,RC_TOTAL]
        aTotalsHeader[ LEN(aTotalsHeader) ] += ;
          TRANSFORM(aReportTotals[ 1, nCol], ;
          aReportData[RP_COLUMNS,nCol,RC_PICT])
      ELSE
        aTotalsHeader[ LEN(aTotalsHeader) ] += ;
        SPACE(aReportData[RP_COLUMNS,nCol,RC_WIDTH])
      ENDIF
      IF nCol < nColumns
        aTotalsHeader[ LEN(aTotalsHeader) ] += " "
      ENDIF
    NEXT
  ENDIF

  PrintArray( aTotalsHeader, .T. )

RETURN NIL

/***
*   PrintArray( <array>, <lIsTotHdg> ) --> NIL
*    Send the input array to the output device.
*/

////////////////
////////////////
//
// Purpose:
//    Internal function
//
// Description:
//    Internal function
//
// Category:
//    Internal Function
//
// Revisions:
//    01/26/94 Added comment blocks
//
////////////////
////////////////

STATIC FUNCTION PrintArray(aArray, lIsTotHdg)
  Local nLine, nPtr := 0

  IF LEN( aArray ) > 0


    // Send aArray to the output device, resetting nLinesLeft
    IF !lIsTotHdg
      IF LEN( aArray ) <= nLinesLeft
        AEVAL( aArray, ;
             { | PrintLine | PrintIt( SPACE(nLeftMargin) + PrintLine ) } )
        nLinesLeft -= LEN( aArray )

      ELSEIF LEN( aArray ) <= nMaxLinesAvail
        NewPage(.F.)                // Recursive
        AEVAL( aArray, ;
             { | PrintLine | PrintIt( SPACE(nLeftMargin) + PrintLine ) } )
        nLinesLeft -= LEN( aArray )

      ELSE
        // Big Momma
        IF nLinesLeft == 0
          NewPage(.F.)             // Recursive
        ENDIF
        nLine := 1
        WHILE nLine < LEN( aArray )
          PrintIt( SPACE(nLeftMargin) + aArray[nLine] )
          nLine++
          nLinesLeft--
          IF nLinesLeft == 0
            NewPage(.F.)             // Recursive
          ENDIF
        END
      ENDIF

    ELSE

      FOR nLine := 1 TO LEN(aArray) STEP 2
        IF nLinesLeft < 2
          NewPage(.F.)               // Recursive
        ENDIF
        AEVAL( aArray, ;
           { | PrintLine | PrintIt( SPACE(nLeftMargin) + PrintLine ) }, ;
            nLine, 2)
        nLinesLeft -= 2
      NEXT
      IF (nGroups > 0 .and. ;
          aReportData[RP_GROUPS,1 ,RG_AEJECT] .and. LEN(aArray)/2 = nGroups)
        NewPage(.F.)                 // Recursive
      ENDIF

    ENDIF
  ENDIF

RETURN NIL

/***
*   MakeStr( <exp>, <cType> ) --> value
*    Convert a value of any data type into string to add to the group header
*/

////////////////
////////////////
//
// Purpose:
//    Internal function
//
// Description:
//    Internal function
//
// Category:
//    Internal Function
//
// Revisions:
//    01/26/94 Added comment blocks
//
////////////////
////////////////

STATIC FUNCTION MakeAStr( uVar, cType )
  LOCAL cString
  DO CASE
    CASE UPPER(cType) == "D"
      cString := DTOC( uVar )

    CASE UPPER(cType) == "L"
      cString := IF( uVar, "T", "F" )

    CASE UPPER(cType) == "N"
      cString := STR( uVar )

    CASE UPPER(cType) == "C"
      cString := uVar

    OTHERWISE
      cString := "INVALID EXPRESSION"
  ENDCASE
RETURN( cString )

/***
*   PrintIt( <cString> ) --> NIL
*   Print a string, THEN send a CRLF
*/

////////////////
////////////////
//
// Purpose:
//    Internal function
//
// Description:
//    Internal function
//
// Category:
//    Internal Function
//
// Revisions:
//    01/26/94 Added comment blocks
//
////////////////
////////////////

STATIC FUNCTION PrintIt( cString )
  IF cString == NIL
    cString := ""
  ELSE
    // prevents output of trailing space, also prevents wrapping of some
    // lines when sent to screen or 80-column printer. Comment out this
    // line for complete Summer 87 compatibility.
    // cString := Trim( cString )
  ENDIF

  QQOUT( cString )
  QOUT()

RETURN NIL

/***
*   EjectPage() --> NIL
*   Eject a page if the form feed option is set
*   Send CHR(12) if TO FILE and FILE_FF is defined.
*/

////////////////
////////////////
//
// Purpose:
//    Internal function
//
// Description:
//    Internal function
//
// Category:
//    Internal Function
//
// Revisions:
//    01/26/94 Added comment blocks
//
////////////////
////////////////

STATIC FUNCTION EjectPage
  IF lFormFeeds
    EJECT

#ifdef FILE_FF
  ELSEIF lToFile
    QQOUT(CHR(12))
#endif

  ENDIF
RETURN NIL


////////////////
////////////////
//
// Purpose:
//    Internal function
//
// Description:
//    Internal function
//
// Category:
//    Internal Function
//
// Revisions:
//    01/26/94 Added comment blocks
//
////////////////
////////////////

STATIC FUNCTION XMLCOUNT( cString, nLineLength, nTabSize, lWrap )
  // Set defaults if none specified
  nLineLength := IF( nLineLength == NIL, 79, nLineLength )
  nTabSize := IF( nTabSize == NIL, 4, nTabSize )
  lWrap := IF( lWrap == NIL, .T., .F. )

  IF nTabSize >= nLineLength
    nTabSize := nLineLength - 1
  ENDIF

RETURN( MLCOUNT( TRIM(cString), nLineLength, nTabSize, lWrap ) )


////////////////
////////////////
//
// Purpose:
//    Internal function
//
// Description:
//    Internal function
//
// Category:
//    Internal Function
//
// Revisions:
//    01/26/94 Added comment blocks
//
////////////////
////////////////

STATIC FUNCTION XMEMOLINE( cString, nLineLength, nLineNumber, nTabSize, lWrap )
  // Set defaults if none specified
  nLineLength := IF( nLineLength == NIL, 79, nLineLength )
  nLineNumber := IF( nLineNumber == NIL, 1, nLineNumber )
  nTabSize := IF( nTabSize == NIL, 4, nTabSize )
  lWrap := IF( lWrap == NIL, .T., lWrap )

  IF nTabSize >= nLineLength
    nTabSize := nLineLength - 1
  ENDIF

RETURN( MEMOLINE( cString, nLineLength, nLineNumber, nTabSize, lWrap ) )

/* NOTES re' FRMBACK.PRG
*  1) Change line 154:
*     FROM:    paths := ListAsArray( s )
*     TO  :    paths := ListAsArray( s, ";" )
*
*  2) Replace the ListAsArray Function with the following:
*
* FUNCTION ListAsArray(cList, cDelim, nWide)
*   LOCAL aArray := {}, lNum
*   cDelim := IF(VALTYPE(cDelim) == 'C', cDelim, ";")
*   cList  := TRIM( STRTRAN(cList, cdelim, chr(13)+chr(10)) )
*   nWide  := IF( (lNum := (VALTYPE(nWide) == 'N' .and. nWide > 0 )), ;
*                        nWide, LEN(cList) )
*   aArray := ARRAY(MLCOUNT(cList,nWide,1,.T.))
*   AEVAL(aArray, {|a,e| aArray[e] := IF(lNum, MEMOLINE(cList,nWide,e,1,.T.),;
*                                     TRIM(MEMOLINE(cList,nWide,e,1,.T.)))})
* RETURN aArray
*/

////////////////
////////////////
//
// Purpose:
//    internal - modified from original supplied with Clipper
//
// Description:
//    Internal.
//
// Category:
//    Internal Function
//
// Revisions:
//    01/26/94 Added comment blocks
//
////////////////
////////////////

/***
*   Frmback.prg
*   Create a report array from a (.frm) file
*   Copyright (c) 1990 Nantucket Corp.  All rights reserved.
*
*   Compile: /n/w/m
*/

// Definitions for buffer sizes
#define  SIZE_FILE_BUFF             1990       // Size of report file
#define  SIZE_LENGTHS_BUFF          110
#define  SIZE_OFFSETS_BUFF          110
#define  SIZE_EXPR_BUFF             1440
#define  SIZE_FIELDS_BUFF           300
#define  SIZE_PARAMS_BUFF           24

// Definitions for offsets into the FILE_BUFF string
#define  LENGTHS_OFFSET             5          // Start of expression length array
#define  OFFSETS_OFFSET             115        // Start of expression position array
#define  EXPR_OFFSET                225        // Start of expression data area
#define  FIELDS_OFFSET              1665       // Start of report columns (fields)
#define  PARAMS_OFFSET              1965       // Start of report parameters block

// These are offsets into the FIELDS_BUFF string to actual values
// Values are added to a block offset FLD_OFFSET that is moved in
// increments of 12
#define  FIELD_WIDTH_OFFSET         1
#define  FIELD_TOTALS_OFFSET        6
#define  FIELD_DECIMALS_OFFSET      7

// These are offsets into FIELDS_BUFF which are used to 'point' into
// the EXPR_BUFF string which contains the textual data
#define  FIELD_CONTENT_EXPR_OFFSET  9
#define  FIELD_HEADER_EXPR_OFFSET   11

// These are actual offsets into the PARAMS_BUFF string which
// are used to 'point' into the EXPR_BUFF string
#define  PAGE_HDR_OFFSET            1
#define  GRP_EXPR_OFFSET            3
#define  SUB_EXPR_OFFSET            5
#define  GRP_HDR_OFFSET             7
#define  SUB_HDR_OFFSET             9

// These are actual offsets into the PARAMS_BUFF string to actual values
#define  PAGE_WIDTH_OFFSET          11
#define  LNS_PER_PAGE_OFFSET        13
#define  LEFT_MRGN_OFFSET           15
#define  RIGHT_MGRN_OFFSET          17
#define  COL_COUNT_OFFSET           19
#define  DBL_SPACE_OFFSET           21
#define  SUMMARY_RPT_OFFSET         22
#define  PE_OFFSET                  23
#define  OPTION_OFFSET              24

// File error definitions
#define  F_OK                       0          // No error
#define  F_EMPTY                   -3          // File is empty
#define  F_ERROR                   -1          // Some kind of error
#define  F_NOEXIST                  2          // File does not exist

/***
*
*  __FrmLoad( cFrmFile ) --> aReport
*  Reads a report (.frm) file and creates a report array
*
*  Notes:
*
*      1.   Report file name has extension.
*      2.   File error number placed in nFileError
*      3.   Offsets start at 1. Offsets are into a Clipper string, 1 to 1990
*      4.   The offsets mentioned in these notes are actual DOS FILE offsets,
*           not like the offsets declared in the body of FrmLoad()
*           which are Clipper STRING offsets.
*      5.   Report file length is 7C6h (1990d) bytes.
*      6.   Expression length array starts at 04h (4d) and can
*           contain upto 55 short (2 byte) numbers.
*      7.   Expression offset index array starts at 72h (114d) and
*           can contain upto 55 short (2 byte) numbers.
*      8.   Expression area starts at offset E0h (224d).
*      9.   Expression area length is 5A0h (1440d).
*     10.   Expressions in expression area are null terminated.
*     11.   Field expression area starts at offset 680h (1664d).
*     12.   Field expressions (column definition) are null terminated.
*     13.   Field expression area can contain up to 25 12-byte blocks.
*/

FUNCTION __FrmLoad( cFrmFile )
   LOCAL aBeckner[19], nCtr
  LOCAL cFieldsBuff
	LOCAL cParamsBuff
	LOCAL nFieldOffset	:= 0
  LOCAL cFileBuff      := SPACE(SIZE_FILE_BUFF)
	LOCAL cGroupExp		:= SPACE(200)
	LOCAL cSubGroupExp	:= SPACE(200)
	LOCAL nColCount		:= 0			   // Number of columns in report
	LOCAL nCount
	LOCAL nFrmHandle							// (.frm) file handle
	LOCAL nBytesRead			            // Read/write and content record counter
  LOCAL nPointer 		:= 0           // Points to an offset into EXPR_BUFF string
  LOCAL nFileError							// Contains current file error
  LOCAL cOptionByte							// Contains option byte

  LOCAL aReport[ RP_COUNT ]				// Create report array
   LOCAL err

  LOCAL s, paths
  LOCAL i

  // Initialize STATIC buffer values
  cLengthsBuff  := ""
  cOffsetsBuff  := ""
  cExprBuff     := ""

  // Default report values
  aReport[ RP_HEADER ]    := {}
	aReport[ RP_WIDTH ]     := 80
	aReport[ RP_LMARGIN ]   := 8
	aReport[ RP_RMARGIN ]   := 0
	aReport[ RP_LINES ]     := 58
  aReport[ RP_SPACING ]   := 1
	aReport[ RP_BEJECT ]    := .T.
	aReport[ RP_AEJECT ]    := .F.
	aReport[ RP_PLAIN ]     := .F.
	aReport[ RP_SUMMARY ]   := .F.
	aReport[ RP_COLUMNS ]   := {}
	aReport[ RP_GROUPS ]    := {}
  aReport[ RP_HEADING ]   := ""

	// Open the report file

	nFrmHandle = FOPEN( cFrmFile )
	nFileError = FERROR()

	IF !( "\" $ cFrmFile .or. ":" $ cFrmFile )
		// if not found and no path in name, go looking

		IF nFileError != F_OK

			s := SET( _SET_DEFAULT )

			IF !Empty( s )
				nFrmHandle := FOPEN( s + "\" + cFrmFile )
				nFileError := FERROR()
			END
		END

		IF nFileError != F_OK

			s := SET( _SET_PATH )
			s := StrTran(s, ",", ";")   // convert any commas in path spec

			paths := ListAsArray( s , ";" )

			FOR i := 1 to Len(paths)
				nFrmHandle := FOPEN( paths[i] + "\" + cFrmFile )
				nFileError := FERROR()

				IF nFileError == F_OK
					EXIT
				END
			NEXT
		END
	END

	// File error
	IF nFileError != F_OK
      err := ErrorNew()
      err:severity := 2
      err:genCode := EG_OPEN
      err:subSystem := "FRMLBL"
      Eval(ErrorBlock(), err)
	ENDIF

	// OPEN ok?
	IF nFileError = F_OK

   	   // Go to START of report file
	   FSEEK(nFrmHandle, 0)

   	   // SEEK ok?
	   nFileError = FERROR()
   	IF nFileError = F_OK

	      // Read entire file into process buffer
   	   nBytesRead = FREAD(nFrmHandle, @cFileBuff, SIZE_FILE_BUFF)

      	// READ ok?
      	IF nBytesRead = 0
         	nFileError = F_EMPTY        // file is empty
      	ELSE
         	nFileError = FERROR()       // check for DOS errors
      	ENDIF

      	IF nFileError = F_OK

         	// Is this a .FRM type file (2 at start and end of file)
          IF BIN2W(SUBSTR(cFileBuff, 1, 2)) = 2 .AND.;
              BIN2W(SUBSTR(cFileBuff, SIZE_FILE_BUFF - 1, 2)) = 2

            	nFileError = F_OK
         	ELSE
					nFileError = F_ERROR
			ENDIF

		ENDIF

	ENDIF

   // Close file
   IF !FCLOSE(nFrmHandle)
      nFileError = FERROR()
   ENDIF

ENDIF

// File existed, was opened and read ok and is a .FRM file
IF nFileError = F_OK

   // Fill processing buffers
   cLengthsBuff = SUBSTR(cFileBuff, LENGTHS_OFFSET, SIZE_LENGTHS_BUFF)
   cOffsetsBuff = SUBSTR(cFileBuff, OFFSETS_OFFSET, SIZE_OFFSETS_BUFF)
   cExprBuff    = SUBSTR(cFileBuff, EXPR_OFFSET, SIZE_EXPR_BUFF)
   cFieldsBuff  = SUBSTR(cFileBuff, FIELDS_OFFSET, SIZE_FIELDS_BUFF)
   cParamsBuff  = SUBSTR(cFileBuff, PARAMS_OFFSET, SIZE_PARAMS_BUFF)


   // Process report attributes
   // Report width
   aReport[ RP_WIDTH ]   := aBeckner[5] := ;
         BIN2W(SUBSTR(cParamsBuff, PAGE_WIDTH_OFFSET, 2))

   // Lines per page
   aReport[ RP_LINES ]   := aBeckner[8] := ;
         BIN2W(SUBSTR(cParamsBuff, LNS_PER_PAGE_OFFSET, 2))

   // Page offset (left margin)
   aReport[ RP_LMARGIN ] := aBeckner[6] := ;
         BIN2W(SUBSTR(cParamsBuff, LEFT_MRGN_OFFSET, 2))

   // Page right margin (not used)
   aReport[ RP_RMARGIN ] := aBeckner[7] := ;
         BIN2W(SUBSTR(cParamsBuff, RIGHT_MGRN_OFFSET, 2))

   nColCount  = aBeckner[19] := ;
         BIN2W(SUBSTR(cParamsBuff, COL_COUNT_OFFSET, 2))

   // Line spacing
   // Spacing is 1, 2, or 3
   aReport[ RP_SPACING ] := ;
         IF(SUBSTR(cParamsBuff, DBL_SPACE_OFFSET, 1) $ "Yy", 2, 1)
   aBeckner[9] := (aReport[RP_SPACING]=2)

   // Summary report flag
   aReport[ RP_SUMMARY ] := aBeckner[15] := ;
         IF(SUBSTR(cParamsBuff, SUMMARY_RPT_OFFSET, 1) $ "Yy", .T., .F.)

   // Process report eject and plain attributes option byte
   cOptionByte = ASC(SUBSTR(cParamsBuff, OPTION_OFFSET, 1))

   aReport[ RP_PLAIN ]  := aBeckner[10] := ;
         (cOptionByte>3)
   cOptionByte -= iif(cOptionByte>3, 4, 0)

   aReport[ RP_AEJECT ] := aBeckner[11] := ;
         (cOptionByte>1)
   cOptionByte -= iif(cOptionByte>1, 2, 0)

   aReport[ RP_BEJECT ] := aBeckner[12] := ;
         (cOptionByte=0)

   // Page heading, report title
   nPointer = BIN2W(SUBSTR(cParamsBuff, PAGE_HDR_OFFSET, 2))

	aReport[ RP_HEADER ] := ;
		ListAsArray(GetExpr( nPointer ), ;
					";", ;
					aReport[ RP_WIDTH ] - aReport[ RP_RMARGIN ] )

   aBeckner[16] := .n.
   aFill(aBeckner, "",  1, 4)
   aFill(aBeckner, "", 13, 2)
   aFill(aBeckner, "", 17, 2)
   FOR nCtr := 1 TO Len(aReport[RP_HEADER])
      aBeckner[nCtr] := aReport[RP_HEADER, nCtr]
   NEXT
   // Process Groups
   // Group
   nPointer = BIN2W(SUBSTR(cParamsBuff, GRP_EXPR_OFFSET, 2))

   IF !EMPTY(cGroupExp := aBeckner[13] := GetExpr( nPointer ))

      // Add a new group array
      AADD( aReport[ RP_GROUPS ], ARRAY( RG_COUNT ))

      // Group expression
      aReport[ RP_GROUPS ][1][ RG_TEXT ] := cGroupExp
      aReport[ RP_GROUPS ][1][ RG_EXP ] := &( "{ || " + cGroupExp + "}" )
      IF USED()
         aReport[ RP_GROUPS ][1][ RG_TYPE ] := ;
                        VALTYPE( EVAL( aReport[ RP_GROUPS ][1][ RG_EXP ] ) )
      ENDIF

      // Group header
      nPointer = BIN2W(SUBSTR(cParamsBuff, GRP_HDR_OFFSET, 2))
      aReport[ RP_GROUPS ][1][ RG_HEADER ] := aBeckner[14] := ;
            GetExpr( nPointer )

      // Page eject after group
      aReport[ RP_GROUPS ][1][ RG_AEJECT ] := aBeckner[16] := ;
            IF(SUBSTR(cParamsBuff, PE_OFFSET, 1) $ "Yy", .T., .F.)

   ENDIF

   // Subgroup
   nPointer = BIN2W(SUBSTR(cParamsBuff, SUB_EXPR_OFFSET, 2))

   IF !EMPTY(cSubGroupExp := aBeckner[17] := GetExpr( nPointer ))

      // Add new group array
      AADD( aReport[ RP_GROUPS ], ARRAY( RG_COUNT ))

      // Subgroup expression
      aReport[ RP_GROUPS ][2][ RG_TEXT ] := cSubGroupExp
      aReport[ RP_GROUPS ][2][ RG_EXP ] := &( "{ || " + cSubGroupExp + "}" )
      IF USED()
         aReport[ RP_GROUPS ][2][ RG_TYPE ] := ;
                        VALTYPE( EVAL( aReport[ RP_GROUPS ][2][ RG_EXP ] ) )
      ENDIF

      // Subgroup header
      nPointer = BIN2W(SUBSTR(cParamsBuff, SUB_HDR_OFFSET, 2))
      aReport[ RP_GROUPS ][2][ RG_HEADER ] := aBeckner[18] := ;
            GetExpr( nPointer )

      // Page eject after subgroup
      aReport[ RP_GROUPS ][2][ RG_AEJECT ] := .F.

   ENDIF

   // Process columns
	 nFieldOffset := 12      // dBASE skips first 12 byte fields block.
   FOR nCount := 1 to nColCount

      AADD( aReport[ RP_COLUMNS ], GetColumn( cFieldsBuff, @nFieldOffset ) )
      aAdd(aBeckner, Array(9))
      aFill(aBeckner[19+nCount], "", 5, 4)
      aBeckner[19+nCount, 1] := aTail(aReport[RP_COLUMNS])[RC_WIDTH]
      aBeckner[19+nCount, 2] := aTail(aReport[RP_COLUMNS])[RC_TOTAL]
      aBeckner[19+nCount, 3] := aTail(aReport[RP_COLUMNS])[RC_DECIMALS]
      aBeckner[19+nCount, 4] := aTail(aReport[RP_COLUMNS])[RC_TEXT]
      FOR nCtr := 1 TO Len(aTail(aReport[RP_COLUMNS])[RC_HEADER])
         aBeckner[19+nCount, 4+nCtr] := ;
               aTail(aReport[RP_COLUMNS])[RC_HEADER, nCtr]
      NEXT
      aBeckner[19+nCount, 9] := nCount*10

   NEXT

ENDIF
   iFRM(aBeckner)

RETURN aReport


/***
*  GetExpr( nPointer ) --> cString
*
*  Reads an expression from EXPR_BUFF via the OFFSETS_BUFF and returns
*  a pointer to offset contained in OFFSETS_BUFF that in turn points
*  to an expression located in the EXPR_BUFF string.
*
*  Notes:
*
*     1. The expression is empty if:
*         a. Passed pointer is equal to 65535
*         b. Character following character pointed to by pointer is CHR(0)
*
*/

////////////////
////////////////
//
// Purpose:
//    Internal function
//
// Description:
//    Internal function
//
// Category:
//    Internal Function
//
// Revisions:
//    01/26/94 Added comment blocks
//
////////////////
////////////////

STATIC FUNCTION GetExpr( nPointer )
   LOCAL nExprOffset   := 0
   LOCAL nExprLength   := 0
   LOCAL nOffsetOffset := 0
   LOCAL cString := ""

   // Stuff for dBASE compatability.
   IF nPointer != 65535

      // Convert DOS FILE offset to CLIPPER string offset
      nPointer++

      // Calculate offset into OFFSETS_BUFF
      IF nPointer > 1
         nOffsetOffset = (nPointer * 2) - 1
      ENDIF

      nExprOffset = BIN2W(SUBSTR(cOffsetsBuff, nOffsetOffset, 2))
      nExprLength = BIN2W(SUBSTR(cLengthsBuff, nOffsetOffset, 2))

      // EXPR_OFFSET points to a NULL, so add one (+1) to get the string
      // and subtract one (-1) from EXPR_LENGTH for correct length

      nExprOffset++
      nExprLength--

      // Extract string
      cString = SUBSTR(cExprBuff, nExprOffset, nExprLength)

      // dBASE does this so we must do it too
      // Character following character pointed to by pointer is NULL
      IF CHR(0) = SUBSTR(cString, 1, 1) .AND. LEN(SUBSTR(cString,1,1)) = 1
         cString = ""
      ENDIF
   ENDIF

   RETURN (cString)


/***
*  GetColumn( <cFieldBuffer>, @<nOffset> ) --> aColumn
*
*  Get a COLUMN element from FIELDS_BUFF string using nOffset to point to
*  the current FIELDS_OFFSET block.
*
*  Notes:
*     1. The Header or Contents expressions are empty if:
*        a. Passed pointer is equal to 65535
*        b. Character following character pointed to by pointer is CHR(0)
*
*/

////////////////
////////////////
//
// Purpose:
//    Internal function
//
// Description:
//    Internal function
//
// Category:
//    Internal Function
//
// Revisions:
//    01/26/94 Added comment blocks
//
////////////////
////////////////

STATIC FUNCTION GetColumn( cFieldsBuffer, nOffset )
   LOCAL nPointer := 0, nNumber := 0, aColumn[ RC_COUNT ], cType

   // Column width
   aColumn[ RC_WIDTH ] := BIN2W(SUBSTR(cFieldsBuffer, nOffset + FIELD_WIDTH_OFFSET, 2))

   // Total column?
   aColumn[ RC_TOTAL ] := IF(SUBSTR(cFieldsBuffer, nOffset + FIELD_TOTALS_OFFSET, 1) $ "Yy", .T., .F.)

   // Decimals width
   aColumn[ RC_DECIMALS ] := BIN2W(SUBSTR(cFieldsBuffer, nOffset + FIELD_DECIMALS_OFFSET, 2))

   // Offset (relative to FIELDS_OFFSET), 'point' to
   // expression area via array OFFSETS[]

   // Content expression
   nPointer = BIN2W(SUBSTR(cFieldsBuffer, nOffset +;
               FIELD_CONTENT_EXPR_OFFSET, 2))
   aColumn[ RC_TEXT ] := GetExpr( nPointer )
   aColumn[ RC_EXP ] := &( "{ || " + GetExpr( nPointer ) + "}" )

   // Header expression
   nPointer = BIN2W(SUBSTR(cFieldsBuffer, nOffset +;
               FIELD_HEADER_EXPR_OFFSET, 2))

   aColumn[ RC_HEADER ] := ListAsArray(GetExpr( nPointer ), ";")

   // Column picture
   // Setup picture only if a database file is open
   IF USED()
      cType := VALTYPE( EVAL(aColumn[ RC_EXP ]) )
      aColumn[ RC_TYPE ] := cType
      DO CASE
      CASE cType = "C" .OR. cType = "M"
         aColumn[ RC_PICT ] := REPLICATE("X", aColumn[ RC_WIDTH ])
      CASE cType = "D"
         aColumn[ RC_PICT ] := "@D"
      CASE cType = "N"
         IF aColumn[ RC_DECIMALS ] != 0
            aColumn[ RC_PICT ] := REPLICATE("9", aColumn[ RC_WIDTH ] - aColumn[ RC_DECIMALS ] -1) + "." + ;
                                  REPLICATE("9", aColumn[ RC_DECIMALS ])
         ELSE
            aColumn[ RC_PICT ] := REPLICATE("9", aColumn[ RC_WIDTH ])
         ENDIF
      CASE cType = "L"
         aColumn[ RC_PICT ] :=  "Y" + REPLICATE("X",aColumn[ RC_WIDTH ]-1)
      ENDCASE
   ENDIF

   // Update offset into ?_buffer
   nOffset += 12

   RETURN ( aColumn )

/***
*  ListAsArray( <cList>, <cDelimiter>, [<nWidth>] ) --> aList
*  Convert a delimited string to an array
*
*/
/*
STATIC FUNCTION ListAsArray( cList, cDelimiter, nWidth )

LOCAL nPos
LOCAL aList := {}						  // Define an empty array
LOCAL lDelimLast := .f.

	IF cDelimiter = NIL
		cDelimiter := ","
	ENDIF

	if nWidth == NIL
		nWidth := Len(cList)
	end

	DO WHILE ( Len(cList) <> 0 )

		nPos := AT(cDelimiter, cList)

		if ( nPos == 0 )
			nPos := Len(cList)
		end

		if ( nPos - 1 > nWidth )
			nPos := nWidth

			while ( nPos > 0 .and. substr(cList, nPos, 1) <> " " )
				nPos --
			end

			if ( nPos == 0 )
				nPos := nWidth
			end
        end

		if ( SUBSTR( cList, nPos, 1 ) == cDelimiter )
			lDelimLast := .t.
			AADD(aList, SUBSTR(cList, 1, nPos - 1)) // Add a new element
		else
            lDelimLast := .f.
			AADD(aList, SUBSTR(cList, 1, nPos)) // Add a new element
		end

		cList := SUBSTR(cList, nPos + 1)

	ENDDO

	if ( lDelimLast )
		AADD(aList, "")
	end

RETURN aList							  // Return the array
*/

////////////////
////////////////
//
// Purpose:
//    Internal function
//
// Description:
//    Internal function
//
// Category:
//    Internal Function
//
// Revisions:
//    01/26/94 Added comment blocks
//
////////////////
////////////////

FUNCTION ListAsArray(cList, cDelim, nWide)
   LOCAL aArray := {}, lNum
   cDelim := IF(VALTYPE(cDelim) == 'C', cDelim, ";")
   cList  := TRIM( STRTRAN(cList, cdelim, chr(13)+chr(10)) )
   nWide  := IF( (lNum := (VALTYPE(nWide) == 'N' .and. nWide > 0 )), ;
                        nWide, LEN(cList) )
   aArray := ARRAY(MLCOUNT(cList,nWide,1,.T.))
   AEVAL(aArray, {|a,e| aArray[e] := IF(lNum, MEMOLINE(cList,nWide,e,1,.T.),;
                                     TRIM(MEMOLINE(cList,nWide,e,1,.T.)))})
RETURN aArray

////////////////
////////////////
//
// Purpose:
//    Internal function
//
// Description:
//    Internal function
//
// Category:
//    Internal Function
//
// Revisions:
//    01/26/94 Added comment blocks
//
////////////////
////////////////

STATIC FUNCTION iFrm(aLoaded)
   STATIC aReport
   IF aLoaded!=NIL
      aReport := aLoaded
   ENDIF
   RETURN aReport
ENDFUNCTION
