/*
    The source code contained within this file is protected under the
    laws of the United States of America and by International Treaty.
    Unless otherwise noted, the source contained herein is:

    Copyright (c)1990-1994 BecknerVision Inc - No Rights Reserved

    Written by John Wm Beckner
    BecknerVision Inc

        ** THIS FILE IS PUBLIC DOMAIN **

        ** Use of this code is at your own risk, absolutely no guarantees
        ** are made to its usefulness  - you must determine for yourself if
        ** this code is of any use to you.  If you do not agree with this, do
        ** not use this code. **
*/

#include "beckner.inc"
#include "inkey.ch"

////////////////
////////////////
//
// Purpose:
//    Fully interactive record adder
//
// Syntax:
//    fFullAdd([<aField>], [<aPrompt>], [<aSayGet>], [<aPicture>],
//             [<aValid>], [<aWhen>], [<lDelete>]) -> NIL
//
// Formal Arguments: (7)
//    Name        Description
//    ___________ ____________
//    aField      Field list [all fields]
//    aPrompt     Prompt names [field names]
//    aSayGet     "S"ay (display-only) or "G"et (modifiable) ["G"]
//    aPicture    Picture strings [none]
//    aValid      Validation strings [none]
//    aWhen       When clause string [none]
//    lDelete     .y. if record can be deleted [.y.]
//
// Returns:
//    NIL
//
// Examples:
//    #include "beckner.inc"
//    FUNCTION TestIt()
//       CreateTest()
//       fFullAdd()           /* lets you add a record */
//       CLOSE Test
//    ENDFUNCTION
//
//    #include "alias.ch"
//    STATIC FUNCTION CreateTest()
//       fCreateDbf("Test/Name/C/32/Balance/N/12/2/Date/D/Over21/L/Notes/M")
//       USE Test NEW EXCLUSIVE
//    ENDFUNCTION
//
// Files:
//    (current work area)
//
// Description:
//    Fully interactive record adder.  You can specify the field list in
//    <aField> or accept the default which is all fields.  Single element
//    arrays are used for S'87 compatibility.  Each subsequent array is on a
//    one-to-one correspondence with <aField>.  The prompts for each field
//    are in <aPrompt> and default to the field names.  <aSayGet> contain 
//    either "S" or "G" for SAY and GET respectively.  SAY fields are only
//    displayed, while GET fields are modifiable.  You can specify PICTURE, 
//    VALID and WHEN clause expressions in <aPicture>, <aValid> and <aWhen>,
//    respectively.  If you want the record to be delete-able, then <lDelete>
//    must be .true. (the default).
//
// See Also:
//    fAddRecord()
//    fDataEdit()
//
// Category:
//    File Function
//
// Revisions:
//    01/26/94 Added comment blocks
//
////////////////
////////////////

FUNCTION fFullAdd(aField, aPrompt, aSayGet, aPicture, aValid, aWhen,;
   lDeleteable)
   LOCAL nCount, lUsePicture := .n., lUseValid := .n., lUseWhen := .n., nCtr
   LOCAL aData := {}, aCoordinates, nRow, nCol, GetList := {}, cScreen, cValid
   LOCAL bKey, aTemp, nTemp, cWhen
   IF aField=NIL
      aField := {}
      aTemp  := dbStruct()
      aEval(aTemp, {|e| aAdd(aField, e[1])})
   ENDIF
   nCount := Len(aField)
   fAddRecord()
   vSave()
   DEFAULT lDeleteable TO .y.,;
         lUseValid TO .y.,;
         lUseWhen TO .y.,;
         lUsePicture TO .y.
   IF aSayGet=NIL
      aSize((aSayGet := {}), nCount)
      aFill(aSayGet, "G")
   ENDIF
   iif(aPrompt=NIL, aPrompt := aClone(aField), NIL)
   aEval(aPrompt, {|e| nTemp := len(e)+2, e := sSetLength(e, 30),;
         e := StrTran(e, " ", ".", nTemp)})
   @ MaxRow(), 0
   @ MaxRow(), 0 SAY 'Record #'+lTrim(Transform(RecNo(), '999,999'))
   IF lDeleteable
      @ MaxRow(), 20 SAY '<ctrl-U> to delete this record and abort'
   ENDIF
   bKey         := SetKey(K_CTRL_U, {||iDelete(lDeleteable)})
   aCoordinates := vWindow(nCount, 76, .y. ,'Adding a New Record')
   nRow         := aCoordinates[1]-1
   nCol         := aCoordinates[2]
   aEval(aPrompt, {|uElement, nElNum| SetPos(nRow+nElNum, nCol),;
         qQout(Trim(uElement))})
   nCol += 31
   aEval(aField, {|e| aAdd(aData, Eval(FieldBlock(e)))})
   FOR nCtr := 1 to nCount
      IF lUsePicture
         IF aPicture[nCtr]=NIL
            @ nRow+nCtr, nCol SAY aData[nCtr]
         ELSE
            @ nRow+nCtr, nCol SAY Transform(aData[nCtr], aPicture[nCtr])
         ENDIF
      ELSE
         IF ValType(aData[nCtr])="C"
            IF len(aData[nCtr])>40
               @ nRow+nCtr, nCol SAY aData[nCtr] picture '@S40'
            ELSE
               @ nRow+nCtr, nCol SAY aData[nCtr]
            ENDIF
         ELSE
            @ nRow+nCtr, nCol SAY aData[nCtr]
         ENDIF
      ENDIF
   NEXT
   FOR nCtr := 1 to nCount
      IF Upper(aSayGet[nCtr])='G'
         cValid := iif(lUseValid, aValid[nCtr], '.y.')
         cWhen  := iif(lUseWhen, aWhen[nCtr], '.y.')
         IF lUsePicture
            IF aPicture[nCtr]!=NIL
               @ nRow+nCtr, nCol GET aData[nCtr];
               picture aPicture[nCtr] VALID &(cValid) WHEN &(cWhen)
            ELSE
               @ nRow+nCtr, nCol GET aData[nCtr] VALID &(cValid) WHEN &(cWhen)
            ENDIF
         ELSE
            @ nRow+nCtr, nCol GET aData[nCtr] VALID &(cValid) WHEN &(cWhen)
         ENDIF
      ENDIF
   NEXT
   READ
   vRestore()
   aEval(aField, {|uElement, nElNum| Eval(fieldblock(uElement),;
   aData[nElNum])})
   UNLOCK
ENDFUNCTION
