/*
 *   Copyright 1992, 1993, 1994 John Melton (G0ORX/N6LYT)
 *              All Rights Reserved
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 1, or (at your option)
 *   any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

/*
	upload.c

	Pacsat Upload for Linux and X-Windows

	This program has been run using the 1.0 version of the
	Linux kernel with the patches from Alan Cox to provide AX.25
	encapsulation in the SLIP protocol(verion 0.12).

	The TNC must be setup for KISS protocol.

	John Melton
	G0ORX, N6LYT

	4 Charlwoods Close
	Copthorne
	West Sussex
	RH10 3QZ
	England

	INTERNET:	g0orx@amsat.org
			n6lyt@amsat.org
			john@images.demon.co.uk
			J.D.Melton@slh0613.icl.wins.co.uk

	History:

	0.1	initial version.
	0.2	added output ready notify code.
	0.3	fixed error conditions - back to IDLE.
	0.4	fixed restart upload of partial file.

*/

#define VERSION_STRING "upload version 0.4 (g0orx/n6lyt)"

#include <xview/xview.h>
#include <xview/panel.h>
#include <xview/font.h>
#include <xview/notice.h>

#include <stdio.h>
#include <unistd.h>
#include <sys/types.h>
#include <fcntl.h>
#include <malloc.h>
#include <dirent.h>
#include <sys/socket.h>
#include <linux/ax25.h>
#include "header.h"
#include "ftl0.h"

Frame frame;
Panel panel;
Panel_item panel_list_item;
Panel_item status;
Xv_Font font;

int nList;

char satellite[16];
char myCall[16];

unsigned long selectedFileId;

#define IDLE			0
#define LOGGING_IN		1
#define REQUESTING_UPLOAD	2
#define UPLOADING		3
#define SENT_END_DATA		4

int state = IDLE;

int fd;
int f;
int fileId;
char fileName[128];
int offset;
int fileLength;
UPLOAD_REQUEST *ul_request;
UPLOAD_RESPONSE *ul_response;
unsigned long uploadFileId;
unsigned char buffer[1024];
int maxFrame;
	
#define MAXHEADER 1024
unsigned char header[MAXHEADER];
int nHeader;

char msg[256];
extern void exit( );
void panel_quit( void );
void button_update( void );
void LoadFile( char * fileName );
void button_upload( void );
void panel_select( Panel_item item, char * string, caddr_t client_data,
			Panel_list_op op, Event *event );
int convert_call(char *name,struct full_sockaddr_ax25 *sax);

Notify_value ReceivedFrame( Notify_client client, int fd );
Notify_value SendFrame( Notify_client client, int fd );
void button_message( void );

main( int argc, char ** argv )
{
	char title[80];

	xv_init( XV_INIT_ARGS, argc, argv, NULL );

	if( getenv( "SATELLITE" ) == (char *)0 )
	{
		printf( "SATELLITE environment variable not set\n" );
		exit( 1 );
	}
	strcpy( satellite, getenv( "SATELLITE" ) );
	strcat( satellite, "-12" );

	if( getenv( "MYCALL" ) == (char *)0 )
	{
		printf( "MYCALL environment variable not set\n" );
		exit( 1 );
	}
	strcpy( myCall, getenv( "MYCALL" ) );
	
	if( getenv( "MAXFRAMEDATA" ) == (char *)0 )
		maxFrame = 100;
	else
		maxFrame = atoi( getenv( "MAXFRAMEDATA" ) );
	if( maxFrame <= 0 )
		maxFrame = 100;

	sprintf( title, "%s: %s" , VERSION_STRING, satellite );

	frame = (Frame)xv_create( (Frame)NULL, FRAME,
				FRAME_LABEL, title,
				NULL );

	font = (Xv_Font)xv_find( frame, FONT,
				FONT_FAMILY, FONT_FAMILY_DEFAULT_FIXEDWIDTH,
				FONT_SIZE, 12,
				NULL );

	panel = (Panel)xv_create( frame, PANEL,
				XV_FONT, font,
				PANEL_LAYOUT, PANEL_HORIZONTAL,
				NULL );

	(void)xv_create( panel, PANEL_BUTTON,
		PANEL_LABEL_STRING, "Quit",
		PANEL_NOTIFY_PROC, panel_quit,
		NULL );

	(void)xv_create( panel, PANEL_BUTTON,
		PANEL_FONT, FONT_FAMILY_DEFAULT_FIXEDWIDTH,
		PANEL_LABEL_STRING, "Update",
		PANEL_NOTIFY_PROC, button_update,
		NULL );

	(void)xv_create( panel, PANEL_BUTTON,
		PANEL_FONT, FONT_FAMILY_DEFAULT_FIXEDWIDTH,
		PANEL_LABEL_STRING, "Message",
		PANEL_NOTIFY_PROC, button_message,
		NULL );

	(void)xv_create( panel, PANEL_BUTTON,
		PANEL_FONT, FONT_FAMILY_DEFAULT_FIXEDWIDTH,
		PANEL_LABEL_STRING, "Upload",
		PANEL_NOTIFY_PROC, button_upload,
		NULL );

	status = (Panel_item)xv_create( panel, PANEL_MESSAGE, 
				XV_WIDTH, 100,
				XV_FONT, font,
				PANEL_LABEL_STRING,
				"Status: Idle  ",
				NULL );

	xv_set( panel, PANEL_LAYOUT, PANEL_VERTICAL, NULL );

	(void)xv_create( panel, PANEL_MESSAGE, 
				XV_WIDTH, 600,
				XV_FONT, font,
				PANEL_LABEL_STRING,
				"  File               Id To         Size Title",
				NULL );

	panel_list_item = (Panel_item)xv_create( panel, PANEL_LIST,
				XV_FONT, font,
				PANEL_LIST_DISPLAY_ROWS, 6,
				PANEL_LIST_WIDTH, 600,
				PANEL_READ_ONLY, 1,
				PANEL_CHOOSE_NONE, TRUE,
				PANEL_CHOOSE_ONE, TRUE,
				PANEL_NOTIFY_PROC, panel_select,
				NULL );

	window_fit_height( panel );
	window_fit_height( frame );
	button_update( );

	xv_main_loop( frame );
}

void panel_quit( )
{
	xv_destroy_safe( frame );
}

void panel_select( Panel_item item, char * string, caddr_t client_data,
			Panel_list_op op, Event *event )
{

	switch( op )
	{
	case PANEL_LIST_OP_SELECT:
		selectedFileId = (unsigned long)client_data;
		break;
	case PANEL_LIST_OP_DESELECT:
		selectedFileId = 0;
		break;
	}
}

void button_update ( )
{
	DIR		*pDir;
	struct dirent	*pDirent;
	int i;

	xv_set( frame, FRAME_BUSY, 1, NULL );
	xv_set( panel_list_item, XV_SHOW, 0, NULL );
	xv_set( panel_list_item, PANEL_CHOOSE_NONE, TRUE, NULL );
	xv_set( panel_list_item, PANEL_CHOOSE_ONE, FALSE, NULL );

	/* flush out the current list */
	for( ; nList>0; nList-- )
		xv_set( panel_list_item, PANEL_LIST_DELETE, 0, NULL );

	/* walk through the directory of files */
	pDir = opendir( "." );
	while( (pDirent = readdir( pDir )) != NULL )
	{
		/* see if it is a .dl file */
		i = 0;
		while( pDirent->d_name[i] && (pDirent->d_name[i] != '.') )
			i++;
		if( strcmp( &pDirent->d_name[i], ".upl" ) == 0 )
			LoadFile( pDirent->d_name );
	}
	closedir( pDir );

	xv_set( panel_list_item, PANEL_LIST_SORT, PANEL_REVERSE, NULL );
	xv_set( panel_list_item, XV_SHOW, 1, NULL );
	xv_set( frame, FRAME_BUSY, 0, NULL );
	if( nList )
	{
		xv_set( panel_list_item, PANEL_CHOOSE_NONE, FALSE, NULL );
		xv_set( panel_list_item, PANEL_CHOOSE_ONE, TRUE, NULL );
		xv_set( panel_list_item, PANEL_LIST_SELECT, 0, TRUE, NULL );
		selectedFileId = xv_get( panel_list_item, PANEL_LIST_CLIENT_DATA, 0, NULL );
	}

}

void LoadFile( char * fileName )
{
	int hFile;
	int nBytes;
	char * pBuffer;
	static struct tm * GMT;
	char szTemp[128];
	int headerSize;
	HEADER *pHeader;
	int j;
	char *p;
	int fileNumber;

	sscanf( fileName, "%x", &fileNumber );

	/* open the file */
	hFile = open( fileName, O_RDONLY );
	if( hFile == -1 )
	{
		perror( fileName );
		return;
	}

	/* extracting the header */
	pBuffer = malloc( 1024 );
	nBytes = read( hFile, pBuffer, 1024 );
	close( hFile );
	pHeader = ExtractHeader( pBuffer, nBytes, &headerSize );

	if( pHeader == NULL )
	{
		printf( "invalid header entry for file: %s\n", fileName );
	}
	else
	{
		/* truncate the source and destination */
		j = 0;
		while( isalnum( pHeader->source[j] ) )
			j++;
      		pHeader->source[j] = '\0';
		j = 0;
		while( isalnum( pHeader->destination[j] ) )
			j++;
      		pHeader->destination[j] = '\0';

		if( strlen( pHeader->title ) == 0 )
			sprintf( szTemp, "%12s %8x %-8s %6d %-30s",
				fileName,
				pHeader->fileId,
				pHeader->destination,
				pHeader->fileSize,
				pHeader->fileName );
		else
			sprintf( szTemp, "%12s %8x %-8s %6d %-30s",
				fileName,
				pHeader->fileId,
				pHeader->destination,
				pHeader->fileSize,
				pHeader->title );

		p = pHeader->source;
		while( *p )
		{
			if( islower( *p ) )
				*p = toupper( *p );
			p++;
		}

		p = pHeader->destination;
		while( *p )
		{
			if( islower( *p ) )
				*p = toupper( *p );
			p++;
		}

		xv_set( panel_list_item, PANEL_LIST_INSERT, nList, NULL );
		xv_set( panel_list_item, PANEL_LIST_STRING, nList, szTemp, NULL );
		xv_set( panel_list_item, PANEL_LIST_CLIENT_DATA, nList, fileNumber, NULL );
		nList++;

		free( pHeader );
	}
	free( pBuffer );
}

void update_status( char * msg )
{
fprintf( stderr, "update_status: %s\n", msg );
	xv_set( status, PANEL_LABEL_STRING, msg, NULL );
fprintf( stderr, "update_status: exit\n" );
}

void button_upload( )
{
	HEADER *pHeader;
	int one;
	int addr_len;
	struct full_sockaddr_ax25 addr;
	int bytes;
	
	sprintf( fileName, "%x.upl", selectedFileId );
	sprintf( msg, "Status: Uploading: %s", fileName );
	update_status( msg );
	f = open( fileName, O_RDWR );
	if( f < 0 )
	{
		update_status( "open failed" );
		perror( fileName );
		return;
	}


	/* get the file length */
	fileLength = lseek( f, 0, 2 );

	/* extracting the header */
	lseek( f, 0 , 0 );
	bytes = read( f, buffer, 1024 );
	lseek( f, 0 , 0 );
	pHeader = ExtractHeader( buffer, bytes, &bytes );
	fileId = pHeader->fileId;
	free( pHeader );

	fd = socket( AF_AX25, SOCK_SEQPACKET, PF_AX25 );
	if( fd < 0 )
	{
		update_status( "socket failed" );
		perror( "socket" );
		close( f );
		return;
	}

	one = 1;
	if( setsockopt( fd, SOL_AX25, AX25_WINDOW, &one, sizeof( one ) ) == -1 )
	{
		update_status( "setsockopt (AX25_WINDOW) failed" );
		perror( "AX25_WINDOW" );
		close( fd );
		close( f );
		return;
	}

#ifdef DEBUG
	one = 1;
	if( setsockopt( fd, SOL_SOCKET, SO_DEBUG, &one, sizeof( one ) ) == -1 )
	{
		update_status( "setsockopt (SO_DEBUG) failed" );
		perror( "SO_DEBUG" );
		close( fd );
		close( f );
		return;
	}
#endif

	addr_len = convert_call( myCall, &addr );

	if( bind( fd, (struct sockaddr *)&addr, addr_len ) == -1 )
	{
		update_status( "bind failed" );
		perror( "bind" );
		close( fd );
		close( f );
		return;
	}

	addr_len = convert_call( satellite, &addr );

	update_status( "Status: Connecting..." );

	if( connect( fd, (struct sockaddr *)&addr, addr_len ) )
	{
		update_status( "connect failed" );
		perror( "connect" );
		close( fd );
		close( f );
		return;
	}

	update_status( "Status: Connected" );

	/* setup state */
	state = LOGGING_IN;

	/* now setup the input notify function */
	update_status( "Status: Enable input func" );
	notify_set_input_func( (Notify_client)panel, ReceivedFrame, fd );

	/* now exit and await the login message */
}

void error( char * msg )
{
	update_status( msg );
	notify_set_input_func( (Notify_client)panel, NOTIFY_FUNC_NULL, fd );
	notify_set_output_func( (Notify_client)panel, NOTIFY_FUNC_NULL, fd );
	close( fd );
	close( f );
	status = IDLE;
}

Notify_value ReceivedFrame( Notify_client client, int fd )
{
	int bytes;
	int type;
	int length;

	bytes = recv( fd, buffer, 512, 0 );
	if( bytes < 0 )
	{
		sprintf( msg, "recv failed: %d", bytes );
		error( msg );
		return NOTIFY_DONE;
	}

	length = buffer[0] + ((buffer[1] << 3) & 0xFF00);
	type = buffer[1] & 0x1F;

	switch( state )
	{
		case LOGGING_IN:
			if( type != LOGIN_RESP )
			{
				sprintf( msg, "Error: Expected LOGIN_RESP got %d", type );
				error( msg );
				return NOTIFY_DONE;
			}
			update_status( "Status: Logged in" );

			/* request an upload */
			sprintf( msg, "Status: Requesting upload: file:%x length:%d",
					fileId, fileLength );
			update_status( msg );
			buffer[0] = sizeof( UPLOAD_REQUEST );
			buffer[1] = UPLOAD_CMD;
			ul_request = (UPLOAD_REQUEST *)&buffer[2];
			ul_request->file = fileId;
			ul_request->length = fileLength;
	
			if( send( fd, (char *)buffer, sizeof( UPLOAD_REQUEST )+2, 0 )
				!= (sizeof( UPLOAD_REQUEST )+2) )
			{
				perror( "send" );
				error( "send failed" );
				return NOTIFY_DONE;
			}
			state = REQUESTING_UPLOAD;
			break;

		case REQUESTING_UPLOAD:
			switch( type )
			{
				case UL_GO_RESP:
					break;
				case UL_ERROR_RESP:
					sprintf( msg, "Status: UL_ERROR_RESP: %d", buffer[2] );
					error( msg );
					return NOTIFY_DONE;
					break;
				default:
					sprintf( msg, "Status: Unexpected GO_RESP: %d", type );
					error( msg );
					return NOTIFY_DONE;
					break;
			}

			ul_response = (UPLOAD_RESPONSE *)&buffer[2];
			offset = ul_response->offset;
			uploadFileId = ul_response->file;
			sprintf( msg, "Status: GO_RESP: file:%x offset:%d",
				uploadFileId, offset );
			update_status( msg );

			/* update the file id in the header */
			/* incase it does not all make it */
			lseek( f, 5, 0 );
			write( f, (char *)&uploadFileId, 4 );

			/* now seek to the start of the data and start uploading */
			lseek( f, offset, 0 );
			state = UPLOADING;
			update_status( "Enable output func" );
			notify_set_output_func( (Notify_client)panel, SendFrame, fd );
			break;

		case SENT_END_DATA:
			switch( type )
			{
				case UL_ACK_RESP:
					sprintf( msg, "Status: File %s uploaded as message %x", fileName, uploadFileId );
					update_status( msg );
					unlink( fileName );
					break;
				case UL_NAK_RESP:
					update_status( "Status: received UL_NAK_RESP" );
					break;
				default:
					sprintf( msg, "Status: unexpected response:%d", type );
					update_status( msg );
					break;
			}
			close( fd );
			notify_set_input_func( (Notify_client)panel, NOTIFY_FUNC_NULL, fd );
			state = IDLE;
			break;
	}
	return NOTIFY_DONE;
}

Notify_value SendFrame( Notify_client client, int fd )
{
	int bytes;

	if( state != UPLOADING )
	{
		/* just ignore - it should go away */
		update_status( "SendFrame - not UPLOADING" );
		return NOTIFY_DONE;
	}

	if( (bytes = read( f, &buffer[2], maxFrame )) > 0 )
	{
		sprintf( msg, "Status: sending data:%d offset:%d", bytes, offset );
		update_status( msg );
		buffer[0] = bytes;
		buffer[1] = DATA;
		/* always send fileId of 0 in header */
		if( offset == 0 )
		{
			buffer[7] = 0;
			buffer[8] = 0;
			buffer[9] = 0;
			buffer[10] = 0;
		}
		if( send( fd, buffer, bytes+2, 0 ) != (bytes+2)  )
		{
			perror( "send" );
			error( "send" );
			return NOTIFY_DONE;
		}
		offset += bytes;
		return NOTIFY_DONE;
	}

	if( bytes < 0 )
	{
		perror( "read" );
		error( "send" );
		return NOTIFY_DONE;
	}

	close( f );

	/* send a data end packet */
	update_status( "Status: sending END_DATA" );
	buffer[0] = 0;
	buffer[1] = DATA_END;
	if( send( fd, buffer, 2, 0 ) != 2  )
	{
		perror( "send" );
		error( "send" );
		return NOTIFY_DONE;
	}

	state = SENT_END_DATA;

	/* turn off the send notify */
	notify_set_output_func( (Notify_client)panel, NOTIFY_FUNC_NULL, fd );
	return NOTIFY_DONE;
}


/*
 *	Library routine for callsign conversion.
 *
 */
 
int convert_call_entry(char *name,unsigned char *buf)
{
	int ct=0;
	int ssid=0;
	unsigned char *p=name;
	while(ct<6)
	{
		if(islower(*p))
			*p=toupper(*p);
		if(!isalnum(*p))
		{
			printf("Invalid symbol in callsign.\n");
			return -1;
		}
		
		buf[ct]=(*p<<1);
		p++;
		ct++;
		if(*p=='-' || *p==0)
			break;
	}
	while(ct<6)
	{
		buf[ct]=' '<<1;
		ct++;
	}
	if(*p==0)
		ssid=0;
	else
	{
		p++;
		if(sscanf(p,"%d",&ssid)!=1 || ssid<0 || ssid>15)
		{
			printf("SSID must follow '-' and be numeric in the range 0-15.\n");
			return -1;
		}
	}
	buf[6]=((ssid+'0')<<1)&0x1E;
	return 0;
}

int convert_call(char *call, struct full_sockaddr_ax25 *sax)
{

	int len=0;
	unsigned char *bp,*np;
	char *addrp;
	int n=0;
	unsigned char *tmp=strdup(call);
	
	if(tmp==NULL)
		return -1;
		
	bp=tmp;
	
	addrp=sax->fsa_ax25.sax25_call.ax25_call;
	do
	{
		/* Fetch one callsign token */
		while(*bp&&isspace(*bp))
			bp++;
		np=bp;
		while(*np&&!isspace(*np))
		np++;
		if(*np)
			*np++=0;
	
		/* Check for the optional 'via' syntax */
		if(n==1 && (strcasecmp(bp,"V")==0 || strcasecmp(bp,"VIA")==0))
		{
			bp=np;
			continue;
		}
		
		/* Process the token */

		if(convert_call_entry(bp,addrp)==-1)
		{
			free(tmp);
			return -1;
		}
			
		/* Move along */
		
		bp=np;
		n++;
		if(n==1)
		{
			addrp=sax->fsa_digipeater[0].ax25_call;	/* First digipeater address */
			len+=sizeof(struct sockaddr_ax25);
		}
		else
		{
			addrp+=sizeof(ax25_address);
			len+=sizeof(ax25_address);
		}
	}
	while(n<AX25_MAX_DIGIS && *bp);
	free(tmp);
	/* Tidy up */
	sax->fsa_ax25.sax25_ndigis=n-1;
	sax->fsa_ax25.sax25_family=AF_AX25;	
	return len;
}

int convert_call_arglist(char *call[], struct full_sockaddr_ax25 *sax)
{

	int len=0;
	unsigned char *bp,*np;
	char *addrp;
	int n=0;
	int argp=0;

	addrp=sax->fsa_ax25.sax25_call.ax25_call;
	do
	{
		/* Fetch one callsign token */
		bp=call[argp++];
		if(bp==NULL)
			break;
	
		/* Check for the optional 'via' syntax */
		if(n==1 && (strcasecmp(bp,"V")==0 || strcasecmp(bp,"VIA")==0))
			continue;
		
		/* Process the token */

		if(convert_call_entry(bp,addrp)==-1)
			return -1;
			
		n++;
		if(n==1)
		{
			addrp=sax->fsa_digipeater[0].ax25_call;	/* First digipeater address */
			len+=sizeof(struct sockaddr_ax25);
		}
		else
		{
			addrp+=sizeof(ax25_address);
			len+=sizeof(ax25_address);
		}
	}
	while(n<AX25_MAX_DIGIS && call[argp]);
	/* Tidy up */
	sax->fsa_ax25.sax25_ndigis=n-1;
	sax->fsa_ax25.sax25_family=AF_AX25;	
	return len;
}

void signal_child( )
{
	int pid, pstatus;

	pid = wait( &pstatus );
}

void button_message( )
{
	int pid;

	signal( SIGCHLD, signal_child );

	if( (pid=fork()) == 0 )
	{
		/* the child process */
		execlp( "message", "message", NULL );
	}
	else if( pid>0 )
	{
		/* success */
	}
	else
		(void)notice_prompt( panel, NULL,
			NOTICE_MESSAGE_STRINGS, "cannot fork message",
			NULL,
			NOTICE_BUTTON, "OK", 100,
			NULL );
}
