/* nmpipe.c - A Named Pipes under UnixWare demonstration program.  To run the 
    demo you need a client and server shell window. Run: nmpipe -c in the 
    client window and nmpipe -s in the server window. They'll rendezvous on 
    the named pipe and exchange 100 message-ack sequences and you can watch 
    what's going on in the printouts.
*/
#include <sys/types.h>
#include <sys/stat.h>
#include <signal.h>
#include <fcntl.h>
#include <stropts.h>

#define FIFO_MODE (S_IRUSR|S_IWUSR|S_IRGRP|S_IWGRP|S_IROTH|S_IWOTH)
#define PIPE_NAME "/tmp/serv1"

static void sig_pipe(int);

/* main - catch pipe signals; determine which type of process we are and then 
    run either client or server. */
int main( int argc, char *argv[] )
{
if( signal( SIGPIPE, sig_pipe ) == SIG_ERR )
    {
    printf( "signal error\n" );
    exit( 1 );
    }
if( strcmp( argv[1], "-c" ) == 0 )
    DoClient();
else
    DoServer();
return( 0 );
}
/* sig_pipe - react to any signal we might get from the pipe. */
static void sig_pipe( int signo )
{
printf( "SIGPIPE!\n" );
exit( 1 );
}
/* DoClient - The client process, opens the name, makes sure that its a stream,
   then sends 100 messages over it to the server, waiting for an ack each time.
   Note how generic this function is, without isastream() you'd never know 
   we were talking over a named pipe.
*/
int DoClient()
{
int fd;
int i;
char buffer[256];

if(( fd = open( PIPE_NAME, O_RDWR )) < 0 )
    {
    printf( "Client couldn't open %s\n", PIPE_NAME );
    return( -1 );
    }
if( isastream( fd ) == 0 )
    {
    printf( "Client %s isn't a stream\n", PIPE_NAME );
    return( -2 );
    }
for( i = 0; i < 100; i++ )
    {
    sprintf( buffer, "Client message %d\n", i );
    if( write( fd, buffer, strlen( buffer ) + 1 ) != strlen( buffer ) + 1 )
        {
        printf( "write error!\n" );
        goto writeerror;
        }
    else
        printf( "Client sent msg: %s", buffer );
    if( read( fd, buffer, sizeof( buffer )) > 0 )
        printf( "received ack %s\n", buffer );
    }
writeerror:
close( fd );
return( 0 );
}
/* DoServer - Create named pipe, then exchange 100 messages with the client. */
int DoServer()
{
int i, ret;
int listenfd;
int readfd;
char buffer[256];
/* Create anonymous stream pipe and named FIFO, and get back handle to pipe. */
if(( listenfd = attach_to_FIFO( PIPE_NAME )) < 0 )
    {
    printf( "Server serv_listen returned %d\n", listenfd );
    return( -1 );
    }
/* Listen to anonymous pipe to get our read/write handle to the named pipe */
if(( readfd = receive_handle( listenfd, NULL )) < 0 )
    {
    printf( "Server serv_accept returned %d\n", readfd );
    return( -2 );
    }
/* Now use the read/write handle to talk to the client. */
printf( "Reading\n" );
for( i = 0; i < 100; i++ )
    {
    if(( ret = read( readfd, buffer, sizeof( buffer ))) > 0 )
        {
        printf( "Server received message %s\n", buffer );
        if( write( readfd, buffer, strlen( buffer ) + 1 ) != 
                                                        strlen( buffer ) + 1 )
            printf( "write error on ack!\n" );
        }
    else
        {
        if( ret == 0 )
            {
            printf( "PIPE CLOSED!\n" );
            break;
            }
        else    
            {
            printf( "read returned %d\n", ret );
            break;
            }
        }
    }
return( 0 );
}
/* receive_handle - Receive read/write named pipe handle via the ioctl call */
int receive_handle( int listenfd, uid_t *uidptr )
{
struct strrecvfd recvfd;
    printf( "listening\n" );
    if( ioctl( listenfd, I_RECVFD, &recvfd ) < 0 )
        return( -1 );
    if( uidptr != NULL )
        *uidptr = recvfd.uid;
    return( recvfd.fd );
}
/* attach_to_FIFO - create FIFO and anonymous pipe, and attach FIFO to pipe. */
int attach_to_FIFO( const char *name )
{
int tempfd, fd[2], len;
    unlink( name );
    printf( "creat\n" );
    if(( tempfd = creat( name, FIFO_MODE )) < 0 )
        return( -1 );
    if( close( tempfd ) < 0 )
        return( -2 );
    printf( "Created anonymous pipe.\n" );
    if( pipe( fd ) < 0 )
        return( -3 );
    if( ioctl( fd[1], I_PUSH, "connld" ) < 0 )
        return( -4 );
    printf( "Attached pipe to FIFO.\n" );
    if( fattach( fd[1], name ) < 0 )
        return( -5 );
    return( fd[0] );
}
